/******************************************************************************


-----------------                                     -----------------
|         MISO  |->P18, 12 -----------------> P18, 12 | MISO          |
|         SSN   |->P18, 14-----------------> P18, 14  | SSN           |
|  Rf05EB       |                                     |  Rf05EB       |
|         SCLK  |->P18, 16 -----------------> P18, 16 | SCLK          |
|         MOSI  |->P18, 18 -----------------> P18, 18 | MOSI          |
|               |                                     |               |
|          GND  |->P18, 20 -----------------> P18, 20 | GND           |
-----------------                                     -----------------

======================================
||                                  ||
|| Start up slave before master!!   ||
||                                  ||
======================================

******************************************************************************/

/*==== INCLUDES ==============================================================*/
#include "spi_rf_tst.h"
#include "spi_rf_tst_menu.h"

/***********************************************************************************
* CONSTANTS
*/


/***********************************************************************************
* LOCAL VARIABLES
*/



/***********************************************************************************
* LOCAL FUNCTIONS
*/
static void appTimerISR(void);



/*==== DECLARATION CONTROL ===================================================*/
void printStats(void);


/*==== GLOBAL VARS============================================================*/

static uint8 appMode;       // Spi operating appMode, either master or slave
static uint8 appTestNo;      // Which test to run


/***********************************************************************************
* @fn          appTimerISR
*
* @brief       32KHz timer interrupt service routine. Signals PER test transmitter
*              application to transmit a packet by setting application state.
*
* @param       none
*
* @return      none
*/
static void appTimerISR(void)
{
    appState = TRANSMIT_PACKET;
}


/***********************************************************************************
* @fn          appStartStop
*
* @brief       Joystick up interrupt service routine. Start or stop 32KHz timer,
*              and thereby start or stop PER test packet transmission.
*
* @param       none
*
* @return      none
*/
void appStartStop(void)
{
    // toggle value
    appStarted ^= 1;

    if(appStarted) {
        halTimer32kIntEnable();
    }
    else {
        halTimer32kIntDisable();
    }
}


/***********************************************************************************
* @fn          appConfigTimer
*
* @brief       Configure timer interrupts for application. Uses 32KHz timer
*
* @param       uint16 rate - Frequency of timer interrupt. This value must be
*              between 1 and 32768 Hz
*
* @return      none
*/
static void appConfigTimer(uint16 rate)
{
    halTimer32kInit(TIMER_32K_CLK_FREQ/rate);
    halTimer32kIntConnect(&appTimerISR);
}

/***********************************************************************************
* @fn          appSlave
*
* @brief       Application code for the slave mode. Puts MCU in endless loop
*
* @param       
*
* @return      none
*/
static void appReceiverSlave()
{
  
    // Initialize BasicRF
    basicRfConfig.myAddr = RX_ADDR;
    if(basicRfInit(&basicRfConfig)==FAILED) {
      HAL_ASSERT(FALSE);
    }
    
    // Set selected test
    appTestNo = appSelectTest(); 
    appSpiSetTest(appMode, appTestNo);
}


/***********************************************************************************
* @fn          appMaster
*
* @brief       Application code for the Master mode. Puts MCU in endless
*              loop
*
* @param       basicRfConfig - file scope variable. Basic RF configuration data
*              txPacket - file scope variable of type perTestPacket_t
*              appState - file scope variable. Holds application state
*              appStarted - file scope variable. Controls start and stop of
*                           transmission
*
* @return      none
*/
static void appTransmitterMaster()
{
 
    uint8 appTxPower;
    uint8 n;

    // Initialize BasicRF
    basicRfConfig.myAddr = TX_ADDR;
    if(basicRfInit(&basicRfConfig)==FAILED) {
      HAL_ASSERT(FALSE);
    }

    // Set TX output power
    appTxPower = appSelectOutputPower();
    halRfSetTxPower(appTxPower);

    // Set burst size
    burstSize = appSelectBurstSize();

    // Basic RF puts on receiver before transmission of packet, and turns off
    // after packet is sent
    basicRfReceiveOff();

    // Config timer and IO
    n = appSelectRate();
    appConfigTimer(n);
    // halJoystickInit();

    // Initalise packet payload
    txPacket.seqNumber = 0;
    for(n = 0; n < sizeof(txPacket.padding); n++) {
        txPacket.padding[n] = n;
    }
 
    // Set selected test
    appTestNo = appSelectTest(); 
    appSpiSetTest(appMode, appTestNo);
     
}


void appSpiRunTst(uint8 testSet, uint8 *pTxData, uint8 *pRxData)
{
    uint8 j = 0;
    uint8 error = 0;
    volatile uint16 wrong;
    uint16 counter;
    
  
    if (testSet == S_M_INTERRUPT) 
    {
      // Interrupt test cases: Set up interrupt
      pTxBuffer = pTxData;
      pRxBuffer = pRxData;
      // Clear pending interrupt
      TCON &= ~0x80; // TCON.URX1IF = 0;
      // Enable interrupt
      URX1IE = 1;
    }
    
    for (counter = 0; counter < burstSize; counter++) 
    
    {
          switch (testSet) {
          case M_S_ACTIVE:
            //spi_m_s_active_master(txBufferMaster);
            for (j = 0; j <= sizeof(txPacket.padding); j++) {
              SSN = LOW;
              U1DBUF = pTxData[j];
              while (U1ACTIVE); 
              SSN = HIGH;
            }
            break;
          case M_S_TXBYTE_RXBYTE:
            for (j = 0; j <= sizeof(txPacket.padding); j++) {
              SSN = LOW;
              U1DBUF = pTxData[j];
              while (!U1TX_BYTE);
              U1TX_BYTE = 0;
              SSN = HIGH;
            }              
            break;
          case M_S_DMA:
            DMAARM = DMAARM0;   // Arm DMA channel 0
            SSN = LOW;
            DMAREQ = 0x01;
            while (!dma0Done);
            while (U1ACTIVE); 
            SSN = HIGH;
            dma0Done = FALSE;
            break;            
          case M_S_INTERRUPT:
            for (j = 0; j <= sizeof(txPacket.padding); j++) {
              halMcuWaitUs(10); // Wait here because IRQ is not fast enough in current implementation
              SSN = LOW;
              U1DBUF = pTxData[j];
              while (!U1TX_BYTE);
              U1TX_BYTE = 0;
              SSN = HIGH;
            }
            break;
          case S_M_ACTIVE:
            for (j = 0; j <= sizeof(txPacket.padding); j++) {
              halMcuWaitUs(1); // Wait to allow slave to keep up
              SSN = LOW;
              U1DBUF = DUMMY_BYTE;
              while (U1ACTIVE);
              pRxData[j] = U1DBUF;
              SSN = HIGH;
            }
            break;
          case S_M_TXBYTE_RXBYTE:
            for (j = 0; j <= sizeof(txPacket.padding); j++) {
              SSN = LOW;
              U1DBUF = DUMMY_BYTE;
              while (!U1TX_BYTE);
              pRxData[j] = U1DBUF;
              SSN = HIGH;
              U1TX_BYTE = 0;
            }
            break;
          case S_M_DMA:
            DMAARM = DMAARM1 | DMAARM0;   // Arm DMA channels 0 and 1
            SSN = LOW;
            DMAREQ = 0x01;
            while (!dma0Done || ! dma1Done);
            while (U1ACTIVE); 
            SSN = HIGH;
            dma0Done = FALSE;
            dma1Done = FALSE;
            break;            
          case S_M_INTERRUPT:
            // Prepare first byte
            SSN = LOW;
            pRxBuffer[0] = U1DBUF;
            U1DBUF = pTxBuffer[0];
            while (!dataReceived);
            SSN = HIGH;
            dataReceived = FALSE;
            break;
          }
          halMcuWaitUs(50); // Delay to allow for error checking on slave
          
          if (testSet >= S_M_ACTIVE) {
            // Error checking
            error = 0;
            for (j = 0; j <= sizeof(txPacket.padding); j++) {
              if (pRxData[j] != pTxData[j]){
                error++;
              }
            }
            if(error != 0) {
              wrong++;
            }
            
            if (error && wrong == 1) {

            }
            for (j = 0; j <= sizeof(txPacket.padding); j++) {
              pRxData[j] = 0xFF;        
            }            
          }          
        }
        if (testSet >= S_M_ACTIVE) {

          counter = 0;
          wrong = 0;
        }
}// appSpiRunTst



/*******************************************************************************
* @fn  main
*
* @brief
*      Main function. 
*
* Parameters:
*
* @param  void
*
* @return void
*
*******************************************************************************/
void main(void)
{ 
    
  // Choose the crystal oscillator as the system clock and set 
  // timer tick speed = system clock speed = fRef (fxosc)
  //int i,j,length;
  halBoardInit();

  halUartInit(HAL_UART_BAUDRATE_115200, 0);
  halJoystickInit();
  halJoystickIntConnect(HAL_JOYSTICK_EVT_PUSHED, printStats);
  halIntOn();
  

  // Drive SSN output as high output in case peer is configured as a slave
  SSN = HIGH;
  P1SEL &= ~0x10; // P1_4 is GPIO (SSN) 
  P1DIR |= 0x10; // Set SSN as output
 
  halLedClear(1);
  
  
  appState = IDLE;
  appStarted = FALSE;
	
  // Config basicRF
  basicRfConfig.panId = PAN_ID;
  basicRfConfig.ackRequest = FALSE;

  // Initialise board peripherals
  //halBoardInit();

    // Initalise hal_rf
    if(halRfInit()==FAILED) {
      HAL_ASSERT(FALSE);
    }
 
  // Print Logo and splash screen on LCD
  utilPrintLogo("RF-SPI Tester");
 
 // Wait for user to press S1 to enter menu
  while (halButtonPushed()!= HAL_BUTTON_1);
  halMcuWaitMs(350);
  halLcdClear();
  
  // Set channel
  basicRfConfig.channel = appSelectChannel();
  
  appMode = appSelectMode();
  
    // Master application
    if(appMode == SPI_MODE_MASTER) {
        // No return from here
        appTransmitterMaster();
    }
    // Slave application
    else if(appMode == SPI_MODE_SLAVE) {
        // No return from here
        appReceiverSlave();
    }
    // Role is undefined. This code should not be reached
    HAL_ASSERT(FALSE);
}

/******************************************************************************
* @fn  URX1_IRQ
*
* @brief
*      
*
* Parameters:
*
* @param  void
*
* @return void
*
******************************************************************************/
#pragma vector=URX1_VECTOR
__interrupt void urx1_IRQ(void)
{
  static uint8 bufferIndex = 0;
 
  TCON &= ~0x80; // TCON.URX1IF = 0;

  pRxBuffer[bufferIndex++] = U1DBUF;
  if (bufferIndex == (N + 1))
  {
    bufferIndex = 0;
    dataReceived = TRUE;
  } 
  else {
    if (appMode == SPI_MODE_MASTER) {
      halMcuWaitUs(5); // Delay to allow slave to keep up
    }
    
    U1DBUF = pTxBuffer[bufferIndex];
  }
}

/******************************************************************************
* @fn  UTX1_IRQ
*
* @brief
*      
*
* Parameters:
*
* @param  void
*
* @return void
*
******************************************************************************/
#pragma vector=UTX1_VECTOR
__interrupt void utx1_IRQ(void)
{
  static uint8 bufferIndex = 0;
  
  IRCON2 &= ~0x04; // IRCON2.UTX1IF = 0

  if (++bufferIndex == (N + 1))
  {
    bufferIndex = 0;
    dataTransmitted = TRUE;
  }
  U1DBUF = pTxBuffer[bufferIndex];
}

/******************************************************************************
* @fn  DMA_IRQ
*
* @brief
*      
*
* Parameters:
*
* @param  void
*
* @return void
*
******************************************************************************/
#pragma vector=DMA_VECTOR
__interrupt void dma_IRQ(void)
{
  DMAIF = 0;              // Clear the CPU DMA interrupt flag
  if (DMAIRQ & 0x01) {
    DMAIRQ &= ~DMAIF0;     // DMA channel 0 module interrupt flag
    dma0Done = TRUE;
  }
  if (DMAIRQ & 0x02) {
    DMAIRQ &= ~DMAIF1;     // DMA channel 0 module interrupt flag
    dma1Done = TRUE;
  }
}

/*==== PRIVATE FUNCTIONS =====================================================*/
void printStats(void)
{  
  // Wait until joystick has been released for 10 ms
  // NOTE: This stalls MCU for a long time
  waitForJoystickIdle();
}


/******************************************************************************
* @fn  waitForJoystickIdle
*
* @brief
*      Wait until joystick has been idle for 10 ms
*
* Parameters:
*
* @param 
*
* @return
*           
*
*
******************************************************************************/
void waitForJoystickIdle(void)
{
  do {
    // Wait until joystick not active
    while (MCU_IO_GET(HAL_BOARD_IO_JOY_MOVE_PORT, HAL_BOARD_IO_JOY_MOVE_PIN));
    halMcuWaitMs(10);
    // Wait 10 ms and see that joystick is still not active
  } while (MCU_IO_GET(HAL_BOARD_IO_JOY_MOVE_PORT, HAL_BOARD_IO_JOY_MOVE_PIN));
}

/*==== END OF FILE ===========================================================*/


