/***********************************************************************************
    Filename:     spi_test_menu.c

    Description:  spi test menu library

***********************************************************************************/

/***********************************************************************************
* INCLUDES
*/

#include "hal_lcd.h"
#include "hal_rf.h"
#include "hal_assert.h"
#include "util_lcd.h"
#include "basic_rf.h"
#include "spi_rf_tst_menu.h"
#include "spi_rf_tst.h"

/***********************************************************************************
* LOCAL CONSTSANTS and MACROS
*/


/***********************************************************************************
* LOCAL VARIABLES
*/


uint8 txBufferMaster[MAX_PAYLOAD_LENGTH]; 
uint8 txBufferSlave[MAX_PAYLOAD_LENGTH]; 
uint8 rxBufferSlave[sizeof(txPacket.padding) + 1];
uint8 rxBufferMaster[sizeof(txPacket.padding) + 1];

static const char *pCurrentRxTrig;
static const char *pCurrentTxTrig;
static const char *pCurrentRxPri;
static const char *pCurrentTxPri;


static const char *currentMode;


// TEST MENU

static const char *currentTest;
static char dmaConfigText[16];



// Other global variables
static DMA_DESC dmaConfig0; // Struct for the DMA configuration (ch. 0)
static DMA_DESC dmaConfig1; // Struct for the DMA configuration (ch. 1)
static uint8 rxtrig;         // Rx DMA trigger
static uint8 txtrig;         // Tx DMA trigger
static uint8 rxpri;          // Rx DMA priority
static uint8 txpri;          // Tx DMA priority


static const menuItem_t pTests[] =
{
  "m_s_active",    M_S_ACTIVE,
  "m_s_txb_rxb",   M_S_TXBYTE_RXBYTE,
  "m_s_dma",       M_S_DMA,
  "m_s_interrupt", M_S_INTERRUPT,
  "s_m_active",    S_M_ACTIVE  ,
  "s_m_txb_rxb",   S_M_TXBYTE_RXBYTE,
  "s_m_dma",       S_M_DMA,
  "s_m_interrupt", S_M_INTERRUPT,
  
};

static const menu_t testMenu =
{
  pTests,
  N_ITEMS(pTests)
};


// Channel menu
static const menuItem_t pChannels[] =
{
  "11 - 2405 MHz", 11,
  "12 - 2410 MHz", 12,
  "13 - 2415 MHz", 13,
  "14 - 2420 MHz", 14,
  "15 - 2425 MHz", 15,
  "16 - 2430 MHz", 16,
  "17 - 2435 MHz", 17,
  "18 - 2440 MHz", 18,
  "19 - 2445 MHz", 19,
  "20 - 2450 MHz", 20,
  "21 - 2455 MHz", 21,
  "22 - 2460 MHz", 22,
  "23 - 2465 MHz", 23,
  "24 - 2470 MHz", 24,
  "25 - 2475 MHz", 25,
  "26 - 2480 MHz", 26,
};

static const menu_t channelMenu =
{
  pChannels,
  N_ITEMS(pChannels)
};

// Mode menu
static const menuItem_t pModes[] =
{
  "Master", SPI_MODE_MASTER,
  "Slave", SPI_MODE_SLAVE,
};

static const menu_t modeMenu =
{
  pModes,
  N_ITEMS(pModes)
};

static const menuItem_t pTrigMenu[] =
{
  "Rx", DMA_TRIG_URX1,
  "Tx", DMA_TRIG_UTX1,
};

static const menu_t trigMenu =
{
  pTrigMenu,
  N_ITEMS(pTrigMenu)
};

uint8 trigValues[] =
{
  TRIG_VAL_1, 
  TRIG_VAL_2
};


static const menuItem_t pPriMenu[] =
{
  "High", DMA_PRI_HIGH,
  "Normal", DMA_PRI_NORMAL,
  "Low", DMA_PRI_LOW,    
};

static const menu_t priMenu =
{
  pPriMenu,
  N_ITEMS(pPriMenu)
};

uint8 priValues[] = 
{ 
  DMA_PRI_HIGH,
  DMA_PRI_NORMAL, 
  DMA_PRI_LOW 
};


// Burst size menu
static const menuItem_t pBurstSizes[] =
{
  "10 Pkts", 0,
  "100 Pkts", 1,
  "1000 Pkts", 2,
  "10000 Pkts", 3,
};

uint32 burstSizes[] =
{
  BURST_SIZE_1,
  BURST_SIZE_2,
  BURST_SIZE_3,
  BURST_SIZE_4
};

static const menu_t burstSizeMenu =
{
  pBurstSizes,
  N_ITEMS(pBurstSizes)
};

// Packet rate menu
#define RATE_ITEM(n)    { #n"/second", n }
static const menuItem_t pRate[] =
{
  RATE_ITEM(100),
  RATE_ITEM(50),
  RATE_ITEM(20),
  RATE_ITEM(10),
};

static const menu_t rateMenu =
{
  pRate,
  N_ITEMS(pRate)
};

// TX output power menu
extern const menu_t powerMenu;

#ifdef INCLUDE_PA
// Gain menu
static const menuItem_t pGainSettings[] =
{
  "high gain", HAL_RF_GAIN_HIGH,
  "low gain", HAL_RF_GAIN_LOW
};

static const menu_t gainMenu =
{
  pGainSettings,
  N_ITEMS(pGainSettings)
};

#endif

/***********************************************************************************
* GLOBAL FUNCTIONS
*/


uint8 appSelectTest(void)
{
    halLcdWriteLine(HAL_LCD_LINE_1, "Select test: ");

    return utilMenuSelect(&testMenu);
}


/******************************************************************************
* @fn  appSelectDmaTrig
*
* @brief
*      Controls DMA trigger choice
*
* Parameters:
*
* @param uint8 Tx 0: Rx, 1: Tx
*
* @return uint8
*           
*
*
******************************************************************************/
uint8 appSelectDmaTrig(uint8 Tx) 
{
  
    uint8 index;
    
    halLcdWriteLine(HAL_LCD_LINE_1, (Tx ? "Tx DMA trigger" : "Rx DMA trigger"));
    
    index = utilMenuSelect(&trigMenu);
    
    if (Tx)
    {
      pCurrentTxTrig = (char const*)index;
    } else {
      pCurrentRxTrig = (char const*)index;
    }
    
    return trigValues[index];
}

/******************************************************************************
* @fn  selectDmaPri
*
* @brief
*      Controls DMA priority choice
*
* Parameters:
*
* @param uint8 Tx 0: Rx, 1: Tx
*
* @return uint8
*           
*
*
******************************************************************************/
uint8 appSelectDmaPri(uint8 Tx) 
{
  
    uint8 index;


    halLcdWriteLine(HAL_LCD_LINE_1, (Tx ? "Tx DMA priority" : "Rx DMA priority"));
 
    
    index = utilMenuSelect(&priMenu);
  
    if (Tx) 
    {
      pCurrentTxPri = (char const*)index;
    } else {
      pCurrentRxPri = (char const*)index;
    }
    
    return priValues[index];
}


/***********************************************************************************
* @fn          appSelectChannel
*
* @brief       Select channel
*
* @param       none
*
* @return      uint8 - Channel chosen
*/
uint8 appSelectChannel(void)
{
    halLcdWriteLine(HAL_LCD_LINE_1, "Channel: ");

    return utilMenuSelect(&channelMenu);
}

/***********************************************************************************
* @fn          appSelectMode
*
* @brief       Select application mode
*
* @param       none
*
* @return      uint8 - Application mode chosen
*/
uint8 appSelectMode(void)
{
    halLcdWriteLine(HAL_LCD_LINE_1, "Operating Mode: ");

    return utilMenuSelect(&modeMenu);
}


/***********************************************************************************
* @fn          appSelectRate
*
* @brief       Select transmit rate
*
* @param       none
*
* @return      uint8 - Rat chosen
*/
uint8 appSelectRate(void)
{
    halLcdWriteLine(HAL_LCD_LINE_1, "Packet rate: ");

    return utilMenuSelect(&rateMenu);
}


/***********************************************************************************
* @fn          appSelectBurstSize
*
* @brief       Select burst size (Number of packets to be transmitted)
*
* @param       none
*
* @return      uint32 - burst size
*/
uint32 appSelectBurstSize(void)
{
  uint8 index=0;
  halLcdWriteLine(HAL_LCD_LINE_1, "Burst Size: ");

  index = utilMenuSelect(&burstSizeMenu);
  return burstSizes[index];
}

/***********************************************************************************
* @fn          appSelectOutputPower
*
* @brief       Select output power
*
* @param       none
*
* @return      uint8 - Output power alternative
*/
uint8 appSelectOutputPower(void)
{
    halLcdWriteLine(HAL_LCD_LINE_1, "TX Output Power: ");

    return utilMenuSelect(&powerMenu);
}


#ifdef INCLUDE_PA
/***********************************************************************************
* @fn          appSelectGain
*
* @brief       Select gain for CC2590/91
*
* @param       none
*
* @return      uint8 - selected gain
*/
uint8 appSelectGain(void)
{
    halLcdWriteLine(HAL_LCD_LINE_1, "Gain: ");

    return utilMenuSelect(&gainMenu);
}
#endif


/**************************************************************************************************
 * @fn      appSpiSetTest
 *
 * @brief   Setup SPI Test Master/Slave Service
 *
 * @param   None
 *
 * @return  None
 **************************************************************************************************/
void appSpiSetTest(uint8 testMode, uint8 testSet)
{
/*==== Used for SPI Test ===============================================================*/
    uint8 j = 0;
    uint8 error = 0;
    volatile uint16 wrong;
    uint16 counter;
    dataReceived = FALSE;
    dataTransmitted = FALSE;
    dma0Done = FALSE;
    dma1Done = FALSE;
    uint32 pktsSent=0;
    
/*==== Used for RF Test ===============================================================*/        
    uint32 segNumber=0;
    int16 perRssiBuf[RSSI_AVG_WINDOW_SIZE] = {0};    // Ring buffer for RSSI
    uint8 perRssiBufCounter = 0;                     // Counter to keep track of the
    // oldest newest byte in RSSI
    // ring buffer
    perRxStats_t rxStats = {0,0,0,0};
    int16 rssi;
    uint8 resetStats=FALSE;
  
    if ((testSet == M_S_DMA && testMode == SPI_MODE_MASTER) || testSet == S_M_DMA) 
    {
     
      txtrig = appSelectDmaTrig(1);
      txpri =  appSelectDmaPri(1);
    }
    
    if ((testSet == M_S_DMA && testMode == SPI_MODE_SLAVE) ||  testSet == S_M_DMA) 
    {
      rxtrig = appSelectDmaTrig(0);
      rxpri =  appSelectDmaPri(0);
    }
    
    if (testSet == S_M_DMA) 
    {
      strncpy(dmaConfigText, pCurrentTxTrig, 2);
      strcat(dmaConfigText, ",");
      strncat(dmaConfigText, pCurrentTxPri, 4);
      strcat(dmaConfigText, ";");
      strncat(dmaConfigText, pCurrentRxTrig, 2);
      strcat(dmaConfigText, ",");
      strncat(dmaConfigText, pCurrentRxPri, 4);
    } else if (testSet == M_S_DMA && testMode == SPI_MODE_MASTER)
    {
      strncpy(dmaConfigText, pCurrentTxTrig, 2);
      strcat(dmaConfigText, ",");
      strncat(dmaConfigText, pCurrentTxPri, 4);
    } else if (testSet == M_S_DMA && testMode == SPI_MODE_SLAVE)
    {
      strncpy(dmaConfigText, pCurrentRxTrig, 2);
      strcat(dmaConfigText, ",");
      strncat(dmaConfigText, pCurrentRxPri, 4);
    }
    //halLcdSpiEna();
    halLcdWriteLine(HAL_LCD_LINE_1, currentTest);
    halLcdWriteLine(HAL_LCD_LINE_2, currentMode);
    if (testSet == M_S_DMA || testSet == S_M_DMA)
    {
      
      halLcdWriteLine(HAL_LCD_LINE_3, dmaConfigText);
    }
    //halLcdSpiDis();
    
    if (testSet == M_S_DMA || testSet == S_M_DMA)
    {
    
    }
    switch (testMode)
    {
      case SPI_MODE_MASTER:
        if (testSet == M_S_DMA || testSet == S_M_DMA) 
        { // DMA testcases: Set up DMA descriptor 0 for Tx
          SET_WORD(dmaConfig0.SRCADDRH, dmaConfig0.SRCADDRL, txBufferMaster);
          SET_WORD(dmaConfig0.DESTADDRH, dmaConfig0.DESTADDRL, &X_U1DBUF);
          
          dmaConfig0.VLEN      = DMA_VLEN_USE_LEN;    // VLEN = 000
   
          SET_WORD(dmaConfig0.LENH, dmaConfig0.LENL, N + 1);
      
          dmaConfig0.WORDSIZE  = DMA_WORDSIZE_uint8;
          dmaConfig0.TRIG      = txtrig;
          dmaConfig0.TMODE     = DMA_TMODE_SINGLE;
          dmaConfig0.SRCINC    = DMA_SRCINC_1;
          dmaConfig0.DESTINC   = DMA_DESTINC_0;
          dmaConfig0.IRQMASK   = DMA_IRQMASK_ENABLE;
          dmaConfig0.M8        = DMA_M8_USE_8_BITS;
          dmaConfig0.PRIORITY  = txpri;
      
          // Save pointer to the DMA configuration struct into DMA-channel 0
          // configuration registers
          SET_WORD(DMA0CFGH, DMA0CFGL, &dmaConfig0); 
          
          if (testSet == S_M_DMA) 
          { // S->M DMA testcase: Set up DMA descriptor 1 for Rx
            SET_WORD(dmaConfig1.SRCADDRH, dmaConfig1.SRCADDRL, &X_U1DBUF);
            SET_WORD(dmaConfig1.DESTADDRH, dmaConfig1.DESTADDRL, rxBufferMaster);
            dmaConfig1.VLEN      = DMA_VLEN_USE_LEN;    // VLEN = 000
       
            SET_WORD(dmaConfig1.LENH, dmaConfig1.LENL, N + 1);
        
            dmaConfig1.WORDSIZE  = DMA_WORDSIZE_uint8;
            dmaConfig1.TRIG      = rxtrig;
            dmaConfig1.TMODE     = DMA_TMODE_SINGLE;
            dmaConfig1.SRCINC    = DMA_SRCINC_0;
            dmaConfig1.DESTINC   = DMA_DESTINC_1;
            dmaConfig1.IRQMASK   = DMA_IRQMASK_ENABLE;
            dmaConfig1.M8        = DMA_M8_USE_8_BITS;
            dmaConfig1.PRIORITY  = rxpri;
            
            // Save pointer to the DMA configuration struct into DMA-channel 0
            // configuration registers
            SET_WORD(DMA1CFGH, DMA1CFGL, &dmaConfig1);
          }
        
        // Set up DMA interrupt              
        DMAIF = 0;          //Clear flags
        DMAIRQ &= ~DMAIF0;   
        DMAIRQ &= ~DMAIF1; 
        // Enable interrupt
        IEN1 |= 0x01; // IEN1.DMAIE = 1
      }
       
      halLcdClear();
      halLcdWriteLines("RF-SPI Tester", "Joystick Push", "start/stop");
            
      // Infinite loop    
      while (TRUE) {
        // Clear LED
        halLedClear(1);
                    
        // Wait for user to start application
        while(!halJoystickPushed() );
        appStartStop();

        halLcdClear();
        halLcdWriteLines("RF-SPI Tester", "Master", NULL);
        
        // Set LED to show that master is active
        halLedSet(1);
                
        // Configure Master on SPI bus
        halMasterSpiInit(); 
        
                
       while(appStarted) 
       {
         if( halJoystickPushed() ) {
                appStartStop();
            }

         if (pktsSent < burstSize) {
           if( appState == TRANSMIT_PACKET ) {
              // Make sure sequence number has network byte order
              UINT32_HTON(txPacket.seqNumber);

              basicRfSendPacket(RX_ADDR, (uint8*)&txPacket, PACKET_SIZE);

              // Change byte order back to host order before increment
              UINT32_NTOH(txPacket.seqNumber);
              txPacket.seqNumber++;

              pktsSent++;
              #ifdef SRF04EB
              utilLcdDisplayValue(HAL_LCD_LINE_2, "Sent: ", (int32)pktsSent, NULL);
              #else
              utilLcdDisplayValue(HAL_LCD_LINE_3, "Sent: ", (int32)pktsSent, NULL);
              #endif
              appState = IDLE;

              halLedToggle(2);
                           
              // Run the SPI Test with same packet as RF
          if (testSet == S_M_INTERRUPT) {
          // Interrupt test cases: Set up interrupt
          pTxBuffer = txBufferMaster;
          pRxBuffer = rxBufferMaster;
          // Clear pending interrupt
          TCON &= ~0x80; // TCON.URX1IF = 0;
          // Enable interrupt
          URX1IE = 1;
        }
        
        for (counter = 0; counter < burstSize; counter++) {
          
          switch (testSet) {
          case M_S_ACTIVE:
            //spi_m_s_active_master(txBufferMaster);
            for (j = 0; j <= sizeof(txPacket.padding); j++) {
              txBufferMaster[j] = j;
              SSN = LOW;
              U1DBUF = txBufferMaster[j];
              while (U1ACTIVE); 
              SSN = HIGH;
            }
            break;
          case M_S_TXBYTE_RXBYTE:
            for (j = 0; j <= sizeof(txPacket.padding); j++) {
              SSN = LOW;
              U1DBUF = txBufferMaster[j];
              while (!U1TX_BYTE);
              U1TX_BYTE = 0;
              SSN = HIGH;
            }              
            break;
          case M_S_DMA:
            DMAARM = DMAARM0;   // Arm DMA channel 0
            SSN = LOW;
            DMAREQ = 0x01;
            while (!dma0Done);
            while (U1ACTIVE); 
            SSN = HIGH;
            dma0Done = FALSE;
            break;            
          case M_S_INTERRUPT:
            for (j = 0; j <= sizeof(txPacket.padding); j++) {
              halMcuWaitUs(10); // Wait here because IRQ is not fast enough in current implementation
              SSN = LOW;
              U1DBUF = txBufferMaster[j];
              while (!U1TX_BYTE);
              U1TX_BYTE = 0;
              SSN = HIGH;
            }
            break;
          case S_M_ACTIVE:
            for (j = 0; j <= sizeof(txPacket.padding); j++) {
              halMcuWaitUs(1); // Wait to allow slave to keep up
              SSN = LOW;
              U1DBUF = DUMMY_BYTE;
              while (U1ACTIVE);
              rxBufferMaster[j] = U1DBUF;
              SSN = HIGH;
            }
            break;
          case S_M_TXBYTE_RXBYTE:
            for (j = 0; j <= sizeof(txPacket.padding); j++) {
              SSN = LOW;
              U1DBUF = DUMMY_BYTE;
              while (!U1TX_BYTE);
              rxBufferMaster[j] = U1DBUF;
              SSN = HIGH;
              U1TX_BYTE = 0;
            }
            break;
          case S_M_DMA:
            DMAARM = DMAARM1 | DMAARM0;   // Arm DMA channels 0 and 1
            SSN = LOW;
            DMAREQ = 0x01;
            while (!dma0Done || ! dma1Done);
            while (U1ACTIVE); 
            SSN = HIGH;
            dma0Done = FALSE;
            dma1Done = FALSE;
            break;            
          case S_M_INTERRUPT:
            // Prepare first byte
            SSN = LOW;
            pRxBuffer[0] = U1DBUF;
            U1DBUF = pTxBuffer[0];
            while (!dataReceived);
            SSN = HIGH;
            dataReceived = FALSE;
            break;
          }
          halMcuWaitUs(50); // Delay to allow for error checking on slave
          
          if (testSet >= S_M_ACTIVE) {
            // Error checking
            error = 0;
            for (j = 0; j <= sizeof(txPacket.padding); j++) {
              if (rxBufferMaster[j] != txBufferMaster[j]){
                error++;
              }
            }
            if(error != 0) {
              wrong++;
            }
            
            if (error && wrong == 1) {

            }
            for (j = 0; j <= N; j++) {
              rxBufferMaster[j] = 0xFF;        
            }            
          }          
        }
        if (testSet >= S_M_ACTIVE) {

          counter = 0;
          wrong = 0;
        }
         // appSpiRunTst(testSet, txPacket.padding, txPacket.padding);
                           
            }
         } else
                appStarted = !appStarted;
        }       

    }
    
    case SPI_MODE_SLAVE:
      if (testSet < S_M_ACTIVE || testSet == S_M_DMA) {
        // M->S and DMA tests only:
        // Clear pending interrupts
        halJoystickIntClear();
        // Enable interrupt
        halJoystickIntEnable(0);
      }
  
      if (testSet == M_S_DMA || testSet == S_M_DMA) {
        // DMA testcases: Set up DMA descriptor 0 for Rx
        SET_WORD(dmaConfig0.SRCADDRH, dmaConfig0.SRCADDRL, &X_U1DBUF);
        SET_WORD(dmaConfig0.DESTADDRH, dmaConfig0.DESTADDRL, rxBufferSlave);
         
        dmaConfig0.VLEN      = DMA_VLEN_USE_LEN;    // VLEN = 000
   
        SET_WORD(dmaConfig0.LENH, dmaConfig0.LENL, N + 1);
    
        dmaConfig0.WORDSIZE  = DMA_WORDSIZE_uint8;
        dmaConfig0.TRIG      = rxtrig;
        dmaConfig0.TMODE     = DMA_TMODE_SINGLE;
        dmaConfig0.SRCINC    = DMA_SRCINC_0;
        dmaConfig0.DESTINC   = DMA_DESTINC_1;
        dmaConfig0.IRQMASK   = DMA_IRQMASK_ENABLE;
        dmaConfig0.M8        = DMA_M8_USE_8_BITS;
        dmaConfig0.PRIORITY  = rxpri;
    
        // Save pointer to the DMA configuration struct into DMA-channel 0
        // configuration registers
        SET_WORD(DMA0CFGH, DMA0CFGL, &dmaConfig0);
        
        if (testSet == S_M_DMA) {
          // S->M DMA testcase: Set up DMA descriptor 1 for Tx
          SET_WORD(dmaConfig1.SRCADDRH, dmaConfig1.SRCADDRL, txBufferSlave);
          SET_WORD(dmaConfig1.DESTADDRH, dmaConfig1.DESTADDRL, &X_U1DBUF);
          dmaConfig1.VLEN      = DMA_VLEN_USE_LEN;    // VLEN = 000
     
          SET_WORD(dmaConfig1.LENH, dmaConfig1.LENL, N + 1);
      
          dmaConfig1.WORDSIZE  = DMA_WORDSIZE_uint8;
          dmaConfig1.TRIG      = txtrig;
          dmaConfig1.TMODE     = DMA_TMODE_SINGLE;
          dmaConfig1.SRCINC    = DMA_SRCINC_1;
          dmaConfig1.DESTINC   = DMA_DESTINC_0;
          dmaConfig1.IRQMASK   = DMA_IRQMASK_ENABLE;
          dmaConfig1.M8        = DMA_M8_USE_8_BITS;
          dmaConfig1.PRIORITY  = txpri;
          
          // Save pointer to the DMA configuration struct into DMA-channel 0
          // configuration registers
          SET_WORD(DMA1CFGH, DMA1CFGL, &dmaConfig1);
          
        }
        
        // Set up DMA interrupt              
        DMAIF = 0;          //Clear flags
        DMAIRQ &= ~DMAIF0;   
        DMAIRQ &= ~DMAIF1; 
        // Enable interrupt
        IEN1 |= 0x01; // IEN1.DMAIE = 1
      }

      // Turn on RF receiver
      basicRfReceiveOn();
      
      halLcdClear();
      halLcdWriteLines("RF-SPI Tester", "Receiver", "Ready");
                  
      while (TRUE) {
          
        // Enable SPI Slave 
        halSlaveSpiInit();              
        
        if (testSet == M_S_INTERRUPT) {
          pTxBuffer = txBufferSlave;
          pRxBuffer = rxBufferSlave;
          // Clear pending interrupts
          TCON &= ~0x80; // TCON.URX1IF = 0;
          // Enable interrupt
          URX1IE = 1;
        }
        else if (testSet == S_M_INTERRUPT) {
          pTxBuffer = txBufferSlave;
          // Clear pending interrupts
          IRCON2 &= ~0x04; // IRCON2.UTX1IF = 0
          // Enable interrupt
          IEN2 |= 0x08; // IEN2.UTX1IE = 1
        }
        else if (testSet == S_M_DMA) {
          DMAARM = DMAARM1 | DMAARM0;   // Arm DMA channels 0 and 1
          DMAREQ = 0x02;
        }            
        // Infinite loop
        halLedSet(1);
               
        while(!basicRfPacketIsReady());
        if(basicRfReceive((uint8*)&rxPacket, MAX_PAYLOAD_LENGTH, &rssi)>0) {
            halLedSet(2);
			
	    // Change byte order from network to host order
	    UINT32_NTOH(rxPacket.seqNumber);
            segNumber = rxPacket.seqNumber;
            
            // If statistics is reset set expected sequence number to
            // received sequence number
            if(resetStats){
              rxStats.expectedSeqNum = segNumber;
              resetStats=FALSE;
            }
        
            // Subtract old RSSI value from sum
            rxStats.rssiSum -= perRssiBuf[perRssiBufCounter];
            // Store new RSSI value in ring buffer, will add it to sum later
            perRssiBuf[perRssiBufCounter] =  rssi;

            // Add the new RSSI value to sum
            rxStats.rssiSum += perRssiBuf[perRssiBufCounter];
            if (++perRssiBufCounter == RSSI_AVG_WINDOW_SIZE) {
                perRssiBufCounter = 0;      // Wrap ring buffer counter
            }

            // Check if received packet is the expected packet
            if (rxStats.expectedSeqNum == segNumber) {
                rxStats.expectedSeqNum++;
            }
            // If there is a jump in the sequence numbering this means some packets in
            // between has been lost.
            else if (rxStats.expectedSeqNum < segNumber){
                rxStats.lostPkts += segNumber - rxStats.expectedSeqNum;
                rxStats.expectedSeqNum = segNumber + 1;
            }
            // If the sequence number is lower than the previous one, we will assume a
            // new data burst has started and we will reset our statistics variables.
            else {
                // Update our expectations assuming this is a new burst
                rxStats.expectedSeqNum = segNumber + 1;
                rxStats.rcvdPkts = 0;
                rxStats.lostPkts = 0;
            }
            rxStats.rcvdPkts++;

            // reset statistics if button 1 is pressed
            if(halButtonPushed()==HAL_BUTTON_1){
                resetStats = TRUE;
                rxStats.rcvdPkts = 1;
                rxStats.lostPkts = 0;
            }
#if 0                       
            // Update LCD
            // PER in units per 1000
            utilLcdDisplayValue(HAL_LCD_LINE_1, "PER: ", (int32)((rxStats.lostPkts*1000)/(rxStats.lostPkts+rxStats.rcvdPkts)), " /1000");
            utilLcdDisplayValue(HAL_LCD_LINE_2, "RSSI: ", (int32)rxStats.rssiSum/32, "dBm");
            #ifndef SRF04EB
            utilLcdDisplayValue(HAL_LCD_LINE_3, "Recv'd: ", (int32)rxStats.rcvdPkts, NULL);
            #endif
#endif 
            halLedClear(2);
        } 
        
        switch (testSet) {
        case M_S_ACTIVE:
          for (j = 0; j <= sizeof(txPacket.padding); j++) {
            while (!U1ACTIVE);
            while (U1ACTIVE);
            rxBufferSlave[j] = U1DBUF;  
          }
          break;
        case M_S_TXBYTE_RXBYTE:
          for (j = 0; j <= sizeof(txPacket.padding); j++) {
            while (!U1RX_BYTE);
            U1RX_BYTE = 0;
            rxBufferSlave[j] = U1DBUF;
          }              
          break;
        case M_S_DMA:
          DMAARM = DMAARM0;   // Arm DMA channel 0
          while (!dma0Done);
          dma0Done = FALSE;
          break;
        case M_S_INTERRUPT:
          while (!dataReceived);
          dataReceived = FALSE;
          break;
        case S_M_ACTIVE:
          for (j = 0; j <= sizeof(txPacket.padding); j++) {
            U1DBUF = txBufferSlave[j];
            while (!U1ACTIVE);
            while (U1ACTIVE);
          }
          break;
        case S_M_TXBYTE_RXBYTE:
          for (j = 0; j <= sizeof(txPacket.padding); j++) {
            U1DBUF = txBufferSlave[j];
            while (!U1RX_BYTE);
            U1RX_BYTE = 0;
          }
          break;
        case S_M_DMA:
          while (!dma1Done) {
            if (dma0Done) {
              DMAARM |= DMAARM0;
            }
          }
          DMAARM |= DMAARM1;
          DMAREQ = 0x02;
          dma1Done = FALSE;
          while (!dma0Done);
          DMAARM |= DMAARM0;
          dma0Done = FALSE;
          break;
        case S_M_INTERRUPT:
          // Prepare first byte
          U1DBUF = txBufferSlave[0];
          while (!dataTransmitted);
          dataTransmitted = FALSE;
          break;
        }

        if (testSet < S_M_ACTIVE || testSet == S_M_DMA) {
          // M->S tests only + DMA:
        
          // Error checking
          error = 0;
          for (j = 0; j <= sizeof(txPacket.padding); j++){
            if (rxBufferSlave[j] != txBufferSlave[j])
              error++;
          }
          if(error != 0) {
            wrong++;
          }
          counter++;
          for (j = 0; j <= sizeof(txPacket.padding); j++) {
            rxBufferSlave[j] = 0xFF;        
          }
        }
      }
      
  }  
} // EoF appSpiTestSet

/***********************************************************************************
  Copyright 2007 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
***********************************************************************************/
