/***********************************************************************************
    Filename:     spi_test_menu.c

    Description:  spi test menu library

***********************************************************************************/

/***********************************************************************************
* INCLUDES
*/

#include "hal_lcd.h"
#include "hal_rf.h"
#include "hal_assert.h"
#include "util_lcd.h"
#include "cc2530_spiTst.h"
#include "cc2530_spiTst_menu.h"

/***********************************************************************************
* LOCAL CONSTSANTS and MACROS
*/


/***********************************************************************************
* LOCAL VARIABLES
*/

uint8 txBufferMaster[] = {  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15,
                           16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,  30,  31,
                           32,  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,
                           48,  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,
                           64,  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,
                           80,  81,  82,  83,  84,  85,  86,  87,  88,  89,  90,  91,  92,  93,  94,  95,
                           96,  97,  98,  99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111,
                          112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127
                         };
uint8 txBufferSlave[] = {  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15,
                          16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,  30,  31,
                          32,  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,
                          48,  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,
                          64,  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,
                          80,  81,  82,  83,  84,  85,  86,  87,  88,  89,  90,  91,  92,  93,  94,  95,
                          96,  97,  98,  99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111,
                         112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127
                        };
uint8 rxBufferMaster[N];
uint8 rxBufferSlave[N];


// TEST MENU



// Other global variables
static DMA_DESC dmaConfig0; // Struct for the DMA configuration (ch. 0)
static DMA_DESC dmaConfig1; // Struct for the DMA configuration (ch. 1)
static uint8 rxpri;          // Rx DMA priority
static uint8 txpri;          // Tx DMA priority


static const menuItem_t pTests[] =
{
  "M -> S",        M_S_ONEWAY,
  "S -> M",        S_M_ONEWAY,
  "M <-> S",       TWOWAY
};

static const menu_t testMenu =
{
  pTests,
  N_ITEMS(pTests)
};

// Mode menu
static const menuItem_t pModes[] =
{
  "Master", SPI_MODE_MASTER,
  "Slave", SPI_MODE_SLAVE,
};

static const menu_t modeMenu =
{
  pModes,
  N_ITEMS(pModes)
};

static const menuItem_t pPriMenu[] =
{
  "High", DMA_PRI_HIGH,
  "Normal", DMA_PRI_NORMAL,
  "Low", DMA_PRI_LOW,    
};

static const menu_t priMenu =
{
  pPriMenu,
  N_ITEMS(pPriMenu)
};

// Burst size menu
static const menuItem_t pBurstSizes[] =
{
  "10 Pkts", 0,
  "100 Pkts", 1,
  "1000 Pkts", 2,
  "10000 Pkts", 3,
};

uint32 burstSizes[] =
{
  BURST_SIZE_1,
  BURST_SIZE_2,
  BURST_SIZE_3,
  BURST_SIZE_4
};

static const menu_t burstSizeMenu =
{
  pBurstSizes,
  N_ITEMS(pBurstSizes)
};

// Packet rate menu
#define RATE_ITEM(n)    { #n"/second", n }
static const menuItem_t pRate[] =
{
  RATE_ITEM(100),
  RATE_ITEM(50),
  RATE_ITEM(20),
  RATE_ITEM(10),
};

static const menu_t rateMenu =
{
  pRate,
  N_ITEMS(pRate)
};

/***********************************************************************************
* GLOBAL FUNCTIONS
*/


uint8 appSelectTest(void)
{
    halLcdWriteLine(HAL_LCD_LINE_1, "Select test: ");

    return utilMenuSelect(&testMenu);
}


/******************************************************************************
* @fn  selectDmaPri
*
* @brief
*      Controls DMA priority choice
*
* @return uint8
*           
*
*
******************************************************************************/
uint8 appSelectDmaPri() 
{
  
    halLcdWriteLine(HAL_LCD_LINE_1, "DMA priority");
 
    return utilMenuSelect(&priMenu);
}


/***********************************************************************************
* @fn          appSelectMode
*
* @brief       Select application mode
*
* @param       none
*
* @return      uint8 - Application mode chosen
*/
uint8 appSelectMode(void)
{
    halLcdWriteLine(HAL_LCD_LINE_1, "Operating Mode: ");

    return utilMenuSelect(&modeMenu);
}


/***********************************************************************************
* @fn          appSelectRate
*
* @brief       Select transmit rate
*
* @param       none
*
* @return      uint8 - Rat chosen
*/
uint8 appSelectRate(void)
{
    halLcdWriteLine(HAL_LCD_LINE_1, "Packet rate: ");

    return utilMenuSelect(&rateMenu);
}


/***********************************************************************************
* @fn          appSelectBurstSize
*
* @brief       Select burst size (Number of packets to be transmitted)
*
* @param       none
*
* @return      uint32 - burst size
*/
uint32 appSelectBurstSize(void)
{
  uint8 index=0;
  halLcdWriteLine(HAL_LCD_LINE_1, "Burst Size: ");

  index = utilMenuSelect(&burstSizeMenu);
  return burstSizes[index];
}


/**************************************************************************************************
 * @fn      appSpiSetTest
 *
 * @brief   Setup SPI Test Master/Slave Service
 *
 * @param   None
 *
 * @return  None
 **************************************************************************************************/
void appSpiSetTest(uint8 testMode, uint8 testSet)
{
    uint8 j = 0;
    uint8 error = 0;
    uint16 counter;
    dataReceived = FALSE;
    dataTransmitted = FALSE;
    dma0Done = FALSE;
    dma1Done = FALSE;
  
    txpri = rxpri = appSelectDmaPri();

    switch (testMode)
    {
    case SPI_MODE_MASTER:
      SET_WORD(dmaConfig0.SRCADDRH, dmaConfig0.SRCADDRL, txBufferMaster);
      SET_WORD(dmaConfig0.DESTADDRH, dmaConfig0.DESTADDRL, &X_U0DBUF);
      
      dmaConfig0.VLEN      = DMA_VLEN_USE_LEN;    // VLEN = 000

      SET_WORD(dmaConfig0.LENH, dmaConfig0.LENL, N);
  
      dmaConfig0.WORDSIZE  = DMA_WORDSIZE_BYTE;
      dmaConfig0.TRIG      = DMA_TRIG_UTX0;
      dmaConfig0.TMODE     = DMA_TMODE_SINGLE;
      dmaConfig0.SRCINC    = DMA_SRCINC_1;
      dmaConfig0.DESTINC   = DMA_DESTINC_0;
      dmaConfig0.IRQMASK   = DMA_IRQMASK_ENABLE;
      dmaConfig0.M8        = DMA_M8_USE_8_BITS;
      dmaConfig0.PRIORITY  = txpri;
  
      // Save pointer to the DMA configuration struct into DMA-channel 0
      // configuration registers
      SET_WORD(DMA0CFGH, DMA0CFGL, &dmaConfig0); 
      
      if (testSet == S_M_ONEWAY || testSet == TWOWAY)
      { // S->M DMA testcase: Set up DMA descriptor 1 for Rx
        SET_WORD(dmaConfig1.SRCADDRH, dmaConfig1.SRCADDRL, &X_U0DBUF);
        SET_WORD(dmaConfig1.DESTADDRH, dmaConfig1.DESTADDRL, rxBufferMaster);

        dmaConfig1.VLEN      = DMA_VLEN_USE_LEN;    // VLEN = 000
   
        SET_WORD(dmaConfig1.LENH, dmaConfig1.LENL, N);
    
        dmaConfig1.WORDSIZE  = DMA_WORDSIZE_BYTE;
        dmaConfig1.TRIG      = DMA_TRIG_URX0;
        dmaConfig1.TMODE     = DMA_TMODE_SINGLE;
        dmaConfig1.SRCINC    = DMA_SRCINC_0;
        dmaConfig1.DESTINC   = DMA_DESTINC_1;
        dmaConfig1.IRQMASK   = DMA_IRQMASK_ENABLE;
        dmaConfig1.M8        = DMA_M8_USE_8_BITS;
        dmaConfig1.PRIORITY  = rxpri;
        
        // Save pointer to the DMA configuration struct into DMA-channel 0
        // configuration registers
        SET_WORD(DMA1CFGH, DMA1CFGL, &dmaConfig1);
      }
      
      // Set up DMA interrupt              
      DMAIF = 0;          //Clear flags
      DMAIRQ &= ~DMAIF0;   
      DMAIRQ &= ~DMAIF1; 
      // Enable interrupt
      IEN1 |= 0x01; // IEN1.DMAIE = 1
     
      halLcdClear();
      halLcdWriteLines("SPI Tester", "Joystick Push", "start");
            
      // Infinite loop    
      while (TRUE)
      {
        // Clear LED
        halLedClear(1);
                    
        // Wait for user to start application
        while (!halJoystickPushed());
        appStartStop();
  
        halLcdClear();
        
        // Set LED to show that master is active
        halLedSet(1);
                
        // Configure Master on SPI bus
        halMasterSpiInit(); 
       
        error = 0;
  
        for (counter = 1; counter <= N_PACKETS; counter++)
        {
          switch (testSet) {
          case M_S_ONEWAY:
            DMAARM = DMAARM0;   // Arm DMA channel 0
            SSN = LOW;
            DMAREQ = 0x01;
            while (!dma0Done);
            while (U0CSR & 0x01); // U0ACTIVE
            SSN = HIGH;
            dma0Done = FALSE;
            break;            
  
          case S_M_ONEWAY:
          case TWOWAY:
            DMAARM = DMAARM0 | DMAARM1;   // Arm DMA channels 0 and 1
            SSN = LOW;
            DMAREQ = 0x01;
            while (!dma0Done || !dma1Done);
            while (U0CSR & 0x01); // U0ACTIVE
            SSN = HIGH;
            dma0Done = FALSE;
            dma1Done = FALSE;
            break;            
          }

          halMcuWaitUs(3000); // Delay to allow for LCD displaying on slave
          
          if (testSet == M_S_ONEWAY || testSet == TWOWAY)
          {
            halLcdDisplayValue(HAL_LCD_LINE_1, NULL, (uint32) counter, " pkts sent");
          }
  
          if (testSet == S_M_ONEWAY || testSet == TWOWAY)
          {
            halLcdDisplayValue(HAL_LCD_LINE_2, NULL, (uint32) counter, " pkts rcved");
            // Error checking
            for (j = 0; j < N; j++) {
              if (rxBufferMaster[j] != txBufferMaster[j]){
                error++;
              }
            }
            
            halLcdDisplayValue(HAL_LCD_LINE_3, "# of errs = ", (uint32) error, NULL);
  
            for (j = 0; j < N; j++) {
              rxBufferMaster[j] = j;        
            }            
          }          
        }
      }
    
    case SPI_MODE_SLAVE:
      if (testSet == M_S_ONEWAY || testSet == TWOWAY)
      {
        // DMA testcases: Set up DMA descriptor 0 for Rx
        SET_WORD(dmaConfig0.SRCADDRH, dmaConfig0.SRCADDRL, &X_U0DBUF);
        SET_WORD(dmaConfig0.DESTADDRH, dmaConfig0.DESTADDRL, rxBufferSlave);
         
        dmaConfig0.VLEN      = DMA_VLEN_USE_LEN;    // VLEN = 000
   
        SET_WORD(dmaConfig0.LENH, dmaConfig0.LENL, N);
    
        dmaConfig0.WORDSIZE  = DMA_WORDSIZE_BYTE;
        dmaConfig0.TRIG      = DMA_TRIG_URX0;
        dmaConfig0.TMODE     = DMA_TMODE_SINGLE;
        dmaConfig0.SRCINC    = DMA_SRCINC_0;
        dmaConfig0.DESTINC   = DMA_DESTINC_1;
        dmaConfig0.IRQMASK   = DMA_IRQMASK_ENABLE;
        dmaConfig0.M8        = DMA_M8_USE_8_BITS;
        dmaConfig0.PRIORITY  = rxpri;
    
        // Save pointer to the DMA configuration struct into DMA-channel 0
        // configuration registers
        SET_WORD(DMA0CFGH, DMA0CFGL, &dmaConfig0);
      }
        
      if (testSet == S_M_ONEWAY || testSet == TWOWAY)
      {
        // S->M DMA testcase: Set up DMA descriptor 1 for Tx
        SET_WORD(dmaConfig1.SRCADDRH, dmaConfig1.SRCADDRL, txBufferSlave);
        SET_WORD(dmaConfig1.DESTADDRH, dmaConfig1.DESTADDRL, &X_U0DBUF);
        dmaConfig1.VLEN      = DMA_VLEN_USE_LEN;    // VLEN = 000
   
        SET_WORD(dmaConfig1.LENH, dmaConfig1.LENL, N);
    
        dmaConfig1.WORDSIZE  = DMA_WORDSIZE_BYTE;
        dmaConfig1.TRIG      = DMA_TRIG_UTX0;
        dmaConfig1.TMODE     = DMA_TMODE_SINGLE;
        dmaConfig1.SRCINC    = DMA_SRCINC_1;
        dmaConfig1.DESTINC   = DMA_DESTINC_0;
        dmaConfig1.IRQMASK   = DMA_IRQMASK_ENABLE;
        dmaConfig1.M8        = DMA_M8_USE_8_BITS;
        dmaConfig1.PRIORITY  = txpri;
        
        // Save pointer to the DMA configuration struct into DMA-channel 0
        // configuration registers
        SET_WORD(DMA1CFGH, DMA1CFGL, &dmaConfig1);
        
      }
        
      // Set up DMA interrupt              
      DMAIF = 0;          //Clear flags
      DMAIRQ &= ~DMAIF0;   
      DMAIRQ &= ~DMAIF1; 
      // Enable interrupt
      IEN1 |= 0x01; // IEN1.DMAIE = 1
      
      halSlaveSpiInit();              
      
      halLcdClear();

      switch (testSet)
      {
      case M_S_ONEWAY:
        halLcdWriteLine(HAL_LCD_LINE_1, "Rdy to rcv");
        break;

      case S_M_ONEWAY:
        halLcdWriteLine(HAL_LCD_LINE_1, "Rdy to xmit");
        break;
      
      case TWOWAY:
        halLcdWriteLine(HAL_LCD_LINE_1, "Rdy to rcv&xmit");
        break;
      }

      // Infinite loop
      halLedSet(1);
      
      counter = 0;
      error = 0;

      while (TRUE)
      {
        switch (testSet) {
        case M_S_ONEWAY:
          DMAARM = DMAARM0;   // Arm DMA channel 0
          while (!dma0Done);
          while (U0CSR & 0x01); // U0ACTIVE
          dma0Done = FALSE;
          break;

        case S_M_ONEWAY:
          DMAARM = DMAARM1;   // Arm DMA channel 1
          DMAREQ = 0x02;
          while (!dma1Done);
          while (U0CSR & 0x01); // U0ACTIVE
          dma1Done = FALSE;
          break;

        case TWOWAY:
          DMAARM = DMAARM0 | DMAARM1;   // Arm DMA channels 0 and 1
          DMAREQ = 0x02;
          while (!dma0Done || !dma1Done);
          while (U0CSR & 0x01); // U0ACTIVE
          dma0Done = FALSE;
          dma1Done = FALSE;
        }

        counter++;

        if (testSet == S_M_ONEWAY || testSet == TWOWAY)
        {
          halLcdDisplayValue(HAL_LCD_LINE_1, NULL, (uint32) counter, " pkts sent");
        }

        if (testSet == M_S_ONEWAY || testSet == TWOWAY)
        {
          halLcdDisplayValue(HAL_LCD_LINE_2, NULL, (uint32) counter, " pkts rcved");
          // Error checking
          for (j = 0; j < N; j++) {
            if (rxBufferSlave[j] != txBufferSlave[j]){
              error++;
            }
          }
          
          halLcdDisplayValue(HAL_LCD_LINE_3, "# of errs = ", (uint32) error, NULL);

          for (j = 0; j < N; j++) {
            rxBufferSlave[j] = j;        
          }            
        }          
      }
    }
}


/***********************************************************************************
  Copyright 2007 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS?WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
***********************************************************************************/
