/*-----------------------------------------------------------------------------
+------------------------------------------------------------------------------
|  Copyright 2010 Texas Instruments Incorporated. All rights reserved.
|
|  IMPORTANT: Your use of this Software is limited to those specific rights
|  granted under the terms of a software license agreement between the user who
|  downloaded the software, his/her employer (which must be your employer) and
|  Texas Instruments Incorporated (the "License"). You may not use this Software
|  unless you agree to abide by the terms of the License. The License limits
|  your use, and you acknowledge, that the Software may not be modified, copied
|  or distributed unless embedded on a Texas Instruments microcontroller or used
|  solely and exclusively in conjunction with a Texas Instruments radio
|  frequency transceiver, which is integrated into your product. Other than for
|  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
|  works of, modify, distribute, perform, display or sell this Software and/or
|  its documentation for any purpose.
|
|  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
|  PROVIDED AS IS?WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
|  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
|  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
|  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
|  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
|  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES INCLUDING
|  BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE OR
|  CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT OF
|  SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
|  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.
|
|  Should you have any questions regarding your right to use this Software,
|  contact Texas Instruments Incorporated at www.TI.com.
|
+------------------------------------------------------------------------------
| Purpose:    DMA configuration and constants
+------------------------------------------------------------------------------
| Decription: This file contains settings and constants used to configure
|             the DMA.
+----------------------------------------------------------------------------*/

/*==== DECLARATION CONTROL ==================================================*/

#ifndef DMA_H
#define DMA_H


/*==== INCLUDES =============================================================*/
#include "hal_types.h"

/*==== CONSTS ===============================================================*/

// Transfer Count (VLEN and LEN)
#define DMA_VLEN_USE_LEN            0x00 // Use LEN for transfer count
#define DMA_VLEN_FIRST_BYTE_P_1     0x01 // Transfer number of uint8s/words commanded by the first uint8/word + 1
#define DMA_VLEN_FIRST_BYTE         0x02 // Transfer number of uint8s/words commanded by the first uint8/word
#define DMA_VLEN_FIRST_BYTE_P_2     0x03 // Transfer number of uint8s/words commanded by the first uint8/word + 2
#define DMA_VLEN_FIRST_BYTE_P_3     0x04 // Transfer number of uint8s/words commanded by the first uint8/word + 3

// uint8 or Word Transfer (WORDSIZE)
#define DMA_WORDSIZE_BYTE           0x00 // Transfer a uint8 at a time
#define DMA_WORDSIZE_WORD           0x01 // Transfer a 16-bit word at a time

// Trigger Event (TRIG)
#define DMA_TRIG_NONE        0 // No trigger, setting DMAREQ.DMAREQx bit starts a single uint8/word transfer or a block transfer
#define DMA_TRIG_PREV        1 // DMA channel is triggered by completion of previous channel
#define DMA_TRIG_T1_CH0      2 // Timer 1, compare, channel 0
#define DMA_TRIG_T1_CH1      3 // Timer 1, compare, channel 1
#define DMA_TRIG_T1_CH2      4 // Timer 1, compare, channel 2
#define DMA_TRIG_T2_OVFL     6 // Timer 2, timer count reaches 0x00
#define DMA_TRIG_T3_CH0      7 // Timer 3, compare, channel 0
#define DMA_TRIG_T3_CH1      8 // Timer 3, compare, channel 1
#define DMA_TRIG_T4_CH0      9 // Timer 4, compare, channel 0
#define DMA_TRIG_T4_CH1     10 // Timer 4, compare, channel 1
#define DMA_TRIG_ST         11 // Sleep Timer compare
#define DMA_TRIG_IOC_0      12 // Port 0 I/O pin input transition
#define DMA_TRIG_IOC_1      13 // Port 1 I/O pin input transition
#define DMA_TRIG_URX0       14 // USART0 RX complete
#define DMA_TRIG_UTX0       15 // USART0 TX complete
#define DMA_TRIG_URX1       16 // USART1 RX complete
#define DMA_TRIG_UTX1       17 // USART1 TX complete
#define DMA_TRIG_FLASH      18 // Flash data write complete
#define DMA_TRIG_RADIO      19 // RF packet uint8 received/transmit
#define DMA_TRIG_ADC_CHALL  20 // ADC end of a conversion in a sequence, sample ready
#define DMA_TRIG_ADC_CH0    21 // ADC end of conversion (AIN0, single-ended or AIN0 - AIN1, differential). Sample ready
#define DMA_TRIG_ADC_CH1    22 // ADC end of conversion (AIN1, single-ended or AIN0 - AIN1, differential). Sample ready
#define DMA_TRIG_ADC_CH2    23 // ADC end of conversion (AIN2, single-ended or AIN2 - AIN3, differential). Sample ready
#define DMA_TRIG_ADC_CH3    24 // ADC end of conversion (AIN3, single-ended or AIN2 - AIN3, differential). Sample ready
#define DMA_TRIG_ADC_CH4    25 // ADC end of conversion (AIN4, single-ended or AIN4 - AIN5, differential). Sample ready
#define DMA_TRIG_ADC_CH5    26 // ADC end of conversion (AIN5, single-ended or AIN4 - AIN5, differential). Sample ready
#define DMA_TRIG_ADC_CH6    27 // ADC end of conversion (AIN6, single-ended or AIN6 - AIN7, differential). Sample ready
#define DMA_TRIG_I2S_RX     27 // I2S RX complete
#define DMA_TRIG_ADC_CH7    28 // ADC end of conversion (AIN7, single-ended or AIN6 - AIN7, differential). Sample ready
#define DMA_TRIG_I2S_TX     28 // I2S TX complete
#define DMA_TRIG_ENC_DW     29 // AES encryption processor requests download input data
#define DMA_TRIG_ENC_UP     30 // AES encryption processor requests upload output data

// Transfer Mode (TMODE)
#define DMA_TMODE_SINGLE          0x00 // On a trigger, transfer a single uint8/word and await the next trigger
#define DMA_TMODE_BLOCK           0x01 // On a trigger, transfer number of uint8s/words specified by the transfer count
#define DMA_TMODE_REPEATED_SINGLE 0x02 // On a trigger, transfer a single uint8/word and await the next trigger. Rearm after transfer count is reached
#define DMA_TMODE_REPEATED_BLOCK  0x03 // On a trigger, transfer number of uint8s/words specified by the transfer count and then rearm

// Source Increment (SRCINC)
#define DMA_SRCINC_0    0x00 // Increment source pointer by 0 uint8s/words after each transfer
#define DMA_SRCINC_1    0x01 // Increment source pointer by 1 uint8s/words after each transfer
#define DMA_SRCINC_2    0x02 // Increment source pointer by 2 uint8s/words after each transfer
#define DMA_SRCINC_M1   0x03 // Decrement source pointer by 1 uint8s/words after each transfer

// Destination Increment (DESTINC)
#define DMA_DESTINC_0   0x00 // Increment destination pointer by 0 uint8s/words after each transfer
#define DMA_DESTINC_1   0x01 // Increment destination pointer by 1 uint8s/words after each transfer
#define DMA_DESTINC_2   0x02 // Increment destination pointer by 2 uint8s/words after each transfer
#define DMA_DESTINC_M1  0x03 // Decrement destination pointer by 1 uint8s/words after each transfer

// Interrupt Mask (IRQMASK)
#define DMA_IRQMASK_DISABLE 0x00 // Disable interrupt generation
#define DMA_IRQMASK_ENABLE  0x01 // Enable interrupt generation upon DMA channel done

// Mode 8 Setting (M8) Only applicable when WORDSIZE = 0
#define DMA_M8_USE_8_BITS 0x00 // Use all 8 bits of first uint8 in source data to determine the transfer count
#define DMA_M8_USE_7_BITS 0x01 // Use 7 LSB of first uint8 in source data to determine the transfer count

#define DMA_PRI_LOW     0x00 // DMA access will always defer to a CPU access
#define DMA_PRI_NORMAL  0x01 // Guarantees that DMA access prevails over CPU on at least every second try
#define DMA_PRI_HIGH    0x02 // DMA access will always prevail over CPU access

/*==== TYPES =================================================================*/

// This makes sure the bitfields members go from the most to least significant bit
#pragma bitfields=reversed

// DMA configuration data structure with the correct bit lengths for each value
typedef struct {
   uint8 SRCADDRH;
   uint8 SRCADDRL;
   uint8 DESTADDRH;
   uint8 DESTADDRL;
   uint8 VLEN      : 3;
   uint8 LENH      : 5;
   uint8 LENL      : 8;
   uint8 WORDSIZE  : 1;
   uint8 TMODE     : 2;
   uint8 TRIG      : 5;
   uint8 SRCINC    : 2;
   uint8 DESTINC   : 2;
   uint8 IRQMASK   : 1;
   uint8 M8        : 1;
   uint8 PRIORITY  : 2;
} DMA_DESC;

// There is no longer need for the bitfields to be reversed, so they are set back to default here.
#pragma bitfields=default

/*==== EXPORTS ===============================================================*/

#endif  /* PER_TEST_DMA_H */

/*==== END OF FILE ===========================================================*/
