/* =============================================================================
 * Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 * ==========================================================================*/

/** ===========================================================================
 * @file Timer_example.c
 *
 * @path $(CSLPATH)\example\timer\src
 *
 * @desc Example for the usage of General purpose timer CSL APIs
 *
 *  ============================================================================
 *   @n Target Platform: EVM
 *  ============================================================================
 * @n <b> Example Description </b>
 * @n The example illustartes the usage of CSL APIs for using timer.
 *    1. This example configures the first instance of timer
 *       as interrupt or event to be sourced normally by the timer. 
 *    2. This enables the timer in one shot mode (for watchdog mode timer is
 *       enabled in continuos mode).
 *    3. The  interrupt handler announces occurance of each interupt 
 *       to the application.
 *    4. The above said procedure is followed for all the modes of the timer
 *       (i.e. 64 bit timer, dual 32 bit timer(chained and unchained)
 *       and watchdog).
 *==============================================================================
 *      
 *   <b> Procedure to run the example </b>
 *   @verbatim
 *      1. Configure the CCS setup to work with the emulator being used
 *      2. Please refer CCS manual for setup configuration and loading 
 *         proper GEL file
 *      3. Launch CCS window
 *      4. Open project Timer_example.pjt
 *      5. Build the project and load the .out file of the project.
 *          
 *   @endverbatim
 *   
 * =============================================================================
 */ 

/** ============================================================================
 *  Revision History
 *  =============== 
 *  21-Apr-2005 PSK  - File Created.
 *  29-Jul-2005 PSK  - Updted changes acooriding to revised timer spec. The 
 *                     number convention TIM12, TIM34 are changed to TIMLO and 
 *                     TIMHI.  
 *  30-Nov-2005 NG   - Updated documentation
 *  06-Feb-2006 ds   - Updated example according to C6455 user guide.(Removed 
 *                     tmrClksrcHi and tmrIpGateHi hwsetup configuration)
 *  23-Mar-2006 ds   - Fixed Bug PSG00001172
 * ============================================================================= 
 */


//#include <string.h>
#include <csl_gpio.h>
#include <csl_tmr.h>
#include <stdio.h>
#include <csl_intc.h>
//#include <soc.h>
#include <csl_intcAux.h>
#include <cslr_dev.h>

/* Handle for the TMR instance */
CSL_TmrHandle         hTmr;

/* NOTE: change the Instance of the timer for different timer instances */
Uint8 IntcInstance = CSL_TMR_0;

/* UART TX Buffer */
Uint8 tx_buf[10];
Uint32 Tx_cnt = 0;

/* GPIO */
CSL_GpioHandle hGpio_tx;

int demoFail = 0;

/* Locals & Forwards */
CSL_Status Uart_Send(Uint8 ch);
void tmrIntcPrdLoLoadDemo(void);
void tmrIntcPrdHiLoadDemo(void);
void tmrIntcChainedDemo(void);
void tmrIntcGptDemo(void);
void tmrWdtModeDemo(void);
void Gpio_Init(void);
CSL_Status Gpio_Tx_Init(void);
CSL_Status Gpio_Tx_Set(int value);

/*
 *****************************************************************************
 *                    INTC related code                             
 *****************************************************************************
 */
    
/* Intc Declarations */ 
CSL_IntcObj                  tmrIntcObj, tmrIntcObj1;
CSL_IntcHandle               tmrIntcHandle, tmrIntcHandle1;
CSL_IntcContext              context, context1;

CSL_IntcEventHandlerRecord   eventHandler[30];
CSL_IntcEventHandlerRecord   record[1];

/* count for ISR */
volatile int intrCnt = 0;

/*
 * =============================================================================
 *   @func  TimerInterruptHandler
 *  
 *   @arg
 *      event - interrupt event id
 *
 *   @desc
 *      Handler for Timer Interrupt
 *
 *   @return
 *      NONE
 *
 * =============================================================================
 */
void TimerInterruptHandler (
    CSL_IntcEventId *event
)
{
    CSL_Status  status;
    CSL_TmrEnamode TimeCountMode = CSL_TMR_ENAMODE_ENABLE;
        if (tx_buf[Tx_cnt])
        {
        	Gpio_Tx_Set(1);
        }
        else
        {
            Gpio_Tx_Set(0);
        }
    CSL_intcEventClear(*event);
    /* Stop the Timer */
    status = CSL_tmrHwControl(hTmr, CSL_TMR_CMD_RESET_TIMLO, NULL);
    if (Tx_cnt==9){
    }
    else{
        /* Start the timer with one shot*/
        CSL_tmrHwControl(hTmr, CSL_TMR_CMD_START_TIMLO, (void *)&TimeCountMode);
    }
    Tx_cnt++;
//    printf("%d start done!\n", intrCnt++);
}


// Uart start Send byte ------------------------------------------
CSL_Status Uart_Send(Uint8 ch)
{
	// prepare data and start tx timer
    tx_buf[0] = 0;
    tx_buf[1] = ch>>0 & 1;
    tx_buf[2] = ch>>1 & 1;
    tx_buf[3] = ch>>2 & 1;
    tx_buf[4] = ch>>3 & 1;
    tx_buf[5] = ch>>4 & 1;
    tx_buf[6] = ch>>5 & 1;
    tx_buf[7] = ch>>6 & 1;
    tx_buf[8] = ch>>7 & 1;
    tx_buf[9] = 1;

    Tx_cnt = 0;
    CSL_TmrEnamode TimeCountMode = CSL_TMR_ENAMODE_ENABLE;


    Gpio_Tx_Set(0);
    Gpio_Tx_Set(1);


    /* Start the timer with one shot*/
    return CSL_tmrHwControl(hTmr, CSL_TMR_CMD_START_TIMLO, (void *)&TimeCountMode);
}


// GPIO ------------------------------------------------------------------------------------------------------
void Gpio_Init(void){
    Bool                        gpioEn;

    /* Unlock the control register */
    CSL_FINST(((CSL_DevRegs*)CSL_DEV_REGS)->PERLOCK, DEV_PERLOCK_LOCKVAL,
               UNLOCK);
    /* Enable the GPIO */
    CSL_FINST(((CSL_DevRegs*)CSL_DEV_REGS)->PERCFG0, DEV_PERCFG0_GPIOCTL,
              ENABLE);
    do {
        gpioEn = (Bool) CSL_FEXT(((CSL_DevRegs*)CSL_DEV_REGS)->PERSTAT0,
                                   DEV_PERSTAT0_GPIOSTAT);
    } while (gpioEn != TRUE);

//    printf("Powersaver clock for GPIO is enabled\n");
    return;
}

// Uart TX GPIO Initial --------------------------------
CSL_Status Gpio_Tx_Init(void)
{
    CSL_GpioPinConfig           config;
    CSL_Status                  status;
    CSL_GpioObj                 gpioObj;
    CSL_GpioHwSetup             hwSetup;
    CSL_GpioContext             pContext;

    /* Initialize the GPIO CSL module */
    status = CSL_gpioInit(&pContext);
    if (status != CSL_SOK) {
//        printf("GPIO: Initialization error.\n");
        demoFail++;
        return status;
    }

    /* Open the CSL module */
    hGpio_tx = CSL_gpioOpen(&gpioObj, CSL_GPIO, NULL, &status);
    if ((hGpio_tx == NULL) || (status != CSL_SOK)) {
//        printf("GPIO: Error opening the instance.\n");
        demoFail++;
        return status;
    }

    /* Setup hardware parameters */
    hwSetup.extendSetup = NULL;

    /* Setup the General Purpose IO */
    status = CSL_gpioHwSetup(hGpio_tx, &hwSetup);

    /* Configure pin 4 */
    config.pinNum = CSL_GPIO_PIN4;
    config.trigger = CSL_GPIO_TRIG_RISING_EDGE;
    config.direction = CSL_GPIO_DIR_OUTPUT;
    status = CSL_gpioHwControl(hGpio_tx, CSL_GPIO_CMD_CONFIG_BIT, &config);
    if (status != CSL_SOK) {
//        printf("GPIO: GPIO pin configuration error.\n");
        demoFail++;
        return status;
    }

    return CSL_SOK;
}

// Set TX GPIO port out put ----------------------------------------------
CSL_Status Gpio_Tx_Set(int value)
{
    CSL_GpioPinNum              pinNum;
    CSL_Status              	status;
    pinNum = CSL_GPIO_PIN4;
//    if(value){
//    	CSL_gpioHwControl (hGpio_tx, CSL_GPIO_CMD_SET_BIT, &pinNum);
//    }
//    else{
//    	CSL_gpioHwControl (hGpio_tx, CSL_GPIO_CMD_CLEAR_BIT, &pinNum);
//    }
    if (value)
    {
        status = CSL_gpioHwControl (hGpio_tx, CSL_GPIO_CMD_SET_BIT, &pinNum);
        if (status != CSL_SOK) {
//            printf("GPIO: Command to set bit... Failed.\n");
            demoFail++;
        }
        return status;
    }
    else
    {
        status = CSL_gpioHwControl (hGpio_tx, CSL_GPIO_CMD_CLEAR_BIT, &pinNum);
        if (status != CSL_SOK) {
//            printf("GPIO: Command to set bit... Failed.\n");
            demoFail++;
        }
        return status;
    }
}

/** ===========================================================================
 *  @n@b   main
 *
 *  @b Description
 *  @n This is the main function for the file.This initializes the CSL for 
 *     INTC, installs interrupt handler for first instance of general purpose 
 *     timer and invokes the routine which demonstrates the usage of CSL APis 
 *     for General purpose timer 
 * ============================================================================
 */
void main (
    void
)
{   
    CSL_IntcGlobalEnableState state;
    
    Bool  timer0En;
    
    /* Unlock the control register */
    CSL_FINST(((CSL_DevRegs*)CSL_DEV_REGS)->PERLOCK, DEV_PERLOCK_LOCKVAL, 
               UNLOCK);
               
    /* Enable the Powersaver clock for TIMER */
    CSL_FINST(((CSL_DevRegs*)CSL_DEV_REGS)->PERCFG0, DEV_PERCFG0_TIMER0CTL, 
               ENABLE);

    do {
        timer0En = (Bool) CSL_FEXT(((CSL_DevRegs*)CSL_DEV_REGS)->PERSTAT0, 
                                   DEV_PERSTAT0_TIMER0STAT);
    } while ((timer0En) != TRUE);

    printf("Powersaver clock for TIMER is enabled\n");
    
    /* Initialize INTC */
    context.numEvtEntries = 1;
    context.eventhandlerRecord = record;

    CSL_intcInit(&context);
  
    /* Enable NMIS */
    CSL_intcGlobalNmiEnable();

    /* Enable all interrupts */
    CSL_intcGlobalEnable(&state);
        
    tmrIntcPrdLoLoadDemo();
//    tmrIntcPrdHiLoadDemo();
//    tmrIntcChainedDemo();
//    tmrIntcGptDemo();
//    tmrWdtModeDemo();

    printf(">>>>>> Example for all the modes of timer are \
                          completed  <<<<<<<< \n");
}

/*
 * =============================================================================
 *   @func  tmrIntcPrdLoLoadDemo
 *  
 *   @arg
 *      NONE
 *
 *   @desc
 *      This funtion is example for the Period register loading with value.
 *
 *   @return
 *      NONE
 *
 * =============================================================================
 */
void tmrIntcPrdLoLoadDemo (
    void
)
{
    CSL_TmrObj TmrObj;
    CSL_Status intStat, status;
    CSL_TmrHwSetup hwSetup = CSL_TMR_HWSETUP_DEFAULTS;
    CSL_IntcEventHandlerRecord EventRecord;
    CSL_IntcParam vectId;
  
    Uint32 LoadValue = 0xe100;
    Uint8 SendData = 0;
  
    CSL_TmrEnamode TimeCountMode = CSL_TMR_ENAMODE_ENABLE;

    /* Clear local data structures */
    memset(&TmrObj, 0, sizeof(CSL_TmrObj));
    printf("Running Gp Timer Interrupt DUAL UNCHAINED Low....\n");
    
    /**************************************************************
    * INTC related code                               *
    **************************************************************/
     
    /* Open INTC */
    vectId = CSL_INTC_VECTID_12;

    tmrIntcHandle = CSL_intcOpen(&tmrIntcObj, CSL_INTC_EVENTID_TINTLO0, &vectId, 
                                 &intStat);
    
    /* Bind ISR to Interrupt */
    EventRecord.handler = (CSL_IntcEventHandler)&TimerInterruptHandler;
    EventRecord.arg = (void *)tmrIntcHandle;
    CSL_intcPlugEventHandler(tmrIntcHandle, &EventRecord);
    
    /* Event Enable */
    CSL_intcHwControl(tmrIntcHandle, CSL_INTC_CMD_EVTENABLE, NULL);
       
    /**************************************************************/

    /* Initialize timer CSL module */
    CSL_tmrInit(NULL);
    
    hTmr =  CSL_tmrOpen(&TmrObj, IntcInstance, NULL, &status);

    CSL_tmrHwSetup(hTmr, &hwSetup);

    /* Stop the Timer */
    status = CSL_tmrHwControl(hTmr, CSL_TMR_CMD_RESET_TIMLO, NULL);
    
    /* set the timer mode to unchained dual mode */
    hwSetup.tmrTimerMode = CSL_TMR_TIMMODE_DUAL_UNCHAINED;

    CSL_tmrHwSetup(hTmr, &hwSetup); 

    /* Load the period register */
    CSL_tmrHwControl(hTmr, CSL_TMR_CMD_LOAD_PRDLO, (void *)&LoadValue);

//    /* Start the timer with one shot*/
//    CSL_tmrHwControl(hTmr, CSL_TMR_CMD_START_TIMLO, (void *)&TimeCountMode);
//    printf("1st start done!\n");
//
//
//    /* INTC related code */
//    while (1) {
//        if (intrCnt > 1){
//            /* Stop the Timer */
//            status = CSL_tmrHwControl(hTmr, CSL_TMR_CMD_RESET_TIMLO, NULL);
//
//            break;
//        }
//    }

    Gpio_Init();
    Gpio_Tx_Init();

    Gpio_Tx_Set(0);
    Gpio_Tx_Set(1);

    while(1)
    {
        Uart_Send(SendData++);
        while(Tx_cnt < 10){
            intrCnt = 0;
        }
    }


      
    /**************************************************************/
    
    CSL_intcHwControl(tmrIntcHandle, CSL_INTC_CMD_EVTDISABLE, NULL);
    
    printf("INTR: The Total number of Events occured are: 0x%d\n", intrCnt);
  
    intrCnt = 0;
  
    /* Stop the Timer */
    CSL_tmrHwControl(hTmr, CSL_TMR_CMD_RESET_TIMLO, NULL);
    
    printf("Example for TIMLO completed\n");

    /* Close the Tmr instance */
    CSL_tmrClose(hTmr);
    CSL_intcClose(tmrIntcHandle);    
}

