/*H***************************************************************************
*
* $Archive:: /boards/dsk6455_v1/examples/dsk_app/dsk_app.c                    $
* $Revision:: 3                                                               $
* $Date:: 4/12/06 11:53a                                                      $
* $Author:: Shilpab                                                           $
*
* DESCRIPTION:
*
*   
* (C) Copyright 2006 by Spectrum Digital Incorporated
* All rights reserved
*
*H***************************************************************************/

/*
 *  dsk_app.c (version 1.00)
 *
 *  This example performs a buffered ping-pong audio transfer using the EDMA3
 *  controller and the AIC23 codec (which is connected through McBSP1).  The
 *  implementation follows the set of the ping-pong buffering configuration
 *  included in the "Peripheral Servicing Example" of the EDMA3 Controller
 *  User's Guide (SPRU966) while using a CSL code flow derived from the CSL
 *  interrupt driven EDMA example included in:
 *
 *  C:\CCStudio_v3.2\boards\dsk6455_v1\csl_c6455\example\edma\edma_interrupt
 *
 *  A ping-pong transfer uses a pair of data buffers for input and another pair
 *  of buffers for output rather than a single buffer for each.  Multiple buffers
 *  allow the DSP to process data in one buffer while data is being transmitted/
 *  received through another buffer to allow the computation to be scheduled at
 *  any time durring buffer transmission rather than between samples (individual
 *  buffer elements).
 *
 *  Resource Usage Summary:
 *
 *  DMA channel 14 (McBSP1 transmit, defined in the DSP datasheet)
 *  DMA channel 15 (McBSP1 receive, defined in the DSP datasheet)
 *  EDMA Transfer Controller 1 (with associated region 1, event queue 1)
 *  PaRAM sets 12 (transmit), 66 (transmit pong reload) and 67 (transmit ping reload)
 *  PaRAM sets 13 (receive), 64 (receive pong reload) and 65 (receive ping reload)
 *  Interrupt Event 72 (CC_INT1)
 *
 *  Two DMA channels are used, one for transmit and one for receive.  Each DMA channel
 *  is hard wired to a fixed peripheral event (McBSP1 transmit and McBSP1 receive in
 *  this case).  When a DMA event is detected, the event is passed to the event
 *  queue of Transfer Controller #1.  The choice of which DMA channel maps to which
 *  transfer controller is made in the DMAQNUMn registers.  As the event is received,
 *  the event index is passed through the DMA Channel to PaRAM Mapping Registers
 *  (DCHMAPn) to arrive at the PaRAM set that will respond to the event.
 *
 *  [This discussion applies to the transmit side only, the receive side is equivalent]
 * 
 *  The event will trigger a Transfer Request (TR) that causes one ACNT sized element
 *  of data to be transferred between the current SRC and DST pointers in the PaRAM set.
 *  In this example, the target of the McBSP transmit events is PaRAM set 12 whose
 *  source is a memory buffer and destination is the McBSP's transmit data register.
 *  After the transfer is completed, the ACNT, BCNT and CCNT fields are incremented/
 *  decremented.  When the McBSP is finished transmitting the data, a new event will
 *   be generated and the process will repeat.
 *
 *  The transfer is considered complete when the ACNT, BCNT and CCNT fields all reach 0.
 *  At this point, a transfer complete event (specified in the TCC field of the PaRAM
 *  set) is generated.  This event is masked through the Event Enable Register (EER)
 *  and Interrupt Enable Register (IER).  If the TCC event is enabled and interrupts
 *  are enabled, an interrupt will be generated.  In this example, both the event and
 *  interrupt associated with the TCC are enabled, and an interrupt will be generated
 *  at the end of each buffer transmission.
 * 
 *  PaRAM set 12 is initially configured at copy data out of the transmit ping buffer
 *  (gBufferXmtPing).  After the interrupt, the PaRAM set contents are reloaded from
 *  the PaRAM set specified in its link field.  In our configuration, the transmit
 *  pong (PaRAM set 66) is loaded.  Its settings are exactly the same, but the source
 *  buffer is different (gBufferXmtPong) and the PaRAM set to be loaded when its
 *  transfer is complete is the ping reload set (PaRAM set 67).  This back and forth
 *  linked transfer implements the ping-pong transfer indefinitely without DSP
 *  intervention.
 *
 *  Other Functions
 *
 *  The example includes a few other functions that are executed in the
 *  background as examples of the multitasking that DSP/BIOS is capable of:
 *
 *  1)  blinkLED() toggles LED #0 every 500ms if DIP switch #0 is depressed.
 *      It is a periodic thread with a period of 500 ticks.
 *
 *  2)  load() simulates a 20-25% dummy load if DIP switch #1 is depressed.
 *      It represents other computation that may need to be done.  It is a
 *      periodic thread with a period of 10ms.
 */


/*
 *  DSP/BIOS is configured using the DSP/BIOS configuration tool.  Settings
 *  for this example are stored in a configuration file called dsk_app.tcf.  At
 *  compile time, Code Composer will auto-generate DSP/BIOS related files
 *  based on these settings.  A header file called dsk_appcfg.h contains the
 *  results of the autogeneration and must be included for proper operation.
 *  The name of the file is taken from dsk_app.tcf and adding cfg.h.
 */
#include "dsk_appcfg.h"

/*
 *  The Board Support Library is divided into several modules, each
 *  of which has its own include file.  The file dsk6455.h must be included
 *  in every program that uses the BSL.  This example also includes
 *  dsk6455_aic23.h because it uses the AIC23 codec module.
 */
#include "dsk6455.h"
#include "dsk6455_led.h"
#include "dsk6455_dip.h"
#include "dsk6455_aic23.h"

#include <stdio.h>
#include <csl.h>
#include <csl_edma3.h>
#include <csl_intc.h>
#include <soc.h>
#include <hwi.h>

#define BUFFSIZE 2048
#define PING     0
#define PONG     1

Int16 gBufferXmtPing[BUFFSIZE]; // Transmit PING buffer
Int16 gBufferXmtPong[BUFFSIZE]; // Transmit PONG buffer
Int16 gBufferRcvPing[BUFFSIZE]; // Receive PING buffer
Int16 gBufferRcvPong[BUFFSIZE]; // Receive PONG buffer

/* Codec configuration settings */
DSK6455_AIC23_Config config = {
    0x0017, // 0 DSK6455_AIC23_LEFTINVOL  Left line input channel volume
    0x0017, // 1 DSK6455_AIC23_RIGHTINVOL Right line input channel volume
    0x01f7, // 2 DSK6455_AIC23_LEFTHPVOL  Left channel headphone volume
    0x01f7, // 3 DSK6455_AIC23_RIGHTHPVOL Right channel headphone volume
    0x0010, // 4 DSK6455_AIC23_ANAPATH    Analog audio path control
    0x0000, // 5 DSK6455_AIC23_DIGPATH    Digital audio path control
    0x0000, // 6 DSK6455_AIC23_POWERDOWN  Power down control
    0x0043, // 7 DSK6455_AIC23_DIGIF      Digital audio interface format
    0x0081, // 8 DSK6455_AIC23_SAMPLERATE Sample rate control
    0x0001  // 9 DSK6455_AIC23_DIGACT     Digital interface activation
};

/* Intc declaration */
CSL_IntcContext                 intcContext; 
CSL_IntcEventHandlerRecord      EventHandler[100];
CSL_IntcObj                     intcObjEdma;
CSL_IntcHandle                  hIntcEdma; 
CSL_IntcGlobalEnableState       state;
CSL_IntcEventHandlerRecord      EventRecord;
CSL_IntcParam                   vectId; 
CSL_Edma3HwDmaChannelSetup      dmahwSetup[CSL_EDMA3_NUM_DMACH] = CSL_EDMA3_DMACHANNELSETUP_DEFAULT;
CSL_Edma3HwSetup                hwSetup = {&dmahwSetup[0],NULL};

/* Globals */

/* Edma handle */
CSL_Edma3Handle     hModule;   

/* PaRAM settings for sets 12 (transmit) and 67 (transmit ping reload) */
CSL_Edma3ParamSetup gParamSetupXmtPing = {  // PaRAM Set Structure for transmit ping buffer
    CSL_EDMA3_OPT_MAKE                      // option -      OPT
        (CSL_EDMA3_ITCCH_DIS,     \
        CSL_EDMA3_TCCH_DIS,       \
        CSL_EDMA3_ITCINT_DIS,     \
        CSL_EDMA3_TCINT_EN,       \
        14, CSL_EDMA3_TCC_NORMAL, \
        CSL_EDMA3_FIFOWIDTH_NONE, \
        CSL_EDMA3_STATIC_DIS,     \
        CSL_EDMA3_SYNC_A,         \
        CSL_EDMA3_ADDRMODE_INCR,  \
        CSL_EDMA3_ADDRMODE_INCR), 
    (Uint32)&gBufferXmtPong,                // srcAddr -     SRC
    CSL_EDMA3_CNT_MAKE(2,BUFFSIZE),         // aCntbCnt -    (ACNT, BCNT)
    (Uint32)0x02900004,                     // dstAddr -     DST
    CSL_EDMA3_BIDX_MAKE(2,0),               // srcDstBidx -  (SRCBIDX, DSTBIDX)
    CSL_EDMA3_LINKBCNTRLD_MAKE(0x4840, 1),  // linkBcntrld - (LINK, BCNTRLD)
    CSL_EDMA3_CIDX_MAKE(0,0),               // srcDstCidx -  (SRCCIDX, DSTCIDX)
    1                                       // cCnt -        CCNT
};

/* PaRAM settings for set 66 (transmit pong reload) */
CSL_Edma3ParamSetup gParamSetupXmtPong = {  // PaRAM Set Structure for transmit pong buffer
    CSL_EDMA3_OPT_MAKE                      // option -      OPT
        (CSL_EDMA3_ITCCH_DIS,     \
        CSL_EDMA3_TCCH_DIS,       \
        CSL_EDMA3_ITCINT_DIS,     \
        CSL_EDMA3_TCINT_EN,       \
        14, CSL_EDMA3_TCC_NORMAL, \
        CSL_EDMA3_FIFOWIDTH_NONE, \
        CSL_EDMA3_STATIC_DIS,     \
        CSL_EDMA3_SYNC_A,         \
        CSL_EDMA3_ADDRMODE_INCR,  \
        CSL_EDMA3_ADDRMODE_INCR), 
    (Uint32)gBufferXmtPing,                 // srcAddr -     SRC
    CSL_EDMA3_CNT_MAKE(2,BUFFSIZE),         // aCntbCnt -    (ACNT, BCNT)
    (Uint32)0x02900004,                     // dstAddr -     DST
    CSL_EDMA3_BIDX_MAKE(2,0),               // srcDstBidx -  (SRCBIDX, DSTBIDX)
    CSL_EDMA3_LINKBCNTRLD_MAKE(0x4860, 1),  // linkBcntrld - (LINK, BCNTRLD)
    CSL_EDMA3_CIDX_MAKE(0,0),               // srcDstCidx -  (SRCCIDX, DSTCIDX)
    1                                       // cCnt -        CCNT
};

/* PaRAM settings for sets 13 (receive) and 65 (receive ping reload) */
CSL_Edma3ParamSetup gParamSetupRcvPing = {  // PaRAM Set Structure for receive ping buffer
    CSL_EDMA3_OPT_MAKE                      // option -      OPT
        (CSL_EDMA3_ITCCH_DIS,     \
        CSL_EDMA3_TCCH_DIS,       \
        CSL_EDMA3_ITCINT_DIS,     \
        CSL_EDMA3_TCINT_EN,       \
        15, CSL_EDMA3_TCC_NORMAL, \
        CSL_EDMA3_FIFOWIDTH_NONE, \
        CSL_EDMA3_STATIC_DIS,     \
        CSL_EDMA3_SYNC_A,         \
        CSL_EDMA3_ADDRMODE_INCR,  \
        CSL_EDMA3_ADDRMODE_INCR), 
    (Uint32)0x02900000,                     // srcAddr -     SRC
    CSL_EDMA3_CNT_MAKE(2,BUFFSIZE),         // aCntbCnt -    (ACNT, BCNT)
    (Uint32)&gBufferRcvPing,                // dstAddr -     DST
    CSL_EDMA3_BIDX_MAKE(0,2),               // srcDstBidx -  (SRCBIDX, DSTBIDX)
    CSL_EDMA3_LINKBCNTRLD_MAKE(0x4800, 1),  // linkBcntrld - (LINK, BCNTRLD)
    CSL_EDMA3_CIDX_MAKE(0,0),               // srcDstCidx -  (SRCCIDX, DSTCIDX)
    1                                       // cCnt -        CCNT
};

/* PaRAM settings for set 64 (transmit pong reload) */
CSL_Edma3ParamSetup gParamSetupRcvPong = {  // PaRAM Set Structure for receive pong buffer
    CSL_EDMA3_OPT_MAKE                      // option -      OPT
        (CSL_EDMA3_ITCCH_DIS,     \
        CSL_EDMA3_TCCH_DIS,       \
        CSL_EDMA3_ITCINT_DIS,     \
        CSL_EDMA3_TCINT_EN,       \
        15, CSL_EDMA3_TCC_NORMAL, \
        CSL_EDMA3_FIFOWIDTH_NONE, \
        CSL_EDMA3_STATIC_DIS,     \
        CSL_EDMA3_SYNC_A,         \
        CSL_EDMA3_ADDRMODE_INCR,  \
        CSL_EDMA3_ADDRMODE_INCR), 
    (Uint32)0x02900000,                     // srcAddr -     SRC
    CSL_EDMA3_CNT_MAKE(2,BUFFSIZE),         // aCntbCnt -    (ACNT, BCNT)
    (Uint32)&gBufferRcvPong,                // dstAddr -     DST
    CSL_EDMA3_BIDX_MAKE(0,2),               // srcDstBidx -  (SRCBIDX, DSTBIDX)
    CSL_EDMA3_LINKBCNTRLD_MAKE(0x4820, 1),  // linkBcntrld - (LINK, BCNTRLD)
    CSL_EDMA3_CIDX_MAKE(0,0),               // srcDstCidx -  (SRCCIDX, DSTCIDX)
    1                                       // cCnt -        CCNT
};

/*
 *  copyData() - Copy one buffer with length elements to another.
 */
void copyData(Int16 *inbuf, Int16 *outbuf, Int16 length)
{
    Int16 i = 0;

	for (i = 0; i < length; i++) {
	    outbuf[i] = inbuf[i];
	}
}

/* ---------------------- Interrupt Service Routine -------------------- */

/*
 *  edmaHwi() - Interrupt service routine for the DMA transfer.  It is
 *              triggered when a complete DMA receive frame has been
 *              transferred.   The edmaHwi ISR is inserted into the interrupt
 *              vector table at compile time through a setting in the DSP/BIOS
 *              configuration under Scheduling --> HWI --> HWI_INT8.  edmaHwi
 *              uses the DSP/BIOS Dispatcher to save register state and make
 *              sure the ISR co-exists with other DSP/BIOS functions.
 */
void edmaHwi(void)
{
	Uint32 intr;
    static Uint32       pingOrPong = PING;
	static Int16        xmtdone = 0, rcvdone = 0;

	/* Check for pending EDMA event interrupts (IPR) */
	intr = *((Uint32*)0x02a01068);

    if (intr & 0x4000)
	{
	    xmtdone = 1;
	}
	if (intr & 0x8000)
	{
	    rcvdone = 1;
	}
	if (xmtdone && rcvdone)
	{
	    if (pingOrPong == PING)
		{

	        pingOrPong = PONG;
	        copyData(gBufferRcvPing, gBufferXmtPing, BUFFSIZE);
			SWI_or(&processBufferSwi, PING);
		} else
		{

		    pingOrPong = PING;
		    copyData(gBufferRcvPong, gBufferXmtPong, BUFFSIZE);
			SWI_or(&processBufferSwi, PONG);
		}
		rcvdone = 0;
		xmtdone = 0;
	}

    /* Clear CPU interrupt event 72 (EVTCLR2) */
	*((Uint32*)0x1800048) = 0x00000100;

	/* Clear processed EDMA event interrupts (ICR) */
	*((Uint32*)0x02a01070) = intr; 
}





/* ------------------------------- Threads ------------------------------ */

/*
 *  processBuffer() - Process audio data once it has been received.            
 */
void processBuffer(void)
{
    Uint32 pingPong;
    
    /* Get contents of mailbox posted by edmaHwi */
    pingPong =  SWI_getmbox();

    /* Copy data from transmit to receive, could process audio here */
    if (pingPong == PING) { 
        /* Toggle + #3 as a visual cue */
        DSK6455_LED_toggle(3);
        
        /* Copy receive PING buffer to transmit PING buffer */
        copyData(gBufferRcvPing, gBufferXmtPing, BUFFSIZE);
    } else {
        /* Toggle LED #2 as a visual cue */
        DSK6455_LED_toggle(2);
        
        /* Copy receive PONG buffer to transmit PONG buffer */
        copyData(gBufferRcvPong, gBufferXmtPong, BUFFSIZE);
    }
}

/*
 *  blinkLED() - Periodic thread (PRD) that toggles LED #0 every 500ms if 
 *               DIP switch #0 is depressed.  The thread is configured
 *               in the DSP/BIOS configuration tool under Scheduling -->
 *               PRD --> PRD_blinkLed.  The period is set there at 500
 *               ticks, with each tick corresponding to 1ms in real
 *               time.
 */
void blinkLED(void)
{
    /* Toggle LED #0 if DIP switch #0 is off (depressed) */
    if (!DSK6455_DIP_get(0))
        DSK6455_LED_toggle(0);
}

/*
 *  load() - PRD that simulates a roughly 20% dummy CPU load if
 *           DIP switch #1 is depressed.  The thread is configured in
 *           the DSP/BIOS configuration tool under Scheduling --> PRD
 *           PRD_load.  The period is set there at 10 ticks, which each tick
 *           corresponding to 1ms in real time.
 */
void load(void)
{
    volatile Uint32 i;  

    if (!DSK6455_DIP_get(1))
        for (i = 0; i < 100000; i++);
}

/*
 *  main() - Main code routine, initializes codec and starts EMDA transfer
 */

void edmaInit()
{
    CSL_Edma3Obj                edmaObj;
    CSL_Edma3ParamHandle        hParamBasic1;
    CSL_Edma3ChannelObj         chObjXmt, chObjRcv;
    CSL_Edma3CmdIntr            regionIntr;
    CSL_Edma3CmdDrae            regionAccess;
    CSL_Edma3ChannelHandle      hChannelXmt, hChannelRcv;
    CSL_Edma3Context            context;
    CSL_Edma3ChannelAttr        chAttrXmt, chAttrRcv;
    CSL_Status                  status;   

    /* Disable global interrupts */
    HWI_disable();

	/* Install HWI handler */
	HWI_dispatchPlug(4, &edmaHwi, 0, NULL);
    HWI_eventMap(4, 72);
	C64_enableIER(0x10);

    /* Module initialization */
    status = CSL_edma3Init(&context);
    if (status != CSL_SOK) {
        printf ("Edma module initialization failed\n");   
        return;
    }
    
    /* Edma module open */
    hModule = CSL_edma3Open(&edmaObj,CSL_EDMA3,NULL,&status);
        
    /* Setup the DRAE masks
     * DRAE enable(Bits 0-15) for the shadow region 1.
     */
    regionAccess.region = CSL_EDMA3_REGION_1 ;
    regionAccess.drae =   0xFFFFFFFF ;   
    regionAccess.draeh =  0xFFFFFFFF ;
    CSL_edma3HwControl(hModule,CSL_EDMA3_CMD_DMAREGION_ENABLE, &regionAccess); 

    /* DMA chanel --> PaRAM mapping, queue assignment */
    dmahwSetup[14].paramNum = 12;
	dmahwSetup[14].que =      CSL_EDMA3_QUE_1;
	dmahwSetup[15].paramNum = 13;
	dmahwSetup[15].que =      CSL_EDMA3_QUE_1;
    CSL_edma3HwSetup(hModule,&hwSetup);

    /* --- Setup Transmit Channel (McBSP1 transmit, param 12) --- */

    /* Channel open */
    chAttrXmt.regionNum = CSL_EDMA3_REGION_1;
    chAttrXmt.chaNum = CSL_EDMA3_CHA_XEVT1;
    hChannelXmt = CSL_edma3ChannelOpen(&chObjXmt, CSL_EDMA3, &chAttrXmt, &status);   
        
    /* Set up parameter block 67 as EDMA transmit ping reload */
    hParamBasic1 = CSL_edma3GetParamHandle(hChannelXmt,67,NULL);        
    status = CSL_edma3ParamSetup(hParamBasic1,&gParamSetupXmtPing);

	/* Set up parameter block 66 as EDMA transmit pong reload */
    hParamBasic1 = CSL_edma3GetParamHandle(hChannelXmt,66,NULL);
    CSL_edma3ParamSetup(hParamBasic1,&gParamSetupXmtPong);

	/* Set up parameter block 12 as EDMA transmit (start with copy of ping) */
    hParamBasic1 = CSL_edma3GetParamHandle(hChannelXmt,12,NULL);
    CSL_edma3ParamSetup(hParamBasic1,&gParamSetupXmtPing);

	/* Set up channel and queue relationships */
    CSL_edma3HwChannelSetupParam(hChannelXmt, 12);
	CSL_edma3HwChannelSetupQue(hChannelXmt, CSL_EDMA3_QUE_1);


    /* --- Setup Receive Chan vnel (McBSP1 receive, param 13) --- */

    /* Channel open */
    chAttrRcv.regionNum = CSL_EDMA3_REGION_1;
    chAttrRcv.chaNum = CSL_EDMA3_CHA_REVT1;
    hChannelRcv = CSL_edma3ChannelOpen(&chObjRcv, CSL_EDMA3, &chAttrRcv, &status);   
        
    /* Set up parameter block 65 as EDMA transmit ping reload */
    hParamBasic1 = CSL_edma3GetParamHandle(hChannelRcv,65,NULL);        
    status = CSL_edma3ParamSetup(hParamBasic1,&gParamSetupRcvPing);

	/* Set up parameter block 64 as EDMA transmit pong reload */
    hParamBasic1 = CSL_edma3GetParamHandle(hChannelRcv,64,NULL);
    CSL_edma3ParamSetup(hParamBasic1,&gParamSetupRcvPong);

	/* Set up parameter block 13 as EDMA transmit (start with copy of ping) */
    hParamBasic1 = CSL_edma3GetParamHandle(hChannelRcv,13,NULL);
    CSL_edma3ParamSetup(hParamBasic1,&gParamSetupRcvPing);

	/* Set up channel and queue relationships */
    CSL_edma3HwChannelSetupParam(hChannelRcv, 13);
	CSL_edma3HwChannelSetupQue(hChannelRcv, CSL_EDMA3_QUE_1);

    /* Enable EDMA region interrupts */
    regionIntr.region = CSL_EDMA3_REGION_1 ;
    regionIntr.intr  = 0x0000C000;
    regionIntr.intrh = 0x00000000;
    CSL_edma3HwControl(hModule,CSL_EDMA3_CMD_INTR_ENABLE,&regionIntr);    

	/* Transmit event clear and enable */
	CSL_edma3HwChannelControl(hChannelXmt,CSL_EDMA3_CMD_CHANNEL_CLEAR,NULL);
    CSL_edma3HwChannelControl(hChannelXmt,CSL_EDMA3_CMD_CHANNEL_ENABLE,NULL);

	/* Receive event clear and enable */
	CSL_edma3HwChannelControl(hChannelRcv,CSL_EDMA3_CMD_CHANNEL_CLEAR,NULL);
    CSL_edma3HwChannelControl(hChannelRcv,CSL_EDMA3_CMD_CHANNEL_ENABLE,NULL);

    /* Clear CPU interrupt event 72 (EVTCLR2) */
	*((Uint32*)0x1800048) = 0x00000100;

    /* Enable CPU interrupt event 72 (EVTMASK2) */
	*((Uint32*)0x1800088) = 0x00000100;

	/* Re-enable global interrupts */
    HWI_enable();

    /* Manually trigger the transmit channel */
    status = CSL_edma3HwChannelControl(hChannelXmt,CSL_EDMA3_CMD_CHANNEL_SET,NULL);

    /* Manually trigger the receive channel */
    status = CSL_edma3HwChannelControl(hChannelRcv,CSL_EDMA3_CMD_CHANNEL_SET,NULL);
}


void main()
{
	DSK6455_AIC23_CodecHandle hCodec;

    /* Initialize the board support library, must be called first */
    DSK6455_init();

    /* Initialize LEDs and DIP switches */
    DSK6455_LED_init();
    DSK6455_DIP_init();
    
    /* Clear buffers */
    memset((void*)gBufferXmtPing,0,sizeof(gBufferXmtPing));
    memset((void*)gBufferXmtPong,0,sizeof(gBufferXmtPong));
    memset((void*)gBufferRcvPing,0,sizeof(gBufferRcvPing));
    memset((void*)gBufferRcvPong,0,sizeof(gBufferRcvPong));

	/* Start the codec */
    hCodec = DSK6455_AIC23_openCodec(0, &config);

	/* Start the EDMA controller */
	edmaInit();
}
