/*H***************************************************************************
*
* $Archive:: /boards/dsk6455_v1/examples/post/post.c                         $
* $Revision:: 1                                                               $
* $Date:: 4/11/06 9:43a                                                       $
* $Author:: Shilpab                                                           $
*
* DESCRIPTION:
*
*   
* (C) Copyright 2005 by Spectrum Digital Incorporated
* All rights reserved
*
*H***************************************************************************/

/*------------------------------------------------------------------------*
 *  ======== post.c ========
 *
 *  The Power On Self Test (POST) is a small program that uses the 6455 DSK
 *  Board Support Library to perform basic tests on the DSP and board
 *  peripherals.  It is typically programmed into the Flash memory so it
 *  runs on boot.  The following tests are performed:
 *
 *  Index    Description
 *    1        LED Test
 *    2        Codec Test
 *    3        External memory Test
 *    4        Flash ID test
 *    5        Flash test
 *    6        EEPROM test
 *
 *  The POST displays the index of the test being run on the LEDs.  If an
 *  error is detected while running the test, the test index will blink
 *  continuously.  If all tests complete successfully, all 4 LEDs will blink
 *  three times and stop will all LEDs on.
 *
 *  This code will only work when built with the large memory model because
 *  of the need to access the external SDRAM.
 *
 *  Please see the 6455 DSK help file under Software/Examples for more
 *  detailed information.
 *

 *  DSP/BIOS is configured using the DSP/BIOS configuration tool.  Settings
 *  for this example are stored in a configuration file called post.cdb.  At
 *  compile time, Code Composer will auto-generate DSP/BIOS related files
 *  based on these settings.  A header file called postcfg.h contains the
 *  results of the autogeneration and must be included for proper operation.
 *  The name of the file is taken from post.cdb and adding cfg.h.
 *------------------------------------------------------------------------*/

#define post_c

/*****************************************************************************
* INCLUDE FILES
*****************************************************************************/
/*---- system and platform files -------------------------------------------*/

#include "postcfg.h"

/*
 *  The 6455 DSK Board Support Library is divided into several modules, each
 *  of which has its own include file.  The file dsk6455.h must be included
 *  in every program that uses the BSL.  This example also includes
 *  dsk6455_aic23.h, dsk6455_led.h and dsk6455_flash.h because it uses
 *  their respective BSL modules.
 */
#include "dsk6455.h"
#include "dsk6455_led.h"
#include "dsk6455_aic23.h"
#include "dsk6455_flash.h"
#include "dsk6455_eeprom.h"


/* Length of sine wave table */
#define SINE_TABLE_SIZE  48

Uint16 buffer[256];

/* Pre-generated sine wave data, 16-bit signed samples */
int sinetable[SINE_TABLE_SIZE] = {
    0x0000, 0x10b4, 0x2120, 0x30fb, 0x3fff, 0x4dea, 0x5a81, 0x658b,
    0x6ed8, 0x763f, 0x7ba1, 0x7ee5, 0x7ffd, 0x7ee5, 0x7ba1, 0x76ef,
    0x6ed8, 0x658b, 0x5a81, 0x4dea, 0x3fff, 0x30fb, 0x2120, 0x10b4,
    0x0000, 0xef4c, 0xdee0, 0xcf06, 0xc002, 0xb216, 0xa57f, 0x9a75,
    0x9128, 0x89c1, 0x845f, 0x811b, 0x8002, 0x811b, 0x845f, 0x89c1,
    0x9128, 0x9a76, 0xa57f, 0xb216, 0xc002, 0xcf06, 0xdee0, 0xef4c
};

/* Codec configuration settings */
DSK6455_AIC23_Config config = { \
    0x0017,  /* 0 DSK6455_AIC23_LEFTINVOL  Left line input channel volume */ \
    0x0017,  /* 1 DSK6455_AIC23_RIGHTINVOL Right line input channel volume */\
    0x00d8,  /* 2 DSK6455_AIC23_LEFTHPVOL  Left channel headphone volume */  \
    0x00d8,  /* 3 DSK6455_AIC23_RIGHTHPVOL Right channel headphone volume */ \
    0x0011,  /* 4 DSK6455_AIC23_ANAPATH    Analog audio path control */      \
    0x0000,  /* 5 DSK6455_AIC23_DIGPATH    Digital audio path control */     \
    0x0000,  /* 6 DSK6455_AIC23_POWERDOWN  Power down control */             \
    0x0043,  /* 7 DSK6455_AIC23_DIGIF      Digital audio interface format */ \
    0x0081,  /* 8 DSK6455_AIC23_SAMPLERATE Sample rate control */            \
    0x0001   /* 9 DSK6455_AIC23_DIGACT     Digital interface activation */   \
};


/*--------------------------------ACCESSORY FUNCTIONS-------------------------------------*/
/*F***************************************************************************
* NAME:  TEST_sleep(Int16 sleeptime) 
*
* DESCRIPTION: Accessory Function
*
* NOTES:
*   
*F***************************************************************************/
void TEST_sleep(Int16 sleeptime)
{
	DSK6455_waitusec( sleeptime * 1000 );
}
/*----------------------------END ACCESSORY FUNCTIONS-------------------------------------*/
/*-----------------------------LED FUNCTIONS-----------------------------------------------*/
/*F***************************************************************************
* NAME:  LED_binary(Int16 ledmask) 
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
void LED_binary(Int16 ledmask)
{
    Int16 i, bit;

    /* Walk through the bits in num setting corresponding LEDs */
    bit = 1;
    for (i = 0; i < 4; i++)
    {
        if (ledmask & bit)
            DSK6455_LED_on(i);
        else
            DSK6455_LED_off(i);
        bit = bit << 1;
    }

}

/*F***************************************************************************
* NAME:  LED_blink(Int16 ledmask, Int16 count)  
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
void LED_blink(Int16 ledmask, Int16 count)
{
    while (count > 0)
    {
        LED_binary(ledmask);
        TEST_sleep(100);
        LED_binary(0);
        TEST_sleep(150);
        count--;
    }
}

/*F***************************************************************************
* NAME:  void LED_error(Int16 ledmask)  
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
void LED_error(Int16 ledmask)
{
    while(1)
        LED_blink(ledmask, 1);
}
/*-----------------------------END OF LED FUNCTIONS-----------------------------------------------*/
/*-----------------------------MEMORY FUNCTIONS---------------------------------------------------*/
/*F***************************************************************************
* NAME:  MEM_fill(Uint32 start, Uint32 len, Uint32 val) 
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 MEM_fill(Uint32 start, Uint32 len, Uint32 val)
{
    Uint32 i, end;

    /* Calculate end of range */
    end = start + len;

    /* Fill a range with a value */
    for (i = start; i < end; i+=sizeof(Uint32))
    {
        *((Uint32 *)i) = val;
    }

    /* Verify the data */
    for (i = start; i < end; i+=sizeof(Uint32))
    {
        if (*((Uint32 *)i) != val)
            return 1;
    }

    return 0;
}
/*F***************************************************************************
* NAME:  MEM_addr(Uint32 start, Uint32 len)  
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 MEM_addr(Uint32 start, Uint32 len)
{
    Uint32 i, end;

    /* Calculate end of range */
    end = start + len;

    /* Fill the range with its address */
    for (i = start; i < end; i+=sizeof(Uint32))
    {
        *((Uint32 *)i) = i;
    }

    /* Verify the data */
    for (i = start; i < end; i+=sizeof(Uint32))
        if (*((Uint32 *)i) != i)
            return 1;

    return 0;
}
/*F***************************************************************************
* NAME: MEM_addrInv(Uint32 start, Uint32 len)  
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 MEM_addrInv(Uint32 start, Uint32 len)
{
    Uint32 i, end;

    /* Calculate end of range */
    end = start + len;

    /* Fill the range with its address */
    for (i = start; i < end; i+=sizeof(Uint32))
    {
        *((Uint32 *)i) = ~i;
    }

    /* Verify the data */
    for (i = start; i < end; i+=sizeof(Uint32))
        if (*((Uint32 *)i) != (~i))
            return 1;

    return 0;
}

/*F***************************************************************************
* NAME:  MEM_test(Uint32 start, Uint32 len, Int16 patterntype)  
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 MEM_test(Uint32 start, Uint32 len, Int16 patterntype)
{
    Int16 status = 0;

    if (!patterntype)
    {
        /* Run the fill tests */
        status |= MEM_fill(start, len, 0x00000000);
        status |= MEM_fill(start, len, 0x55555555);
        status |= MEM_fill(start, len, 0xAAAAAAAA);
        status |= MEM_fill(start, len, 0xFFFFFFFF);
    } else
    {
        /* Run the address tests */
        status |= MEM_addr(start, len);
        status |= MEM_addrInv(start, len);
    }

    return status;
}
/*F***************************************************************************
* NAME:  MEM_walking(Uint32 add)
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 MEM_walking(Uint32 add)
{
    Int16 i;
    Uint32 mask, *pdata;

    pdata = (Uint32 *)add;

    /* Walking ones and zeros */
    mask = 1;
    for (i = 0; i < 32; i++)
    {
        /* Test one in bit position i */
        *pdata = mask;
        if (*pdata != mask)
            return 1;

        /* Test zero in bit position i */
        *pdata = ~mask;
        if (*pdata != (~mask))
            return 1;

        mask = mask << 1;
    }

    return 0;
}

/*F***************************************************************************
* NAME:  MEM_walking(Uint32 add)
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 MEM_bytestrobe(Uint32 add)
{
    Uint32 *pdata;
    Uint8 *pcheck;

    /* Write pattern */
    pdata = (Uint32 *)add;
    *pdata = 0x12345678;

    /* Do dummy write */
    pdata = (Uint32 *)DSK6455_FLASH_BASE;
    *pdata = 0x80808080;

    /* Check pattern */
    pcheck = (Uint8 *)add;
    if (*pcheck++ != 0x78)
        return 0x10;
    if (*pcheck++ != 0x56)
        return 0x20;
    if (*pcheck++ != 0x34)
        return 0x40;
    if (*pcheck++ != 0x12)
        return 0x80;

    return 0;
}

/*---------------------------END OF MEMORY FUNCTIONS------------------------------------------*/
/*---------------------------START OF POST TESTS----------------------------------------------*/
/*F***************************************************************************
* NAME:  TEST_led()
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/

Int16 TEST_led()
{
    Uint32 i, j;

    DSK6455_LED_init();
    for (i = 0; i < 20; i++)
    {
        LED_binary(0xaa);
        for (j = 0; j < 1000000; j++);
        LED_binary(0x55);
        for (j = 0; j < 1000000; j++);
    }
    LED_binary(0xff);

    return 0;
}
/*F***************************************************************************
* NAME:  TEST_extMem()
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/

Int16 TEST_extMem()
{
    Int16 status = 0;
    
    /* Check external memory (byte 0xe0000000 to byte 0x1000000) */
    status |= MEM_test(0xe0000000, 0x1000000, 0);
    status |= MEM_test(0xe0000000, 0x1000000, 1);
    status |= MEM_walking(0xe0000000);
    status |= MEM_walking(0xe0000004);
    status |= MEM_bytestrobe(0xE0000000);
     
    return status;
}
/*F***************************************************************************
* NAME:  TEST_codec()
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/

Int16 TEST_codec()
{
    DSK6455_AIC23_CodecHandle hCodec;
    Int16 i, j;

    /* Start the codec */
    hCodec = DSK6455_AIC23_openCodec(0, &config);

    /* Generate a 1KHz sine wave for 1 second */
    for (i = 0; i < 1000; i++)
    {
        for (j = 0; j < SINE_TABLE_SIZE; j++)
        {
            while (!DSK6455_AIC23_write32(hCodec, sinetable[j]));
            while (!DSK6455_AIC23_write32(hCodec, sinetable[j]));
        }
    }

    /* Close codec */
    DSK6455_AIC23_closeCodec(hCodec);

    return 0;
}

/*F***************************************************************************
* NAME:  TEST_flashID()
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 TEST_flashID()
{
    Uint8 MfgId,DevId;
    Int16 i;

    /* Reset flash */
    *((Uint8 *)DSK6455_FLASH_BASE) = 0xf0;

    /* Configure to read manufacturer ID */
    *((Uint8 *)DSK6455_FLASH_BASE) = 0xaa;
    *((Uint8 *)DSK6455_FLASH_BASE) = 0x55;
    *((Uint8 *)DSK6455_FLASH_BASE) = 0x90;

    /* Insert small delay for device to respond */
    for (i = 0; i < 10; i++);

    /* Read IDs */
    MfgId = *((Uint8 *)DSK6455_FLASH_BASE);
    DevId = *((Uint8 *)DSK6455_FLASH_BASE + 1);

    /* Reset flash */
    *((Uint8 *)DSK6455_FLASH_BASE) = 0xf0;

    /* Check IDs */
    if ((MfgId != 0x0001) || (DevId != 0xa3))
        return 1;

    /* Test passed */
    return 0;
}

/*F***************************************************************************
* NAME:  TEST_flash()  
*
* DESCRIPTION:
*
* NOTES:CALLED BY FLASH DIAGNOSTICS IN THE DIAGNOSTIC UTILITY
*   
*F***************************************************************************/
Int16 TEST_flash()
{
    Uint16 i, j, k, page;
    Uint32 startaddr, addr;

    /* Program the Flash page by page, 256 words at a time */
    for (page = 0; page < 1; page++)
    {
        /* Erase all of the Flash */
        DSK6455_FLASH_erase(DSK6455_FLASH_BASE + DSK6455_FLASH_SECTORSIZE, DSK6455_FLASH_SECTORSIZE *8); 

        addr = DSK6455_FLASH_BASE + DSK6455_FLASH_SECTORSIZE;
        for (i = 0; i < 8; i++)
        {
            for (j = 0; j < 128; j++)
            {
                startaddr = addr;
                for (k = 0; k < 256; k++)
                {
                    buffer[k] = addr + i + page;
                    addr += 2;
                }
                DSK6455_FLASH_write((Uint32)buffer, startaddr, 512);
            }
        }
    }

    /* Verify the Flash page by page, 256 words at a time */
    for (page = 0; page < 1; page++)
    {
        addr = DSK6455_FLASH_BASE + DSK6455_FLASH_SECTORSIZE;
        for (i = 0; i < 8 ; i++)
        {
            for (j = 0; j < 128; j++)
            {
                startaddr = addr;
                DSK6455_FLASH_read(startaddr, (Uint32)buffer, 512);
                for (k = 0; k < 256; k++)
                {
                    if (buffer[k] != ((addr + i + page) & 0xffff))
                        return page + 1;
                    addr += 2;
                }
            }
        }
    }

    return 0;
  
}

/*F***************************************************************************
* NAME:  TEST_eeprom()  
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/

Int16 TEST_eeprom()
{
    Uint16 page, i;
    Uint8 *pdata;

	DSK6455_EEPROM_init();

    /* Write a pattern */
    for (page = 0; page < 4; page++)
    {
        pdata = (Uint8 *)buffer;
        for (i = 0; i < 64; i++)
            *pdata++ = (page + i) & 0xff;
        DSK6455_EEPROM_write((Uint32)buffer, page << 6, 64);
    }

    /* Verify the pattern */
    for (page = 0; page < 4; page++)
    {
        DSK6455_EEPROM_read(page << 6, (Uint32)buffer, 64);
        pdata = (Uint8 *)buffer;
        for (i = 0; i < 64; i++)
            if (*pdata++ != (page + i) & 0xff)
                return 1;  // Fail
    }

    return 0;
}
/*---------------------------END OF POST TESTS----------------------------------------------*/
/*F***************************************************************************
* NAME: TEST_execute()  
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/

void TEST_execute(Int16 (*funchandle)(), Int16 ledmask, Int16 insertdelay)
{
    Int16 status;
    

    /* Display test ID */
    LED_binary(ledmask);
    
    /* Call test function */
    status = funchandle();

    /* Pause so LEDs can be read */
    if (insertdelay)
        TEST_sleep(100);
    
    /* Check for test fail */
    if (status)
        LED_error(ledmask);
}
/*------------------------------START OF MAIN FUNCTION--------------------------------*/
main()
{
    /* Call BSL init */
    DSK6455_init();

    /* Set initial LED state */
    DSK6455_LED_init();

    /* Run the tests sequentially */

    TEST_execute(TEST_led,     1, 0);  /* LED */
    TEST_execute(TEST_codec,     2, 0);  /* Codec */
    TEST_execute(TEST_extMem,     3, 0);  /* External memory (typical) */
    TEST_execute(TEST_flashID,     4, 0);  /* Flash ID */
    TEST_execute(TEST_flash,      5, 0);   /*flash */
    TEST_execute(TEST_eeprom,      6, 0);   /*eeprom */

   /* All tests complete, board passes */
    LED_blink(0xf, 3);
    
    /* Leave all LEDs on */
    LED_binary(0xf);

    asm( " .long 0x1001E000" );

}
/*------------------------------END OF MAIN FUNCTION--------------------------------*/

