/*H***************************************************************************
*
* $Archive:: /boards/dsk6455_v1/lib/DSK6455BSL/dsk6455m_i2c.c                 $
* $Revision:: 1                                                               $
* $Date:: 4/11/06 9:46a                                                       $
*
* DESCRIPTION: I2C implementation.
* 
* NOTES :  BSL Version# 3.24 
*
* (C) Copyright 2005 by Spectrum Digital Incorporated
* All rights reserved
*
*H***************************************************************************/
#include "dsk6455m.h"
#include "dsk6455m_i2c.h"

/* ------------------------------------------------------------------------ *
 *                                                                          *
 *  DSK6455M_I2C_init( )                                                  *
 *                                                                          *
 *      Enable and initalize the I2C module                                 *
 *                                                                          *
 * ------------------------------------------------------------------------ */
Int16 DSK6455M_I2C_init( )
{
    I2C_ICMDR   = 0;                                // Reset I2C
    I2C_ICPSC   = 125;                              // Run I2C module at 1MHz (input SYSCLK2/6)
    I2C_ICCLKL  = 5;                                // Config clk LOW for 50kHz
    I2C_ICCLKH  = 5;                                // Config clk HIGH for 50kHz
    I2C_ICMDR  |= ICMDR_IRS;                        // Release I2C from reset

    return 0;
}

/* ------------------------------------------------------------------------ *
 *                                                                          *
 *  DSK6455_I2C_close( )                                                 *
 *                                                                          *
 * ------------------------------------------------------------------------ */
Int16 DSK6455M_I2C_close( )
{
    I2C_ICMDR = 0;                                  // Reset I2C
    return 0;
}

/* ------------------------------------------------------------------------ *
 *                                                                          *
 *  DSK6455M_I2C_write( i2caddr, data, len )                              *
 *                                                                          *
 *      I2C write in Master mode                                            *
 *                                                                          *
 *      i2caddr <- I2C slave address                                        *
 *      data    <- I2C data ptr                                             *
 *      len     <- # of bytes to write                                      *
 *                                                                          *
 * ------------------------------------------------------------------------ */
Int16 DSK6455M_I2C_write( Uint16 i2caddr, Uint8* data, Uint16 len )
{
    Uint16 i;

    I2C_ICCNT = len;                                // Set len
    I2C_ICSAR = i2caddr;                            // Set I2C slave address
    I2C_ICMDR = ICMDR_STT                           // Config for master write
              | ICMDR_TRX
              | ICMDR_MST
              | ICMDR_IRS
              | ICMDR_FREE
              ;

    DSK6455_wait( 10 );                                  // Short delay

    for ( i = 0 ; i < len ; i++ )
    {
        I2C_ICDXR = data[i];                        // Write
        while ( ( I2C_ICSTR & ICSTR_ICXRDY ) == 0 );// Wait for Tx Ready
    }

    I2C_ICMDR |= ICMDR_STP;                         // Generate STOP

    return 0;
}

/* ------------------------------------------------------------------------ *
 *                                                                          *
 *  DSK6455M_I2C_read( i2caddr, data, len )                               *
 *                                                                          *
 *      I2C read in Master mode                                             *
 *                                                                          *
 *      i2caddr <- I2C slave address                                        *
 *      data    <- I2C data ptr                                             *
 *      len     <- # of bytes to write                                      *
 *                                                                          *
 * ------------------------------------------------------------------------ */
Int16 DSK6455M_I2C_read( Uint16 i2caddr, Uint8* data, Uint16 len )
{
    Uint16 i;

    I2C_ICCNT = len;                                // Set len
    I2C_ICSAR = i2caddr;                            // Set I2C slave address
    I2C_ICMDR = ICMDR_STT                           // Config for master read
              | ICMDR_MST
              | ICMDR_IRS
              | ICMDR_FREE
              ;

    for ( i = 0 ; i < len ; i++ )
    {
        while ( ( I2C_ICSTR & ICSTR_ICRRDY ) == 0 );// Wait for Rx Ready
        data[i] = I2C_ICDRR;                        // Read
    }

    I2C_ICMDR |= ICMDR_STP;                         // Generate STOP

    return 0;
}
