/****************************************************************************\
 *           Copyright (C) 2012 Texas Instruments Incorporated.             *
 *                           All Rights Reserved                            *
 *                                                                          *
 * GENERAL DISCLAIMER                                                       *
 * ------------------                                                       *
 * All software and related documentation is provided "AS IS" and without   *
 * warranty or support of any kind and Texas Instruments expressly disclaims*
 * all other warranties, express or implied, including, but not limited to, *
 * the implied warranties of merchantability and fitness for a particular   *
 * purpose.  Under no circumstances shall Texas Instruments be liable for   *
 * any incidental, special or consequential damages that result from the    *
 * use or inability to use the software or related documentation, even if   *
 * Texas Instruments has been advised of the liability.                     *
 ****************************************************************************
This example shows I2C master mode operations:
1. read/write I2C EEPROM
2. read I2C temperature sensor
 ****************************************************************************
 * Created by :                                                             *
 *            Brighton Feng                                                 *
 *            Texas Instruments                                             * 
 *            Jan 5, 2012                                                   *
 *  August 24, 2013 Brighton Feng  Update for K2                            *
***************************************************************************/
#include <stdio.h>
#include <csl_bootcfgAux.h>
#include "K2_common.h"
#include "K2_Board_Init.h"
#include "K2_I2C_init_drv.h"
#include "I2C_EEPROM_Test.h"
#include "I2C_loopback_test.h"

//the number of I2C used for external connection test
#define I2C_NUM 		0

#define I2C_LOOPBACK_TEST 		1
#define I2C_EEPROM_TEST 		1
#define I2C_TEMP_SENSOR_TEST 	0

#if I2C_TEMP_SENSOR_TEST
#define I2C_TEMP_SENSOR_ADDRESS 0x48
#endif

//MMU memory ranges tables
MMU_Memory_Map_Range memory_ranges[]=
{
	{//MSMC RAM
    .uiVirtualAddress   = 0x0C000000,
    .ullPhysicalAddress = 0x0C000000,
    .uiByteCnt          = MSMC_RAM_SIZE_BYTES,
    .attribute          = MMU_MEM_ATTR_NORMAL_CACHE_RA_WB_WA,
    .accessPermission   = MMU_MEM_ATTR_RW,
    .exectuePermission  = MMU_MEM_ATTR_X,
    .shareAttr          = MMU_MEM_ATTR_OUTER_SHARE,
    .isGlobal           = MMU_MEM_ATTR_GLOBAL,
    .isSecure           = MMU_MEM_ATTR_SECURE,
	},                     
	{//DDR3A               
    .uiVirtualAddress   = 0x80000000,
    .ullPhysicalAddress = 0x800000000ULL, 	
#if (DDR3A_SIZE_BYTES>0x80000000ULL)
    .uiByteCnt          = 0x80000000, 	
#else
    .uiByteCnt          = DDR3A_SIZE_BYTES, 	
#endif
    .attribute          = MMU_MEM_ATTR_NORMAL_CACHE_RA_WB_WA,
    .accessPermission   = MMU_MEM_ATTR_RW,
    .exectuePermission  = MMU_MEM_ATTR_XN,
    .shareAttr          = MMU_MEM_ATTR_OUTER_SHARE,
    .isGlobal           = MMU_MEM_ATTR_GLOBAL,
    .isSecure           = MMU_MEM_ATTR_SECURE,
	}
};

MMU_Long_Format_Config mmu_cfg=
{
	.memory_map        = memory_ranges,
	.uiNumMemMapRanges = sizeof(memory_ranges)/sizeof(MMU_Memory_Map_Range),
	.ullpMMU3rdLevelTT = NULL, 	//No level 3 translation table
	.bAlignCheck       = TRUE,
	.tableCacheAttr    = MMU_TAB_ATTR_CACHE_WB_WA,
	.tableShareAttr	   = MMU_MEM_ATTR_INNER_SHARE,
};

int main()
{
	int i;
	
	/*common initialization for internal modules in K2 device 
	enable GIC, memory protection interrupts, EDC for MSMC RAM */
	K2_common_device_init();
	/*initialize GIC interface for CPU, enable IRQ, FIQ, PMU*/
	K2_common_CPU_init();
	KeyStone_Exception_cfg(TRUE);

	//Main core speed= MAIN_PLL_REF_CLK_MHZ*MAIN_PLL_MULTIPLIER/MAIN_PLL_DIVISOR
	KeyStone_main_PLL_init(MAIN_PLL_REF_CLK_MHZ, MAIN_PLL_MULTIPLIER, MAIN_PLL_DIVISOR);

#ifndef DEVICE_K2E 	//K2E only has main PLL for both ARM and DSP cores
	//ARM core speed= ARM_PLL_REF_CLK_MHZ*ARM_PLL_MULTIPLIER/ARM_PLL_DIVISOR
	K2_ARM_PLL_init(ARM_PLL_REF_CLK_MHZ, ARM_PLL_MULTIPLIER, ARM_PLL_DIVISOR);
	//K2_ARM_PLL_init(125, 8, 1);
#endif
	K2_DDR3A_config(NULL, NULL);

	/*MMU configure for ARM core*/
	MMU_long_format_init(&mmu_cfg);
	CP15_ICacheEnable();
	CP15_DCacheEnable();

#if I2C_LOOPBACK_TEST
	for(i=0; i< CSL_I2C_PER_CNT; i++)	
		I2C_loopback_test(i);
#endif

	K2_I2C_Master_Init(I2C_NUM, 400);

#if I2C_EEPROM_TEST
	Uint32 uiAddress;
	puts("I2C EEPROM test start...");
	for(uiAddress=I2C_EEPROM_SIZE_BYTES-(4*1024); uiAddress<I2C_EEPROM_SIZE_BYTES; uiAddress+= 1024)
		I2C_EEPROM_Test(I2C_NUM, I2C_EEPROM_ADDRESS, uiAddress, uiAddress+1024);
#endif
	
#if I2C_TEMP_SENSOR_TEST
	TMP_Sensor_init(I2C_NUM, I2C_TEMP_SENSOR_ADDRESS, 75, 80);
	printf("temperature is %d",GetTemperature(I2C_TEMP_SENSOR_ADDRESS));
#endif

	puts("I2C test complete.");

	return 0;
}


