/******************************************************************************
  Copyright (C), 2013, Texas Instrument.
 ******************************************************************************
  File Name     : K2_DDR_Init.c
  Version       : Initial Draft
  Author        : Brighton Feng
  Created       : May 18, 2013
  Last Modified :
  Description   : KeyStone 2 DDR configuration
  History       :
  1.Date        : May 18, 2013
    Author      : Brighton Feng
    Modification: Created file
******************************************************************************/
#ifndef _K2_DDR_INIT_H_
#define _K2_DDR_INIT_H_
#include "K2_common.h"

/**************************************************************************\
* Register Overlay Structure for DDR3 PHY
\**************************************************************************/
/*registers for a data byte*/
typedef struct  {
	volatile Uint32 GCR   ;  
	volatile Uint32 GSR0  ;  
	volatile Uint32 GSR1  ;  
	volatile Uint8 RSVD0[20];
	volatile Uint32 LCDLR0;  
	volatile Uint32 LCDLR1;  
	volatile Uint32 LCDLR2;  
	volatile Uint32 MDLR  ;  
	volatile Uint32 GTR   ;  
	volatile Uint8 RSVD1[12];
}DDR3_DAT_Regs;

/*registers for a data byte*/
typedef struct  {
	volatile Uint32 CR0;  
	volatile Uint32 CR1;  
	volatile Uint32 SR0;  
	volatile Uint32 SR1;  
}DDR3_ZQ_Regs;

typedef struct  {
	volatile Uint8 RSVD0[4];
	volatile Uint32 PIR    ;//DDR3_PHY_CFG_BASE + 0x00000004
	volatile Uint32 PGCR0  ;//DDR3_PHY_CFG_BASE + 0x00000008
	volatile Uint32 PGCR1  ;//DDR3_PHY_CFG_BASE + 0x0000000C
	volatile Uint32 PGSR0  ;//DDR3_PHY_CFG_BASE + 0x00000010
	volatile Uint32 PGSR1  ;//DDR3_PHY_CFG_BASE + 0x00000014
	volatile Uint32 PLLCR  ;//DDR3_PHY_CFG_BASE + 0x00000018
	volatile Uint32 PTR0   ;//DDR3_PHY_CFG_BASE + 0x0000001C
	volatile Uint32 PTR1   ;//DDR3_PHY_CFG_BASE + 0x00000020
	volatile Uint32 PTR2   ;//DDR3_PHY_CFG_BASE + 0x00000024
	volatile Uint32 PTR3   ;//DDR3_PHY_CFG_BASE + 0x00000028
	volatile Uint32 PTR4   ;//DDR3_PHY_CFG_BASE + 0x0000002C
	volatile Uint8 RSVD1[0x00000040-(0x0000002C+4)];
	volatile Uint32 DSGCR  ;//DDR3_PHY_CFG_BASE + 0x00000040
	volatile Uint32 DCR    ;//DDR3_PHY_CFG_BASE + 0x00000044
	volatile Uint32 DTPR0  ;//DDR3_PHY_CFG_BASE + 0x00000048
	volatile Uint32 DTPR1  ;//DDR3_PHY_CFG_BASE + 0x0000004C
	volatile Uint32 DTPR2  ;//DDR3_PHY_CFG_BASE + 0x00000050
	volatile Uint32 MR0    ;//DDR3_PHY_CFG_BASE + 0x00000054
	volatile Uint32 MR1    ;//DDR3_PHY_CFG_BASE + 0x00000058
	volatile Uint32 MR2    ;//DDR3_PHY_CFG_BASE + 0x0000005C
	volatile Uint8 RSVD2[0x00000068-(0x0000005C+4)];
	volatile Uint32 DTCR   ;//DDR3_PHY_CFG_BASE + 0x00000068
	volatile Uint8 RSVD3[0x0000008C-(0x00000068+4)];
	volatile Uint32 PGCR2  ;//DDR3_PHY_CFG_BASE + 0x0000008C
	volatile Uint8 RSVD4[0x00000180-(0x0000008C+4)];
	DDR3_ZQ_Regs ZQ[4]; 	//DDR3_PHY_CFG_BASE + 0x00000180
	DDR3_DAT_Regs DAT[9];  //DDR3_PHY_CFG_BASE + 0x000001C0
} DDR3_PHY_Regs;

#define DDR3_ECC_CTRL_REG_RMW_EN_SHIFT 			28
#define DDR3_ECC_CTRL_REG_RMW_EN_MASK 			(1<<28)
#define DDR3_ECC_CTRL_REG_ECC_VERIFY_EN_SHIFT 	29
#define DDR3_ECC_CTRL_REG_ECC_VERIFY_EN_MASK 	(1<<29)

#define DDR3_PHY_PGSR0_APLOCK_SHIFT  (31)
#define DDR3_PHY_PGSR0_WEERR_SHIFT   (27)
#define DDR3_PHY_PGSR0_REERR_SHIFT   (26)
#define DDR3_PHY_PGSR0_WDERR_SHIFT   (25)
#define DDR3_PHY_PGSR0_RDERR_SHIFT   (24)
#define DDR3_PHY_PGSR0_WLAERR_SHIFT  (23)
#define DDR3_PHY_PGSR0_QSGERR_SHIFT  (22)
#define DDR3_PHY_PGSR0_WLERR_SHIFT   (21)
#define DDR3_PHY_PGSR0_ZCERR_SHIFT   (20)
#define DDR3_PHY_PGSR0_WEDONE_SHIFT  (11)
#define DDR3_PHY_PGSR0_REDONE_SHIFT  (10)
#define DDR3_PHY_PGSR0_WDDONE_SHIFT  (9 )
#define DDR3_PHY_PGSR0_RDDONE_SHIFT  (8 )
#define DDR3_PHY_PGSR0_WLADONE_SHIFT (7 )
#define DDR3_PHY_PGSR0_QSGDONE_SHIFT (6 )
#define DDR3_PHY_PGSR0_WLDONE_SHIFT  (5 )
#define DDR3_PHY_PGSR0_DIDONE_SHIFT  (4 )
#define DDR3_PHY_PGSR0_ZCDONE_SHIFT  (3 )
#define DDR3_PHY_PGSR0_DCDONE_SHIFT  (2 )
#define DDR3_PHY_PGSR0_PLDONE_SHIFT  (1 )
#define DDR3_PHY_PGSR0_IDONE_SHIFT   (0 )
#define DDR3_PHY_PGSR0_APLOCK_MASK  (1<<DDR3_PHY_PGSR0_APLOCK_SHIFT )
#define DDR3_PHY_PGSR0_WEERR_MASK   (1<<DDR3_PHY_PGSR0_WEERR_SHIFT  )
#define DDR3_PHY_PGSR0_REERR_MASK   (1<<DDR3_PHY_PGSR0_REERR_SHIFT  )
#define DDR3_PHY_PGSR0_WDERR_MASK   (1<<DDR3_PHY_PGSR0_WDERR_SHIFT  )
#define DDR3_PHY_PGSR0_RDERR_MASK   (1<<DDR3_PHY_PGSR0_RDERR_SHIFT  )
#define DDR3_PHY_PGSR0_WLAERR_MASK  (1<<DDR3_PHY_PGSR0_WLAERR_SHIFT )
#define DDR3_PHY_PGSR0_QSGERR_MASK  (1<<DDR3_PHY_PGSR0_QSGERR_SHIFT )
#define DDR3_PHY_PGSR0_WLERR_MASK   (1<<DDR3_PHY_PGSR0_WLERR_SHIFT  )
#define DDR3_PHY_PGSR0_ZCERR_MASK   (1<<DDR3_PHY_PGSR0_ZCERR_SHIFT  )
#define DDR3_PHY_PGSR0_WEDONE_MASK  (1<<DDR3_PHY_PGSR0_WEDONE_SHIFT )
#define DDR3_PHY_PGSR0_REDONE_MASK  (1<<DDR3_PHY_PGSR0_REDONE_SHIFT )
#define DDR3_PHY_PGSR0_WDDONE_MASK  (1<<DDR3_PHY_PGSR0_WDDONE_SHIFT )
#define DDR3_PHY_PGSR0_RDDONE_MASK  (1<<DDR3_PHY_PGSR0_RDDONE_SHIFT )
#define DDR3_PHY_PGSR0_WLADONE_MASK (1<<DDR3_PHY_PGSR0_WLADONE_SHIFT)
#define DDR3_PHY_PGSR0_QSGDONE_MASK (1<<DDR3_PHY_PGSR0_QSGDONE_SHIFT)
#define DDR3_PHY_PGSR0_WLDONE_MASK  (1<<DDR3_PHY_PGSR0_WLDONE_SHIFT )
#define DDR3_PHY_PGSR0_DIDONE_MASK  (1<<DDR3_PHY_PGSR0_DIDONE_SHIFT )
#define DDR3_PHY_PGSR0_ZCDONE_MASK  (1<<DDR3_PHY_PGSR0_ZCDONE_SHIFT )
#define DDR3_PHY_PGSR0_DCDONE_MASK  (1<<DDR3_PHY_PGSR0_DCDONE_SHIFT )
#define DDR3_PHY_PGSR0_PLDONE_MASK  (1<<DDR3_PHY_PGSR0_PLDONE_SHIFT )
#define DDR3_PHY_PGSR0_IDONE_MASK   (1<<DDR3_PHY_PGSR0_IDONE_SHIFT  )

#define DDR3_PHY_PLLCR_FRQSEL_SHIFT (18)
#define DDR3_PHY_PLLCR_FRQSEL_MASK  (0x3<<DDR3_PHY_PLLCR_FRQSEL_SHIFT)

#define DDR3_PHY_PGCR1_ZCKSEL_SHIFT (23)
#define DDR3_PHY_PGCR1_IODDRM_SHIFT (7)
#define DDR3_PHY_PGCR1_WLSTEP_SHIFT (2)
#define DDR3_PHY_PGCR1_ZCKSEL_MASK  (0x3<<DDR3_PHY_PGCR1_ZCKSEL_SHIFT)
#define DDR3_PHY_PGCR1_IODDRM_MASK  (0x3<<DDR3_PHY_PGCR1_IODDRM_SHIFT)
#define DDR3_PHY_PGCR1_WLSTEP_MASK  (0x1<<DDR3_PHY_PGCR1_WLSTEP_SHIFT)
#define DDR3_PHY_PGCR2_TREFPRD_SHIFT (0)
#define DDR3_PHY_PGCR2_TREFPRD_MASK  (0x1FFFF<<DDR3_PHY_PGCR2_TREFPRD_SHIFT)

#define DDR3_PHY_PTR3_TDINIT0_SHIFT (0)
#define DDR3_PHY_PTR3_TDINIT0_MASK  (0xFFFFF<<DDR3_PHY_PTR3_TDINIT0_SHIFT)
#define DDR3_PHY_PTR3_TDINIT1_SHIFT (20)
#define DDR3_PHY_PTR3_TDINIT1_MASK  (0x1FF<<DDR3_PHY_PTR3_TDINIT1_SHIFT)
#define DDR3_PHY_PTR4_TDINIT2_SHIFT (0)
#define DDR3_PHY_PTR4_TDINIT2_MASK  (0x3FFFF<<DDR3_PHY_PTR4_TDINIT2_SHIFT)
#define DDR3_PHY_PTR4_TDINIT3_SHIFT (18)
#define DDR3_PHY_PTR4_TDINIT3_MASK  (0x3FF<<DDR3_PHY_PTR4_TDINIT3_SHIFT)

#define DDR3_PHY_DCR_UDIMM_SHIFT    (29)
#define DDR3_PHY_DCR_NOSRA_SHIFT    (27)
#define DDR3_PHY_DCR_BYTEMASK_SHIFT (10)
#define DDR3_PHY_DCR_MPRDQ_SHIFT    (7)
#define DDR3_PHY_DCR_PDQ_SHIFT      (4)
#define DDR3_PHY_DCR_UDIMM_MASK     (0x1<<DDR3_PHY_DCR_UDIMM_SHIFT    )
#define DDR3_PHY_DCR_NOSRA_MASK     (0x1<<DDR3_PHY_DCR_NOSRA_SHIFT    )
#define DDR3_PHY_DCR_BYTEMASK_MASK  (0xFF<<DDR3_PHY_DCR_BYTEMASK_SHIFT )
#define DDR3_PHY_DCR_MPRDQ_MASK     (0x1<<DDR3_PHY_DCR_MPRDQ_SHIFT    )
#define DDR3_PHY_DCR_PDQ_MASK       (0x7<<DDR3_PHY_DCR_PDQ_SHIFT      )

#define DDR3_PHY_DTPR0_TRC_SHIFT    (26)
#define DDR3_PHY_DTPR0_TRRD_SHIFT   (22)
#define DDR3_PHY_DTPR0_TRAS_SHIFT   (16)
#define DDR3_PHY_DTPR0_TRCD_SHIFT   (12)
#define DDR3_PHY_DTPR0_TRP_SHIFT    (8 )
#define DDR3_PHY_DTPR0_TWTR_SHIFT   (4 )
#define DDR3_PHY_DTPR0_TRTP_SHIFT   (0 )
#define DDR3_PHY_DTPR1_TWLO_SHIFT   (26)
#define DDR3_PHY_DTPR1_TWLMRD_SHIFT (20)
#define DDR3_PHY_DTPR1_TRFC_SHIFT   (11)
#define DDR3_PHY_DTPR1_TFAW_SHIFT   (5 )
#define DDR3_PHY_DTPR1_TMOD_SHIFT   (2 )
#define DDR3_PHY_DTPR1_TMRD_SHIFT   (0 )
#define DDR3_PHY_DTPR2_TCCD_SHIFT   (31)
#define DDR3_PHY_DTPR2_TRTW_SHIFT   (30)
#define DDR3_PHY_DTPR2_TRTODT_SHIFT (29)
#define DDR3_PHY_DTPR2_TDLLK_SHIFT  (19)
#define DDR3_PHY_DTPR2_TCKE_SHIFT   (15)
#define DDR3_PHY_DTPR2_TXP_SHIFT    (10)
#define DDR3_PHY_DTPR2_TXS_SHIFT    (0 )
#define DDR3_PHY_DTPR0_TRC_MASK    (0x3F<<DDR3_PHY_DTPR0_TRC_SHIFT   ) 
#define DDR3_PHY_DTPR0_TRRD_MASK   (0xF <<DDR3_PHY_DTPR0_TRRD_SHIFT  )  
#define DDR3_PHY_DTPR0_TRAS_MASK   (0x3F<<DDR3_PHY_DTPR0_TRAS_SHIFT  ) 
#define DDR3_PHY_DTPR0_TRCD_MASK   (0xF <<DDR3_PHY_DTPR0_TRCD_SHIFT  )  
#define DDR3_PHY_DTPR0_TRP_MASK    (0xF <<DDR3_PHY_DTPR0_TRP_SHIFT   )  
#define DDR3_PHY_DTPR0_TWTR_MASK   (0xF <<DDR3_PHY_DTPR0_TWTR_SHIFT  )  
#define DDR3_PHY_DTPR0_TRTP_MASK   (0xF <<DDR3_PHY_DTPR0_TRTP_SHIFT  )  
#define DDR3_PHY_DTPR1_TWLO_MASK   (0xF  <<DDR3_PHY_DTPR1_TWLO_SHIFT  )  
#define DDR3_PHY_DTPR1_TWLMRD_MASK (0x3F <<DDR3_PHY_DTPR1_TWLMRD_SHIFT) 
#define DDR3_PHY_DTPR1_TRFC_MASK   (0x1FF<<DDR3_PHY_DTPR1_TRFC_SHIFT  )
#define DDR3_PHY_DTPR1_TFAW_MASK   (0x3F <<DDR3_PHY_DTPR1_TFAW_SHIFT  ) 
#define DDR3_PHY_DTPR1_TMOD_MASK   (0x7  <<DDR3_PHY_DTPR1_TMOD_SHIFT  )  
#define DDR3_PHY_DTPR1_TMRD_MASK   (0x3  <<DDR3_PHY_DTPR1_TMRD_SHIFT  )  
#define DDR3_PHY_DTPR2_TCCD_MASK   (0x1  <<DDR3_PHY_DTPR2_TCCD_SHIFT  )  
#define DDR3_PHY_DTPR2_TRTW_MASK   (0x1  <<DDR3_PHY_DTPR2_TRTW_SHIFT  )  
#define DDR3_PHY_DTPR2_TRTODT_MASK (0x1  <<DDR3_PHY_DTPR2_TRTODT_SHIFT)  
#define DDR3_PHY_DTPR2_TDLLK_MASK  (0x3FF<<DDR3_PHY_DTPR2_TDLLK_SHIFT )
#define DDR3_PHY_DTPR2_TCKE_MASK   (0xF  <<DDR3_PHY_DTPR2_TCKE_SHIFT  )  
#define DDR3_PHY_DTPR2_TXP_MASK    (0x1F <<DDR3_PHY_DTPR2_TXP_SHIFT   ) 
#define DDR3_PHY_DTPR2_TXS_MASK    (0x3FF<<DDR3_PHY_DTPR2_TXS_SHIFT   )

#define DDR3_PHY_MR0_PD_SHIFT     (12)
#define DDR3_PHY_MR0_WR_SHIFT     (9 )
#define DDR3_PHY_MR0_DR_SHIFT     (8 )
#define DDR3_PHY_MR0_TM_SHIFT     (7 )
#define DDR3_PHY_MR0_CL123_SHIFT  (4 )
#define DDR3_PHY_MR0_BT_SHIFT     (3 )
#define DDR3_PHY_MR0_CL0_SHIFT    (2 )
#define DDR3_PHY_MR0_BL_SHIFT     (0 )
#define DDR3_PHY_MR1_QOFF_SHIFT   (12)
#define DDR3_PHY_MR1_TDQS_SHIFT   (11)
#define DDR3_PHY_MR1_RTT2_SHIFT   (9 )
#define DDR3_PHY_MR1_LEVEL_SHIFT  (7 )
#define DDR3_PHY_MR1_RTT1_SHIFT   (6 )
#define DDR3_PHY_MR1_DIC1_SHIFT   (5 )
#define DDR3_PHY_MR1_AL_SHIFT     (3 )
#define DDR3_PHY_MR1_RTT0_SHIFT   (2 )
#define DDR3_PHY_MR1_DIC0_SHIFT   (1 )
#define DDR3_PHY_MR1_DE_SHIFT     (0 )
#define DDR3_PHY_MR2_RTTWR_SHIFT  (9 )
#define DDR3_PHY_MR2_RSVD_SHIFT   (8 )
#define DDR3_PHY_MR2_SRT_SHIFT    (7 )
#define DDR3_PHY_MR2_ASR_SHIFT    (6 )
#define DDR3_PHY_MR2_CWL_SHIFT    (3 )
#define DDR3_PHY_MR2_PASR_SHIFT   (0 )

#define DDR3_PHY_MR0_PD_MASK     (0x1<<DDR3_PHY_MR0_PD_SHIFT   )
#define DDR3_PHY_MR0_WR_MASK     (0x7<<DDR3_PHY_MR0_WR_SHIFT   )
#define DDR3_PHY_MR0_DR_MASK     (0x1<<DDR3_PHY_MR0_DR_SHIFT   )
#define DDR3_PHY_MR0_TM_MASK     (0x1<<DDR3_PHY_MR0_TM_SHIFT   )
#define DDR3_PHY_MR0_CL123_MASK  (0x7<<DDR3_PHY_MR0_CL123_SHIFT)
#define DDR3_PHY_MR0_BT_MASK     (0x1<<DDR3_PHY_MR0_BT_SHIFT   )
#define DDR3_PHY_MR0_CL0_MASK    (0x1<<DDR3_PHY_MR0_CL0_SHIFT  )
#define DDR3_PHY_MR0_BL_MASK     (0x3<<DDR3_PHY_MR0_BL_SHIFT   )
#define DDR3_PHY_MR1_QOFF_MASK   (0x1<<DDR3_PHY_MR1_QOFF_SHIFT )
#define DDR3_PHY_MR1_TDQS_MASK   (0x1<<DDR3_PHY_MR1_TDQS_SHIFT )
#define DDR3_PHY_MR1_RTT2_MASK   (0x1<<DDR3_PHY_MR1_RTT2_SHIFT )
#define DDR3_PHY_MR1_LEVEL_MASK  (0x1<<DDR3_PHY_MR1_LEVEL_SHIFT)
#define DDR3_PHY_MR1_RTT1_MASK   (0x1<<DDR3_PHY_MR1_RTT1_SHIFT )
#define DDR3_PHY_MR1_DIC1_MASK   (0x1<<DDR3_PHY_MR1_DIC1_SHIFT )
#define DDR3_PHY_MR1_AL_MASK     (0x3<<DDR3_PHY_MR1_AL_SHIFT   )
#define DDR3_PHY_MR1_RTT0_MASK   (0x1<<DDR3_PHY_MR1_RTT0_SHIFT )
#define DDR3_PHY_MR1_DIC0_MASK   (0x1<<DDR3_PHY_MR1_DIC0_SHIFT )
#define DDR3_PHY_MR1_DE_MASK     (0x1<<DDR3_PHY_MR1_DE_SHIFT   )
#define DDR3_PHY_MR2_RTTWR_MASK  (0x3<<DDR3_PHY_MR2_RTTWR_SHIFT)
#define DDR3_PHY_MR2_RSVD_MASK   (0x1<<DDR3_PHY_MR2_RSVD_SHIFT )
#define DDR3_PHY_MR2_SRT_MASK    (0x1<<DDR3_PHY_MR2_SRT_SHIFT  )
#define DDR3_PHY_MR2_ASR_MASK    (0x1<<DDR3_PHY_MR2_ASR_SHIFT  )
#define DDR3_PHY_MR2_CWL_MASK    (0x7<<DDR3_PHY_MR2_CWL_SHIFT  )
#define DDR3_PHY_MR2_PASR_MASK   (0x7<<DDR3_PHY_MR2_PASR_SHIFT )

#define DDR3_PHY_DTCR_RFSHDT_SHIFT  (28)
#define DDR3_PHY_DTCR_RANKEN_SHIFT  (24)
#define DDR3_PHY_DTCR_DTEXG_SHIFT   (23)
#define DDR3_PHY_DTCR_DTEXD_SHIFT   (22)
#define DDR3_PHY_DTCR_DTDSTP_SHIFT  (21)
#define DDR3_PHY_DTCR_DTDEN_SHIFT   (20)
#define DDR3_PHY_DTCR_DTDBS_SHIFT   (16)
#define DDR3_PHY_DTCR_DTBDC_SHIFT   (13)
#define DDR3_PHY_DTCR_DTWBDDM_SHIFT (12)
#define DDR3_PHY_DTCR_DTWDQM_SHIFT  (8 )
#define DDR3_PHY_DTCR_DTCMPD_SHIFT  (7 )
#define DDR3_PHY_DTCR_DTMPR_SHIFT   (6 )
#define DDR3_PHY_DTCR_DTRANK_SHIFT  (4 )
#define DDR3_PHY_DTCR_DTRPTN_SHIFT  (0 )
#define DDR3_PHY_DTCR_RFSHDT_MASK   (0xF<<DDR3_PHY_DTCR_RFSHDT_SHIFT )
#define DDR3_PHY_DTCR_RANKEN_MASK   (0xF<<DDR3_PHY_DTCR_RANKEN_SHIFT )
#define DDR3_PHY_DTCR_DTEXG_MASK    (0x1<<DDR3_PHY_DTCR_DTEXG_SHIFT  )
#define DDR3_PHY_DTCR_DTEXD_MASK    (0x1<<DDR3_PHY_DTCR_DTEXD_SHIFT  )
#define DDR3_PHY_DTCR_DTDSTP_MASK   (0x1<<DDR3_PHY_DTCR_DTDSTP_SHIFT )
#define DDR3_PHY_DTCR_DTDEN_MASK    (0x1<<DDR3_PHY_DTCR_DTDEN_SHIFT  )
#define DDR3_PHY_DTCR_DTDBS_MASK    (0xF<<DDR3_PHY_DTCR_DTDBS_SHIFT  )
#define DDR3_PHY_DTCR_DTBDC_MASK    (0x1<<DDR3_PHY_DTCR_DTBDC_SHIFT  )
#define DDR3_PHY_DTCR_DTWBDDM_MASK  (0x1<<DDR3_PHY_DTCR_DTWBDDM_SHIFT)
#define DDR3_PHY_DTCR_DTWDQM_MASK   (0xF<<DDR3_PHY_DTCR_DTWDQM_SHIFT )
#define DDR3_PHY_DTCR_DTCMPD_MASK   (0x1<<DDR3_PHY_DTCR_DTCMPD_SHIFT )
#define DDR3_PHY_DTCR_DTMPR_MASK    (0x1<<DDR3_PHY_DTCR_DTMPR_SHIFT  )
#define DDR3_PHY_DTCR_DTRANK_MASK   (0x3<<DDR3_PHY_DTCR_DTRANK_SHIFT )
#define DDR3_PHY_DTCR_DTRPTN_MASK   (0xF<<DDR3_PHY_DTCR_DTRPTN_SHIFT )

#define DDR3_PHY_DAT_GCR_DXEN_SHIFT (0)
#define DDR3_PHY_DAT_GCR_DXEN_MASK  (0x1<<DDR3_PHY_DAT_GCR_DXEN_SHIFT)

/*DDR3 memory termination resistor value*/
typedef enum
{
	DDR_TERM_DISABLED= 0, 						
	DDR_TERM_RZQ_OVER_4, 						
	DDR_TERM_RZQ_OVER_2, 						
	DDR_TERM_RZQ_OVER_6, 						
	DDR_TERM_RZQ_OVER_12, 						
	DDR_TERM_RZQ_OVER_8 						
}DDR3_TERM;

/*DDR memmory Dynamic On-Die Termination*/
typedef enum
{
	DDR_DYN_ODT_DISABLED= 0, 						
	DDR_DYN_ODT_OVER_4, 						
	DDR_DYN_ODT_OVER_2 						
}DDR3_DYN_ODT;

/*DDR memory Driver Impedance Control*/
typedef enum
{
	DDR_DRIVE_RZQ_OVER_6= 0, 						
	DDR_DRIVE_RZQ_OVER_7 						
}DDR3_DRIVE;

/*CAS Write Latency*/
typedef enum
{
	DDR_CWL_5= 0, 						
	DDR_CWL_6, 						
	DDR_CWL_7, 						
	DDR_CWL_8, 						
	DDR_CWL_9, 						
	DDR_CWL_10, 						
	DDR_CWL_11, 						
	DDR_CWL_12 						
}DDR3_CWL;

/*CAS Latency*/
typedef enum
{
	DDR_CL_5= 2, 						
	DDR_CL_6= 4, 						
	DDR_CL_7= 6, 						
	DDR_CL_8= 8, 						
	DDR_CL_9= 10, 						
	DDR_CL_10= 12, 						
	DDR_CL_11= 14, 						
	DDR_CL_12= 1, 						
	DDR_CL_13= 3, 						
	DDR_CL_14= 5, 						
	DDR_CL_15= 7, 						
	DDR_CL_16= 9 						
}DDR3_CL;

/*DDR3 data bus width*/
typedef enum
{
	DDR_BUS_WIDTH_64= 0, 						
	DDR_BUS_WIDTH_32, 						
	DDR_BUS_WIDTH_16					
}DDR3_BUS_WIDTH;

/*Internal SDRAM bank setup bits*/
typedef enum
{
	DDR_1_BANK= 0, 						
	DDR_2_BANK, 						
	DDR_4_BANK, 						
	DDR_8_BANK					
}DDR3_NUM_BANK;

/*External chip select setup*/
typedef enum
{
	DDR_1_CE= 0, 						
	DDR_2_CE				
}DDR3_NUM_CE;

/*Page/row size, column address bits*/
typedef enum
{
	DDR_PAGE_SIZE_8_BIT_256_WORD = 0,
	DDR_PAGE_SIZE_9_BIT_512_WORD,
	DDR_PAGE_SIZE_10_BIT_1024_WORD,
	DDR_PAGE_SIZE_11_BIT_2048_WORD
}DDR3_PAGE_SIZE;

/*DDR MAX temperature
if the DDR operating temperatue exceeds 85C, refresh rate should be double*/
typedef enum
{
	DDR_TEMP_UNDER_85C = 0,
	DDR_TEMP_OVER_85C
}DDR3_Temperature;

typedef enum
{
	MAP_TO_COS1 = 1,
	MAP_TO_COS2
}DDR3_Priority_to_COS_Map;

typedef enum
{
	/*enable calculation for accesses outside of these address ranges*/
	EN_ECC_OUT_OF_DEFINED_RANGES= 0, 
	/*Enable ECC calculation within address ranges defined in ECC Address Range 1 and 2 registers*/
	EN_ECC_WITHIN_DEFINED_RANGES 						
}DDR3_ECC_Range_Mode;

/*DDR ECC configuration*/
typedef struct
{
	/*please note, this is not DDR virtual mapping address in CPU.
	this is DDR controller internal address from 0~8GB.
	Lower 17 bits will be ignored and cleared*/
	unsigned long long startAddr;
	unsigned long long byteCnt;
}DDR3_ECC_Addr_Range;

/*DDR ECC configuration*/
typedef struct
{
	DDR3_ECC_Range_Mode rangeMode;
	DDR3_ECC_Addr_Range addressRange[2];

	/*thereshold number and window of 1-bit ECC error for trigger interrupt*/
	unsigned int one_bit_ECC_INT_threshold; 
	unsigned int one_bit_ECC_INT_window; 
}DDR3_ECC_Config;

/*DDR3 arbitration configuration for multiple master sharing DDR3*/
typedef struct
{
	/*maximum DDR clock cycles for one request to wait before its priority is 
	raised to highest.*/
	Uint32 old_request_max_wait_cycles;  /*0~4080*/
	Uint32 COS1_request_max_wait_cycles; /*0~4080*/
	Uint32 COS2_request_max_wait_cycles; /*0~4080*/
	DDR3_Priority_to_COS_Map priority_to_COS_map[8];
}DDR3_Arbitration_Config;

/*some DDR3 timing are specified as maximum of nanosecond and number of clock*/
typedef struct
{
	unsigned int nCK; //timing value in number of clock
	float ns;   //timing value in nanosecond
}DDR3_MAX_time;

/*DDR3 configuration*/
typedef struct
{
	/*------------------register pointers------------------*/
	PLL_ControlRegs * PLL_Regs;
	CSL_Emif4fvRegs * ControlRegs;
	DDR3_PHY_Regs * PHY_Regs;

	/*----------------DDR PLL configuration----------------*/
	/*DDR3 data rate will be confiugred as: 
	data_rate=ref_clock_MHz/PLL_input_divisor*PLL_multiplier/PLL_output_divisor*4
	clock_speed=ref_clock_MHz/PLL_input_divisor*PLL_multiplier/PLL_output_divisor*2
	Please make, 1.5625GHz < ref_clock_MHz/PLL_input_divisor*PLL_multiplier < 3.125GHz */
	float ref_clock_MHz; //reference clock speed in MHz
	unsigned int PLL_input_divisor;
	unsigned int PLL_multiplier;
	unsigned int PLL_output_divisor;

	/*--------------SDRAM basic configuration--------------*/
	DDR3_BUS_WIDTH busWidth;
	DDR3_NUM_BANK num_bank;
	DDR3_NUM_CE num_CE;
	Bool bUDIMM_address_mirror;	//if UDIMM Address Mirroring is used or not
	DDR3_PAGE_SIZE pageSize;
	DDR3_CL CL; //CAS_latency
	DDR3_CWL CWL; //CAS_write_latency
	DDR3_DRIVE drive;
	DDR3_TERM DDR_TERM;
	DDR3_DYN_ODT DYN_ODT;

	/*-----------board related timing parameter-------------*/
	float round_trip_delay_ns; //round trip delay in nanosecond
	DDR3_Temperature temperature;
	
	/*-----DDR Timing values from DDR memory data sheet----*/
	/*the unit of following parameters is ns (nanosecond)*/
	float tRCD  ;
	float tRP   ;
	float tRAS  ;
	float tRC   ;
	float tWR   ;
	float tFAW  ;
	float tRFC  ;
	float tWLO  ;
	/*the unit of following parameters is clock number*/
	unsigned int tCCD  ;
	unsigned int tMRD  ;
	unsigned int tDLLK ;
	unsigned int tZQOPER ;
	unsigned int tZQCS ;
	unsigned int tWLMRD;
	/*DDR3 timing parameters specified as maximum of nanosecond and number of clock*/
	DDR3_MAX_time tWTR;
	DDR3_MAX_time tRTP;
	DDR3_MAX_time tXP ;
	DDR3_MAX_time tXPDLL;
	DDR3_MAX_time tXS ;
	DDR3_MAX_time tCKE;
	DDR3_MAX_time tMOD  ;
	DDR3_MAX_time tRRD;

	/*optional configuration, default configuration will be used 
	if the pointer of the configuration structure is NULL*/
	DDR3_Arbitration_Config * arbitration_cfg;
	DDR3_ECC_Config * ecc_cfg;
}DDR3_Config;

/*----------------------------------------------*
 * project-wide global variables                *
 *----------------------------------------------*/
#if (2==CSL_DDR3_PER_CNT)
/*for Device has two DDR3 interfaces*/
extern DDR3_PHY_Regs * gDDR3_PHY_Regs[CSL_DDR3_PER_CNT];
#else
/*for Device has one DDR3 interface*/
extern DDR3_PHY_Regs * gDDR3_PHY_Regs;
#endif

/*----------------------------------------------*
 * routines' implementations                    *
 *----------------------------------------------*/
extern int K2_DDR_PHY_status_check(DDR3_PHY_Regs * PHY_Regs);
extern void K2_DDR_Init(DDR3_Config * DDR_cfg);

/*get DDR ECC ranges accoridng to "ecc_cfg", return number valid ranges*/
extern Uint32 get_DDR_ECC_ranges(DDR3_ECC_Config * ecc_cfg, 
	DDR3_ECC_Addr_Range addressRanges[], unsigned long long ullTotalByteCount);

#endif

