/* 
 * Copyright (c) 2012, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*
 *  ======== sleep6x.c ========
 *  This example shows registration and notification of sleep modes, and 
 *  activation of the Power_STANDBY, Power_SLEEP, and Power_DEEPSLEEP modes on
 *  the evm6748.
 */
#include <xdc/std.h>
#include <xdc/runtime/System.h>
#include <xdc/runtime/Types.h>
#include <xdc/runtime/Timestamp.h>
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Clock.h>
#include <ti/sysbios/family/c674/Power.h>
#include <xdc/cfg/global.h>

#define REG(x) (*((volatile unsigned *)(x)))

/* RTC register and bit definitions */
#define RTC_SECOND_REG	0x01C23000
#define RTC_ALMSEC_REG	0x01C23020
#define RTC_CTRL_REG	0x01C23040
#define RTC_STATUS_REG	0x01C23044
#define RTC_INTR_REG	0x01C23048
#define RTC_OSC_REG	0x01C23054
#define RTC_KICK0R_REG	0x01C2306C
#define RTC_KICK1R_REG	0x01C23070
#define KICK0_UNLOCK	0x83E70B13
#define KICK1_UNLOCK	0x95A4F1E0
#define RTC_OSC_SWRESET	0x00000020
#define RTC_START	0x00000081
#define RTC_ALARM_INTR	0x00000008
#define RTC_ALARM_BIT	0x00000040

/* PINMUX register and bit definitions */
#define PINMUX0_REG	0x01C14120
#define MUX_MASK	0xF0000000
#define MUX_RTC_ALARM	0x20000000

/* RTC alarm delay for DEEPSLEEP */
#define ALARM_DELAY	0x15

Power_NotifyResponse myNotifyFxn(Power_Event eventType, Arg eventArg1,
    Arg eventArg2, Arg clientArg);

Void task();

/*
 *  ======== main ========
 */
Void main(int argc, char *argv[])
{
    BIOS_start();
}

/*
 *  ======== task ========
 */
Void task()
{
    Power_NotifyHandle notifyHandle1;
    Power_NotifyHandle notifyHandle2;
    Power_NotifyHandle notifyHandle3;
    Power_NotifyHandle notifyHandle4;
    Fxn delayedCompletionFxn;
    Power_Status status;
    UInt current;
    UInt new;

    /* register for sleep notifications */
    status = Power_registerNotify(Power_GOINGTOSLEEP, 0, myNotifyFxn, 0, 
        &notifyHandle1, &delayedCompletionFxn);
    if (status == Power_SOK) {
        System_printf("registered for GOINGTOSLEEP event\n");
    }

    status = Power_registerNotify(Power_AWAKEFROMSLEEP, 0, myNotifyFxn, 0, 
        &notifyHandle2, &delayedCompletionFxn);
    if (status == Power_SOK) {
        System_printf("registered for AWAKEFROMSLEEP event\n");
    }

    status = Power_registerNotify(Power_GOINGTODEEPSLEEP, 0, myNotifyFxn, 0, 
        &notifyHandle3, &delayedCompletionFxn);
    if (status == Power_SOK) {
        System_printf("registered for GOINGTODEEPSLEEP event\n");
    }

    status = Power_registerNotify(Power_AWAKEFROMDEEPSLEEP, 0, myNotifyFxn, 0, 
        &notifyHandle4, &delayedCompletionFxn);
    if (status == Power_SOK) {
        System_printf("registered for AWAKEFROMDEEPSLEEP event\n");
    }

    System_printf("current Clock ticks = %d\n", Clock_getTicks());

    System_printf("going to STANDBY ...\n");

    /* go to STANDBY until next Clock tick ... */
    status = Power_sleepDSP(Power_STANDBY, 0, 0);

    System_printf("awake from STANDBY, status = 0x%x\n", status);

    System_printf("current Clock ticks = %d\n", Clock_getTicks());

    System_printf("going to SLEEP ...\n");

    /* go to SLEEP until next Clock tick ... */
    status = Power_sleepDSP(Power_SLEEP, 0, 0);

    System_printf("awake from SLEEP, status = 0x%x\n", status);

    System_printf("current Clock ticks = %d\n", Clock_getTicks());

    /* unlock RTC registers */
    REG(RTC_KICK0R_REG) = KICK0_UNLOCK;  /* unlock RTC registers */
    REG(RTC_KICK1R_REG) = KICK1_UNLOCK;
  
    /* wait for next Clock tick to start RTC configuration */
    current = new = Clock_getTicks();
    while (new == current) {
        new = Clock_getTicks();
    }

    /* initiate SW reset of RTC */
    REG(RTC_OSC_REG) |= RTC_OSC_SWRESET;

    /* configure PINMUX for RTC_ALARM functionality */
    REG(PINMUX0_REG) &= ~MUX_MASK;
    REG(PINMUX0_REG) |= MUX_RTC_ALARM;

    /* wait for RTC reset completion */
    current = new = Clock_getTicks();
    while (new == current) {
        new = Clock_getTicks();
    }

    /* set alarm time (in seconds) */
    REG(RTC_ALMSEC_REG) = ALARM_DELAY;

    /* enable alarm interrupt to signal DEEPSLEEP */
    REG(RTC_INTR_REG) = RTC_ALARM_INTR;

    /* enable isolation and start the RTC */ 
    REG(RTC_CTRL_REG) |= RTC_START;

    System_printf("current RTC time = %x sec\n", REG(RTC_SECOND_REG));

    System_printf("going to DEEPSLEEP ...\n");

    /* go to DEEPSLEEP until RTC alarm signal ... */
    status = Power_sleepDSP(Power_DEEPSLEEP, Power_RTC_ALARM, 0);

    System_printf("awake from DEEPSLEEP, status = 0x%x\n", status);

    System_printf("current RTC time = %x sec\n", REG(RTC_SECOND_REG));

    /* un-register for sleep notifications */
    Power_unregisterNotify(notifyHandle1);
    Power_unregisterNotify(notifyHandle2);
    Power_unregisterNotify(notifyHandle3);
    Power_unregisterNotify(notifyHandle4);
}


/*
 *  ======== myNotifyFxn ========
 */
/* ARGSUSED */
Power_NotifyResponse myNotifyFxn(Power_Event eventType, Arg eventArg1,
    Arg eventArg2, Arg clientArg)
{
    switch (eventType) 
    {
        case Power_GOINGTOSLEEP :  
            System_printf("    myNotifyFxn - GOINGTOSLEEP\n");
            break;

        case Power_AWAKEFROMSLEEP  :  
            System_printf("    myNotifyFxn - AWAKEFROMSLEEP\n");
            break;

        case Power_GOINGTODEEPSLEEP :  
            System_printf("    myNotifyFxn - GOINGTODEEPSLEEP\n");
            break;

        case Power_AWAKEFROMDEEPSLEEP  :  
            System_printf("    myNotifyFxn - AWAKEFROMDEEPSLEEP\n");
            break;
    }

    return(Power_NOTIFYDONE);
}
