/******************************************************************************
 * Copyright (c) 2011-2012 Texas Instruments Incorporated - http://www.ti.com
 *
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 *****************************************************************************/

#include <cerrno>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "platform.h"
#include "resource_mgr.h"

/* The default values for the input strings are given below */
#define DEF_INIT_CONFIG_PLL1_PLLM      20
#define DEF_INIT_CONFIG_UART_BAUDRATE  0

#define DEF_NAND_TEST_BLOCK_NUMBER    512

#define DEF_NOR_TEST_SECTOR_NUMBER    10

#define DEF_EEPROM_TEST_SLAVE_ADDRESS 0x51

#define DEF_LED_TEST_LOOP_COUNT    1000000
#define DEF_LED_TEST_LOOP_DELAY    10000000

#define DEF_EXT_MEM_TEST_BASE_ADDR 0x80000000
#define DEF_EXT_MEM_TEST_LENGTH    0x1FFFFFFF

#define DEF_INT_MEM_TEST_CORE_ID   1


#define CHIP_LEVEL_REG  0x02620000
#define DEVSTAT  (*(unsigned int*)(CHIP_LEVEL_REG + 0x0020))
#define KICK0    (*(unsigned int*)(CHIP_LEVEL_REG + 0x0038))
#define KICK1    (*(unsigned int*)(CHIP_LEVEL_REG + 0x003C))

#define IPCGR0    (*(unsigned int*)(CHIP_LEVEL_REG + 0x0240))
#define IPCGR1    (*(unsigned int*)(CHIP_LEVEL_REG + 0x0244))

#define KICK0_UNLOCK (0x83E70B13)
#define KICK1_UNLOCK (0x95A4F1E0)


typedef struct test_config {
    uint8_t print_info;
    uint8_t print_current_core_id;
    uint8_t print_switch_state;
    uint8_t test_eeprom;
    uint8_t test_nand;
    uint8_t test_nor;
    uint8_t test_led;
    uint8_t test_uart;
    uint8_t test_external_mem;
    uint8_t test_internal_mem;

    uint32_t init_config_pll1_pllm;
    uint32_t init_config_uart_baudrate;
    uint32_t nand_test_block_number;
    uint32_t nor_test_sector_number;
    uint32_t eeprom_test_slave_address;
    uint32_t led_test_loop_count;
    uint32_t led_test_loop_delay;
    uint32_t ext_mem_test_base_addr;
    uint32_t ext_mem_test_length;
    uint32_t int_mem_test_core_id;
} test_config;


/* OSAL functions for Platform Library */
uint8_t *Osal_platformMalloc (uint32_t num_bytes, uint32_t alignment)
{
	return malloc(num_bytes);
}

void Osal_platformFree (uint8_t *dataPtr, uint32_t num_bytes)
{
    /* Free up the memory */
    if (dataPtr)
    {
        free(dataPtr);
    }
}

void Osal_platformSpiCsEnter(void)
{
    /* Get the hardware semaphore.
     *
     * Acquire Multi core CPPI synchronization lock
     */
    while ((CSL_semAcquireDirect (PLATFORM_SPI_HW_SEM)) == 0);

    return;
}

void Osal_platformSpiCsExit (void)
{
    /* Release the hardware semaphore
     *
     * Release multi-core lock.
     */
    CSL_semReleaseSemaphore (PLATFORM_SPI_HW_SEM);

    return;
}

void user_delay(unsigned int cycle)
{
	int i;
	for(i=0; i<cycle; i++)
		__asm(" nop");
}

/******************************************************************************
 * Function:    test_led  
 ******************************************************************************/
static void test_led(test_config * args, platform_info * p_info)
{
    int32_t j;
    volatile uint32_t tmp;
    int32_t max_loop = args->led_test_loop_count;

    tmp = 0;
    do {
        for (j = PLATFORM_USER_LED_CLASS; j < PLATFORM_END_LED_CLASS; j++) {
                platform_led(0, PLATFORM_LED_ON, (LED_CLASS_E)j);
                if(tmp != 0)//core 1 up
                	platform_led(1, PLATFORM_LED_ON, (LED_CLASS_E)j);
                user_delay(args->led_test_loop_delay);
                platform_led(0, PLATFORM_LED_OFF, (LED_CLASS_E) j);
                if(tmp != 0)//core 1 up
                	platform_led(1, PLATFORM_LED_OFF, (LED_CLASS_E)j);
                user_delay(args->led_test_loop_delay);
        }
        tmp = *(uint32_t *)(0x02620240);

    } while (--max_loop);

    /* turn on all the LEDS (default state) after the test */
}



/******************************************************************************
 * Function:    main  
 ******************************************************************************/
void main (int argc, char *argv[])
{
	platform_init_flags  init_flags;
	platform_init_config init_config;
	platform_info        p_info;
	test_config args;

	/* Set default values */
	memset(&args, 0x01, sizeof(test_config));
	memset(&init_flags, 0x01, sizeof(platform_init_flags));
	args.init_config_pll1_pllm     = DEF_INIT_CONFIG_PLL1_PLLM;
	args.init_config_uart_baudrate = DEF_INIT_CONFIG_UART_BAUDRATE;
	args.nand_test_block_number    = DEF_NAND_TEST_BLOCK_NUMBER;
	args.nor_test_sector_number    = DEF_NOR_TEST_SECTOR_NUMBER;
	args.eeprom_test_slave_address = DEF_EEPROM_TEST_SLAVE_ADDRESS;
	args.led_test_loop_count       = DEF_LED_TEST_LOOP_COUNT;
	args.led_test_loop_delay       = DEF_LED_TEST_LOOP_DELAY;
	args.ext_mem_test_base_addr    = DEF_EXT_MEM_TEST_BASE_ADDR;
	args.ext_mem_test_length       = DEF_EXT_MEM_TEST_LENGTH;
	args.int_mem_test_core_id      = DEF_INT_MEM_TEST_CORE_ID;

	init_config.pllm    = args.init_config_pll1_pllm;
	init_flags.ddr = 1;
	init_flags.phy = 1;
	init_flags.pll = 1;
	init_flags.tcsl = 1;
	init_flags.ecc = 0;

	//initialize platform
	if (platform_init(&init_flags, &init_config) != Platform_EOK) {
		printf("Platform failed to initialize, errno = 0x%x \n", platform_errno);
	}

	platform_write_configure(PLATFORM_WRITE_PRINTF);
	platform_uart_init();
	platform_uart_set_baudrate(args.init_config_uart_baudrate);
	platform_get_info(&p_info);

	//boot core 1
	//write _c_int00 address to core 1 boot magic address
	*(uint32_t *)(0x118FFFFC) = 0x1180a5a0;
	//send a IPC to core 1 to boot up
	IPCGR1 = 0x1;

	if(args.test_led) {
		test_led(&args, &p_info);
	}


}

