/*H***************************************************************************
*
* $Archive:: /boards/dsk5509a/lib/dsk5509_srom.c                              $
* $Revision:: 1                                                               $
* $Date:: 11/10/05 11:38a                                                     $
* $Author:: Shilpab                                                           $
*
* DESCRIPTION:
*
* 
* NOTES : 
*
* (C) Copyright 2005 by Spectrum Digital Incorporated
* All rights reserved
*
*H***************************************************************************/

/*
 *  ======== dsk5509_srom.c ========
 *  DSK5509 SROM module implementation
 */

#include <csl.h>
#include <csl_mcbsp.h>
#include <csl_gpio.h>

#include "dsk5509.h"
#include "dsk5509_srom.h"

static MCBSP_Config mcbspCfgSPI = {
    MCBSP_SPCR1_RMK(
        MCBSP_SPCR1_DLB_OFF,                   /* DLB      = 0 */
        MCBSP_SPCR1_RJUST_RZF,                 /* RJUST    = 0 */
        MCBSP_SPCR1_CLKSTP_DELAY,              /* CLKSTP   = 3 */
        MCBSP_SPCR1_DXENA_NA,                  /* DXENA    = 0 */
        MCBSP_SPCR1_ABIS_DISABLE,              /* ABIS     = 0 */
        MCBSP_SPCR1_RINTM_RRDY,                /* RINTM    = 0 */
        0,                                     /* RSYNCER  = 0 */
        MCBSP_SPCR1_RRST_DISABLE               /* RRST     = 0 */
    ),
    MCBSP_SPCR2_RMK(
        MCBSP_SPCR2_FREE_YES,                  /* FREE     = 1 */
        MCBSP_SPCR2_SOFT_NO,                   /* SOFT     = 0 */
        MCBSP_SPCR2_FRST_RESET,                /* FRST     = 0 */
        MCBSP_SPCR2_GRST_RESET,                /* GRST     = 0 */
        MCBSP_SPCR2_XINTM_XRDY,                /* XINTM    = 0 */
        0,                                     /* XSYNCER  = 0 */
        MCBSP_SPCR2_XRST_DISABLE               /* XRST     = 0 */
    ),
    MCBSP_RCR1_RMK(
        MCBSP_RCR1_RFRLEN1_OF(0),              /* RFRLEN1  = 0 */
        MCBSP_RCR1_RWDLEN1_8BIT                /* RWDLEN1  = 0 */
    ),
    MCBSP_RCR2_RMK(
        MCBSP_RCR2_RPHASE_SINGLE,              /* RPHASE   = 0 */
        MCBSP_RCR2_RFRLEN2_OF(0),              /* RFRLEN2  = 0 */
        MCBSP_RCR2_RWDLEN2_8BIT,               /* RWDLEN2  = 0 */
        MCBSP_RCR2_RCOMPAND_MSB,               /* RCOMPAND = 0 */
        MCBSP_RCR2_RFIG_YES,                   /* RFIG     = 0 */
        MCBSP_RCR2_RDATDLY_1BIT                /* RDATDLY  = 1 */
    ),
    MCBSP_XCR1_RMK(
        MCBSP_XCR1_XFRLEN1_OF(0),              /* XFRLEN1  = 0 */
        MCBSP_XCR1_XWDLEN1_8BIT                /* XWDLEN1  = 0 */
    ),
    MCBSP_XCR2_RMK(
        MCBSP_XCR2_XPHASE_SINGLE,              /* XPHASE   = 0 */
        MCBSP_XCR2_XFRLEN2_OF(0),              /* XFRLEN2  = 0 */
        MCBSP_XCR2_XWDLEN2_8BIT,               /* XWDLEN2  = 0 */
        MCBSP_XCR2_XCOMPAND_MSB,               /* XCOMPAND = 0 */
        MCBSP_XCR2_XFIG_YES,                   /* XFIG     = 0 */
        MCBSP_XCR2_XDATDLY_1BIT                /* XDATDLY  = 1 */
    ),
    MCBSP_SRGR1_RMK(
        MCBSP_SRGR1_FWID_OF(0),                /* FWID     = 0 */
        MCBSP_SRGR1_CLKGDV_OF(63)              /* CLKGDV   = 63 */
    ),
    MCBSP_SRGR2_RMK(
        MCBSP_SRGR2_GSYNC_FREE,                /* FREE     = 0 */
        MCBSP_SRGR2_CLKSP_RISING,              /* CLKSP    = 0 */
        MCBSP_SRGR2_CLKSM_INTERNAL,            /* CLKSM    = 1 */
        MCBSP_SRGR2_FSGM_DXR2XSR,              /* FSGM     = 0 */
        MCBSP_SRGR2_FPER_OF(0)                 /* FPER     = 0 */
    ),
    MCBSP_MCR1_DEFAULT,
    MCBSP_MCR2_DEFAULT,
    MCBSP_PCR_RMK(
        MCBSP_PCR_IDLEEN_RESET,                /* IDLEEN   = 0 */
        MCBSP_PCR_XIOEN_SP,                    /* XIOEN    = 0 */
        MCBSP_PCR_RIOEN_SP,                    /* RIOEN    = 0 */
        MCBSP_PCR_FSXM_INTERNAL,               /* FSXM     = 1 */
        MCBSP_PCR_FSRM_EXTERNAL,               /* FSRM     = 0 */
        MCBSP_PCR_CLKXM_OUTPUT,                /* CLKXM    = 1 */
        MCBSP_PCR_CLKRM_INPUT,                 /* CLKRM    = 0 */
        MCBSP_PCR_SCLKME_NO,                   /* SCLKME   = 0 */
        0,                                     /* DXSTAT   = 0 */
        MCBSP_PCR_FSXP_ACTIVELOW,              /* FSXP     = 1 */
        MCBSP_PCR_FSRP_ACTIVELOW,              /* FSRP     = 1 */
        MCBSP_PCR_CLKXP_RISING,                /* CLKXP    = 0 */
        MCBSP_PCR_CLKRP_RISING                 /* CLKRP    = 1 */
    ),
    MCBSP_RCERA_DEFAULT,
    MCBSP_RCERB_DEFAULT,
    MCBSP_RCERC_DEFAULT,
    MCBSP_RCERD_DEFAULT,
    MCBSP_RCERE_DEFAULT,
    MCBSP_RCERF_DEFAULT,
    MCBSP_RCERG_DEFAULT,
    MCBSP_RCERH_DEFAULT,
    MCBSP_XCERA_DEFAULT,
    MCBSP_XCERB_DEFAULT,
    MCBSP_XCERC_DEFAULT,
    MCBSP_XCERD_DEFAULT,
    MCBSP_XCERE_DEFAULT,
    MCBSP_XCERF_DEFAULT,
    MCBSP_XCERG_DEFAULT,
    MCBSP_XCERH_DEFAULT
};

MCBSP_Handle DSK5509_SROM_hMcbsp;

static Uint16 srombuf[DSK5509_SROM_BLOCKSIZE + 4], statusbuf[4];

static sromenable()
{
    GPIO_pinWrite(GPIO_PIN4, 0);
}

static sromdisable()
{
    GPIO_pinWrite(GPIO_PIN4, 1);
}

static void sromcycle(Uint16 *buf, Int16 len)
{
    Int16 i;
    Uint16 sdata;

    /* Clear out any garbage read data */
    if (MCBSP_rrdy(DSK5509_SROM_hMcbsp))
        MCBSP_read16(DSK5509_SROM_hMcbsp);

    for (i = 0; i < len; i++)
    {
        /* Wait until transmitter free */
        while (!MCBSP_xrdy(DSK5509_SROM_hMcbsp));
        MCBSP_write16(DSK5509_SROM_hMcbsp, buf[i]);

        /* Wait for receive data */
        while (!MCBSP_rrdy(DSK5509_SROM_hMcbsp));
        sdata = MCBSP_read16(DSK5509_SROM_hMcbsp);
        if ( i> 0)
            buf[i - 1] = sdata;
    }
}

static Uint16 readstatus()
{
    /* Issue read status command */
    statusbuf[0] = DSK5509_SROM_RDSR;
    sromenable();
    sromcycle(statusbuf, 2);
    sromdisable();

    return statusbuf[0];
}

static void writestatus(Uint16 value)
{
    /* Issue read status command */
    statusbuf[0] = DSK5509_SROM_WRSR;
    statusbuf[1] = value;
    sromenable();
    sromcycle(statusbuf, 2);
    sromdisable();
}

void DSK5509_SROM_open()
{
    /* Enable the SROM data path */
    DSK5509_rset(DSK5509_MISC, DSK5509_rget(DSK5509_MISC) & 0xfe);

    /* Enable GPIO 4 (chip select) as output, high */
    GPIO_pinWrite(GPIO_PIN4, 1);
    GPIO_pinDirection(GPIO_PIN4, GPIO_OUTPUT);
    GPIO_pinEnable(GPIO_PIN4);

    /* Open McBSP handle */
    DSK5509_SROM_hMcbsp = MCBSP_open(MCBSP_PORT0, MCBSP_OPEN_RESET);

    /* Configure McBSP in clock stop SPI mode */
    MCBSP_config(DSK5509_SROM_hMcbsp, &mcbspCfgSPI);

    /* Start the McBSP */
    MCBSP_start(DSK5509_SROM_hMcbsp, MCBSP_XMIT_START | MCBSP_RCV_START |
        MCBSP_SRGR_START, 500);

    /* Disable write protection */
    writestatus(0);
}

void DSK5509_SROM_close()
{
    /* Close the McBSP */
    MCBSP_close(DSK5509_SROM_hMcbsp);
}

/*
 *  ======== DSK5509_SROM_read ========
 *  Read data from SROM
 */
void DSK5509_SROM_read(Uint32 src, Uint32 dst, Uint32 length)
{
    Uint16 *psrc, *pdst;
    Uint32 i;

    /* Establish destination */
    pdst = (Uint16 *)dst;

    /* Create command block for program operation */
    srombuf[0] = DSK5509_SROM_READ;
    srombuf[1] = (src >> 8) & 0xff;
    srombuf[2] = src & 0xff;

    /* Execute read command */
    sromenable();
    sromcycle(srombuf, 3 + length);
    sromdisable();

    psrc = srombuf + 2;
    for (i = 0; i < length; i++)
        *pdst++ = *psrc++;
}

/*
 *  ======== DSK5509_SROM_write ========
 *  Write data to SROM
 */
void DSK5509_SROM_write(Uint32 src, Uint32 dst, Uint32 length)
{
    Uint16 *psrc;
    Uint32 i, bytesleft, bytestoprogram;

    /* Establish source */
    psrc = (Uint16 *)src;

    bytesleft = length;
    while (bytesleft > 0)
    {
        bytestoprogram = bytesleft;

        /* Most to program is DSK5509_SROM_BLOCKSIZE */
        if (bytestoprogram > DSK5509_SROM_BLOCKSIZE)
            bytestoprogram = DSK5509_SROM_BLOCKSIZE;

        /* Make sure you don't run off the end of a block */
        if ((dst & DSK5509_SROM_BLOCKMASK) != ((dst + bytestoprogram) & DSK5509_SROM_BLOCKMASK))
            bytestoprogram -= (dst + bytestoprogram) - ((dst + bytestoprogram) & DSK5509_SROM_BLOCKMASK);

        /* Issue WPEN */
        srombuf[0] = DSK5509_SROM_WREN;
        sromenable();
        sromcycle(srombuf, 1);

        /* Create command block for program operation */
        srombuf[0] = DSK5509_SROM_PROGRAM;
        srombuf[1] = (dst >> 8) & 0xff;
        srombuf[2] = dst & 0xff;
        for (i = 0; i < bytestoprogram; i++)
            srombuf[3 + i] = *psrc++;

        /* Execute write command */
        sromenable();
        sromcycle(srombuf, 3 + bytestoprogram);
        sromdisable();

        /* Wait while busy */
        while((readstatus() & 1) != 0);

        /* Get ready for next iteration */
        bytesleft -= bytestoprogram;
        dst += bytestoprogram;
    }
}

/*
 *  ======== DSK5509_SROM_erase ========
 *  Erase a block of the SROM
 */
void DSK5509_SROM_erase(Uint32 start, Uint32 length)
{
}
