/*
 *  Copyright (C) 2021 Texas Instruments Incorporated
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* This example demonstrates the CAN message transmission and reception in
 * digital loop back mode with the following configuration.
 *
 * CAN FD Message Format.
 * Message ID Type is Standard, Msg Id 0xC0.
 * MCAN is configured in Interrupt Mode.
 * MCAN Interrupt Line Number 0.
 * Arbitration Bit Rate 1Mbps.
 * Data Bit Rate 5Mbps.
 * Buffer mode is used for Tx and RX to store message in message RAM.
 *
 * Message is transmitted and received back internally using internal loopback
 * mode. When the received message id and the data matches with the transmitted
 * one, then the example is completed.
 *
 */

#include <stdio.h>
#include <kernel/dpl/DebugP.h>
#include <kernel/dpl/AddrTranslateP.h>
#include <kernel/dpl/SemaphoreP.h>
#include <drivers/mcan.h>
#include "ti_drivers_config.h"
#include "ti_drivers_open_close.h"
#include "ti_board_open_close.h"

#define APP_MCAN_BASE_ADDR                       (CONFIG_MCAN0_BASE_ADDR)
#define APP_MCAN_INTR_NUM                        (CONFIG_MCAN0_INTR)
#define APP_MCAN_MSG_LOOP_COUNT                  (100U)

// 将消息内存内存部分分配给过滤元素、缓冲区、fifo，ID过滤最大可设为128
#define APP_MCAN_STD_ID_FILTER_CNT               (1U)

#define APP_MCAN_EXT_ID_FILTER_CNT               (0U) // 扩展ID过滤器数量 最大为64

#define APP_MCAN_TX_BUFF_CNT                     (1U) // 发送buffer数量 最大为32
#define APP_MCAN_TX_FIFO_CNT                     (0U) // 发送FIFO数量，buffer+FIFO数量不超过32个 

#define APP_MCAN_TX_EVENT_FIFO_CNT               (0U) // 发送时间FIFO数量，最大为32

#define APP_MCAN_FIFO_0_CNT                      (0U) // RX FIFO 0数量，最大为64
//  RX FIFO 1数量  最大为64 其余的内存被分配给rx缓冲区，它的最大大小同样为64 
#define APP_MCAN_FIFO_1_CNT                      (0U)

// 标准帧ID
#define APP_MCAN_STD_ID                          (0xC0U)  // 扩展ID
#define APP_MCAN_STD_ID_MASK                     (0x7FFU) // 扩展ID标识
#define APP_MCAN_STD_ID_SHIFT                    (18U)

#define APP_MCAN_EXT_ID_MASK                     (0x1FFFFFFFU) // 经典位掩码滤波器

// 标准CAN一帧最大8字节，CAN FD可到64字节，发送长度按照下表发送
static const uint8_t gMcanDataSize[16U] = {0U,  1U,  2U,  3U,
                                           4U,  5U,  6U,  7U,
                                           8U,  12U, 16U, 20U,
                                           24U, 32U, 48U, 64U};

// 表示传输完成的信号量 
static SemaphoreP_Object gMcanTxDoneSem, gMcanRxDoneSem;
static HwiP_Object       gMcanHwiObject;
static uint32_t          gMcanBaseAddr;

// 静态函数声明
static void    App_mcanIntrISR(void *arg);
static void    App_mcanConfig(Bool enableInternalLpbk);
static void    App_mcanInitMsgRamConfigParams(MCAN_MsgRAMConfigParams *msgRAMConfigParams);
static void    App_mcanEnableIntr(void);
static void    App_mcanConfigTxMsg(MCAN_TxBufElement *txMsg);
static void    App_mcanCompareMsg(MCAN_TxBufElement *txMsg,MCAN_RxBufElement *rxMsg);
static void    App_mcanInitStdFilterElemParams( MCAN_StdMsgIDFilterElement *stdFiltElem,uint32_t bufNum);

void mcan_loopback_interrupt_main(void *args)
{
    int32_t                 status = SystemP_SUCCESS;
    HwiP_Params             hwiPrms;
    MCAN_TxBufElement       txMsg;
    MCAN_ProtocolStatus     protStatus;
    MCAN_RxBufElement       rxMsg;
    MCAN_RxNewDataStatus    newDataStatus;
    MCAN_ErrCntStatus       errCounter;
    uint32_t                i, bufNum, fifoNum, bitPos = 0U;

    // 1 调试串口初始化，在ti_board_open_close.c和 ti_drivers_open_close.c 文件中定义.
    // 上述两个文件在makefile过程中，通过example.syscfg文件生产，编译完成后通过make clean清除。
    Drivers_open();
    Board_driversOpen();

    DebugP_log("[MCAN] Loopback Interrupt mode, application started ...\r\n");

    // Tx/Rx 信号量结构
    status = SemaphoreP_constructBinary(&gMcanTxDoneSem, 0);
    DebugP_assert(SystemP_SUCCESS == status);
    status = SemaphoreP_constructBinary(&gMcanRxDoneSem, 0);
    DebugP_assert(SystemP_SUCCESS == status);

    // 中断寄存器
    HwiP_Params_init(&hwiPrms);
    hwiPrms.intNum      = APP_MCAN_INTR_NUM;
    hwiPrms.callback    = &App_mcanIntrISR;
    status              = HwiP_construct(&gMcanHwiObject, &hwiPrms);
    DebugP_assert(status == SystemP_SUCCESS);

    // 获取CAN 寄存器地址
    gMcanBaseAddr = (uint32_t) AddrTranslateP_getLocalAddr(APP_MCAN_BASE_ADDR);

    // CAN初始化相关参数, 不使能内回环模式，使能外部发送模式
    App_mcanConfig(TRUE);

    // 使能中断
    App_mcanEnableIntr();

    // 发送和接收字节，然后比对是否一致
    for (i = 0U; i < APP_MCAN_MSG_LOOP_COUNT; i++)
    {
        App_mcanConfigTxMsg(&txMsg);// 组织CAN帧 要发送的数据

        bufNum = 0U; // 选择buffer序号，最大32
        // 中断发送函数
        status = MCAN_txBufTransIntrEnable(gMcanBaseAddr, bufNum, (uint32_t)TRUE);
        DebugP_assert(status == CSL_PASS);

        // 将组织好的帧放入 Tx buffer中
        MCAN_writeMsgRam(gMcanBaseAddr, MCAN_MEM_TYPE_BUF, bufNum, &txMsg);

        // 添加传输请求时，此功能将触发传输
        status = MCAN_txBufAddReq(gMcanBaseAddr, bufNum);
        DebugP_assert(status == CSL_PASS);

        // 等待发送完成的信号量
        SemaphoreP_pend(&gMcanTxDoneSem, SystemP_WAIT_FOREVER);

        // 检测发送过程状态 是否有错
        MCAN_getProtocolStatus(gMcanBaseAddr, &protStatus);
        if (((MCAN_ERR_CODE_NO_ERROR != protStatus.lastErrCode) ||
             (MCAN_ERR_CODE_NO_CHANGE != protStatus.lastErrCode)) &&
            ((MCAN_ERR_CODE_NO_ERROR != protStatus.dlec) ||
             (MCAN_ERR_CODE_NO_CHANGE != protStatus.dlec)) &&
            (0U != protStatus.pxe))
        {
             DebugP_assert(FALSE);
        }

        // 等待发送完成的信号量
        SemaphoreP_pend(&gMcanRxDoneSem, SystemP_WAIT_FOREVER);

        // 检测接收过程状态 是否有错
        MCAN_getErrCounters(gMcanBaseAddr, &errCounter);
	//DebugP_log("recErrCnt:%d,canErrLogCnt:%d \r\n",errCounter.recErrCnt,errCounter.canErrLogCnt);
        DebugP_assert((0U == errCounter.recErrCnt) &&
                      (0U == errCounter.canErrLogCnt));

        //  从rx buffer中提取数据
        MCAN_getNewDataStatus(gMcanBaseAddr, &newDataStatus);
        MCAN_clearNewDataStatus(gMcanBaseAddr, &newDataStatus);

        // 确定包含数据的buffer
        bufNum = 0U;
        fifoNum = MCAN_RX_FIFO_NUM_0;

        bitPos = (1U << bufNum);
        if (bitPos == (newDataStatus.statusLow & bitPos))
        {
            MCAN_readMsgRam(gMcanBaseAddr, MCAN_MEM_TYPE_BUF, bufNum, fifoNum, &rxMsg);
        }
        else
        {
            DebugP_assert(FALSE);
        }

        // 比较发送和接收字节是否一致
        App_mcanCompareMsg(&txMsg, &rxMsg);
    }
    // 回收信号量
    HwiP_destruct(&gMcanHwiObject);
    SemaphoreP_destruct(&gMcanTxDoneSem);
    SemaphoreP_destruct(&gMcanRxDoneSem);

    DebugP_log("All tests have passed!!\r\n");

    Board_driversClose();
    Drivers_close();

    return;
}

static void App_mcanConfig(Bool enableInternalLpbk)
{
    MCAN_StdMsgIDFilterElement stdFiltElem[APP_MCAN_STD_ID_FILTER_CNT] = {0U};
    MCAN_InitParams            initParams = {0U};
    MCAN_ConfigParams          configParams = {0U};
    MCAN_MsgRAMConfigParams    msgRAMConfigParams = {0U};
    MCAN_BitTimingParams       bitTimes = {0U};
    uint32_t                   i;

    // MCAN部分参数初始化 
    MCAN_initOperModeParams(&initParams);
    // CAN FD模式使能，可变波特率使能
    initParams.fdMode          = TRUE;
    initParams.brsEnable       = TRUE;

    // CAN 过滤器参数设置
    MCAN_initGlobalFilterConfigParams(&configParams);

    // CAN 波特设置 如果是CAN FD模式 控制区1M 数据区5M
    MCAN_initSetBitTimeParams(&bitTimes);

    // CAN 发送和接收 buffer FIFO配置
    App_mcanInitMsgRamConfigParams(&msgRAMConfigParams);

    // 初始化接收过滤器设置，应和发送ID一样 针对扩展ID
    for (i = 0U; i < APP_MCAN_STD_ID_FILTER_CNT; i++)
    {
        App_mcanInitStdFilterElemParams(&stdFiltElem[i], i);
    }
    // 等待内存初始化完成
    while (FALSE == MCAN_isMemInitDone(gMcanBaseAddr))
    {}

    // 将mcan置于软件初始化模式下（配置完成后会恢复正常模式）
    MCAN_setOpMode(gMcanBaseAddr, MCAN_OPERATION_MODE_SW_INIT);
    while (MCAN_OPERATION_MODE_SW_INIT != MCAN_getOpMode(gMcanBaseAddr))
    {}

    // CAN初始化
    MCAN_init(gMcanBaseAddr, &initParams);
    // 配置CAN 过滤器相关参数
    MCAN_config(gMcanBaseAddr, &configParams);
    // 波特率相关参数的时间偏移设置
    MCAN_setBitTime(gMcanBaseAddr, &bitTimes);
     // 配置buffer FIFO
    MCAN_msgRAMConfig(gMcanBaseAddr, &msgRAMConfigParams);
    // 设置扩展帧过滤
    MCAN_setExtIDAndMask(gMcanBaseAddr, APP_MCAN_EXT_ID_MASK);

    // 配置扩展帧过滤
    for (i = 0U; i < APP_MCAN_STD_ID_FILTER_CNT; i++)
    {
        MCAN_addStdMsgIDFilter(gMcanBaseAddr, i, &stdFiltElem[i]);
    }
    // 配置内回环模式
    if (TRUE == enableInternalLpbk)
    {
        MCAN_lpbkModeEnable(gMcanBaseAddr, MCAN_LPBK_MODE_INTERNAL, TRUE);
    }

    // 恢复can 为正常模式
    MCAN_setOpMode(gMcanBaseAddr, MCAN_OPERATION_MODE_NORMAL);
    while (MCAN_OPERATION_MODE_NORMAL != MCAN_getOpMode(gMcanBaseAddr))
    {}

    return;
}

static void App_mcanConfigTxMsg(MCAN_TxBufElement *txMsg)
{
    uint32_t i;

    // CAN帧 ID区组织填入，此处默认了开启CAN FD模式 和可变波特率
    MCAN_initTxBufElement(txMsg);
    // 此处修改初始化的数值，改为真正发送的数值
    txMsg->id  = ((APP_MCAN_STD_ID & MCAN_STD_ID_MASK) << MCAN_STD_ID_SHIFT);
    txMsg->dlc = MCAN_DATA_SIZE_64BYTES; // 标准CAN最大8字节 CAN FD最大64字节
    txMsg->fdf = TRUE; // 是否为CAN FD帧
    txMsg->xtd = FALSE; // 是否为扩展帧
    for (i = 0U; i < gMcanDataSize[MCAN_DATA_SIZE_64BYTES]; i++)
    {
        txMsg->data[i] = i;
    }

    return;
}

static void App_mcanInitStdFilterElemParams(MCAN_StdMsgIDFilterElement *stdFiltElem,
                                            uint32_t bufNum)
{
    // efid1 定义了标准帧ID 范围为0xc0
    stdFiltElem->sfid1 = APP_MCAN_STD_ID;
    // 如果FIFO模式被选择，则efid2应该被设置
    stdFiltElem->sfid2 = bufNum;
    // 在buffer中存储消息
    stdFiltElem->sfec  = MCAN_STD_FILT_ELEM_BUFFER;
    // 如果消息被存储在buffer中，eft的配置无效
    stdFiltElem->sft   = MCAN_STD_FILT_TYPE_RANGE;

    return;
}

static void App_mcanEnableIntr(void)
{
    MCAN_enableIntr(gMcanBaseAddr, MCAN_INTR_MASK_ALL, (uint32_t)TRUE);
    MCAN_enableIntr(gMcanBaseAddr,
                    MCAN_INTR_SRC_RES_ADDR_ACCESS, (uint32_t)FALSE);
    // 中断线0
    MCAN_selectIntrLine(gMcanBaseAddr, MCAN_INTR_MASK_ALL, MCAN_INTR_LINE_NUM_0);
    // 使能中断线0
    MCAN_enableIntrLine(gMcanBaseAddr, MCAN_INTR_LINE_NUM_0, (uint32_t)TRUE);

    return;
}

static void App_mcanInitMsgRamConfigParams(MCAN_MsgRAMConfigParams
                                           *msgRAMConfigParams)
{
    int32_t status;

    // 初始化硬件层buffer FIFO相关参数，每个buffer和FIFO为64字节
    MCAN_initMsgRamConfigParams(msgRAMConfigParams);

    // 发送使用5个FIFO 0个buffer
    msgRAMConfigParams->lss = APP_MCAN_STD_ID_FILTER_CNT;
    msgRAMConfigParams->lse = APP_MCAN_EXT_ID_FILTER_CNT;
    msgRAMConfigParams->txBufCnt = APP_MCAN_TX_BUFF_CNT;
    msgRAMConfigParams->txFIFOCnt = APP_MCAN_TX_FIFO_CNT;
    // 选择Buffer/FIFO模式 接收用5个FIFO
    msgRAMConfigParams->txBufMode = MCAN_TX_MEM_TYPE_BUF;
    msgRAMConfigParams->txEventFIFOCnt = APP_MCAN_TX_EVENT_FIFO_CNT;
    msgRAMConfigParams->rxFIFO0Cnt = APP_MCAN_FIFO_0_CNT;
    msgRAMConfigParams->rxFIFO1Cnt = APP_MCAN_FIFO_1_CNT;
    // 选择FIFO 阻塞模式
    msgRAMConfigParams->rxFIFO0OpMode = MCAN_RX_FIFO_OPERATION_MODE_BLOCKING;
    msgRAMConfigParams->rxFIFO1OpMode = MCAN_RX_FIFO_OPERATION_MODE_BLOCKING;
    // 计算发送和接收 buffer FIFO的内存地址
    status = MCAN_calcMsgRamParamsStartAddr(msgRAMConfigParams);
    DebugP_assert(status == CSL_PASS);

    return;
}

static void App_mcanCompareMsg(MCAN_TxBufElement *txMsg,
                               MCAN_RxBufElement *rxMsg)
{
    uint32_t i;

    if (((txMsg->id >> APP_MCAN_STD_ID_SHIFT) & APP_MCAN_STD_ID_MASK) ==
            ((rxMsg->id >> APP_MCAN_STD_ID_SHIFT) & APP_MCAN_STD_ID_MASK))
    {
        for (i = 0U; i < gMcanDataSize[MCAN_DATA_SIZE_64BYTES]; i++)
        {
            DebugP_log("i=%d,tx date=%d rx date=%d \r\n",i,txMsg->data[i],rxMsg->data[i]);
	    if (txMsg->data[i] != rxMsg->data[i])
            {
                DebugP_logError("Data mismatch !!!\r\n");
                DebugP_assert(FALSE);
            }
        }
    }
    else
    {
        DebugP_logError("Message ID mismatch !!!\r\n");
        DebugP_assert(FALSE);
    }

    return;
}

static void App_mcanIntrISR(void *arg)
{
    uint32_t intrStatus;

    intrStatus = MCAN_getIntrStatus(gMcanBaseAddr);
    MCAN_clearIntrStatus(gMcanBaseAddr, intrStatus);

    if (MCAN_INTR_SRC_TRANS_COMPLETE ==
        (intrStatus & MCAN_INTR_SRC_TRANS_COMPLETE))
    {
        SemaphoreP_post(&gMcanTxDoneSem);
    }

    // 如果使用FIFO0/FIFO1，需要用MCAN_INTR_SRC_RX_FIFO0_NEW_MSG/MCAN_INTR_SRC_RX_FIFO1_NEW_MSG 
    // 替代MCAN_INTR_SRC_DEDICATED_RX_BUFF_MSG macro
    if (MCAN_INTR_SRC_DEDICATED_RX_BUFF_MSG ==
        (intrStatus & MCAN_INTR_SRC_DEDICATED_RX_BUFF_MSG))
    {
        SemaphoreP_post(&gMcanRxDoneSem);
    }

    return;
}
