#include "hFiles.h"
#include "dataHome.h"
#include "sysContext.h"
//#include "listHelper.h"

#include "chlEnc.h"
#include "sysCfg.h"

#define CELL_BUFFER_NUM (14*64)
#define CELL_BUFFER_SIZE 896

#define FRAME_HEAD_INFO 0x1D80
#define BCH_PARITY_AREA_SIZE 13

#ifdef ALLOC_IRAM_CELL_BUFFER
#pragma DATA_ALIGN(pBuffMem, 128);
static unsigned char pBuffMem[(CELL_BUFFER_NUM+MAX_CELL_TYPE_NUM)*CELL_BUFFER_SIZE];
#else
static unsigned char *pBuffMem = NULL;
#endif

#pragma DATA_ALIGN(pCellMem, 128);
static unsigned char pCellMem[(CELL_BUFFER_NUM+MAX_CELL_TYPE_NUM)*sizeof(CELL)];

const unsigned short cellTypeInfoArr[MAX_CELL_TYPE_NUM] = \
{ \
	0x1C80, \
	0x1A80, \
	0x1880, \
	0x1680, \
	0x1480, \
	0x1280, \
	0x1080, \
	0x2080, \
	0x20C0, \
	0x20E0, \
	0x30C0, \
	0x30E0 \
};

const unsigned int cellSizeArr[MAX_CELL_TYPE_NUM] = { 491,491,491,491,491,491,491,491,743,869,743,869 };

const unsigned char Walsh[MAX_FRAME_NUMBER][8] = \
{ \
	{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, \
	{0x55,0x55,0x55,0x55,0x55,0x55,0x55,0x55}, \
	{0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33}, \
	{0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66}, \
	{0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F}, \
	{0x5A,0x5A,0x5A,0x5A,0x5A,0x5A,0x5A,0x5A}, \
	{0x3C,0x3C,0x3C,0x3C,0x3C,0x3C,0x3C,0x3C}, \
	{0x69,0x69,0x69,0x69,0x69,0x69,0x69,0x69}, \
	{0x00,0xFF,0x00,0xFF,0x00,0xFF,0x00,0xFF}, \
	{0x55,0xAA,0x55,0xAA,0x55,0xAA,0x55,0xAA}, \
	{0x33,0xCC,0x33,0xCC,0x33,0xCC,0x33,0xCC}, \
	{0x66,0x99,0x66,0x99,0x66,0x99,0x66,0x99}, \
	{0x0F,0xF0,0x0F,0xF0,0x0F,0xF0,0x0F,0xF0}, \
	{0x5A,0xA5,0x5A,0xA5,0x5A,0xA5,0x5A,0xA5}, \
	{0x3C,0xC3,0x3C,0xC3,0x3C,0xC3,0x3C,0xC3}, \
	{0x69,0x96,0x69,0x96,0x69,0x96,0x69,0x96}, \
	{0xFF,0xFF,0x00,0x00,0xFF,0xFF,0x00,0x00}, \
	{0xAA,0xAA,0x55,0x55,0xAA,0xAA,0x55,0x55}, \
	{0xCC,0xCC,0x33,0x33,0xCC,0xCC,0x33,0x33}, \
	{0x99,0x99,0x66,0x66,0x99,0x99,0x66,0x66}, \
	{0xF0,0xF0,0x0F,0x0F,0xF0,0xF0,0x0F,0x0F}, \
	{0xA5,0xA5,0x5A,0x5A,0xA5,0xA5,0x5A,0x5A}, \
	{0xC3,0xC3,0x3C,0x3C,0xC3,0xC3,0x3C,0x3C}, \
	{0x96,0x96,0x69,0x69,0x96,0x96,0x69,0x69}, \
	{0xFF,0x00,0x00,0xFF,0xFF,0x00,0x00,0xFF}, \
	{0xAA,0x55,0x55,0xAA,0xAA,0x55,0x55,0xAA}, \
	{0xCC,0x33,0x33,0xCC,0xCC,0x33,0x33,0xCC}, \
	{0x99,0x66,0x66,0x99,0x99,0x66,0x66,0x99}, \
	{0xF0,0x0F,0x0F,0xF0,0xF0,0x0F,0x0F,0xF0}, \
	{0xA5,0x5A,0x5A,0xA5,0xA5,0x5A,0x5A,0xA5}, \
	{0xC3,0x3C,0x3C,0xC3,0xC3,0x3C,0x3C,0xC3}, \
	{0x96,0x69,0x69,0x96,0x96,0x69,0x69,0x96}, \
	{0xFF,0xFF,0xFF,0xFF,0x00,0x00,0x00,0x00}, \
	{0xAA,0xAA,0xAA,0xAA,0x55,0x55,0x55,0x55}, \
	{0xCC,0xCC,0xCC,0xCC,0x33,0x33,0x33,0x33}, \
	{0x99,0x99,0x99,0x99,0x66,0x66,0x66,0x66}, \
	{0xF0,0xF0,0xF0,0xF0,0x0F,0x0F,0x0F,0x0F}, \
	{0xA5,0xA5,0xA5,0xA5,0x5A,0x5A,0x5A,0x5A}, \
	{0xC3,0xC3,0xC3,0xC3,0x3C,0x3C,0x3C,0x3C}, \
	{0x96,0x96,0x96,0x96,0x69,0x69,0x69,0x69}, \
	{0xFF,0x00,0xFF,0x00,0x00,0xFF,0x00,0xFF}, \
	{0xAA,0x55,0xAA,0x55,0x55,0xAA,0x55,0xAA}, \
	{0xCC,0x33,0xCC,0x33,0x33,0xCC,0x33,0xCC}, \
	{0x99,0x66,0x99,0x66,0x66,0x99,0x66,0x99}, \
	{0xF0,0x0F,0xF0,0x0F,0x0F,0xF0,0x0F,0xF0}, \
	{0xA5,0x5A,0xA5,0x5A,0x5A,0xA5,0x5A,0xA5}, \
	{0xC3,0x3C,0xC3,0x3C,0x3C,0xC3,0x3C,0xC3}, \
	{0x96,0x69,0x96,0x69,0x69,0x96,0x69,0x96}, \
	{0x00,0x00,0xFF,0xFF,0xFF,0xFF,0x00,0x00}, \
	{0x55,0x55,0xAA,0xAA,0xAA,0xAA,0x55,0x55}, \
	{0x33,0x33,0xCC,0xCC,0xCC,0xCC,0x33,0x33}, \
	{0x66,0x66,0x99,0x99,0x99,0x99,0x66,0x66}, \
	{0x0F,0x0F,0xF0,0xF0,0xF0,0xF0,0x0F,0x0F}, \
	{0x5A,0x5A,0xA5,0xA5,0xA5,0xA5,0x5A,0x5A}, \
	{0x3C,0x3C,0xC3,0xC3,0xC3,0xC3,0x3C,0x3C}, \
	{0x69,0x69,0x96,0x96,0x96,0x96,0x69,0x69}, \
	{0x00,0xFF,0xFF,0x00,0xFF,0x00,0x00,0xFF}, \
	{0x55,0xAA,0xAA,0x55,0xAA,0x55,0x55,0xAA}, \
	{0x33,0xCC,0xCC,0x33,0xCC,0x33,0x33,0xCC}, \
	{0x66,0x99,0x99,0x66,0x99,0x66,0x66,0x99}, \
	{0x0F,0xF0,0xF0,0x0F,0xF0,0x0F,0x0F,0xF0}, \
	{0x5A,0xA5,0xA5,0x5A,0xA5,0x5A,0x5A,0xA5}, \
	{0x3C,0xC3,0xC3,0x3C,0xC3,0x3C,0x3C,0xC3}, \
	{0x69,0x96,0x96,0x69,0x96,0x69,0x69,0x96} \
};

FRAMEAREA frameArea[MAX_FRAME_NUMBER] = \
{ \
/* 0*/{5,80,1,9},
/* 1*/{11,16,2,9},
/* 2*/{22,16,3,9},
/* 3*/{45,0,4,0},
/* 4*/{90,0,5,0},
/* 5*/{180,0,6,0},
/* 6*/{360,0,7,0},
/* 7*/{720,0,8,0},
/* 8*/{720,0,9,0},
/* 9*/{720,0,10,0},
/*10*/{1080,0,11,0},
/*11*/{1080,0,12,0},
/*12*/{0,0,0,0}, 
/*13*/{0,0,0,0}, 
/*14*/{0,0,0,0}, 
/*15*/{0,0,0,0}, 
/*16*/{1,592,1,8}, 
/*17*/{2,592,2,8}, 
/*18*/{4,592,3,8}, 
/*19*/{8,592,4,8}, 
/*20*/{16,592,5,8},
/*21*/{32,592,6,8}, 
/*22*/{64,592,7,8}, 
/*23*/{1,592,1,9}, 
/*24*/{2,592,2,9},
/*25*/{4,592,3,9},
/*26*/{8,592,4,9},
/*27*/{16,592,5,9},
/*28*/{32,592,6,9},
/*29*/{64,592,7,9},
/*30*/{1,592,1,10},
/*31*/{2,592,2,10},
/*32*/{4,592,3,10},
/*33*/{8,592,4,10},
/*34*/{16,592,5,10},
/*35*/{32,592,6,10},
/*36*/{64,592,7,10},
/*37*/{1,74,1,5}, 
/*38*/{2,148,2,6}, 
/*39*/{4,148,3,6}, 
/*40*/{72,864,7,11}, 
/*41*/{72,864,7,12}, 
/*42*/{0,0,0,0}, 
/*43*/{0,0,0,0}, 
/*44*/{0,0,0,0}, 
/*45*/{0,0,0,0}, 
/*46*/{144,864,8,11},
/*47*/{144,864,8,12},
/*48*/{0,0,0,0},
/*49*/{0,0,0,0},
/*50*/{0,0,0,0},
/*51*/{0,0,0,0},
/*52*/{144,864,9,11},
/*53*/{144,864,9,12},
/*54*/{0,0,0,0},
/*55*/{0,0,0,0},
/*56*/{0,0,0,0},
/*57*/{0,0,0,0},
/*58*/{144,864,10,11},
/*59*/{144,864,10,12}, 
/*60*/{0,0,0,0},
/*61*/{0,0,0,0},
/*62*/{0,0,0,0},
/*63*/{0,0,0,0} 
};
unsigned int frameAreaUpdateTime = 0;
unsigned int frameAreaVer = 0;

typedef struct _QUE
{
	int count;
	void *pHead;
	void *pTail;
	HANDLE hEvent;
} QUE, *PQUE;

static QUE emptyCellQ;

typedef struct _QELEM
{
	void *pNext;
} QELEM,*PQELEM;
//--------------------------------------------------------------------
// enqElemQ()
//
// Enqueue a element onto the supplied queue
//--------------------------------------------------------------------
static inline void enqElemQ(PQUE pQ, void *pElem)
{

	((PQELEM)pElem)->pNext = 0;

	if(!pQ->count)
	{
		// Queue is empty - Initialize it with this one element
		pQ->pHead = pElem;
		pQ->pTail = pElem;
		pQ->count++;
	}
	else
	{
		// Queue is not empty - Push onto END
		((PQELEM)(pQ->pTail))->pNext = pElem;
		pQ->pTail = pElem;
		pQ->count++;
	}

	if(pQ->hEvent) PostEvent(pQ->hEvent);

}


//--------------------------------------------------------------------
// deqElemQ()
//
// Dequeue a element from the supplied queue
//--------------------------------------------------------------------
static inline void* deqElemQ(PQUE pQ)
{
	void *pElem = pQ->pHead;

	if(pElem)
	{
		pQ->pHead = ((PQELEM)pElem)->pNext;
		if(!pQ->pHead)
			pQ->pTail = 0;
		pQ->count--;
		((PQELEM)pElem)->pNext = 0;
	}

	return pElem;
}

typedef struct _XFRAME
{
	struct _XFRAME *pNext;
	unsigned char cellBuffer[2+8+2]; // '2'-cell info, '8'-walsh code, '2'-rsv.
	CELL firstCell;
	int number;
	unsigned int xAreaCellNum;
	unsigned int yAreaCellNum;
	QUE xAreaCellQ;
	QUE yAreaCellQ;
} XFRAME,*PXFRAME;

typedef struct _XFRAMEQ
{
	int count;
	PXFRAME pHead;
	PXFRAME pTail;
} XFRAMEQ;

static XFRAMEQ XFrameQ;

static inline void enqXFrameQ(PXFRAME pFrame)
{

	pFrame->pNext = 0;

	if(!XFrameQ.count)
	{
		XFrameQ.pHead = pFrame;
		XFrameQ.pTail = pFrame;
		XFrameQ.count = 1;
	}
	else
	{
		XFrameQ.pTail->pNext = pFrame;
		XFrameQ.pTail = pFrame;
		XFrameQ.count++;
	}

}

static inline PXFRAME deqXFrameQ()
{
	PXFRAME pFrame = NULL;

	if(XFrameQ.pHead == XFrameQ.pTail) // We are at the end of XFrameQ
	{
		if(pFrame = XFrameQ.pHead)
		{
			// reuse this frame
			pFrame->xAreaCellNum = frameArea[pFrame->number].xAreaCellNum + 1;
			pFrame->yAreaCellNum = frameArea[pFrame->number].yAreaCellNum;

			enqElemQ(&pFrame->xAreaCellQ, &pFrame->firstCell);
		}

		// This is the last one, so, will not be dequeued from the queue.
		pFrame = NULL;
	}
	else
	{
		pFrame = XFrameQ.pHead;
		XFrameQ.pHead = XFrameQ.pHead->pNext;
		XFrameQ.count--;
	}

	return pFrame;

}

#pragma DATA_ALIGN(XFramePool, 128);
static XFRAME XFramePool[MAX_FRAME_NUMBER];

static QUE emptyXFrameQ;

HANDLE hEventGate = NULL;

static inline void initXFrame(PXFRAME pFrm, int frmNum)
{
	int i;

	pFrm->number = frmNum;
	pFrm->xAreaCellNum = frameArea[frmNum].xAreaCellNum + 1; // '1' is the head
	pFrm->yAreaCellNum = frameArea[frmNum].yAreaCellNum;

	pFrm->firstCell.buffer = pFrm->cellBuffer;
	pFrm->firstCell.offset = 2;  // sizeof(unsigned short)
	pFrm->firstCell.len = 2 + 8;
	pFrm->firstCell.type = \
		(frmNum<<FRAME_NUMBER_FIELD_OFFSET) | CELL_TYPE_XAREA | CELL_TYPE_HEAD;
	pFrm->firstCell.fPseudo = true; //So, this cell will not be enqueued onto emptyCellQ, when used.

	*((unsigned short *)pFrm->cellBuffer) = FRAME_HEAD_INFO;
	for(i=0; i<8; i++) pFrm->cellBuffer[2+i] = Walsh[frmNum][i];

	enqElemQ(&pFrm->xAreaCellQ, &pFrm->firstCell);
}

static inline PCELL makeDefaultCell(unsigned char type)
{
	PCELL pCell;
	int num = numFromType(type);
	int area = areaFromType(type);

	int cellType;

	if(!area)
		cellType = frameArea[num].xAreaCellType;
	else
		cellType = frameArea[num].yAreaCellType;

	pCell = (PCELL)(pCellMem+(CELL_BUFFER_NUM+cellType-1)*sizeof(CELL));

	return pCell;
}

void InitDataHomeModule()
{
	int i;
	unsigned char *pBuffTmp;
	unsigned char *pCellTmp;
	PXFRAME pFrame;
	PCELL pCell;

#ifndef ALLOC_IRAM_CELL_BUFFER
	pBuffMem = (unsigned char *)CELL_BUFFER_DDR_ADDR;
#endif

	mmzero(&XFramePool, sizeof(XFramePool));
	mmzero(&emptyCellQ, sizeof(emptyCellQ));
	
	// Get temp pointers
	pBuffTmp = pBuffMem;
	pCellTmp = pCellMem;

	// Break memory array into cell buffers and push onto free queue
	for(i=0; i<CELL_BUFFER_NUM; i++)
	{
		pCell = (PCELL)pCellTmp;
		pCellTmp += sizeof(CELL);

		pCell->offset = 2; // sizeof(unsigned short), the reserved room will be filled with cell info
		pCell->len = pCell->offset;
		pCell->fPseudo = false;
		pCell->type = 0;

		pCell->buffer = pBuffTmp;
		pBuffTmp += CELL_BUFFER_SIZE;

		enqElemQ(&emptyCellQ, pCell);
	}

	// Initialize the default cells
	for(i=0; i<MAX_CELL_TYPE_NUM; i++)
	{
		pCell = (PCELL)pCellTmp;
		pCellTmp += sizeof(CELL);

		pCell->offset = 2;
		pCell->len = pCell->offset + cellSizeArr[i];
		#ifdef _BCH_ENCODE_
		pCell->len += BCH_PARITY_AREA_SIZE;
		#endif
		pCell->len = (pCell->len+1)&(~1);
		pCell->fPseudo = true;
		pCell->type = 0;

		pCell->buffer = pBuffTmp;
		pBuffTmp += CELL_BUFFER_SIZE;

		*((volatile unsigned short*)&pCell->buffer[0]) = cellTypeInfoArr[i];
		mmzero((void *)&pCell->buffer[2], pCell->len);
	}

	mmzero(&emptyXFrameQ, sizeof(emptyXFrameQ));
	for(i=0; i<MAX_FRAME_NUMBER; i++)
	{
		enqElemQ(&emptyXFrameQ, &XFramePool[i]);
	}
	emptyXFrameQ.hEvent = CreateEvent(0);
	
	mmzero(&XFrameQ, sizeof(XFrameQ));
	pFrame = (PXFRAME)deqElemQ(&emptyXFrameQ);
	if(pFrame)
	{
		initXFrame(pFrame, DEFAULT_FRAME_NUMBER);
		enqXFrameQ(pFrame);
	}

	hEventGate = CreateEvent(1);
	
}

PCELL GetCell(HANDLE hEvent)
{
	PCELL pCell = NULL;
	PendEvent(hEventGate,TO_FOREVER);

	pCell = (PCELL)deqElemQ(&emptyCellQ);

	if(!pCell && hEvent && !emptyCellQ.hEvent)
	{
		emptyCellQ.hEvent = hEvent;
		PostEvent(hEventGate);
		// We must wait until some empty cell is available.
		while(!pCell && 
			PendEvent(emptyCellQ.hEvent,TO_FOREVER))
		{
			PendEvent(hEventGate,TO_FOREVER);
			ResetEvent(emptyCellQ.hEvent, 0);
			pCell = (PCELL)deqElemQ(&emptyCellQ);
			if(pCell) emptyCellQ.hEvent = NULL;
			PostEvent(hEventGate);
		}
		
	}
	else
		PostEvent(hEventGate);

	return pCell;
}

void FreeCell(PCELL pCell)
{
	if(pCell && !pCell->fPseudo)
	{
		// reset the cell's members
		pCell->len = pCell->offset;
		pCell->type = 0; // its the invalid type

		// enter the critical section
		PendEvent(hEventGate,TO_FOREVER);
		
		enqElemQ(&emptyCellQ,pCell);

		PostEvent(hEventGate);
	}
}

PCELL GetFilledCell(bool fMust)
{
	PXFRAME pFrame;
	PCELL pCell = NULL;
	unsigned char type;

	PendEvent(hEventGate,TO_FOREVER);

	pFrame = XFrameQ.pHead;

	if(pFrame)
	{
		if(pFrame->xAreaCellNum)
		{
			pCell = (PCELL)deqElemQ(&pFrame->xAreaCellQ);
			if(!pCell && fMust)
			{
				type = (pFrame->number << FRAME_NUMBER_FIELD_OFFSET)| \
										CELL_TYPE_XAREA | \
										CELL_TYPE_DATA;
				pCell = makeDefaultCell(type);
			}

			if(pCell) pFrame->xAreaCellNum--;
		}
		else if(pFrame->yAreaCellNum)
		{
			pCell = (PCELL)deqElemQ(&pFrame->yAreaCellQ);
			if(!pCell && fMust)
			{
				type = (pFrame->number << FRAME_NUMBER_FIELD_OFFSET)| \
										CELL_TYPE_YAREA | \
										CELL_TYPE_DATA;
				pCell = makeDefaultCell(type);
			}

			if(pCell) pFrame->yAreaCellNum--;
		}

		// If the XFrame is empty, free it!
		if(!(pFrame->xAreaCellNum | pFrame->yAreaCellNum))
		{
			pFrame = deqXFrameQ();
			if(pFrame) enqElemQ(&emptyXFrameQ,pFrame);
		}
	}

	PostEvent(hEventGate);

	return pCell;

}

#ifdef _BCH_ENCODE_
#pragma DATA_ALIGN(rsvBuffer, 128);
static unsigned char rsvBuffer[CELL_BUFFER_SIZE];

bool InsertFilledCell(PCELL pCell, bool fBlock)
{
	static int i = 0, j = 0, k = 0, size = 0;
	static PCELL pcellArr[4] = {0,0,0,0};
	static PXFRAME pFrmCache = 0;
	static bool fFailed = false;
	static int iEnterNum = 0;

	int temp;
	bool sChanged = false;
	PXFRAME pFrame = NULL;
	unsigned char num, area;
	unsigned char *buffN[4];
	
	if(!pCell && i>0) 
	{
		k++;
		if((k>1&&i>=3)||
			(k>2&&i>=2)||
			(k>3&&i>=1))
		{
			if(iEnterNum>0) return true;
			
			PendEvent(hEventGate,TO_FOREVER);
			goto pre_insert;
		}
	}

	if(!pCell) return true;
	
	num = numFromType(pCell->type);
	area = areaFromType(pCell->type);

	if(!area)
	{
		*((unsigned short*)(pCell->buffer)) = cellTypeInfoArr[frameArea[num].xAreaCellType-1];
		pCell->len = pCell->offset + cellSizeArr[frameArea[num].xAreaCellType - 1];
	}
	else
	{
		*((unsigned short*)(pCell->buffer)) = cellTypeInfoArr[frameArea[num].yAreaCellType-1];
		pCell->len = pCell->offset + cellSizeArr[frameArea[num].yAreaCellType - 1];
	}
	
	PendEvent(hEventGate,TO_FOREVER);

	iEnterNum++;

	if(pCell) pcellArr[i++] = pCell;
pre_insert:
	if(fFailed)
	{
		if(pCell)
		{
			bch_en(pcellArr[i-1]->len-pcellArr[i-1]->offset,
					&pcellArr[i-1]->buffer[pcellArr[i-1]->offset],
					rsvBuffer,
					rsvBuffer,
					rsvBuffer);
			pcellArr[i-1]->len += BCH_PARITY_AREA_SIZE;
		}
		fFailed = false;
	}
	else
	{
		if(!size)
		{
			size = pcellArr[i-1]->len-pcellArr[i-1]->offset;
		}
		else if(size != pcellArr[i-1]->len-pcellArr[i-1]->offset)
		{
			sChanged = true;
		}
		
		if(sChanged || !pCell)
		{
			if(sChanged) i--;
			
			for(temp=0; temp<4; temp++)
			{
				if(temp<i)
				{
					buffN[temp] = &pcellArr[temp]->buffer[pcellArr[temp]->offset];
					pcellArr[temp]->len += BCH_PARITY_AREA_SIZE;
				}
				else
				{
					buffN[temp] = rsvBuffer;
				}
			}
			bch_en(size, buffN[0], buffN[1], buffN[2], buffN[3]);

		}
		else
		{
			if(i == 4)
			{
				bch_en(pcellArr[0]->len-pcellArr[0]->offset,
						&pcellArr[0]->buffer[pcellArr[0]->offset],
						&pcellArr[1]->buffer[pcellArr[1]->offset],
						&pcellArr[2]->buffer[pcellArr[2]->offset],
						&pcellArr[3]->buffer[pcellArr[3]->offset]);
				
				pcellArr[0]->len += BCH_PARITY_AREA_SIZE;
				pcellArr[1]->len += BCH_PARITY_AREA_SIZE;
				pcellArr[2]->len += BCH_PARITY_AREA_SIZE;
				pcellArr[3]->len += BCH_PARITY_AREA_SIZE;

			}
			else
			{
				iEnterNum = 0;
				PostEvent(hEventGate);
				return true;	
			}
		}
	}

insert_next:
	pCell = pcellArr[j];
	num = numFromType(pCell->type);
	area = areaFromType(pCell->type);
	
	/*
	// aligned with 16-bits
	while(pCell->len % 2)
	{
		pCell->buffer[pCell->len] = 0; // padded with '0'
		pCell->len++;
	}
	*/
	
	// We try to find an existing XFrame, and insert the cell into it...
	// Firstly, take our chance on the cached XFrame.
	if(pFrmCache && pFrmCache->number == num)
	{
		if(!area)
		{
			if(pFrmCache->xAreaCellQ.count >= pFrmCache->xAreaCellNum)
				goto lbl_new_frame;

			enqElemQ(&pFrmCache->xAreaCellQ,pCell);
			
			if(++j == i)
			{
				goto lbl_exit;
			}
			else
			{
				goto insert_next;
			}
		}
		else
		{
			if(pFrmCache->yAreaCellQ.count >= pFrmCache->yAreaCellNum)
				goto lbl_new_frame;

			enqElemQ(&pFrmCache->yAreaCellQ,pCell);
			
			if(++j == i)
			{
				goto lbl_exit;
			}
			else
			{
				goto insert_next;
			}
		}
	}
	else
	{
		pFrame = XFrameQ.pHead;
		while(pFrame)
		{
			if(pFrame->number == num) 
			{
				if(!area)
				{
					if(pFrame->xAreaCellQ.count < pFrame->xAreaCellNum)
					{
						enqElemQ(&pFrame->xAreaCellQ,pCell);
						pFrmCache = pFrame;

						if(++j == i)
						{
							goto lbl_exit;
						}
						else
						{
							goto insert_next;
						}
					}
				}
				else
				{
					if(pFrame->yAreaCellQ.count < pFrame->yAreaCellNum)
					{
						enqElemQ(&pFrame->yAreaCellQ,pCell);
						pFrmCache = pFrame;

						if(++j == i)
						{
							goto lbl_exit;
						}
						else
						{
							goto insert_next;
						}
					}	
				}
			}
			pFrame = pFrame->pNext;
		}	
	}
	
lbl_new_frame:
	// Here, we must create a new XFrame, then insert the cell into it...
	pFrame = (PXFRAME)deqElemQ(&emptyXFrameQ);
	if(!pFrame && fBlock && emptyXFrameQ.hEvent)
	{
		ResetEvent(emptyXFrameQ.hEvent, 0);
		PostEvent(hEventGate);
		// We are blocked, until some task free a XFrame
		while(!pFrame && 
			PendEvent(emptyXFrameQ.hEvent, TO_FOREVER))
		{
			PendEvent(hEventGate, TO_FOREVER);
			pFrame = (PXFRAME)deqElemQ(&emptyXFrameQ);
			PostEvent(hEventGate);
		}
		PendEvent(hEventGate, TO_FOREVER);
	}

	if(pFrame)
	{
		initXFrame(pFrame, num);
		enqXFrameQ(pFrame);
		if(!area)
			enqElemQ(&pFrame->xAreaCellQ, pCell);
		else
			enqElemQ(&pFrame->yAreaCellQ, pCell);

		pFrmCache = pFrame;

		if(++j == i)
		{
			goto lbl_exit;
		}
		else
		{
			goto insert_next;
		}
	}
	
	fFailed = true;
	iEnterNum = 0;
	PostEvent(hEventGate);
	return false;

lbl_exit:
	if(sChanged)
	{
		size = pcellArr[i]->len - pcellArr[i]->offset;
		pcellArr[0] = pcellArr[i];
		i = 1;
		j = k = 0;
	}
	else
	{
		i = j = k = size = 0;
	}
	
	iEnterNum = 0;
	PostEvent(hEventGate);
	return true;
}
#endif

#ifndef _BCH_ENCODE_

bool InsertFilledCell(PCELL pCell, bool fBlock)
{
	static PXFRAME pFrmCache = 0;

	PXFRAME pFrame = NULL;
	unsigned char num = numFromType(pCell->type);
	unsigned char area = areaFromType(pCell->type);

	if(!area)
	{
		*((unsigned short*)(pCell->buffer)) = cellTypeInfoArr[frameArea[num].xAreaCellType-1];
		pCell->len = pCell->offset + cellSizeArr[frameArea[num].xAreaCellType - 1];
	}
	else
	{
		*((unsigned short*)(pCell->buffer)) = cellTypeInfoArr[frameArea[num].yAreaCellType-1];
		pCell->len = pCell->offset + cellSizeArr[frameArea[num].yAreaCellType - 1];
	}


	while(pCell->len % 2)
	{
		pCell->buffer[pCell->len] = 0; // padded with '0'
		pCell->len++;
	}
	
	PendEvent(hEventGate,TO_FOREVER);

	// We try to find an existing XFrame, and insert the cell into it...
	// Firstly, take our chance on the cached XFrame.
	if(pFrmCache && pFrmCache->number == num)
	{
		if(!area)
		{
			if(pFrmCache->xAreaCellQ.count >= pFrmCache->xAreaCellNum)
				goto lbl_new_frame;

			enqElemQ(&pFrmCache->xAreaCellQ,pCell);
			PostEvent(hEventGate);
			return true;
		}
		else
		{
			if(pFrmCache->yAreaCellQ.count >= pFrmCache->yAreaCellNum)
				goto lbl_new_frame;

			enqElemQ(&pFrmCache->yAreaCellQ,pCell);
			PostEvent(hEventGate);
			return true;
		}
	}
	else
	{
		pFrame = XFrameQ.pHead;
		while(pFrame)
		{
			if(pFrame->number == num) 
			{
				if(!area)
				{
					if(pFrame->xAreaCellQ.count < pFrame->xAreaCellNum)
					{
						enqElemQ(&pFrame->xAreaCellQ,pCell);
						pFrmCache = pFrame;
						PostEvent(hEventGate);
						return true;
					}
				}
				else
				{
					if(pFrame->yAreaCellQ.count < pFrame->yAreaCellNum)
					{
						enqElemQ(&pFrame->yAreaCellQ,pCell);
						pFrmCache = pFrame;
						PostEvent(hEventGate);
						return true;
					}	
				}
			}
			pFrame = pFrame->pNext;
		}	
	}
	
lbl_new_frame:
	// Here, we must create a new XFrame, then insert the cell into it...
	pFrame = (PXFRAME)deqElemQ(&emptyXFrameQ);
	if(!pFrame && fBlock && emptyXFrameQ.hEvent)
	{
		ResetEvent(emptyXFrameQ.hEvent, 0);
		PostEvent(hEventGate);
		// We are blocked, until some task free a XFrame
		while(!pFrame && 
			PendEvent(emptyXFrameQ.hEvent, TO_FOREVER))
		{
			PendEvent(hEventGate, TO_FOREVER);
			pFrame = (PXFRAME)deqElemQ(&emptyXFrameQ);
			PostEvent(hEventGate);
		}
		PendEvent(hEventGate, TO_FOREVER);
	}

	if(pFrame)
	{
		initXFrame(pFrame, num);
		enqXFrameQ(pFrame);
		if(!area)
			enqElemQ(&pFrame->xAreaCellQ, pCell);
		else
			enqElemQ(&pFrame->yAreaCellQ, pCell);

		pFrmCache = pFrame;
		PostEvent(hEventGate);
		return true;
	}

	PostEvent(hEventGate);
	return false;
}

#endif
void InsertFrame(int num)
{
	PXFRAME pFrame = NULL;
	
	if(num<0 || num>63) return;
	
	PendEvent(hEventGate,TO_FOREVER);
	pFrame = (PXFRAME)deqElemQ(&emptyXFrameQ);
	if(!pFrame && emptyXFrameQ.hEvent)
	{
		ResetEvent(emptyXFrameQ.hEvent, 0);
		PostEvent(hEventGate);
		// We are blocked, until some task free a XFrame
		while(!pFrame && 
			PendEvent(emptyXFrameQ.hEvent, TO_FOREVER))
		{
			PendEvent(hEventGate, TO_FOREVER);
			pFrame = (PXFRAME)deqElemQ(&emptyXFrameQ);
			PostEvent(hEventGate);
		}
		PendEvent(hEventGate, TO_FOREVER);
	}

	initXFrame(pFrame, num);
	enqXFrameQ(pFrame);
	
	PostEvent(hEventGate);
}

void FillCellContent(int mode, unsigned char cVal)
{
	int i, j;
	unsigned char uVal = 0;
	PCELL pCell = (PCELL)(pCellMem+(CELL_BUFFER_NUM)*sizeof(CELL));
	switch(mode)
	{
	case CONTENT_MODE_CIRCULAR:
		for(i=0;i<MAX_CELL_TYPE_NUM;i++)
		{
			for(uVal=cVal,j=2;j<pCell->len;j++)
			{
				pCell->buffer[j] = uVal++;
			}
			pCell++;
		}
		break;
	case CONTENT_MODE_FIXED:
		for(i=0;i<MAX_CELL_TYPE_NUM;i++)
		{
			mmset(&pCell->buffer[2],pCell->len-2,cVal);
			pCell++;
		}
		break;
	default:
		break;
	}
}

static struct
{
	HANDLE hEvent;
	unsigned int nCyc;
	unsigned int chlMode;
	unsigned int datMode;
	unsigned int sCellCount;
	unsigned int sFrameCount;
	unsigned char numArr[MAX_FRAME_NUMBER];
	unsigned char nMax;
	bool fRun;
} dataGenInfo;

void ResetDataGenStatistics()
{
	dataGenInfo.sCellCount = 0;
	dataGenInfo.sFrameCount = 0;
}

unsigned int GetGenCellCount()
{
	return dataGenInfo.sCellCount;
}

unsigned int GetGenFrameCount()
{
	return dataGenInfo.sFrameCount;
}

bool SetDataGenInfo(unsigned int nCyc, unsigned int chlMode, unsigned int datMode, unsigned char *pFrmNum, unsigned char nMax, unsigned int *pTotalCell)
{
	int i;
	unsigned int cellNum = 0;

	if(pFrmNum && nMax<=MAX_FRAME_NUMBER)
	{
		for(i=0; i<nMax; i++)
		{
			if(pFrmNum[i]>MAX_FRAME_NUMBER)
				return false;
			else
				dataGenInfo.numArr[i] = pFrmNum[i];
		}
		dataGenInfo.nMax = nMax;
		dataGenInfo.chlMode = chlMode;
		dataGenInfo.datMode = datMode;
		dataGenInfo.nCyc = nCyc;

		if(pTotalCell)
		{
			if(chlMode==LOW_SNR_CHNL_MODE)
			{
				for(i=0; i<nMax; i++)
				{
					cellNum += frameArea[dataGenInfo.numArr[i]].xAreaCellNum;
				}
			}
			else
			{
				for(i=0; i<nMax; i++)
				{
					cellNum += frameArea[dataGenInfo.numArr[i]].xAreaCellNum;
					cellNum += frameArea[dataGenInfo.numArr[i]].yAreaCellNum;
				}
			}
			
			*pTotalCell = cellNum*nCyc;
		}
		return true;
	}

	return false;
}

void StartDataGen()
{
	if(!dataGenInfo.fRun)
	{
		dataGenInfo.fRun = true;
		if(dataGenInfo.hEvent)
			PostEvent(dataGenInfo.hEvent);
	}
}

void RestartDataGen()
{
	ResetDataGenStatistics();
	
	if(!dataGenInfo.fRun)
	{
		dataGenInfo.fRun = true;
		if(dataGenInfo.hEvent)
			PostEvent(dataGenInfo.hEvent);
	}
}

void StopDataGen()
{
	if(dataGenInfo.hEvent)
		ResetEvent(dataGenInfo.hEvent, 0);
	dataGenInfo.fRun = false;
}

void DataGenTask()
{
	int i, j;
	unsigned int xNum, yNum;
	unsigned char number, xType, yType, uVal;
	unsigned short cellFlag = TEST_CELL_FLAG;

	static unsigned int uIdentity = 0;
	PCELL pCell = 0;
	
	{ // Initialize dataGenInfo
	mmzero(&dataGenInfo, sizeof(dataGenInfo));
	dataGenInfo.hEvent = CreateEvent(0);
	}

	while(true)
	{
		if(dataGenInfo.fRun || PendEvent(dataGenInfo.hEvent, TO_FOREVER))
		{
			if(dataGenInfo.nCyc)
			{
				for(i=0; i<dataGenInfo.nMax; i++)
				{
					number = dataGenInfo.numArr[i];
					xType = frameArea[number].xAreaCellType;
					yType = frameArea[number].yAreaCellType;
					xNum = frameArea[number].xAreaCellNum;
					yNum = frameArea[number].yAreaCellNum;

					for(j=0; j<xNum; j++)
					{
						pCell = GetCell(dataGenInfo.hEvent);
						if(pCell)
						{
							pCell->type = (number<<FRAME_NUMBER_FIELD_OFFSET)|CELL_TYPE_XAREA|CELL_TYPE_DATA;
							mmcopy(&pCell->buffer[pCell->len], &cellFlag, sizeof(unsigned short));
							pCell->len += sizeof(unsigned short);
							mmcopy(&pCell->buffer[pCell->len], &uIdentity, sizeof(unsigned int));
							pCell->len += sizeof(unsigned int);
							uIdentity++;

							switch(FillMode(dataGenInfo.datMode))
							{
							case DATA_FILL_MODE_CIRCULAR:
								uVal = FillData(dataGenInfo.datMode);
								while(pCell->len<(cellSizeArr[xType-1]+pCell->offset))
								{
									pCell->buffer[pCell->len++] = uVal++;
								}
								break;
							case DATA_FILL_MODE_FIXED:
								uVal = FillData(dataGenInfo.datMode);
								mmset(&pCell->buffer[pCell->len], (cellSizeArr[xType-1]-sizeof(unsigned short)-sizeof(unsigned int)), uVal);
								pCell->len += (cellSizeArr[xType-1]-sizeof(unsigned short)-sizeof(unsigned int));
								break;
							default:
								break;
							}

							while(!InsertFilledCell(pCell,true)) TaskSleep(8);

							dataGenInfo.sCellCount++;
						}
					}

					if(dataGenInfo.chlMode == HIGH_SNR_CHNL_MODE)
					{
						for(j=0; j<yNum; j++)
						{
							pCell = GetCell(dataGenInfo.hEvent);
							if(pCell)
							{
								pCell->type = (number<<FRAME_NUMBER_FIELD_OFFSET)|CELL_TYPE_YAREA|CELL_TYPE_DATA;
								mmcopy(&pCell->buffer[pCell->len], &cellFlag, sizeof(unsigned short));
								pCell->len += sizeof(unsigned short);
								mmcopy(&pCell->buffer[pCell->len], &uIdentity, sizeof(unsigned int));
								pCell->len += sizeof(unsigned int);
								uIdentity++;

								switch(FillMode(dataGenInfo.datMode))
								{
								case DATA_FILL_MODE_CIRCULAR:
									uVal = FillData(dataGenInfo.datMode);
									while(pCell->len < (cellSizeArr[yType-1]+pCell->offset))
									{
										pCell->buffer[pCell->len++] = uVal++;
									}
									break;
								case DATA_FILL_MODE_FIXED:
									uVal = FillData(dataGenInfo.datMode);
									mmset(&pCell->buffer[pCell->len], (cellSizeArr[yType-1]-sizeof(unsigned short)-sizeof(unsigned int)), uVal);
									pCell->len += (cellSizeArr[yType-1]-sizeof(unsigned short)-sizeof(unsigned int));
									break;
								default:
									break;
								}

								while(!InsertFilledCell(pCell,true)) TaskSleep(8);

								dataGenInfo.sCellCount++;
							}
						}
					}
					dataGenInfo.sFrameCount++;
				}
				dataGenInfo.nCyc--;
			}
			else
			{
				for(i=0; i<dataGenInfo.nMax; i++)
				{
					if(!dataGenInfo.fRun) break;
					InsertFrame(dataGenInfo.numArr[i]);
				}
			}
		}
	}
}

#define IsValidModeTabVer(uVer) true

bool UpdateFrameModeTab(void *pTab, unsigned int uVer, unsigned int uTime)
{
	if(pTab && uTime>frameAreaUpdateTime && IsValidModeTabVer(uVer))
	{
		mmcopy(frameArea, pTab, MAX_FRAME_NUMBER*sizeof(FRAMEAREA));
		frameAreaUpdateTime = uTime;
		frameAreaVer = uVer;
		return true;
	}

	return false;
}

void GetFrameModeTab(void *pTab, unsigned int *pVer, unsigned int *pTime)
{
	if(pTab)
	{
		mmcopy(pTab, frameArea, MAX_FRAME_NUMBER*sizeof(FRAMEAREA));
	}
	if(pTime) *pTime = frameAreaUpdateTime;
	if(pVer) *pVer = frameAreaVer;
	
}

void GetFramVer(unsigned int *pVer)
{
	if(pVer) *pVer = frameAreaVer;
}
unsigned int GetWorkStatus()
{
	return dataGenInfo.nCyc;
}
