
#include <std.h>
#include <tsk.h>

#include "i2c.h"

#define I2C_WAITFOREVER  1

 /* Spin in a delay loop for delay iterations */
void I2C_wait(Uint32 delay)
{
    volatile Uint32 i, n;

    n = 0;
    for (i = 0; i < delay; i++)
    {
        n = n + 1;
    }
}

/* ------------------------------------------------------------------------ *
 *      Enable and initalize the I2C module                                 *
 * ------------------------------------------------------------------------ */
Int16 I2C_init( )
{
    I2C_ICMDR   = 0;                                // Reset I2C
    I2C_ICPSC   = CSL_I2C_CLOCK_PRESCALAR;          // Run I2C module at 10MHz (input SYSCLK2/6)
    I2C_ICCLKL  = CSL_I2C_CLOCK_CLKL;               // Config clk LOW for 50kHz
    I2C_ICCLKH  = CSL_I2C_CLOCK_CLKH;               // Config clk HIGH for 50kHz
    I2C_ICMDR  |= ICMDR_IRS;                        // Release I2C from reset

    return 0;
}

/* ------------------------------------------------------------------------ *
 *      I2C write in Master mode                                            *
 *                                                                          *
 *      i2caddr <- I2C slave address                                        *
 *      data    <- I2C data ptr                                             *
 *      len     <- # of bytes to write                                      *
 * ------------------------------------------------------------------------ */
Int16 I2C_write( Uint16 i2caddr, Uint8* data, Uint16 len )
{
    Uint16 i;

    I2C_ICCNT = len;                                // Set len
    I2C_ICSAR = i2caddr;                            // Set I2C slave address
    I2C_ICMDR = ICMDR_STT                           // Config for master write
              | ICMDR_TRX
              | ICMDR_MST
              | ICMDR_IRS
              | ICMDR_FREE
              ;

    I2C_wait( 10 );                                  // Short delay

    for ( i = 0 ; i < len ; i++ )
    {
        I2C_ICDXR = data[i];                        // Write
        while ( ( I2C_ICSTR & ICSTR_ICXRDY ) == 0 );// Wait for Tx Ready
    }

    I2C_ICMDR |= ICMDR_STP;                         // Generate STOP

    return 0;
}

/* ------------------------------------------------------------------------ *
 *      I2C read in Master mode                                             *
 *                                                                          *
 *      i2caddr <- I2C slave address                                        *
 *      data    <- I2C data ptr                                             *
 *      len     <- # of bytes to write                                      *
 * ------------------------------------------------------------------------ */
Int16 I2C_read( Uint16 i2caddr, Uint8* data, Uint16 len )
{
    Uint16 i;
#ifndef  I2C_WAITFOREVER
    Uint32 times;
#endif

    I2C_ICCNT = len;                                // Set len
    I2C_ICSAR = i2caddr;                            // Set I2C slave address
    I2C_ICMDR = ICMDR_STT                           // Config for master read
              | ICMDR_MST
              | ICMDR_IRS
              | ICMDR_FREE
              ;

    if ( ( I2C_ICSTR & ICSTR_NACK ) == 0 )
    {
        I2C_ICMDR |= ICMDR_STP;                         // Generate STOP
        return 1;
    }
	
#ifdef  I2C_WAITFOREVER
    for ( i = 0 ; i < len ; i++ )
    {
        while ( ( I2C_ICSTR & ICSTR_ICRRDY ) == 0 );// Wait for Rx Ready
        data[i] = I2C_ICDRR;                        // Read
    }
#else
    for ( i = 0 ; i < len ; i++ )
    {
        times = 0;
	  while (1)
        {
             I2C_wait( 1000 );
             times++;
		if(( I2C_ICSTR & ICSTR_ICRRDY ) == 1) break; 
		if(times == 100) return 1;//return failed if attempt some times 
        }// Wait for Rx Ready
        data[i] = I2C_ICDRR;                        // Read
    }
#endif	

    I2C_ICMDR |= ICMDR_STP;                         // Generate STOP

    return 0;
}

Int16 I2C_read_slave_regs( Uint16 i2caddr, Uint8 regaddr, Uint8* data, Uint16 len )
{
    Uint16 i;
#ifndef  I2C_WAITFOREVER
    Uint32 times;
#endif

    I2C_ICCNT = 1;                                // Set len
    I2C_ICSAR = i2caddr;                            // Set I2C slave address
    I2C_ICMDR = ICMDR_STT                           // Config for master write
              | ICMDR_TRX
              | ICMDR_MST
              | ICMDR_IRS
              | ICMDR_FREE
              ;

    I2C_wait( 10 );                                  // Short delay

    I2C_ICDXR = regaddr;                        // Write

    I2C_wait( 10 );                                  // Short delay

    I2C_ICCNT = len;                                // Set len
    I2C_ICSAR = i2caddr;                            // Set I2C slave address
    I2C_ICMDR = ICMDR_STT                           // Config for master read
              | ICMDR_MST
              | ICMDR_IRS
              | ICMDR_FREE
              ;
	
#ifdef  I2C_WAITFOREVER
    for ( i = 0 ; i < len ; i++ )
    {
        while ( ( I2C_ICSTR & ICSTR_ICRRDY ) == 0 );// Wait for Rx Ready
        data[i] = I2C_ICDRR;                        // Read
    }
	
#else

    for ( i = 0 ; i < len ; i++ )
    {
        times = 0;
	  while (1)
        {
             I2C_wait( 1000 );
             times++;
		if(( I2C_ICSTR & ICSTR_ICRRDY ) == 1) break; 
		if(times == 0x100000) return 1;//return failed if attempt some times 
        }// Wait for Rx Ready
        data[i] = I2C_ICDRR;                        // Read
    }
#endif
	
    I2C_ICMDR |= ICMDR_STP;                         // Generate STOP

    return 0;
}


