#include "hFiles.h"

#include <mem.h>
#include <ecm.h>
#include <sem.h>
#include <c64.h>
#include <hwi.h>
#include <lck.h>
#include <tsk.h>

#include "sysContext.h"

static uint g_segID = 0;
static LCK_Handle g_hsysLCK = NULL;
extern unsigned int IRAM;
PVOID alloc_mem(uint size, uint align)
{
	PVOID pmem = (PVOID)MEM_alloc(g_segID, size, align);
	if(pmem == MEM_ILLEGAL) pmem = Mem_NULL;

	return pmem;
}
VOID free_mem(void *pmem, uint size)
{
	MEM_free(g_segID, pmem, size);
}

VOID mmzero(void *pmem, uint size)
{
	uint *pdwDst;
	uchar *pbDst = pmem;

	// Copy 'till aligned
	while((((uint)pbDst)&0x3) && size > 0)
	{
		*pbDst++ = 0;
		size--;
	}

	if(size)
	{
		pdwDst = (uint *)pbDst;
		// Copy 'till less than 4 bytes
		while(size > 3)
		{
			*pdwDst++ = 0;
			size -= 4;
		}

		if(size)
		{
			pbDst = (uchar *)pdwDst;
			// Copy 'till done
			while(size--) *pbDst++ = 0;
		}
	}
}

VOID mmset(void *pmem, uint size, uchar value)
{
	uint *pdwDst;
	uchar *pbDst = pmem;

	while((((uint)pbDst)&0x3) && size > 0)
	{
		*pbDst++ = value;
		size--;
	}

	if(size)
	{
		pdwDst = (uint *)pbDst;
		while(size > 3)
		{
			*pdwDst++ = (value<<24)|(value<<16)|(value<<8)|value;
			size -= 4;
		}

		if(size)
		{
			pbDst = (uchar *)pdwDst;
			while(size--) *pbDst++ = value;
		}
	}
}

VOID mmset_16(void *pmem, uint size, unsigned short value)//16BIT
{
	unsigned short *pbDst = pmem;

	while(size > 0)
	{
		*pbDst++ = value;
		size--;
	}
}
VOID mmcopy(void *pdst, const void *psrc, uint size)
{
	uint *pdwsrc, *pdwdst;
	uchar *pbsrc, *pbdst;

	// Fast case
	if(!(((uint)pdst)&0x3) && !(((uint)psrc)&0x3))
	{
		pdwsrc = (uint*)psrc;
		pdwdst = (uint*)pdst;

		while(size>3)
		{
			*pdwdst++ = *pdwsrc++;
			size -= 4;
		}

		if(size)
		{
			pbsrc = (uchar *)pdwsrc;
			pbdst = (uchar *)pdwdst;

			while(size--)
				*pbdst++ = *pbsrc++;
		}
	}
	else
	{
		pbsrc = (uchar*)psrc;
		pbdst = (uchar*)pdst;

		while(size--)
			*pbdst++ = *pbsrc++;
	}
}

void strCopy(char *pDst, const char* pSrc)
{
	int i = 0;
	while(*pSrc) {
		pDst[i] = *pSrc++;
	}
	pDst[i] = '\0';
}

void sysCritON()
{
	if(g_hsysLCK) LCK_pend(g_hsysLCK, TO_FOREVER);
}

void sysCritOFF()
{
	if(g_hsysLCK) LCK_post(g_hsysLCK);
}

bool InitSysContext()
{
	bool bResult = true;
	
	if(!g_hsysLCK) g_hsysLCK = LCK_create(NULL);
	if(!g_hsysLCK) bResult = false;

	return bResult;
}

void _llEnter()
{
	HWI_disable();
}
void _llExit()
{
	HWI_enable();
}

HEVENT CreateEvent(int count)
{
	return (HEVENT)SEM_create(count, NULL);
}
void DeleteEvent(HEVENT hEvt)
{
	SEM_delete((SEM_Handle)hEvt);
}

bool PendEvent(HEVENT hEvt, uint timeout)
{
	return SEM_pend((SEM_Handle)hEvt, timeout);
}
VOID PostEvent(HEVENT hEvt)
{
	SEM_post((SEM_Handle)hEvt);
}
VOID ResetEvent(HEVENT hEvt, int count)
{
	SEM_reset((SEM_Handle)hEvt, count);
}

VOID tsk_sleep(uint uTicks)
{
	TSK_sleep(uTicks);
}

PVOID create_task(void(*pFun)(), char *name, int pri, uint stk, uint arg)
{
	struct TSK_Attrs ta;

	// assign to default attributes
	ta = TSK_ATTRS;

	// create the system tasks
	ta.priority = pri;
	ta.stack = 0;
	ta.stacksize = stk;
	ta.stackseg = 0;
	ta.environ = 0;
	ta.name = name;
	ta.exitflag = 0;

	return ((PVOID)TSK_create((Fxn)pFun, &ta, arg, 0, 0));
}

uint  Interrupt_create(PINTSETUP pInt)
{
	uint      i, vectid;
	HWI_Attrs   hwi_attrs;
	ECM_Attrs   ecm_attrs;

	if(!pInt || !pInt->sysEvtCnt || !pInt->pFxn)
	{
		/* Invalid Interrupt setup arguments. Return error */
		return 1;
	}

	if(pInt->sysEvtCnt == 1)
	{
		/* One to One mapping between the system event and the
		 * interrupt vector. We can use HWI module of the bios
		 * to configure this interrupt.
		 */
		if( pInt->intvectId > MIN_INTVECT_ID && pInt->intvectId < MAX_INTVECT_ID )
		{
			HWI_eventMap(pInt->intvectId, pInt->sysEvtId[0]);

			if(pInt->arg)
			{
				hwi_attrs.intrMask = 1; /* default value */
				hwi_attrs.ccMask   = 1; /* default value */
				hwi_attrs.arg = (Arg)pInt->arg;
				HWI_dispatchPlug(pInt->intvectId, (Fxn)pInt->pFxn, -1, &hwi_attrs);
			}
			else
				HWI_dispatchPlug(pInt->intvectId, (Fxn)pInt->pFxn, -1, 0);
		}
		else
		{
			/* The interrupt vector configured is invalid. 
			 * Return error.
			 */
			return 1;
		}
	}
	else
	{
		/* multiple interrupt vectors combined to a single 
		 * system event. We will have to use the ecm module
		 * of bios to configure this interrupt.
		 */
		if(pInt->arg)
		{
			ecm_attrs.arg = (Arg)pInt->arg;
		}

		/* Validate the Interrupt enable flags and set it.  */
		if(!pInt->f_enable || pInt->f_enable == 1)
		{
			ecm_attrs.unmask = pInt->f_enable;
		}
		else
		{
			/* Disable interrupts by default */
			ecm_attrs.unmask = 0;
		}

		for(i = 0; i < pInt->sysEvtCnt; i++)
		{
			if(pInt->sysEvtId[i] > MIN_EVENT_ID && pInt->sysEvtId[i] < MAX_EVENT_ID )
			{
				ECM_dispatchPlug(pInt->sysEvtId[i],(ECM_Fxn)pInt->pFxn,&ecm_attrs);
			}
			else
			{
				/* Invalid system event number.
				 * Return error.
				 */
				return 1;
			}
			vectid++;
		}
	}

	/* Return success */
	return 0;
}
uint  DeleteInterrupt(PINTSETUP pInt)
{
	/* No way to currently delete interrupts */
	return 0;
}

uint  EnableInterrupt(PINTSETUP pInt)
{
	uint      i;

	if(!pInt || !pInt->sysEvtCnt || !pInt->pFxn)
	{
		/* Invalid Interrupt setup arguments. Return error */
		return 1;
	}

	if(pInt->sysEvtCnt == 1)
	{
		/* One to One mapping between the system event and the
		* interrupt vector. We can use HWI module of the bios
		* to configure this interrupt.
		*/
		/* Nothing to do. HWI interrupts are enabled by default. */
	}
	else
	{
		/* multiple interrupt vectors combined to a single 
		* system event. we will have to use the ecm module
		* of bios to configure this interrupt.
		*/
		for (i = 0; i < pInt->sysEvtCnt; i++)
		{
			ECM_enableEvent(pInt->sysEvtId[i]);
		}
	}

	/* Return success */
	return 0;
}

uint  DisableInterrupt(PINTSETUP pInt)
{
	uint      i;

	if(!pInt || !pInt->sysEvtCnt || !pInt->pFxn)
	{
		/* Invalid Interrupt setup arguments. Return error */
		return 1;
	}

	if(pInt->sysEvtCnt == 1)
	{
		/* One to One mapping between the system event and the
		 * interrupt vector. we can use HWI module of the bios
		 * to configure this interrupt.
		 */
		/* Nothing to do. HWI interrupts cannot be disabled on
		 * per interrupt number basis. 
		 */
	}
	else
	{
		/* multiple interrupt vectors combined to a single 
		 * system event. We will have to use the ecm module
		 * of bios to configure this interrupt.
		 */
		for (i = 0; i < pInt->sysEvtCnt; i++)
		{
			ECM_disableEvent(pInt->sysEvtId[i]);
		}
	}

	/* Return success */
	return 0;
}

/* Spin in a delay loop for delay iterations */
void delay(Uint32 delay)
{
    volatile Uint32 i, n;

    n = 0;
    for (i = 0; i < delay; i++)
    {
        n = n + 1;
    }
}
void EnableSysInterrupt(vectID) { C64_enableIER(1 << vectID); }
void DisableSysInterrupt(vectID) { C64_disableIER(1 << vectID); }


