/*  ============================================================================
*   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2008
*
*   Use of this software is controlled by the terms and conditions found in the
*   license agreement under which this software has been supplied.
*  ============================================================================
*/

/** @file csl_sar_DmaExample.c
 *
 *  @brief Test code to verify the SAR functionality in DMA mode
 *
 *
 * \page    page14  CSL SAR EXAMPLE DOCUMENTATION
 *
 * \section SAR2   SAR EXAMPLE2 - DMA MODE TEST
 *
 * \subsection SAR2x    TEST DESCRIPTION:
 *		This test verifies the functionality of CSL SAR(Successive Approximation
 * Register) in DMA mode. SAR module on C5505/C5515 DSP is used to convert the
 * voltage generated by the key pad available on the C5505/C5515 EVM to digital
 * values. This key pad is having 10 button with each button having a different
 * digital value corresponding to the voltage generated when it is pressed.
 *
 * During the test C5505/C5515 key pad button voltage is read and the converted
 * digital value is displayed in the CCS "stdout" window. This digital value
 * corresponds to the voltage of the button pressed while running the test.
 * SAR module is having six channels to read the data. This test configures
 * channel four for reading the button voltage in DMA mode. DMA channel should
 * configured for SAR data read before starting the SAR data conversion. DMA
 * module in this test uses interrupt to indicate the completion of DMA data
 * transfer. CSL INTC module should be configured and DMA ISR should registered
 * before starting the SAR data conversion. After successful configuration of
 * the SAR, DMA data transfer is started using DMA_start() API and SAR voltage
 * conversion is started using the CSL API SAR_startConversion(). Digital value
 * of the button voltage is read by the DMA module and is copied to the data
 * buffer 'readBuffer'. Even though the digital value generated by SAR is
 * maximum 10 bits, DMA reads 32 bits from the SAR. Due to word swap nature
 * of the DMA, value read from the SAR will be stored in the upper word
 * of the data buffer. This value will be displayed in the CCS "stdout" window.
 * Digital value is read from SAR and is displayed on CCS "stdout" window
 * only one time.
 *
 * @verbatim
   Digital values corresponding to the button voltage will vary slightly
   depending on the system clock value. Below are the range of values for
   each button available on the C5505/C5515 EVM key pad.

    No button pushed: 0x3fb - 0x3ff
    SHIFT:    0x230 - 0x238
    STOP:     0x0
    RWD:      0x20f - 0x216
    MENU:     0x367 - 0x374
    DN:       0x2bc - 0x2c6
    SEL/PLAY: 0x1d3 - 0x1d8
    UP:       0x24f - 0x258
    REC:      0xd5  - 0xd8
    FWD:      0x163 - 0x167
    MODE:     0x30b - 0x316

   @endverbatim
 *
 *
 * NOTE: THIS TEST HAS BEEN DEVELOPED TO WORK WITH CHIP VERSIONS C5505 AND
 * C5515. MAKE SURE THAT PROPER CHIP VERSION MACRO IS DEFINED IN THE FILE
 * c55xx_csl\inc\csl_general.h.
 *
 * \subsection SAR2y    TEST PROCEDURE:
 *  @li Open the CCS and connect the target (C5505/C5515 EVM)
 *  @li Open the project "CSL_SAR_DmaExample.pjt" and build it
 *  @li Load the program on to the target
 *  @li Set the PLL frequency to 12.288MHz
 *  @li Press any one of the buttons present on C5505 EVM key pad.
 *  @li Run the program and observe the test result
 *  @li Repeat the test at PLL frequencies 40, 60, 75 and 100MHz
 *  @li Repeat the test in Release mode
 *
 * \subsection SAR2z    TEST RESULT:
 *  @li All the CSL APIs should return success
 *  @li Digital value corresponding to a button voltage should be within the
 *      range specified in the test procedure.
 *
 * ============================================================================
 */

/* ============================================================================
 * Revision History
 * ================
 * 01-Oct-2008 created
 * 15-May-2009 Modified for testing DMA Word swap work around on C5505 PG1.4
 * ============================================================================
 */

#include <csl_sar.h>
#include <csl_dma.h>
#include <stdio.h>
#include <csl_intc.h>
#include <csl_general.h>

/* CSL example test failed       */
#define CSL_TEST_FAILED         (1)
/* CSL example test passed       */
#define CSL_TEST_PASSED         (0)

#define CSL_DMA_SAR_BUFFER_SIZE  2
/* SAR object Structure          */
CSL_SarHandleObj SarObj;

/* DMA Interrupt Service Routine */
interrupt void dma_isr(void);

/* Reference the start of the interrupt vector table */
/* This symbol is defined in file vectors.asm       */
extern void VECSTART(void);

#pragma DATA_ALIGN(readBuffer, 4);
Uint16 readBuffer[CSL_DMA_SAR_BUFFER_SIZE];
CSL_DMA_ChannelObj  dmaObj;                 /* DMA object structure */
CSL_DMA_Handle 		dmaHandle;              /* DMA handle           */
CSL_DMA_Config 		dmaConfig;              /* DMA config structure */
int                 i=0;
int                 j=0;

/* To test Keypad voltage Measurement in DMA mode */
int  sar_test_dma_keypad_voltage();

   /////INSTRUMENTATION FOR BATCH TESTING -- Part 1 --   
   /////  Define PaSs_StAtE variable for catching errors as program executes.
   /////  Define PaSs flag for holding final pass/fail result at program completion.
        volatile Int16 PaSs_StAtE = 0x0001; // Init to 1. Reset to 0 at any monitored execution error.
        volatile Int16 PaSs = 0x0000; // Init to 0.  Updated later with PaSs_StAtE when and if
   /////                                  program flow reaches expected exit point(s).
   /////
void main(void)
{
    int    result;

    /* To Test Keypad Voltage Measurement in DMA mode */

	printf("CSL SAR Module DMA Mode Test\n\n");
	printf("Press any Button in the EVM Keypad\n");

    /* Initialize buffer with 0xFFFF value */
    for(j=0; j < CSL_DMA_SAR_BUFFER_SIZE ;j++)
	{
        readBuffer[j] = 0xFFFF;
	}

        /*enable the corresponding DMA clock from PCGCR Registers*/
    CSL_FINST(CSL_SYSCTRL_REGS->PCGCR2,
                         SYS_PCGCR2_DMA2CG,ACTIVE );

    result = sar_test_dma_keypad_voltage();
    if(result != CSL_TEST_PASSED)
    {
        printf("\nCSL SAR Module DMA Mode Test Failed!!\n");
   /////INSTRUMENTATION FOR BATCH TESTING -- Part 2 --   
   /////  Reseting PaSs_StAtE to 0 if error detected here.
        PaSs_StAtE = 0x0000; // Was intialized to 1 at declaration.
   /////
    }
    else
    {
        printf("\nCSL SAR Module DMA Mode Test Passed!!\n");
    }

   /////INSTRUMENTATION FOR BATCH TESTING -- Part 3 -- 
   /////  At program exit, copy "PaSs_StAtE" into "PaSs".
        PaSs = PaSs_StAtE; //If flow gets here, override PaSs' initial 0 with 
   /////                   // pass/fail value determined during program execution.
   /////  Note:  Program should next exit to C$$EXIT and halt, where DSS, under
   /////   control of a host PC script, will read and record the PaSs' value.  
   /////
}

/* Testing of SAR A/D Keypad Voltage Measurement in DMA mode*/
int  sar_test_dma_keypad_voltage(void)
{

    CSL_Status    status;
    CSL_SarHandleObj *SarHandle;
    CSL_SarChSetup param;
    int result;
    int chanNo;

    result = CSL_TEST_FAILED;

    /* Initialize the SAR module */
    status = SAR_init();
    if(status != CSL_SOK)
    {
        printf("SAR Init Failed!!\n");
        return(result);
    }

    /* Open SAR channel */
    status = SAR_chanOpen(&SarObj,CSL_SAR_CHAN_3);
    SarHandle = &SarObj;
    if(status != CSL_SOK){
        printf("SAR_chanOpen Failed!!\n");
        return result;
    }
    /* Initialize channel */
    status = SAR_chanInit(SarHandle);
    if(status != CSL_SOK)
    {
        printf("SAR_chanInit Failed!!\n");
        return(result);
    }
    param.OpMode = CSL_SAR_DMA;
    param.MultiCh = CSL_SAR_NO_DISCHARGE;
    param.RefVoltage = CSL_SAR_REF_VIN;
    param.SysClkDiv = 11 ;
    /* Configuration for SAR module */
    status = SAR_chanSetup(SarHandle,&param);
    if(status != CSL_SOK)
    {
        printf("SAR_chanConfig Failed!!\n");
        return(result);
    }


    /* DMA configuration */
    status = DMA_init();
    if (status != CSL_SOK)
    {
        printf("DMA_init() Failed \n");
        return(result);
   /////INSTRUMENTATION FOR BATCH TESTING -- Part 2 --   
   /////  Reseting PaSs_StAtE to 0 if error detected here.
        PaSs_StAtE = 0x0000; // Was intialized to 1 at declaration.
   /////
    }

#if (defined(CHIP_C5505_C5515) || defined(CHIP_C5504_C5514))
	dmaConfig.pingPongMode = CSL_DMA_PING_PONG_DISABLE;
#endif
    dmaConfig.autoMode     = CSL_DMA_AUTORELOAD_ENABLE;
    dmaConfig.burstLen     = CSL_DMA_TXBURST_1WORD;
    dmaConfig.trigger      = CSL_DMA_EVENT_TRIGGER;
    dmaConfig.dmaEvt       = CSL_DMA_EVT_SAR_AD;
    dmaConfig.dmaInt       = CSL_DMA_INTERRUPT_ENABLE;
    dmaConfig.chanDir      = CSL_DMA_READ;
    dmaConfig.trfType      = CSL_DMA_TRANSFER_IO_MEMORY;
    dmaConfig.dataLen      = CSL_DMA_SAR_BUFFER_SIZE * 2;
    dmaConfig.srcAddr      = (Uint32)&(SarHandle->baseAddr->SARDATA);
    dmaConfig.destAddr     = (Uint32)readBuffer;

    dmaHandle = DMA_open(CSL_DMA_CHAN8,&dmaObj, &status);
    if (dmaHandle == NULL)
    {
        printf("DMA_open() Failed \n");
        return result;
    }

    /* Interrupt configuration */
    IRQ_globalDisable();

	/* Clear any pending interrupts */
	IRQ_clearAll();

	/* Disable all the interrupts */
	IRQ_disableAll();

    IRQ_setVecs((Uint32)&VECSTART);
    IRQ_clear(DMA_EVENT);
    IRQ_clear(SAR_EVENT);
    /* Register the ISR */
    IRQ_plug (DMA_EVENT, &dma_isr);
    /* Enable DMA Interrupt */
    IRQ_enable(DMA_EVENT);
    IRQ_globalEnable();


    status = DMA_config(dmaHandle, &dmaConfig);
    if (status != CSL_SOK)
    {
        printf("DMA_config() Failed \n");
        return result;
    }
    /* Set channel cycle set */
    status = SAR_chanCycSet(SarHandle,CSL_SAR_SINGLE_CONVERSION);
    if(status != CSL_SOK)
    {
        printf("SAR_chanCycSet Failed!!\n");
        return(result);
    }
    /* set ADC Measurement parameters */
    status = SAR_A2DMeasParamSet(SarHandle,CSL_KEYPAD_MEAS,&chanNo);
    if(status != CSL_SOK)
    {
        printf("SAR_A2DMeasParamSet Failed!!\n");
        return(result);
    }
    printf("Channel Number selected %d\n",chanNo);

    status = DMA_start(dmaHandle);
    if (status != CSL_SOK)
    {
        printf("DMA_start() Failed \n");
        return(result);
    }
    /* Enable Interrupt globally */
    //IRQ_globalEnable();
    /* start the conversion */
    status = SAR_startConversion(SarHandle);
    if(status != CSL_SOK)
    {
        printf("SAR_startConversion Failed!!\n");
        return(result);
    }
    /* Wait for Interrupt  */
    while(i != 1);

    printf("Voltage Read 0x%x\n",readBuffer[1]);

    IRQ_globalDisable();

	/* Clear any pending interrupts */
	IRQ_clearAll();

	/* Disable all the interrupts */
	IRQ_disableAll();

    /* Stop the conversion */
    status = SAR_stopConversion(SarHandle);
    if(status != CSL_SOK)
    {
        printf("SAR_stopConversion Failed!!\n");
        return(result);
    }
    /* Close the channel */
    status = SAR_chanClose(SarHandle);
    if(status != CSL_SOK)
    {
        printf("SAR_chanClose Failed!!\n");
        return(result);
    }
    /* Deinit */
    status = SAR_deInit();
    if(status != CSL_SOK)
    {
        printf("SAR_deInit Failed!!\n");
        return(result);
    }
    result = CSL_TEST_PASSED;
    return(result);
}

/* DMA ISR */
interrupt void dma_isr(void)
{
    i++;
    /* Clearing Interrupt */
    IRQ_clear(SAR_EVENT);
}

