/*  ============================================================================
*   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2008
*
*   Use of this software is controlled by the terms and conditions found in the
*   license agreement under which this software has been supplied.
*  ============================================================================
*/

/** @file intc_example.c
 *
 *  @brief INTC functional layer sample source file for SAR
 *
 *
 * \page    page7  INTC EXAMPLE DOCUMENTATION
 *
 * \section INTC   INTC EXAMPLE
 *
 * \subsection INTCx    TEST DESCRIPTION:
 *		This test verifies the functionality of CSL INTC module. INTC module
 * is used to configure and enable\disable the interrupts available on the
 * C5505/C5515 DSP.
 *
 * This test verifies the INTC module using CSL SAR. SAR module on C5505/C5515
 * DSP is used to convert the voltage generated by the key pad available on the
 * C5505/C5515 EVM to digital values. During the test C5505/C5515 EVM key pad
 * button voltage is read using interrupts. SAR interrupts are configured and
 * ISR is registered using INTC module APIs. SAR is configured and the data
 * conversion is started using SAR_startConversion(). Interrupt is generated
 * when there is a valid value in the SAR data register. Digital value is read
 * from the SAR data register using the Interrupt Service Routine and is
 * displayed in the CCS 'stdout' window. Generation of SAR interrupt will prove
 * the functionality of INTC module.
 *
 * Digital values corresponding to the button voltage will vary slightly
 * depending on the system clock value. Below are the range of values for
 * each button available on the C5505/C5515 EVM key pad.
 *
 *  No button pushed: 0x3fb - 0x3ff
 *  SHIFT:    0x230 - 0x238
 *  STOP:     0x0
 *  RWD:      0x20f - 0x216
 *  MENU:     0x367 - 0x374
 *  DN:       0x2bc - 0x2c6
 *  SEL/PLAY: 0x1d3 - 0x1d8
 *  UP:       0x24f - 0x258
 *  REC:      0xd5  - 0xd8
 *  FWD:      0x163 - 0x167
 *  MODE:     0x30b - 0x316
 *
 * NOTE: THIS TEST HAS BEEN DEVELOPED TO WORK WITH CHIP VERSIONS C5505 AND
 * C5515. MAKE SURE THAT PROPER PROPER CHIP VERSION MACRO SHOULD IS DEFINED
 * IN THE FILE c55xx_csl\inc\csl_general.h.
 *
 * \subsection INTCy    TEST PROCEDURE:
 *  @li Open the CCS and connect the target (C5505/C5515 EVM)
 *  @li Open the project "CSL_INTC_Example.pjt" and build it
 *  @li Load the program on to the target
 *  @li Set the PLL frequency to 12.288MHz
 *  @li Press any one of the buttons present on C5505/C5515 EVM key pad.
 *  @li Run the program and observe the test result
 *  @li Repeat the test at PLL frequencies 40, 60, 75 and 100MHz
 *  @li Repeat the test in Release mode
 *
 * \subsection INTCz    TEST RESULT:
 *  @li All the CSL APIs should return success
 *  @li Digital value should read in the ISR and displayed in the CCS "stdout"
 *      window
 *
 *
 */

/* ============================================================================
 * Revision History
 * ================
 * 22-Sep-2008  Craeted
 * ============================================================================
 */

#include <csl_sar.h>
#include <csl_intc.h>
#include <csl_general.h>
#include <stdio.h>


#define CSL_TEST_FAILED         (1)
#define CSL_TEST_PASSED         (0)

/* Global Structure Declaration*/

/** Sar object structure */
CSL_SarHandleObj SarObj;
/** Sar object pointer */
CSL_SarHandleObj *SarHandle;
/* SAR A/D Data Register Address */
Uint32 sarRegDataAddr = 0x7014;

Uint16 readBuffer;
int i = 0;
//---------Function prototypes---------
/** Interrupt Service Routine */
interrupt void sarISR(void);

/* Reference the start of the interrupt vector table */
/* This symbol is defined in file vectors.asm       */
extern void VECSTART(void);
/* Example Function */
int  sar_test_Int_keypad_voltage();

   /////INSTRUMENTATION FOR BATCH TESTING -- Part 1 --
   /////  Define PaSs_StAtE variable for catching errors as program executes.
   /////  Define PaSs flag for holding final pass/fail result at program completion.
        volatile Int16 PaSs_StAtE = 0x0001; // Init to 1. Reset to 0 at any monitored execution error.
        volatile Int16 PaSs = 0x0000; // Init to 0.  Updated later with PaSs_StAtE when and if
   /////                                  program flow reaches expected exit point(s).
   /////
void main(void)
{
    int status;

	printf("CSL INTC MODULE TEST\n\n");

    status = sar_test_Int_keypad_voltage();
    if(CSL_TEST_PASSED == status)
    {
        printf("sar_test_Int_keypad_voltage is passed\n");
    }
    else
    {
		printf("sar_test_Int_keypad_voltage is failed\n");

   /////INSTRUMENTATION FOR BATCH TESTING -- Part 2 --
   /////  Reseting PaSs_StAtE to 0 if error detected here.
        PaSs_StAtE = 0x0000; // Was intialized to 1 at declaration.
    }
   /////

	printf("\nCSL INTC MODULE TEST COMPLETED!!\n");

   /////INSTRUMENTATION FOR BATCH TESTING -- Part 3 --
   /////  At program exit, copy "PaSs_StAtE" into "PaSs".
        PaSs = PaSs_StAtE; //If flow gets here, override PaSs' initial 0 with
   /////                   // pass/fail value determined during program execution.
   /////  Note:  Program should next exit to C$$EXIT and halt, where DSS, under
   /////   control of a host PC script, will read and record the PaSs' value.
   /////
}

/* Testing of SAR A/D Keypad Voltage Measurement */
int	sar_test_Int_keypad_voltage(void)
{
    Bool flag = 1;
    CSL_Status    status;
    CSL_SarChSetup param;
    int chanNo;
    CSL_IRQ_Config  irqConfig;
    CSL_IRQ_Config  irqQuery;

    printf("Testing INTC using SAR module\n");
	printf("Press Any Button on the EVM Keypad\n");

    /* Disable interrupt */
    IRQ_globalDisable();

	/* Clear any pending interrupts */
	IRQ_clearAll();

	/* Disable all the interrupts */
	IRQ_disableAll();

    /* Initialize Interrupt Vector table */
    IRQ_setVecs((Uint32)(&VECSTART));

    /* IRQ init call*/
    IRQ_init((CSL_IRQ_Dispatch*)0x0000,0);
    /* IRQ_config */
    irqConfig.funcAddr = &sarISR;
    irqConfig.funcArg = 2;
    IRQ_config(SAR_EVENT,&irqConfig);

    /* Getconfig */
    IRQ_getConfig(SAR_EVENT,&irqQuery);
    if(irqQuery.funcAddr != irqConfig.funcAddr)
    {
        printf("Wrong ISR address\n");
    }
    /* Initialize the SAR module */
    status = SAR_init();
    if(status != CSL_SOK)
    {
        printf("SAR Init Failed!!\n");
        return CSL_TEST_FAILED ;
    }

    /* Open SAR channel */
    status = SAR_chanOpen(&SarObj,CSL_SAR_CHAN_3);
    SarHandle = &SarObj;
    if(status != CSL_SOK)
    {
        printf("SAR_chanOpen Failed!!\n");
        return CSL_TEST_FAILED;
    }

    /* Initialize channel */
    status = SAR_chanInit(SarHandle);
    if(status != CSL_SOK)
    {
    printf("SAR_chanInit Failed!!\n");
    return CSL_TEST_FAILED;
    }

    /* Clear any pending Interrupt */
    IRQ_clear(SAR_EVENT);
    IRQ_test(SAR_EVENT,&flag);
    IRQ_map(SAR_EVENT);
    /* Set Arguments */
    status = IRQ_setArg(SAR_EVENT,sarRegDataAddr);
    if(status != CSL_SOK)
    {
        printf("IRQ_setArg failed!!\n");
        return CSL_TEST_FAILED ;
    }
    /* Register the ISR */
    IRQ_plug(SAR_EVENT,&sarISR);

    param.OpMode =  CSL_SAR_INTERRUPT;
    param.MultiCh = CSL_SAR_NO_DISCHARGE;
    param.RefVoltage = CSL_SAR_REF_VIN;
    param.SysClkDiv = 0x0b ;
    /* Configuration for SAR module */
    status = SAR_chanSetup(SarHandle,&param);
    if(status != CSL_SOK)
    {
        printf("SAR_chanConfig Failed!!\n");
        return CSL_TEST_FAILED;
    }

    /* Set channel cycle set */
    status = SAR_chanCycSet(SarHandle,CSL_SAR_SINGLE_CONVERSION);
    if(status != CSL_SOK)
    {
        printf("SAR_chanCycSet Failed!!\n");
        return CSL_TEST_FAILED;
    }
    /* set ADC Measurement parameters */
    status = SAR_A2DMeasParamSet(SarHandle,CSL_KEYPAD_MEAS,&chanNo);
    if(status != CSL_SOK)
    {
        printf("SAR_A2DMeasParamSet Failed!!\n");
        return CSL_TEST_FAILED;
    }

    /* Enabling Interrupt */
    IRQ_enable(SAR_EVENT);
    IRQ_globalEnable();

    /* start the conversion */
    status = SAR_startConversion(SarHandle);
    if(status != CSL_SOK)
    {
        printf("SAR_startConversion Failed!!\n");
        return CSL_TEST_FAILED;
    }

    while(i!=1);

    printf("After ISR :SAR ADC read data 0x%x\n",readBuffer);

    /* Stop the conversion */
    status = SAR_stopConversion(SarHandle);
    if(status != CSL_SOK)
    {
        printf("SAR_stopConversion Failed!!\n");
        return CSL_TEST_FAILED;
    }
    /* Close the channel */
    status = SAR_chanClose(SarHandle);
    if(status != CSL_SOK)
    {
        printf("SAR_chanClose Failed!!\n");
        return CSL_TEST_FAILED;
    }
    /* Deinit */
    status = SAR_deInit();
    if(status != CSL_SOK)
    {
        printf("SAR_deInit Failed!!\n");
        return CSL_TEST_FAILED;
    }

    return CSL_TEST_PASSED;
}


// ISR to read ADC data
interrupt void sarISR(void)
{
    Uint32 RegAddr;
    i = 0;
    /* Get Arguments which was set using IRQ_setArg */
    IRQ_getArg(SAR_EVENT,&RegAddr);
    if(RegAddr != sarRegDataAddr)
    {
        printf("IRQ_getArg is failed\n");
   /////INSTRUMENTATION FOR BATCH TESTING -- Part 2 --
   /////  Reseting PaSs_StAtE to 0 if error detected here.
        PaSs_StAtE = 0x0000; // Was intialized to 1 at declaration.
   /////
    }
    IRQ_disable(SAR_EVENT);
    SAR_readData(SarHandle, &readBuffer);
    i++;
}


