/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2008
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *  ============================================================================
 */

/** @file csl_rtc_example.c
 *
 *  @brief Test for RTC timer and alarm functionality.
 *
 *
 * \page    page13  CSL RTC EXAMPLE DOCUMENTATION
 *
 * \section RTC1   RTC EXAMPLE1 - TIMER & ALARM TEST
 *
 * \subsection RTC1x    TEST DESCRIPTION:
 *		This test code verifies the functionality of real time clock (RTC).
 * RTC module will be configured and started to increment the configured time.
 * Time is read from RTC module to verify whether the timer is running or not.
 * Date and time read from the RTC is displayed in the CCS "stdout" window.
 * Manual inspection is required to verify whether the timer is running
 * or not. Time and Date will be displayed in the format HH:MM:SS:mmmm, DD-MM-YY
 * respectively. 'm' in the time format represents millisecond component of
 * the time. In the current test code time is set to 12:12:12:12 and date is
 * set to 16-10-08. Test code will stop running after displaying the time for
 * 255 iterations. RTC is configured to generate an interrupt for each second.
 * Message with interrupt count will be displayed in the CCS "stdout"  window
 * to notify the RTC second interrupt.
 *		This test also verifies the generation of alarm interrupt. Alarm time
 * is set to 12:12:17:512. So there will be an alarm interrupt at 5 secs after
 * starting the timer. A message will be displayed in the CCS "stdout" window
 * to notify the RTC alarm interrupt
 *
 * NOTE: THIS TEST HAS BEEN DEVELOPED TO WORK WITH CHIP VERSIONS C5505 AND
 * C5515. MAKE SURE THAT PROPER CHIP VERSION MACRO IS DEFINED IN THE FILE
 * c55xx_csl\inc\csl_general.h.
 *
 * \subsection RTC1y    TEST PROCEDURE:
 *  @li Open the CCS and connect the target (C5505/C5515 EVM)
 *  @li Open the project "CSL_RTC_Example_Out.pjt" and build it
 *  @li Load the program on to the target
 *  @li Set the PLL frequency to 12.288MHz
 *  @li Run the program and observe the test result
 *  @li Repeat the test at PLL frequencies 40, 60, 75 and 100MHz
 *  @li Repeat the test in Release mode
 *
 * \subsection RTC1z    TEST RESULT:
 * @li All the CSL RTC APIs should return success
 * @li RTC time should be displayed in the CCS stdout window for 255 iterations.
 *     There should be an increment in the time for each value displayed.
 * @li An interrupt should be generated by the RTC for each second, which will
 *     be displayed on the CCS "stdout" window
 * @li An alarm interrupt should be generated when time reaches 12:12:17:512
 *     which will be displayed on the CCS "stdout" window
 *
 * ============================================================================
 */

/* ============================================================================
 * Revision History
 * ================
 * 16-Oct-2008 Created
 * ============================================================================
 */

#include <stdio.h>
#include <csl_rtc.h>
#include <csl_intc.h>
#include <csl_general.h>

#define RTC_TIME_PRINT_CYCLE    (0xFFu)
#define RTC_CALL_BACK           (1u)

CSL_RtcTime	     InitTime;
CSL_RtcDate 	 InitDate;
CSL_RtcTime 	 GetTime;
CSL_RtcDate 	 GetDate;
CSL_RtcConfig    rtcConfig;
CSL_RtcConfig    rtcGetConfig;
CSL_RtcAlarm     AlarmTime;
CSL_RtcIsrAddr   isrAddr;
CSL_RtcIsrDispatchTable      rtcDispatchTable;
volatile Uint32 rtcTimeCount = RTC_TIME_PRINT_CYCLE;
Uint16    secIntrCnt = 0;

/* Reference the start of the interrupt vector table */
extern void VECSTART(void);
/* Prototype declaration for ISR function */
interrupt void rtc_isr(void);

void rtc_msIntc(void);
void rtc_secIntc(void);
void rtc_minIntc(void);
void rtc_hourIntc(void);
void rtc_dayIntc(void);
void rtc_extEvt(void);
void rtc_alarmEvt(void);

   /////INSTRUMENTATION FOR BATCH TESTING -- Part 1 --   
   /////  Define PaSs_StAtE variable for catching errors as program executes.
   /////  Define PaSs flag for holding final pass/fail result at program completion.
        volatile Int16 PaSs_StAtE = 0x0001; // Init to 1. Reset to 0 at any monitored execution error.
        volatile Int16 PaSs = 0x0000; // Init to 0.  Updated later with PaSs_StAtE when and if
   /////                                  program flow reaches expected exit point(s).
   /////
void main()
{
	CSL_Status    status;
	Uint16        iteration;

	iteration = 1;

	printf("CSL RTC TESTS\n\n");
	printf("This test demonstrates RTC TIMER and ALARM functionality\n");
	printf("RTC Time will be read and displayed 255 times\n");
	printf("RTC interrupt will be generated for each Second\n");
	printf("RTC ALARM interrupt will be generated at Time 12:12:17:512\n\n");

	/* Set the RTC config structure */
	rtcConfig.rtcyear  = 8;
	rtcConfig.rtcmonth = 8;
	rtcConfig.rtcday   = 8;
	rtcConfig.rtchour  = 8;
	rtcConfig.rtcmin   = 8;
	rtcConfig.rtcsec   = 8;
	rtcConfig.rtcmSec  = 8;

	rtcConfig.rtcyeara  = 8;
	rtcConfig.rtcmontha = 8;
	rtcConfig.rtcdaya   = 8;
	rtcConfig.rtchoura  = 8;
	rtcConfig.rtcmina   = 8;
	rtcConfig.rtcseca   = 8;
	rtcConfig.rtcmSeca  = 10;

	rtcConfig.rtcintcr  = 0x803F;

	/* Set the RTC init structure */
    InitDate.year  = 8;
    InitDate.month = 10;
    InitDate.day   = 16;

    InitTime.hours = 12;
    InitTime.mins  = 12;
    InitTime.secs  = 12;
    InitTime.mSecs = 12;

	/* Set the RTC alarm time */
    AlarmTime.year  = 8;
    AlarmTime.month = 10;
    AlarmTime.day   = 16;
    AlarmTime.hours = 12;
    AlarmTime.mins  = 12;
    AlarmTime.secs  = 17;
    AlarmTime.mSecs = 512;

    /* Register the ISR function */
    isrAddr.MilEvtAddr    = rtc_msIntc;
    isrAddr.SecEvtAddr    = rtc_secIntc;
    isrAddr.MinEvtAddr    = rtc_minIntc;
    isrAddr.HourEvtAddr   = rtc_hourIntc;
    isrAddr.DayEvtAddr    = rtc_dayIntc;
    isrAddr.ExtEvtAddr    = rtc_extEvt;
    isrAddr.AlarmEvtAddr  = rtc_alarmEvt;

    status = RTC_setCallback(&rtcDispatchTable, &isrAddr);
	if(status != CSL_SOK)
	{
		printf("RTC_setCallback Failed\n");
		return;
	}
	else
	{
		printf("RTC_setCallback Successful\n");
	}

	/* Configure and enable the RTC interrupts using INTC module */
    IRQ_globalDisable();

	/* Clear any pending interrupts */
	IRQ_clearAll();

	/* Disable all the interrupts */
	IRQ_disableAll();

	IRQ_setVecs((Uint32)&VECSTART);
	IRQ_clear(RTC_EVENT);

	IRQ_plug (RTC_EVENT, &rtc_isr);

	IRQ_enable(RTC_EVENT);
	IRQ_globalEnable();

	/* Reset the RTC */
	RTC_reset();

	/* Configure the RTC module */
	status = RTC_config(&rtcConfig);
	if(status != CSL_SOK)
	{
		printf("RTC_config Failed\n");
		return;
	}
	else
	{
		printf("RTC_config Successful\n");
	}

	/* Read the configuration values from the RTC module */
	status = RTC_getConfig(&rtcGetConfig);
	if(status != CSL_SOK)
	{
		printf("RTC_getConfig Failed\n");
		return;
	}
	else
	{
		printf("RTC_getConfig Successful\n");
	}

	/* Set the RTC time */
	status = RTC_setTime(&InitTime);
	if(status != CSL_SOK)
	{
		printf("RTC_setTime Failed\n");
		return;
	}
	else
	{
		printf("RTC_setTime Successful\n");
	}

	/* Set the RTC date */
	status = RTC_setDate(&InitDate);
	if(status != CSL_SOK)
	{
		printf("RTC_setDate Failed\n");
		return;
	}
	else
	{
		printf("RTC_setDate Successful\n");
	}

	/* Set the RTC Alarm time */
	status = RTC_setAlarm(&AlarmTime);
	if(status != CSL_SOK)
	{
		printf("RTC_setAlarm Failed\n");
		return;
	}
	else
	{
		printf("RTC_setAlarm Successful\n");
	}

	/* Set the RTC interrupts */
	status = RTC_setPeriodicInterval(CSL_RTC_MINS_PERIODIC_INTERRUPT);
	if(status != CSL_SOK)
	{
		printf("RTC_setPeriodicInterval Failed\n");
		return;
	}
	else
	{
		printf("RTC_setPeriodicInterval Successful\n");
	}

	/* Enable the RTC SEC interrupts */
	status = RTC_eventEnable(CSL_RTC_SECEVENT_INTERRUPT);
	if(status != CSL_SOK)
	{
		printf("RTC_eventEnable for SEC EVENT Failed\n");
		return;
	}
	else
	{
		printf("RTC_eventEnable for SEC EVENT Successful\n");
	}

	/* Enable the RTC alarm interrupts */
	status = RTC_eventEnable(CSL_RTC_ALARM_INTERRUPT);
	if(status != CSL_SOK)
	{
		printf("RTC_eventEnable for ALARM EVENT Failed\n");
		return;
	}
	else
	{
		printf("RTC_eventEnable for ALARM EVENT Successful\n");
	}

	printf("\nStarting the RTC\n\n");
	/* Start the RTC */
	RTC_start();

	/* This loop will display the RTC time for 255 times */
	while(rtcTimeCount--)
	{
	 	status = RTC_getTime(&GetTime);
		if(status != CSL_SOK)
		{
			printf("RTC_getTime Failed\n");
			return;
		}

	 	status = RTC_getDate(&GetDate);
		if(status != CSL_SOK)
		{
			printf("RTC_getDate Failed\n");
			return;
		}

		printf("Iteration %d: ",iteration++);

	    printf("Time and Date is : %02d:%02d:%02d:%04d, %02d-%02d-%02d\n",
		GetTime.hours,GetTime.mins,GetTime.secs,GetTime.mSecs,GetDate.day,GetDate.month,GetDate.year);
	}

    IRQ_globalDisable();

	/* Clear any pending interrupts */
	IRQ_clearAll();

	/* Disable all the interrupts */
	IRQ_disableAll();

	/* Stop the RTC */
	RTC_stop();

	printf("\nCSL RTC TESTS COMPLETED\n");
   /////INSTRUMENTATION FOR BATCH TESTING -- Part 3 -- 
   /////  At program exit, copy "PaSs_StAtE" into "PaSs".
        PaSs = PaSs_StAtE; //If flow gets here, override PaSs' initial 0 with 
   /////                   // pass/fail value determined during program execution.
   /////  Note:  Program should next exit to C$$EXIT and halt, where DSS, under
   /////   control of a host PC script, will read and record the PaSs' value.  
   /////
}

interrupt void rtc_isr(void)
{

#ifdef RTC_CALL_BACK
    CSL_RTCEventType rtcEventType;

    rtcEventType = RTC_getEventId();

    if (((void (*)(void))(rtcDispatchTable.isr[rtcEventType])))
     {
         ((void (*)(void))(rtcDispatchTable.isr[rtcEventType]))();
     }
#else
    Uint16 statusRegVal;

    statusRegVal = CSL_RTC_REGS->RTCINTFL;

    /* check for alarm interrupt */
    if (CSL_RTC_RTCINTFL_ALARMFL_MASK ==
                (statusRegVal & (Uint16)CSL_RTC_RTCINTFL_ALARMFL_MASK ))
    {
		CSL_FINST(CSL_RTC_REGS->RTCINTFL, RTC_RTCINTFL_ALARMFL, SET);
    }
    /* check for external event interrupt */
    else if (CSL_RTC_RTCINTFL_EXTFL_MASK ==
                (statusRegVal &(Uint16)CSL_RTC_RTCINTFL_EXTFL_MASK ))
    {
		CSL_FINST(CSL_RTC_REGS->RTCINTFL, RTC_RTCINTFL_EXTFL, SET);
    }
    /* check for day interrupt */
    else if (CSL_RTC_RTCINTFL_DAYFL_MASK ==
                (statusRegVal & CSL_RTC_RTCINTFL_DAYFL_MASK))
    {
		CSL_FINST(CSL_RTC_REGS->RTCINTFL, RTC_RTCINTFL_DAYFL, SET);
    }
    /* check for hour interrupt */
    else if (CSL_RTC_RTCINTFL_HOURFL_MASK ==
                (statusRegVal & CSL_RTC_RTCINTFL_HOURFL_MASK))
    {
		CSL_FINST(CSL_RTC_REGS->RTCINTFL, RTC_RTCINTFL_HOURFL, SET);
    }
    /* check for minute interrupt */
    else if (CSL_RTC_RTCINTFL_MINFL_MASK ==
                (statusRegVal & CSL_RTC_RTCINTFL_MINFL_MASK ))
    {
		CSL_FINST(CSL_RTC_REGS->RTCINTFL, RTC_RTCINTFL_MINFL, SET);
    }
    /* check for seconds interrupt */
    else if (CSL_RTC_RTCINTFL_SECFL_MASK ==
                (statusRegVal & CSL_RTC_RTCINTFL_SECFL_MASK ))
    {
		CSL_FINST(CSL_RTC_REGS->RTCINTFL, RTC_RTCINTFL_SECFL, SET);
    }
    /* check for milliseconds interrupt */
    else if (CSL_RTC_RTCINTFL_MSFL_MASK ==
                (statusRegVal & CSL_RTC_RTCINTFL_MSFL_MASK ))
    {
		CSL_FINST(CSL_RTC_REGS->RTCINTFL, RTC_RTCINTFL_MSFL, SET);
    }
#endif
}

void rtc_msIntc(void)
{
    CSL_FINST(CSL_RTC_REGS->RTCINTFL, RTC_RTCINTFL_MSFL, SET);
}

void rtc_secIntc(void)
{
    CSL_FINST(CSL_RTC_REGS->RTCINTFL, RTC_RTCINTFL_SECFL, SET);
	secIntrCnt++;
	printf("\nRTC Sec Interrupt %d\n\n",secIntrCnt);
}

void rtc_minIntc(void)
{
    CSL_FINST(CSL_RTC_REGS->RTCINTFL, RTC_RTCINTFL_MINFL, SET);
}

void rtc_hourIntc(void)
{
    CSL_FINST(CSL_RTC_REGS->RTCINTFL, RTC_RTCINTFL_HOURFL, SET);
}

void rtc_dayIntc(void)
{
    CSL_FINST(CSL_RTC_REGS->RTCINTFL, RTC_RTCINTFL_DAYFL, SET);
}

void rtc_extEvt(void)
{
    CSL_FINST(CSL_RTC_REGS->RTCINTFL, RTC_RTCINTFL_EXTFL, SET);
}

void rtc_alarmEvt(void)
{
    CSL_FINST(CSL_RTC_REGS->RTCINTFL, RTC_RTCINTFL_ALARMFL, SET);
	printf("\nRTC Alarm Interrupt\n\n");
}
