/************************************************************************** 
 * @file  hal_adc.c
 * 
 * Copyright 2008 Texas Instruments, Inc.
***************************************************************************/
#include  "msp430.h"
#include "hal_adc.h"


#if ADC_INTF == ADC_MSP2274        // Interface to internal ADC
/*******************************************************************************
* @brief  Configure the internal ADC for measuring the supply voltage using 
          the internal voltage divider.
* 
* @param  none
* 
* @return calculated battery voltage with 10 bit resolution.
*******************************************************************************/
unsigned char Read_BattVoltage(void) {
    signed long temp;
    // Perform ADC measurement on internal temp sensor
    ADC10CTL0 &= ~ENC;                        // disable ADC to change ports
    ADC10CTL1 = INCH_11 + ADC10DIV_3;         // Temp Sensor ADC10CLK/4
    ADC10CTL0 |= ENC + ADC10SC;               // Sampling and conversion start
    __bis_SR_register(LPM0_bits + GIE);       // LPM0 with interrupts enabled

    temp = ADC10MEM;
    
    temp = (((temp*REF_MVOLT)>>9)/10)-100;

    return (unsigned char)temp;
}

/*******************************************************************************
* @brief  Configure the internal ADC for measuring the supply voltage using 
*         the internal voltage divider.
* 
* @param  none
* 
* @return calculated temperature in 0.1degC steps.
*******************************************************************************/
signed int Read_Internal_Temp_Sensor(void) {
    signed long temp;
    /* Perform ADC measurement on internal temp sensor */
    ADC10CTL0 &= ~ENC;                         // disable ADC to change ports
    ADC10CTL1 = INCH_10 + ADC10DIV_3;          // Temp Sensor ADC10CLK/4
    ADC10CTL0 |= ENC + ADC10SC;                // Sampling and conversion start
    __bis_SR_register(LPM0_bits + GIE);        // LPM0 with interrupts enabled

    temp = ADC10MEM;
    /* 
     * VTEMP=0.00355(TEMPC)+0.986
     * TEMPC = (VTEMP-0.986) / 0.00355 (TEMPC = (mVTEMP-986) / 3.55)
     * ADC10*2500/1024-986)/3.55; (    // ADC10*2500-986*1024)/3.55*1024)
     * ADC10*2500*10-986*10240)/3.55*10240
    */
    temp = ((signed long)ADC10MEM*REF_MVOLT*100-986*102400)/36352;
    return (signed int)temp;
}

/*******************************************************************************
* @brief  Enable ADC, setup all the ADC paramters and wait for reference to 
*         settle
* 
* @param  none
* 
* @return none
*******************************************************************************/
void Enable_ADC(void) {
  
  ADC10CTL1 = INCH_10 + ADC10DIV_3;         // Temp Sensor ADC10CLK/4
  ADC10CTL0 = SREF_1 + ADC10SHT_3 + REFON + ADC10ON + ADC10IE + REF2_5V;

  /* If sensors need a special IO configuration this code could be place here */

  /* Timer based delay function to let the reference settle down */
  TACCR0 = 30;                              // Delay to allow Ref to settle
  TACCTL0 |= CCIE;                          // Compare-mode interrupt
  TACTL = TASSEL_2 + MC_1;                  // TACLK = SMCLK, Up mode
  __bis_SR_register(LPM0_bits + GIE);       // LPM0, TA0_ISR will force exit
  TACCTL0 &= ~CCIE;             
 
}

/*******************************************************************************
* @brief  Disable ADC
* 
* @param  none
* 
* @return none
*******************************************************************************/
void Disable_ADC(void) {
  
  ADC10CTL1 = 0;                             // disable ADC
  ADC10CTL0 = 0;                             // disable ADC
}

/*******************************************************************************
* @brief  ADC10 interrupt service routine 
* 
*******************************************************************************/
#pragma vector=ADC10_VECTOR
__interrupt void ADC10_ISR(void) {
  ADC10CTL0 &= ~ADC10IFG;
  _BIC_SR_IRQ(LPM0_bits);                 // Clear LPM0 bits from 0(SR)
}
#endif