/************************************************************************** 
 * @file  hal_usb.c
 * 
 * Copyright 2008 Texas Instruments, Inc.
***************************************************************************/

/* ****************************************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR REPRESENTATIONS, 
* EITHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING ANY IMPLIED WARRANTIES OF 
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY 
* OR COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET POSSESSION, AND 
* NON-INFRINGEMENT OF ANY THIRD PARTY INTELLECTUAL PROPERTY RIGHTS WITH REGARD
* TO THE PROGRAM OR YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL OR 
* INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY THEORY OF LIABILITY AND WHETHER OR 
* NOT TI HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY 
* OUT OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. EXCLUDED 
* DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF REMOVAL OR REINSTALLATION, 
* COMPUTER TIME, LABOR COSTS, LOSS OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS,
* OR LOSS OF USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S AGGREGATE 
* LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF YOUR USE OF THE PROGRAM 
* EXCEED FIVE HUNDRED DOLLARS (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted by Texas 
* Instruments is distributed as "freeware".  You may, only under TI's copyright
* in the Program, use and modify the Program without any charge or restriction.  
* You may distribute to third parties, provided that you transfer a copy of this
* license to the third party and the third party agrees to these terms by its 
* first use of the Program. You must reproduce the copyright notice and any 
* other legend of ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains copyrighted material, 
* trade secrets and other TI proprietary information and is protected by 
* copyright laws, international copyright treaties, and trade secret laws, as
* well as other intellectual property laws.  To protect TI's rights in the 
* Program, you agree not to decompile, reverse engineer, disassemble or 
* otherwise translate any object code versions of the Program to a human-
* readable form.  You agree that in no event will you alter, remove or destroy
* any copyright notice included in the Program.  TI reserves all rights not 
* specifically granted under this license. Except as specifically provided 
* herein, nothing in this agreement shall be construed as conferring by 
* implication, estoppel, or otherwise, upon you, any license or other right
* under any TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
*******************************************************************************/
#include  "msp430.h"
#include "hal_usb.h"

#define USB_UART_BUFFER_SIZE 40

char halUsbReceiveBuffer[USB_UART_BUFFER_SIZE];
unsigned char bufferSize=0;
volatile char uart_eol = 0;

#if UART_SER_INTF == F5_UART_INTF_USCIA2        // Interface to UART
/**************************************************************************
 * @brief  Initializes the serial communications peripheral and GPIO ports 
 *         to communicate with the TUSB3410.
 * 
 * @param  none
 * 
 * @return none
 **************************************************************************/
void halUsbInit(void)
{
  volatile unsigned char i;
  
  for (i = 0;i < USB_UART_BUFFER_SIZE; i++)
    halUsbReceiveBuffer[i]='\0';
	  
  bufferSize = 0;
  USB_PORT_SEL |= USB_PIN_RXD + USB_PIN_TXD;
  USB_PORT_DIR |= USB_PIN_TXD;
  USB_PORT_DIR &= ~USB_PIN_RXD;
  
  UCA2CTL1 |= UCSWRST;                        //Reset State                      
  UCA2CTL1 |= UCSSEL_2;                       // SMCLK
  UCA2CTL0 = UCMODE_0;
  UCA2CTL0 &= ~UC7BIT;                        // 8bit char
  
  UCA2BR0 = 0x45; //1MHz/9600=104,8Mhz/9600=0x41,8Mhz/115200=0x45,8Mhz/57600=138
  UCA2BR1 = 0;    //1MHz/9600=0,  8Mhz/9600=0x03,8Mhz/115200=0x00,8Mhz/57600=0
 
  UCA2MCTL = UCBRS0;                          // Modulation UCBRSx = 1
  UCA2MCTL = 0xE;
  UCA2CTL1 &= ~UCSWRST;  
  UCA2IE |= UCRXIE;                           // Enable USCI_A0 RX interrupt  
  
  __bis_SR_register(GIE);                     // Enable Interrupts
}

/***************************************************************************
 * @brief  Disables the serial communications peripheral and clears the GPIO
 *         settings used to communicate with the TUSB3410.
 * 
 * @param  none
 * 
 * @return none
 **************************************************************************/
void halUsbShutDown(void)
{
  UCA2IE &= ~UCRXIE;
  UCA2CTL1 = UCSWRST;                          //Reset State                         
  USB_PORT_SEL &= ~( USB_PIN_RXD + USB_PIN_TXD );
  USB_PORT_DIR |= USB_PIN_TXD;
  USB_PORT_DIR |= USB_PIN_RXD;
  USB_PORT_OUT &= ~(USB_PIN_TXD + USB_PIN_RXD);
}

/**************************************************************************
 * @brief  Sends a character over UART to the TUSB3410.
 * 
 * @param  character The character to be sent. 
 * 
 * @return none
 **************************************************************************/
void halUsbSendChar(char character)
{
  while (!(UCA2IFG & UCTXIFG));  
  UCA2TXBUF = character;
}

/**********************************************************************//**
 * @brief  Sends a string of characters to the TUSB3410
 * 
 * @param  string[] The array of characters to be transmit to the TUSB3410.
 * 
 * @param  length   The length of the string.
 * 
 * @return none
 **************************************************************************/
void halUsbSendString(char string[], unsigned char length)
{
  volatile unsigned char i;
  for (i=0; i < length; i++)
    halUsbSendChar(string[i]);  
}

/************************************************************************/
#pragma vector=USCI_A2_VECTOR
__interrupt void USCI_A2_ISR(void) {
  if(UCA2RXBUF == 8) {
    if(bufferSize>0) {
      bufferSize--;
    }
    while (!(UCA2IFG & UCTXIFG));   // print a back space
    UCA2TXBUF = UCA2RXBUF;
    while (!(UCA2IFG & UCTXIFG));   // print a white space
    UCA2TXBUF = 32;
    while (!(UCA2IFG & UCTXIFG));   // print a white space
    UCA2TXBUF = UCA2RXBUF;
  } else {
    halUsbReceiveBuffer[bufferSize++] = UCA2RXBUF; // store inputs
    if(UCA2RXBUF == 13) {     // if its a "return" then activate main-loop
      uart_eol = 1;
     __bic_SR_register_on_exit(LPM3_bits);   
    } else {                  // if normail character, loop it back for ui
       UCA2TXBUF = UCA2RXBUF;
    }
  }
}

#elif UART_SER_INTF == F5_UART_INTF_USCIA1        // Interface to UART
/**************************************************************************
 * @brief  Initializes the serial communications peripheral and GPIO ports 
 *         to communicate with the TUSB3410.
 * 
 * @param  none
 * 
 * @return none
 **************************************************************************/
void halUsbInit(void)
{
  volatile unsigned char i;
  
  for (i = 0;i < USB_UART_BUFFER_SIZE; i++)
    halUsbReceiveBuffer[i]='\0';
	  
  bufferSize = 0;
  USB_PORT_SEL |= USB_PIN_RXD + USB_PIN_TXD;
  USB_PORT_DIR |= USB_PIN_TXD;
  USB_PORT_DIR &= ~USB_PIN_RXD;
  
  UCA1CTL1 |= UCSWRST;                        //Reset State                      
  UCA1CTL1 |= UCSSEL_2;                       // SMCLK
  UCA1CTL0 = UCMODE_0;
  UCA1CTL0 &= ~UC7BIT;                        // 8bit char
  
  UCA1BR0 = 138; //1MHz/9600=104,8Mhz/9600=0x41,8Mhz/115200=0x45,8Mhz/57600=138
  UCA1BR1 = 0;    //1MHz/9600=0,  8Mhz/9600=0x03,8Mhz/115200=0x00,8Mhz/57600=0
 
  UCA1MCTL = UCBRS0;                          // Modulation UCBRSx = 1
  UCA1MCTL = 0xE;
  UCA1CTL1 &= ~UCSWRST;  
  UCA1IE |= UCRXIE;                           // Enable USCI_A0 RX interrupt  
  
  __bis_SR_register(GIE);                   // Enable Interrupts
}

/***************************************************************************
 * @brief  Disables the serial communications peripheral and clears the GPIO
 *         settings used to communicate with the TUSB3410.
 * 
 * @param  none
 * 
 * @return none
 **************************************************************************/
void halUsbShutDown(void)
{
  UCA1IE &= ~UCRXIE;
  UCA1CTL1 = UCSWRST;                          //Reset State                         
  USB_PORT_SEL &= ~( USB_PIN_RXD + USB_PIN_TXD );
  USB_PORT_DIR |= USB_PIN_TXD;
  USB_PORT_DIR |= USB_PIN_RXD;
  USB_PORT_OUT &= ~(USB_PIN_TXD + USB_PIN_RXD);
}

/**************************************************************************
 * @brief  Sends a character over UART to the TUSB3410.
 * 
 * @param  character The character to be sent. 
 * 
 * @return none
 **************************************************************************/
void halUsbSendChar(char character)
{
  while (!(UCA1IFG & UCTXIFG));  
  UCA1TXBUF = character;
}

/**********************************************************************//**
 * @brief  Sends a string of characters to the TUSB3410
 * 
 * @param  string[] The array of characters to be transmit to the TUSB3410.
 * 
 * @param  length   The length of the string.
 * 
 * @return none
 **************************************************************************/
void halUsbSendString(char string[], unsigned char length)
{
  volatile unsigned char i;
  for (i=0; i < length; i++)
    halUsbSendChar(string[i]);  
}

/************************************************************************/
#pragma vector=USCI_A1_VECTOR
__interrupt void USCI_A1_ISR (void)
{
    if(UCA1RXBUF == 8) {
    if(bufferSize>0) {
      bufferSize--;
    }
    while (!(UCA1IFG & UCTXIFG));   // print a back space
    UCA1TXBUF = UCA1RXBUF;
    while (!(UCA1IFG & UCTXIFG));   // print a white space
    UCA1TXBUF = 32;
    while (!(UCA1IFG & UCTXIFG));   // print a white space
    UCA1TXBUF = UCA1RXBUF;
  } else {
    halUsbReceiveBuffer[bufferSize++] = UCA1RXBUF; // store inputs
    if(UCA1RXBUF == 13) {     // if its a "return" then activate main-loop
      uart_eol = 1;
     __bic_SR_register_on_exit(LPM3_bits);   
    } else {                  // if normail character, loop it back for ui
       UCA1TXBUF = UCA1RXBUF;
    }
  }
}

#elif UART_SER_INTF == F5_UART_INTF_USCIA0        // Interface to UART
/**************************************************************************
 * @brief  Initializes the serial communications peripheral and GPIO ports 
 *         to communicate with the TUSB3410.
 * 
 * @param  none
 * 
 * @return none
 **************************************************************************/
void halUsbInit(void)
{
  volatile unsigned char i;
  
  for (i = 0;i < USB_UART_BUFFER_SIZE; i++)
    halUsbReceiveBuffer[i]='\0';
	  
  bufferSize = 0;
  USB_PORT_SEL |= USB_PIN_RXD + USB_PIN_TXD;
  USB_PORT_DIR |= USB_PIN_TXD;
  USB_PORT_DIR &= ~USB_PIN_RXD;
  
  UCA0CTL1 |= UCSWRST;                        //Reset State                      
  UCA0CTL1 |= UCSSEL_2;                       // SMCLK
  UCA0CTL0 = UCMODE_0;
  UCA0CTL0 &= ~UC7BIT;                        // 8bit char
  
  UCA0BR0 = 138; //1MHz/9600=104,8Mhz/9600=0x41,8Mhz/115200=0x45,8Mhz/57600=138
  UCA0BR1 = 0;    //1MHz/9600=0,  8Mhz/9600=0x03,8Mhz/115200=0x00,8Mhz/57600=0
 
  UCA0MCTL = UCBRS0;                          // Modulation UCBRSx = 1
  UCA0MCTL = 0xE;
  UCA0CTL1 &= ~UCSWRST;  
  UCA0IE |= UCRXIE;                           // Enable USCI_A0 RX interrupt  
  
  __bis_SR_register(GIE);                   // Enable Interrupts
}

/***************************************************************************
 * @brief  Disables the serial communications peripheral and clears the GPIO
 *         settings used to communicate with the TUSB3410.
 * 
 * @param  none
 * 
 * @return none
 **************************************************************************/
void halUsbShutDown(void)
{
  UCA0IE &= ~UCRXIE;
  UCA0CTL1 = UCSWRST;                          //Reset State                         
  USB_PORT_SEL &= ~( USB_PIN_RXD + USB_PIN_TXD );
  USB_PORT_DIR |= USB_PIN_TXD;
  USB_PORT_DIR |= USB_PIN_RXD;
  USB_PORT_OUT &= ~(USB_PIN_TXD + USB_PIN_RXD);
}

/**************************************************************************
 * @brief  Sends a character over UART to the TUSB3410.
 * 
 * @param  character The character to be sent. 
 * 
 * @return none
 **************************************************************************/
void halUsbSendChar(char character)
{
  while (!(UCA0IFG & UCTXIFG));  
  UCA0TXBUF = character;
}

/**********************************************************************//**
 * @brief  Sends a string of characters to the TUSB3410
 * 
 * @param  string[] The array of characters to be transmit to the TUSB3410.
 * 
 * @param  length   The length of the string.
 * 
 * @return none
 **************************************************************************/
void halUsbSendString(char string[], unsigned char length)
{
  volatile unsigned char i;
  for (i=0; i < length; i++)
    halUsbSendChar(string[i]);  
}

/************************************************************************/
#pragma vector=USCI_A0_VECTOR
__interrupt void USCI_A1_ISR (void)
{
    if(UCA0RXBUF == 8) {
    if(bufferSize>0) {
      bufferSize--;
    }
    while (!(UCA0IFG & UCTXIFG));   // print a back space
    UCA0TXBUF = UCA0RXBUF;
    while (!(UCA0IFG & UCTXIFG));   // print a white space
    UCA0TXBUF = 32;
    while (!(UCA0IFG & UCTXIFG));   // print a white space
    UCA0TXBUF = UCA0RXBUF;
  } else {
    halUsbReceiveBuffer[bufferSize++] = UCA0RXBUF; // store inputs
    if(UCA0RXBUF == 13) {     // if its a "return" then activate main-loop
      uart_eol = 1;
     __bic_SR_register_on_exit(LPM3_bits);   
    } else {                  // if normail character, loop it back for ui
       UCA0TXBUF = UCA0RXBUF;
    }
  }
}
#elif UART_SER_INTF == F2_UART_INTF_USCIA0        // Interface to UART 

/**************************************************************************
 * @brief  Initializes the serial communications peripheral and GPIO ports 
 *         to communicate with the TUSB3410.
 * 
 * @param  none
 * 
 * @return none
 **************************************************************************/
void halUsbInit(void)
{
  volatile unsigned char i;
  
  for (i = 0;i < USB_UART_BUFFER_SIZE; i++)
    halUsbReceiveBuffer[i]='\0';
	  
  bufferSize = 0;
  USB_PORT_SEL |= USB_PIN_RXD + USB_PIN_TXD;
  USB_PORT_DIR |= USB_PIN_TXD;
  USB_PORT_DIR &= ~USB_PIN_RXD;
  
  UCA0CTL1 |= UCSWRST;                          //Reset State                      
  UCA0CTL0 = UCMODE_0;
  UCA0CTL0 &= ~UC7BIT;                        // 8bit char
  UCA0CTL1 |= UCSSEL_2;                       // SMCLK
  UCA0BR0 = 0xA5;            //1MHz/9600=104  // 6.5Mhz/9600=0xA5
  UCA0BR1 = 0x02;            //1MHz/9600=0    // 6.5Mhz/9600=0x02
  UCA0MCTL = UCBRS0;                          // Modulation UCBRSx = 1
  UCA0CTL1 &= ~UCSWRST;  
  IE2 |= UCA0RXIE;                          // Enable USCI_A0 RX interrupt
  
  __bis_SR_register(GIE);                   // Enable Interrupts
}

/***************************************************************************
 * @brief  Disables the serial communications peripheral and clears the GPIO
 *         settings used to communicate with the TUSB3410.
 * 
 * @param  none
 * 
 * @return none
 **************************************************************************/
void halUsbShutDown(void)
{
  UC0IE &= ~UCA0RXIE;
  UCA0CTL1 = UCSWRST;                          //Reset State                         
  USB_PORT_SEL &= ~( USB_PIN_RXD + USB_PIN_TXD );
  USB_PORT_DIR |= USB_PIN_TXD;
  USB_PORT_DIR |= USB_PIN_RXD;
  USB_PORT_OUT &= ~(USB_PIN_TXD + USB_PIN_RXD);
}

/**************************************************************************
 * @brief  Sends a character over UART to the TUSB3410.
 * 
 * @param  character The character to be sent. 
 * 
 * @return none
 **************************************************************************/
void halUsbSendChar(char character)
{
  while (!(IFG2 & UCA0TXIFG));  
  UCA0TXBUF = character;

}

/**********************************************************************//**
 * @brief  Sends a string of characters to the TUSB3410
 * 
 * @param  string[] The array of characters to be transmit to the TUSB3410.
 * 
 * @param  length   The length of the string.
 * 
 * @return none
 **************************************************************************/
void halUsbSendString(char string[], unsigned char length)
{
  volatile unsigned char i;
  for (i=0; i < length; i++)
    halUsbSendChar(string[i]);  
}

///************************************************************************/
#pragma vector=USCIAB0RX_VECTOR
__interrupt void USCI0RX_ISR(void) {
  if(UCA0RXBUF == 8) {
    if(bufferSize>0) {
      bufferSize--;
    }
    while (!(IFG2 & UCA0TXIFG));   // print a back space
    UCA0TXBUF = UCA0RXBUF;
    while (!(IFG2 & UCA0TXIFG));   // print a white space
    UCA0TXBUF = 32;
    while (!(IFG2 & UCA0TXIFG));   // print a white space
    UCA0TXBUF = UCA0RXBUF;
  } else {
    halUsbReceiveBuffer[bufferSize++] = UCA0RXBUF; // store inputs
    if(UCA0RXBUF == 13) {     // if its a "return" then activate main-loop
      uart_eol = 1;
     __bic_SR_register_on_exit(LPM3_bits);   
    } else {                  // if normail character, loop it back for ui
       UCA0TXBUF = UCA0RXBUF;
    }
  }
}

#elif UART_SER_INTF == FR5x_UART_INTF_USCIA0        // Interface to UART
/**************************************************************************
 * @brief  Initializes the serial communications peripheral and GPIO ports 
 *         to communicate with the TUSB3410.
 * 
 * @param  none
 * 
 * @return none
 **************************************************************************/
void halUsbInit(void)
{
  volatile unsigned char i;
  
  for (i = 0;i < USB_UART_BUFFER_SIZE; i++)
    halUsbReceiveBuffer[i]='\0';
	  
  bufferSize = 0;
  USB_PORT_SEL1 |=  (USB_PIN_RXD + USB_PIN_TXD);
  USB_PORT_SEL0 &= ~(USB_PIN_RXD + USB_PIN_TXD);
  USB_PORT_DIR |= USB_PIN_TXD;
  USB_PORT_DIR &= ~USB_PIN_RXD;
  
  UCA0CTL1 |= UCSWRST; 
  UCA0CTL1 = UCSSEL_1;                      // Set ACLK = 32768 as UCBRCLK
  UCA0BR0 = 3;                              // 9600 baud
  UCA0BR1 = 0; 
  UCA0MCTLW |= 0x5300;                      // 32768/9600 - INT(32768/9600)=0.41
                                            // UCBRSx value = 0x53 (See UG)
  UCA0CTL1 &= ~UCSWRST;                     // release from reset
  UCA0IE |= UCRXIE;                         // Enable RX interrupt 
  
  __bis_SR_register(GIE);                   // Enable Interrupts
}

/***************************************************************************
 * @brief  Disables the serial communications peripheral and clears the GPIO
 *         settings used to communicate with the TUSB3410.
 * 
 * @param  none
 * 
 * @return none
 **************************************************************************/
void halUsbShutDown(void)
{
  UCA0IE &= ~UCRXIE;
  UCA0CTL1 = UCSWRST;                          //Reset State                         
  USB_PORT_SEL1 &= ~(USB_PIN_RXD + USB_PIN_TXD);  
  USB_PORT_SEL0 &= ~(USB_PIN_RXD + USB_PIN_TXD);  
  USB_PORT_DIR |= USB_PIN_TXD;
  USB_PORT_DIR |= USB_PIN_RXD;
  USB_PORT_OUT &= ~(USB_PIN_TXD + USB_PIN_RXD);
}

/**************************************************************************
 * @brief  Sends a character over UART to the TUSB3410.
 * 
 * @param  character The character to be sent. 
 * 
 * @return none
 **************************************************************************/
void halUsbSendChar(char character)
{
  while (!(UCA0IFG & UCTXIFG));  
  UCA0TXBUF = character;
}

/**********************************************************************//**
 * @brief  Sends a string of characters to the TUSB3410
 * 
 * @param  string[] The array of characters to be transmit to the TUSB3410.
 * 
 * @param  length   The length of the string.
 * 
 * @return none
 **************************************************************************/
void halUsbSendString(char string[], unsigned char length)
{
  volatile unsigned char i;
  for (i=0; i < length; i++)
    halUsbSendChar(string[i]);  
}

/************************************************************************/
#pragma vector=USCI_A0_VECTOR
__interrupt void USCI_A1_ISR (void)
{
    if(UCA0RXBUF == 8) {
    if(bufferSize>0) {
      bufferSize--;
    }
    while (!(UCA0IFG & UCTXIFG));   // print a back space
    UCA0TXBUF = UCA0RXBUF;
    while (!(UCA0IFG & UCTXIFG));   // print a white space
    UCA0TXBUF = 32;
    while (!(UCA0IFG & UCTXIFG));   // print a white space
    UCA0TXBUF = UCA0RXBUF;
  } else {
    halUsbReceiveBuffer[bufferSize++] = UCA0RXBUF; // store inputs
    if(UCA0RXBUF == 13) {     // if its a "return" then activate main-loop
      uart_eol = 1;
     __bic_SR_register_on_exit(LPM3_bits);   
    } else {                  // if normail character, loop it back for ui
       UCA0TXBUF = UCA0RXBUF;
    }
  }
}

#endif