/*******************************************************************************
*  Description:  This file contains functions that allow the MSP430 device to
*  access the SPI interface of the CC1100/CC2500.  There are multiple
*  instances of each function; the one to be compiled is selected by the
*  system variable CC_RF_SER_INTF, defined in "CC_hardware_board.h".
*
*  MSP430/CC1100-2500 Interface Code Library v1.3
*
*  W. Goh & Thomas Almholt
*  Texas Instruments, Inc.
*  December 2009, October 2010, October 2011
*  IAR Embedded Workbench v5.10
********************************************************************************
* Version:  1.3
* Comments: Added support for FRAM devices
* Author  : Thomas Almholt, Texas Instruments, October 2010
*
* Version:  1.2
* Comments: Added support for antenna diversity thru the use of multiple radios
*           Added support for collision detection, and various help functions.
*           Changed comments for better readability
* Author  : Thomas Almholt, Texas Instruments, October 2010
*
* Version:  1.1
* Comments: Fixed several bugs where it is stuck in a infinite while loop
*           Added support for 5xx
*
* Version:  1.00
* Comments: Initial Release Version
*******************************************************************************/
#include "CCx_hal.h"
#include "CCxx00_spi.h"
#include "CCxx00_def.h"
#include "CCxx00_msp430.h"

/*******************************************************************************
*  void CC_Wait(void)
*
*  DESCRIPTION:
* Delay function. # of CPU cycles delayed is similar to "cycles". Specifically,
* it's ((cycles-15) % 6) + 15.  Not exact, but gives a sense of the real-time
* delay.  Also, if MCLK ~1MHz, "cycles" is similar to # of useconds delayed.
*******************************************************************************/
void CC_Wait(unsigned int cycles) {
  cycles = cycles * 8;
  while(cycles>1)                  // 15 cycles consumed by overhead
    cycles--;                       // 6 cycles consumed each iteration
}

/******************************************************************************
* If USART0 is used
******************************************************************************/
#if CC_RF_SER_INTF == CC_SER_INTF_USART0
/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART0 SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
   if (RF_IDx==RF_SLOTB) {
      CC_CSBn_PxOUT |= CC_CSBn_PIN;
      CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
      CC_CSAn_PxOUT |= CC_CSAn_PIN;
      CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  ME1 |= USPIE0;                            // Enable USART0 SPI mode
  UCTL0 = SWRST;                            // Disable USART state machine
  UCTL0 |= CHAR + SYNC + MM;                // 8-bit SPI Master **SWRST**
  UTCTL0 |= CKPH + SSEL1 + SSEL0 + STC;     // SMCLK, 3-pin mode
  UBR00 = 0x02;                             // UCLK/2
  UBR10 = 0x00;                             // 0
  UMCTL0 = 0x00;                            // No modulation
  CC_SPI_USART0_PxSEL |= CC_SPI_USART0_SIMO | CC_SPI_USART0_SOMI 
                              | CC_SPI_USART0_UCLK;
                                            // SPI option select
  CC_SPI_USART0_PxDIR |= CC_SPI_USART0_SIMO + CC_SPI_USART0_UCLK;
                                            // SPI TX out direction
  UCTL0 &= ~SWRST;                          // Initialize USART state machine
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART0 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG1&UTXIFG0));                  // Wait for TX to finish
  U0TXBUF = addr;                           // Send address
  while (!(IFG1&UTXIFG0));                  // Wait for TX to finish
  U0TXBUF = value;                          // Send value
  while(!(UTCTL0&TXEPT));                   // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART0 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG1 & UTXIFG0));                // Wait for TX to finish
  U0TXBUF = addr | WRITE_BURST;   // Send address
  for (i = 0; i < count; i++)
  {
    while (!(IFG1 & UTXIFG0));              // Wait for TX to finish
    U0TXBUF = buffer[i];                    // Send data
  }
  while(!(UTCTL0 & TXEPT));
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART0 SPI port
*******************************************************************************/
char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG1 & UTXIFG0));                // Wait for TX to finish
  U0TXBUF = (addr | READ_SINGLE);           // Send address
  while (!(IFG1 & UTXIFG0));                // Wait for TX to finish
  U0TXBUF = 0;                              // Dummy write so we can read data
  while(!(UTCTL0 & TXEPT));                 // Wait for TX complete
  x = U0RXBUF;                              // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART0 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG1 & UTXIFG0));                // Wait for TXBUF ready
  U0TXBUF = (addr | READ_BURST);  // Send address
  while(!(UTCTL0 & TXEPT));                 // Wait for TX complete
  U0TXBUF = 0;                              // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  IFG1 &= ~URXIFG0;                         // Clear flag
  while (!(IFG1&URXIFG0));                  // Wait for end of 1st data byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    U0TXBUF = 0;                            // Initiate next data RX, meanwhile
    buffer[i] = U0RXBUF;                    // Store data from last data RX
    while (!(IFG1&URXIFG0));                // Wait for end of data RX
  }
  buffer[count-1] = U0RXBUF;                // Store last RX byte in buffer
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
 MSP430F2xx USART0 SPI port
*******************************************************************************/
char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG1 & UTXIFG0));                // Wait for TX to finish
  U0TXBUF = (addr | READ_BURST);            // Send address
  while (!(IFG1 & UTXIFG0));                // Wait for TX to finish
  U0TXBUF = 0;                              // Dummy write so we can read data
  while(!(UTCTL0 & TXEPT));                 // Wait for TX complete
  status = U0RXBUF;                         // Read data
 if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
 MSP430F2xx USART0 SPI port
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG1 & UTXIFG0));                // Wait for TX to finish
  U0TXBUF = strobe;                         // Send strobe
  // Strobe addr is now being TX'ed
  IFG1 &= ~URXIFG0;                         // Clear flag
  while(!(UTCTL0 & TXEPT));                 // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr". See Sec. 27.1 of CC1100 datasheet
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG1 & UTXIFG0));                // Wait for TX to finish
  U0TXBUF = SRES;                 // Send strobe
  // Strobe addr is now being TX'ed
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

// END USART0

/******************************************************************************
* If USART1 is used
******************************************************************************/
#elif CC_RF_SER_INTF == CC_SER_INTF_USART1

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
   if (RF_IDx==RF_SLOTB) {
      CC_CSBn_PxOUT |= CC_CSBn_PIN;
      CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
      CC_CSAn_PxOUT |= CC_CSAn_PIN;
      CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  ME2 |= USPIE1;                            // Enable USART1 SPI mode
  UCTL1 = SWRST;                            // Disable USART state machine
  UCTL1 |= CHAR + SYNC + MM;                // 8-bit SPI Master **SWRST**
  UTCTL1 |= CKPH + SSEL1 + SSEL0 + STC;     // SMCLK, 3-pin mode
  UBR01 = 0x02;                             // UCLK/2
  UBR11 = 0x00;                             // 0
  UMCTL1 = 0x00;                            // No modulation
  CC_SPI_USART1_PxSEL |= CC_SPI_USART1_SIMO
                          | CC_SPI_USART1_SOMI
                          | CC_SPI_USART1_UCLK;
                                            // SPI option select
  CC_SPI_USART1_PxDIR |= CC_SPI_USART1_SIMO + CC_SPI_USART1_UCLK;
                                            // SPI TXD out direction
  UCTL1 &= ~SWRST;                          // Initialize USART state machine
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UTXIFG1));                  // Wait for TX to finish
  U1TXBUF = addr;                           // Send address
  while (!(IFG2&UTXIFG1));                  // Wait for TX to finish
  U1TXBUF = value;                          // Load data for TX after addr
  while(!(UTCTL1&TXEPT));                   // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UTXIFG1));                  // Wait for TX to finish
  U1TXBUF = addr | WRITE_BURST;   // Send address
  for (i = 0; i < count; i++)
  {
    while (!(IFG2&UTXIFG1));                // Wait for TX to finish
    U1TXBUF = buffer[i];                    // Send data
  }
  while(!(UTCTL1&TXEPT));                   // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UTXIFG1));                  // Wait for TX to finish
  U1TXBUF = (addr | READ_SINGLE); // Send address
  while (!(IFG2&UTXIFG1));                  // Wait for TX to finish
  U1TXBUF = 0;                              // Load dummy byte for TX after addr
  while(!(UTCTL1&TXEPT));                   // Wait for TX complete
  x = U1RXBUF;                              // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UTXIFG1));                  // Wait for TXBUF ready
  U1TXBUF = (addr | READ_BURST);  // Send address
  while(!(UTCTL1&TXEPT));                   // Wait for TX complete
  U1TXBUF = 0;                              // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  IFG2 &= ~URXIFG1;                         // Clear flag
  while (!(IFG2&URXIFG1));                  // Wait for end of 1st data byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    U1TXBUF = 0;                            // Initiate next data RX, meanwhile
    buffer[i] = U1RXBUF;                    // Store data from last data RX
    while (!(IFG2&URXIFG1));                // Wait for end of data RX
  }
  buffer[count-1] = U1RXBUF;                // Store last RX byte in buffer
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UTXIFG1));                  // Wait for TX to finish
  U1TXBUF = (addr | READ_BURST);  // Send address
  while (!(IFG2&UTXIFG1));                  // Wait for TX to finish
  U1TXBUF = 0;                              // Dummy write so we can read data
  while(!(UTCTL1&TXEPT));                   // Wait for TX complete
  status = U1RXBUF;                         // Read data
 if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART0 SPI port
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG1 & UTXIFG0));                // Wait for TX to finish
  U0TXBUF = strobe;                         // Send strobe
  // Strobe addr is now being TX'ed
  IFG1 &= ~URXIFG0;                         // Clear flag
  while(!(UTCTL0 & TXEPT));                 // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART0 SPI port
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG1 & UTXIFG0));                // Wait for TX to finish
  U1TXBUF = strobe;                         // Send strobe
  // Strobe addr is now being TX'ed
  while(!(UTCTL1&TXEPT));                   // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr". See Sec. 27.1 of CC1100 datasheet
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART0 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2 & UTXIFG1));                // Wait for TX to finish
  U1TXBUF = SRES;                 // Send strobe
  // Strobe addr is now being TX'ed
  while(!(UTCTL1&TXEPT));                   // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

// END CC_SER_INTF_USART1

/******************************************************************************
* If USCIA0 is used
*   |-- If 5xx
*         |-- Use 5xx Init
*   |-- Else
*         |-- Use 2xx, 4xx Init
*******************************************************************************/
#elif CC_RF_SER_INTF == CC_SER_INTF_USCIA0

/******************************************************************************
* Support for 5xx USCI_A0
*******************************************************************************/
#ifdef MSP430_F5xx

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_A0 SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
   if (RF_IDx==RF_SLOTB) {
      CC_CSBn_PxOUT |= CC_CSBn_PIN;
      CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
      CC_CSAn_PxOUT |= CC_CSAn_PIN;
      CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  UCA0CTL1 |= UCSWRST;                      // **Disable USCI state machine**
  UCA0CTL0 |= UCMST+UCCKPH+UCMSB+UCSYNC;    // 3-pin, 8-bit SPI master
  UCA0CTL1 |= UCSSEL_2;                     // SMCLK
  UCA0BR0 = 0x02;                           // UCLK/2
  UCA0BR1 = 0;
  UCA0MCTL = 0;
  CC_SPI_USCIA0_PxSEL |= CC_SPI_USCIA0_SIMO
                         | CC_SPI_USCIA0_SOMI
                         | CC_SPI_USCIA0_UCLK;
                                            // SPI option select
  CC_SPI_USCIA0_PxDIR |= CC_SPI_USCIA0_SIMO | CC_SPI_USCIA0_UCLK;
                                            // SPI TXD out direction
  UCA0CTL1 &= ~UCSWRST;                     // **Initialize USCI state machine**
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_A0 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA0TXBUF = addr;                         // Send address
  while (!(UCA0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA0TXBUF = value;                        // Send data
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_A0 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA0TXBUF = addr | WRITE_BURST; // Send address
  for (i = 0; i < count; i++)
  {
    while (!(UCA0IFG&UCTXIFG));               // Wait for TXBUF ready
    UCA0TXBUF = buffer[i];                  // Send data
  }
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_A0 SPI port
*******************************************************************************/
char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA0TXBUF = (addr | READ_SINGLE);// Send address
  while (!(UCA0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA0TXBUF = 0;                            // Dummy write so we can read data
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  x = UCA0RXBUF;                            // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA0TXBUF = (addr | READ_BURST);// Send address
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  UCA0TXBUF = 0;                            // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  UCA0IFG &= ~UCRXIFG;                      // Clear flag
  while (!(UCA0IFG&UCRXIFG));               // Wait for end of addr byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    UCA0TXBUF = 0;                          //Initiate next data RX, meanwhile..
    buffer[i] = UCA0RXBUF;                  // Store data from last data RX
    while (!(UCA0IFG&UCRXIFG));             // Wait for RX to finish
  }
  buffer[count-1] = UCA0RXBUF;              // Store last RX byte in buffer
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA0TXBUF = (addr | READ_BURST);// Send address
  while (!(UCA0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA0TXBUF = 0;                            // Dummy write so we can read data
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  status = UCA0RXBUF;                       // Read data
 if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA0TXBUF = strobe;                       // Send strobe
  // Strobe addr is now being TX'ed
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr". See Sec. 27.1 of CC1100 datasheet
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA0TXBUF = SRES;               // Send strobe
  // Strobe addr is now being TX'ed
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/* End of support for 5xx USCI_A0 */
#else

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_A0 SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
   if (RF_IDx==RF_SLOTB) {
      CC_CSBn_PxOUT |= CC_CSBn_PIN;
      CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
      CC_CSAn_PxOUT |= CC_CSAn_PIN;
      CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  UCA0CTL1 |= UCSWRST;                      // **Disable USCI state machine**
  UCA0CTL0 |= UCMST+UCCKPH+UCMSB+UCSYNC;    // 3-pin, 8-bit SPI master
  UCA0CTL1 |= UCSSEL_2;                     // SMCLK
  UCA0BR0 = 0x02;                           // UCLK/2
  UCA0BR1 = 0;
  UCA0MCTL = 0;
  CC_SPI_USCIA0_PxSEL |= CC_SPI_USCIA0_SIMO
                         | CC_SPI_USCIA0_SOMI
                         | CC_SPI_USCIA0_UCLK;
                                            // SPI option select
  CC_SPI_USCIA0_PxDIR |= CC_SPI_USCIA0_SIMO | CC_SPI_USCIA0_UCLK;
                                            // SPI TXD out direction
  UCA0CTL1 &= ~UCSWRST;                     // **Initialize USCI state machine**
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_A0 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UCA0TXIFG));                // Wait for TXBUF ready
  UCA0TXBUF = addr;                         // Send address
  while (!(IFG2&UCA0TXIFG));                // Wait for TXBUF ready
  UCA0TXBUF = value;                        // Send data
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_A0 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UCA0TXIFG));                // Wait for TXBUF ready
  UCA0TXBUF = addr | WRITE_BURST; // Send address
  for (i = 0; i < count; i++)
  {
    while (!(IFG2&UCA0TXIFG));              // Wait for TXBUF ready
    UCA0TXBUF = buffer[i];                  // Send data
  }
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_A0 SPI port
*******************************************************************************/
char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UCA0TXIFG));                // Wait for TX to finish
  UCA0TXBUF = (addr | READ_SINGLE);// Send address
  while (!(IFG2&UCA0TXIFG));                // Wait for TX to finish
  UCA0TXBUF = 0;                            // Dummy write so we can read data
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  x = UCA0RXBUF;                            // Read data
   if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    

  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_A0 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    

  while (!(IFG2&UCA0TXIFG));                // Wait for TX to finish
  UCA0TXBUF = (addr | READ_BURST);// Send address
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  UCA0TXBUF = 0;                            // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  IFG2 &= ~UCA0RXIFG;                       // Clear flag
  while (!(IFG2&UCA0RXIFG));                // Wait for end of addr byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    UCA0TXBUF = 0;                          //Initiate next data RX, meanwhile..
    buffer[i] = UCA0RXBUF;                  // Store data from last data RX
    while (!(IFG2&UCA0RXIFG));              // Wait for RX to finish
  }
  buffer[count-1] = UCA0RXBUF;              // Store last RX byte in buffer
   if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    

  while (!(IFG2&UCA0TXIFG));                // Wait for TX to finish
  UCA0TXBUF = (addr | READ_BURST);// Send address
  while (!(IFG2&UCA0TXIFG));                // Wait for TX to finish
  UCA0TXBUF = 0;                            // Dummy write so we can read data
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  status = UCA0RXBUF;                       // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    

  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    

  while (!(IFG2&UCA0TXIFG));                // Wait for TX to finish
  UCA0TXBUF = strobe;                       // Send strobe
  // Strobe addr is now being TX'ed
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr". See Sec. 27.1 of CC1100 datasheet
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UCA0TXIFG));                // Wait for TXBUF ready
  UCA0TXBUF = SRES;               // Send strobe
  // Strobe addr is now being TX'ed
  while (UCA0STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

#endif
// END CC_SER_INTF_USCIA0

//******************************************************************************
// If USCIA1 is used
//******************************************************************************
#elif CC_RF_SER_INTF == CC_SER_INTF_USCIA1

//******************************************************************************
// Support for 5xx USCI_A1
//******************************************************************************
#ifdef MSP430_F5xx

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIA1 SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
   if (RF_IDx==RF_SLOTB) {
      CC_CSBn_PxOUT |= CC_CSBn_PIN;
      CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
      CC_CSAn_PxOUT |= CC_CSAn_PIN;
      CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  UCA1CTL1 |= UCSWRST;                      // **Disable USCI state machine**
  UCA1CTL0 |= UCMST+UCCKPH+UCMSB+UCSYNC;    // 3-pin, 8-bit SPI master
  UCA1CTL1 |= UCSSEL_2;                     // SMCLK
  UCA1BR0 = 0x02;                           // UCLK/2
  UCA1BR1 = 0;
  UCA1MCTL = 0;
  CC_SPI_USCIA1_PxSEL |= CC_SPI_USCIA1_SIMO
                         | CC_SPI_USCIA1_SOMI;
  CC_SPI_USCIA1_PxSEL_UCLK |= CC_SPI_USCIA1_UCLK;
                                            // SPI option select
  CC_SPI_USCIA1_PxDIR |= CC_SPI_USCIA1_SIMO;
  CC_SPI_USCIA1_PxDIR_UCLK |=  CC_SPI_USCIA1_UCLK;
                                            // SPI TXD out direction
  UCA1CTL1 &= ~UCSWRST;                     // **Initialize USCI state machine**
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIA1 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA1TXBUF = addr;                         // Send address
  while (!(UCA1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA1TXBUF = value;                        // Send data
  while (UCA1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIA1 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA1TXBUF = addr | WRITE_BURST; // Send address
  for (i = 0; i < count; i++)
  {
    while (!(UCA1IFG&UCTXIFG));               // Wait for TXBUF ready
    UCA1TXBUF = buffer[i];                  // Send data
  }
  while (UCA1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIA1 SPI port
*******************************************************************************/
char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA1TXBUF = (addr | READ_SINGLE);// Send address
  while (!(UCA1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA1TXBUF = 0;                            // Dummy write so we can read data
  while (UCA1STAT & UCBUSY);                // Wait for TX complete
  x = UCA1RXBUF;                            // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USART1 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA1TXBUF = (addr | READ_BURST);// Send address
  while (UCA1STAT & UCBUSY);                // Wait for TX complete
  UCA1TXBUF = 0;                            // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  UCA1IFG &= ~UCRXIFG;                      // Clear flag
  while (!(UCA1IFG&UCRXIFG));               // Wait for end of addr byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    UCA1TXBUF = 0;                          //Initiate next data RX, meanwhile..
    buffer[i] = UCA1RXBUF;                  // Store data from last data RX
    while (!(UCA1IFG&UCRXIFG));             // Wait for RX to finish
  }
  buffer[count-1] = UCA1RXBUF;              // Store last RX byte in buffer
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USART1 SPI port
*******************************************************************************/
char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA1TXBUF = (addr | READ_BURST);// Send address
  while (!(UCA1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA1TXBUF = 0;                            // Dummy write so we can read data
  while (UCA1STAT & UCBUSY);                // Wait for TX complete
  status = UCA1RXBUF;                       // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USART1 SPI port
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA1TXBUF = strobe;                       // Send strobe
  // Strobe addr is now being TX'ed
  while (UCA1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr". See Sec. 27.1 of CC1100 datasheet
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA1TXBUF = SRES;               // Send strobe
  // Strobe addr is now being TX'ed
  while (UCA1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/* End of support for 5xx USCI_A1 */
#else

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
   if (RF_IDx==RF_SLOTB) {
      CC_CSBn_PxOUT |= CC_CSBn_PIN;
      CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
      CC_CSAn_PxOUT |= CC_CSAn_PIN;
      CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  UCA1CTL1 |= UCSWRST;                      // **Disable USCI state machine**
  UCA1CTL0 |= UCMST+UCCKPH+UCMSB+UCSYNC;    // 3-pin, 8-bit SPI master
  UCA1CTL1 |= UCSSEL_2;                     // SMCLK
  UCA1BR0 = 0x02;                           // UCLK/2
  UCA1BR1 = 0;
  UCA1MCTL = 0;
  CC_SPI_USCIA1_PxSEL |= CC_SPI_USCIA1_SIMO
                         | CC_SPI_USCIA1_SOMI
                         | CC_SPI_USCIA1_UCLK;
                                            // SPI option select
  CC_SPI_USCIA1_PxDIR |= CC_SPI_USCIA1_SIMO | CC_SPI_USCIA1_UCLK;
                                            // SPI TXD out direction
  UCA1CTL1 &= ~UCSWRST;                     // **Initialize USCI state machine**
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UC1IFG&UCA1TXIFG));              // Wait for TXBUF ready
  UCA1TXBUF = addr;                         // Send address
  while (!(UC1IFG&UCA1TXIFG));              // Wait for TXBUF ready
  UCA1TXBUF = value;                        // Send data
  while (UCA1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UC1IFG&UCA1TXIFG));              // Wait for TXBUF ready
  UCA1TXBUF = addr | WRITE_BURST; // Send address
  for (i = 0; i < count; i++)
  {
    while (!(UC1IFG&UCA1TXIFG));            // Wait for TXBUF ready
    UCA1TXBUF = buffer[i];                  // Send data
  }
  while (UCA1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCIA1 SPI port
*******************************************************************************/
char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UC1IFG&UCA1TXIFG));              // Wait for TXBUF ready
  UCA1TXBUF = (addr | READ_SINGLE);// Send address
  while (!(UC1IFG&UCA1TXIFG));              // Wait for TX to finish
  UCA1TXBUF = 0;                            // Dummy write so we can read data
  while (UCA1STAT & UCBUSY);                // Wait for TX complete
  x = UCA1RXBUF;                            // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
 while (!(UC1IFG&UCA1TXIFG));              // Wait for TXBUF ready
  UCA1TXBUF = (addr | READ_BURST);// Send address
  while (UCA1STAT & UCBUSY);                // Wait for TX complete
  UCA1TXBUF = 0;                            // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  UC1IFG &= ~UCA1RXIFG;                     // Clear flag
  while (!(UC1IFG&UCA1RXIFG));              // Wait for end of 1st data byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    UCA1TXBUF = 0;                          //Initiate next data RX, meanwhile..
    buffer[i] = UCA1RXBUF;                  // Store data from last data RX
    while (!(UC1IFG&UCA1RXIFG));            // Wait for RX to finish
  }
  buffer[count-1] = UCA1RXBUF;              // Store last RX byte in buffer
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UC1IFG&UCA1TXIFG));              // Wait for TXBUF ready
  UCA1TXBUF = (addr | READ_BURST);// Send address
  while (!(UC1IFG&UCA1TXIFG));              // Wait for TXBUF ready
  UCA1TXBUF = 0;                            // Dummy write so we can read data
  while (UCA1STAT & UCBUSY);                // Wait for TX complete
  status = UCA1RXBUF;                       // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UC1IFG&UCA1TXIFG));              // Wait for TXBUF ready
  UCA1TXBUF = strobe;                       // Send strobe
  // Strobe addr is now being TX'ed
  while (UCA1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr". See Sec. 27.1 of CC1100 datasheet
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART1 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UC1IFG&UCA1TXIFG));              // Wait for TXBUF ready
  UCA1TXBUF = SRES;               // Send strobe
  // Strobe addr is now being TX'ed
  while (UCA1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

#endif
//******************************************************************************
// End of USCIA1
//******************************************************************************

#elif CC_RF_SER_INTF == CC_SER_INTF_USCIA2

//******************************************************************************
// Support for 5xx USCI_A2
//******************************************************************************
#ifdef MSP430_F5xx

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIA2 SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
   if (RF_IDx==RF_SLOTB) {
      CC_CSBn_PxOUT |= CC_CSBn_PIN;
      CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
      CC_CSAn_PxOUT |= CC_CSAn_PIN;
      CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  UCA2CTL1 |= UCSWRST;                      // **Disable USCI state machine**
  UCA2CTL0 |= UCMST+UCCKPH+UCMSB+UCSYNC;    // 3-pin, 8-bit SPI master
  UCA2CTL1 |= UCSSEL_2;                     // SMCLK
  UCA2BR0 = 0x02;                           // UCLK/2
  UCA2BR1 = 0;
  CC_SPI_USCIA2_PxSEL |= CC_SPI_USCIA2_SOMI
                         | CC_SPI_USCIA2_UCLK
                         | CC_SPI_USCIA2_SIMO;
                                            // SPI option select
  CC_SPI_USCIA2_PxDIR |= CC_SPI_USCIA2_UCLK | CC_SPI_USCIA2_SIMO;
                                            // SPI TXD out direction
  UCA2CTL1 &= ~UCSWRST;                     // **Initialize USCI state machine**
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIA2 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
   while (!(UCA2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA2TXBUF = addr;                         // Send address
  while (!(UCA2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA2TXBUF = value;                        // Send data
  while (UCA2STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}


/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIA2 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA2TXBUF = addr | WRITE_BURST; // Send address
  for (i = 0; i < count; i++)
  {
    while (!(UCA2IFG&UCTXIFG));             // Wait for TXBUF ready
    UCA2TXBUF = buffer[i];                  // Send data
  }
  while (UCA2STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCIA2 SPI port
*******************************************************************************/
char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA2TXBUF = (addr | READ_SINGLE);// Send address
  while (!(UCA2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA2TXBUF = 0;                            // Dummy write so we can read data
  while (UCA2STAT & UCBUSY);                // Wait for TX complete
  x = UCA2RXBUF;                            // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART2 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
   while (!(UCA2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA2TXBUF = (addr | READ_BURST);// Send address
  while (UCA2STAT & UCBUSY);                // Wait for TX complete
  UCA2TXBUF = 0;                            // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  UCA2IFG &= ~UCRXIFG;                      // Clear flag
  while (!(UCA2IFG&UCRXIFG));               // Wait for end of addr byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    UCA2TXBUF = 0;                          //Initiate next data RX, meanwhile..
    buffer[i] = UCA2RXBUF;                  // Store data from last data RX
    while (!(UCA2IFG&UCRXIFG));             // Wait for RX to finish
  }
  buffer[count-1] = UCA2RXBUF;              // Store last RX byte in buffer
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART2 SPI port
*******************************************************************************/
char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA2TXBUF = (addr | READ_BURST);// Send address
  while (!(UCA2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA2TXBUF = 0;                            // Dummy write so we can read data
  while (UCA2STAT & UCBUSY);                // Wait for TX complete
  status = UCA2RXBUF;                       // Read data
 if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
 MSP430F2xx USART2 SPI port
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA2TXBUF = strobe;                       // Send strobe
  // Strobe addr is now being TX'ed
  while (UCA2STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr". See Sec. 27.1 of CC1100 datasheet
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA2TXBUF = SRES;               // Send strobe
  // Strobe addr is now being TX'ed
  while (UCA2STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/* End of support for 5xx USCI_A2 */
#endif

/*******************************************************************************
* If USCIA3 is used
*   |-- If 5xx
*         |-- Use 5xx Init
*******************************************************************************/
#elif CC_RF_SER_INTF == CC_SER_INTF_USCIA3

/*******************************************************************************
* Support for 5xx USCI_A3
*******************************************************************************/
#ifdef MSP430_F5xx

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIA3 SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
   if (RF_IDx==RF_SLOTB) {
      CC_CSBn_PxOUT |= CC_CSBn_PIN;
      CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
      CC_CSAn_PxOUT |= CC_CSAn_PIN;
      CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    
  UCA3CTL1 |= UCSWRST;                      // **Disable USCI state machine**
  UCA3CTL0 |= UCMST+UCCKPH+UCMSB+UCSYNC;    // 3-pin, 8-bit SPI master
  UCA3CTL1 |= UCSSEL_2;                     // SMCLK
  UCA3BR0 = 0x02;                           // UCLK/2
  UCA3BR1 = 0;
  CC_SPI_USCIA3_PxSEL |= CC_SPI_USCIA3_SOMI
                         | CC_SPI_USCIA3_UCLK
                         | CC_SPI_USCIA3_SIMO;
                                            // SPI option select
  CC_SPI_USCIA3_PxDIR |= CC_SPI_USCIA3_UCLK | CC_SPI_USCIA3_SIMO;
                                            // SPI TXD out direction
  UCA3CTL1 &= ~UCSWRST;                     // **Initialize USCI state machine**
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIA3 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA3TXBUF = addr;                         // Send address
  while (!(UCA3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA3TXBUF = value;                        // Send data
  while (UCA3STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIA3 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA3TXBUF = addr | WRITE_BURST; // Send address
  for (i = 0; i < count; i++)
  {
    while (!(UCA3IFG&UCTXIFG));             // Wait for TXBUF ready
    UCA3TXBUF = buffer[i];                  // Send data
  }
  while (UCA3STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIA3 SPI port
*******************************************************************************/
char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA3TXBUF = (addr | READ_SINGLE);// Send address
  while (!(UCA3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA3TXBUF = 0;                            // Dummy write so we can read data
  while (UCA3STAT & UCBUSY);                // Wait for TX complete
  x = UCA3RXBUF;                            // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART3 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA3TXBUF = (addr | READ_BURST);// Send address
  while (UCA3STAT & UCBUSY);                // Wait for TX complete
  UCA3TXBUF = 0;                            // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  UCA3IFG &= ~UCRXIFG;                      // Clear flag
  while (!(UCA3IFG&UCRXIFG));               // Wait for end of addr byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    UCA3TXBUF = 0;                          //Initiate next data RX, meanwhile..
    buffer[i] = UCA3RXBUF;                  // Store data from last data RX
    while (!(UCA3IFG&UCRXIFG));             // Wait for RX to finish
  }
  buffer[count-1] = UCA3RXBUF;              // Store last RX byte in buffer
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USART3 SPI port
*******************************************************************************/
char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA3TXBUF = (addr | READ_BURST);// Send address
  while (!(UCA3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA3TXBUF = 0;                            // Dummy write so we can read data
  while (UCA3STAT & UCBUSY);                // Wait for TX complete
  status = UCA3RXBUF;                       // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
 MSP430F2xx USART3 SPI port
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA3TXBUF = strobe;                       // Send strobe
  // Strobe addr is now being TX'ed
  while (UCA3STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr". See Sec. 27.1 of CC1100 datasheet
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCA3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCA3TXBUF = SRES;               // Send strobe
  // Strobe addr is now being TX'ed
  while (UCA3STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/* End of support for 5xx USCI_A3 */
#endif


#elif CC_RF_SER_INTF == CC_SER_INTF_FRxUSCIB0

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430FR57x USCI_B0 SPI port

#define CC_FRx1SPI_USCIB0_PxSEL  P1SEL
#define CC_FRx1SPI_USCIB0_PxDIR  P1DIR
#define CC_FRx1SPI_USCIB0_PxIN   P1IN
#define CC_FRxSPI_USCIB0_SIMO   BIT6
#define CC_FRxSPI_USCIB0_SOMI   BIT7
#define CC_FRx2SPI_USCIB0_PxSEL  P2SEL
#define CC_FRx2SPI_USCIB0_PxDIR  P2DIR
#define CC_FRx2SPI_USCIB0_PxIN   P2IN
#define CC_FRxSPI_USCIB0_UCLK   BIT2
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;
    CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;
    CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  UCB0CTLW0 |= UCSWRST;                      // **Disable USCI state machine**
  UCB0CTLW0 |= UCMST+UCCKPH+UCMSB+UCSYNC;    // 3-pin, 8-bit SPI master
  UCB0CTLW0 |= UCSSEL_2;                    // SMCLK
  UCB0BR0 = 0x02;                           // UCLK/2
  UCB0BR1 = 0;
  
  CC_FRx1SPI_USCIB0_PxSEL1 |= CC_FRxSPI_USCIB0_SIMO | CC_FRxSPI_USCIB0_SOMI;
  CC_FRx1SPI_USCIB0_PxSEL0 &= ~(CC_FRxSPI_USCIB0_SIMO | CC_FRxSPI_USCIB0_SOMI);
  CC_FRx2SPI_USCIB0_PxSEL1 |= CC_FRxSPI_USCIB0_UCLK;
  CC_FRx2SPI_USCIB0_PxSEL0 &= ~CC_FRxSPI_USCIB0_UCLK;
  // SPI option select
  CC_FRx1SPI_USCIB0_PxDIR |= CC_FRxSPI_USCIB0_SIMO;
  CC_FRx2SPI_USCIB0_PxDIR |= CC_FRxSPI_USCIB0_UCLK;
  // SPI TXD out direction
  UCB0CTL1 &= ~UCSWRST;                     // **Initialize USCI state machine**
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx == RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = addr;                         // Send address
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = value;                        // Send data
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}
/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = addr | WRITE_BURST; // Send address
  for (i = 0; i < count; i++)
  {
    while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
    UCB0TXBUF = buffer[i];                  // Send data
  }
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
unsigned char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = (addr | READ_SINGLE);// Send address
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = 0;                            // Dummy write so we can read data
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  x = UCB0RXBUF;                            // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  char i;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = (addr | READ_BURST);// Send address
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  UCB0TXBUF = 0;                            // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  UCB0IFG &= ~UCRXIFG;                      // Clear flag
  while (!(UCB0IFG&UCRXIFG));               // Wait for end of 1st data byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    UCB0TXBUF = 0;                          //Initiate next data RX, meanwhile..
    buffer[i] = UCB0RXBUF;                  // Store data from last data RX
    while (!(UCB0IFG&UCRXIFG));              // Wait for RX to finish
  }
  buffer[count-1] = UCB0RXBUF;              // Store last RX byte in buffer
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
unsigned char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = (addr | READ_BURST);// Send address
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = 0;                            // Dummy write so we can read data
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  status = UCB0RXBUF;                       // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
unsigned char CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = strobe;                       // Send strobe
  // Strobe addr is now being TX'ed
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return 0;
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_B0 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(4500);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB0IFG&UCTXIFG));             // Wait for TXBUF ready
  UCB0TXBUF = RF_SRES;                    // Send strobe
  // Strobe addr is now being TX'ed
  while (UCB0STAT & UCBUSY);              // Wait for TX to complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}


#elif CC_RF_SER_INTF == CC_SER_INTF_USCIB0

/*******************************************************************************
*  Support for 5xx USCI_B0
*******************************************************************************/

#ifdef MSP430_F5xx

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;
    CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;
    CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    
  
  UCB0CTL1 |= UCSWRST;                      // **Disable USCI state machine**
  UCB0CTL0 |= UCMST+UCCKPH+UCMSB+UCSYNC;    // 3-pin, 8-bit SPI master
  UCB0CTL1 |= UCSSEL_2;                     // SMCLK
  UCB0BR0 = 0x02;                           // UCLK/2
  UCB0BR1 = 0;
  CC_SPI_USCIB0_PxSEL |= CC_SPI_USCIB0_SIMO
    | CC_SPI_USCIB0_SOMI
      | CC_SPI_USCIB0_UCLK;
  // SPI option select
  CC_SPI_USCIB0_PxDIR |= CC_SPI_USCIB0_SIMO | CC_SPI_USCIB0_UCLK;
  // SPI TXD out direction
  UCB0CTL1 &= ~UCSWRST;                     // **Initialize USCI state machine**
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = addr;                         // Send address
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = value;                        // Send data
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}
/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = addr | WRITE_BURST; // Send address
  for (i = 0; i < count; i++)
  {
    while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
    UCB0TXBUF = buffer[i];                  // Send data
  }
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
unsigned char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = (addr | READ_SINGLE);// Send address
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = 0;                            // Dummy write so we can read data
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  x = UCB0RXBUF;                            // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  char i;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = (addr | READ_BURST);// Send address
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  UCB0TXBUF = 0;                            // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  UCB0IFG &= ~UCRXIFG;                      // Clear flag
  while (!(UCB0IFG&UCRXIFG));               // Wait for end of 1st data byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    UCB0TXBUF = 0;                          //Initiate next data RX, meanwhile..
    buffer[i] = UCB0RXBUF;                  // Store data from last data RX
    while (!(UCB0IFG&UCRXIFG));              // Wait for RX to finish
  }
  buffer[count-1] = UCB0RXBUF;              // Store last RX byte in buffer
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
unsigned char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = (addr | READ_BURST);// Send address
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = 0;                            // Dummy write so we can read data
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  status = UCB0RXBUF;                       // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
unsigned char CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = strobe;                       // Send strobe
  // Strobe addr is now being TX'ed
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return 0;
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_B0 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(4500);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB0IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB0TXBUF = RF_SRES;               // Send strobe
  // Strobe addr is now being TX'ed
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

#else
/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_B0 SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
  
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;
    CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;
    CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  UCB0CTL1 |= UCSWRST;                      // **Disable USCI state machine**
  UCB0CTL0 |= UCMST+UCCKPH+UCMSB+UCSYNC;    // 3-pin, 8-bit SPI master
  UCB0CTL1 |= UCSSEL_2;                     // SMCLK
  UCB0BR0 = 0x04;                           // UCLK/4
  UCB0BR1 = 0;
  CC_SPI_USCIB0_PxSEL |= CC_SPI_USCIB0_SIMO
                         | CC_SPI_USCIB0_SOMI
                         | CC_SPI_USCIB0_UCLK;
                                            // SPI option select
  CC_SPI_USCIB0_PxDIR |= CC_SPI_USCIB0_SIMO | CC_SPI_USCIB0_UCLK;
                                            // SPI TXD out direction
  UCB0CTL1 &= ~UCSWRST;                     // **Initialize USCI state machine**
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_B0 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UCB0TXIFG));                // Wait for TXBUF ready
  UCB0TXBUF = addr;                         // Send address
  while (!(IFG2&UCB0TXIFG));                // Wait for TXBUF ready
  UCB0TXBUF = value;                        // Send data
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer, \
*                          unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_B0 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UCB0TXIFG));                // Wait for TXBUF ready
  UCB0TXBUF = addr | WRITE_BURST; // Send address
  for (i = 0; i < count; i++)
  {
    while (!(IFG2&UCB0TXIFG));              // Wait for TXBUF ready
    UCB0TXBUF = buffer[i];                  // Send data
  }
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_B0 SPI port
*******************************************************************************/
unsigned char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UCB0TXIFG));                // Wait for TXBUF ready
  UCB0TXBUF = (addr | READ_SINGLE);// Send address
  while (!(IFG2&UCB0TXIFG));                // Wait for TXBUF ready
  UCB0TXBUF = 0;                            // Dummy write so we can read data
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  x = UCB0RXBUF;                            // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    

  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer, \
*                          unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;          // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;          // /CS enable
  }    
  while (!(IFG2&UCB0TXIFG));                // Wait for TXBUF ready
  UCB0TXBUF = (addr | READ_BURST);   // Send address
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  UCB0TXBUF = 0;                            // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  IFG2 &= ~UCB0RXIFG;                       // Clear flag
  while (!(IFG2&UCB0RXIFG));                // Wait for end of 1st data byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    UCB0TXBUF = 0;                          //Initiate next data RX, meanwhile..
    buffer[i] = UCB0RXBUF;                  // Store data from last data RX
    while (!(IFG2&UCB0RXIFG));              // Wait for RX to finish
  }
  buffer[count-1] = UCB0RXBUF;              // Store last RX byte in buffer
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;           // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;           // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_B0 SPI port
*******************************************************************************/
unsigned char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UCB0TXIFG));                // Wait for TXBUF ready
  UCB0TXBUF = (addr | READ_BURST);// Send address
  while (!(IFG2&UCB0TXIFG));                // Wait for TXBUF ready
  UCB0TXBUF = 0;                            // Dummy write so we can read data
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  status = UCB0RXBUF;                       // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;           // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;           // /CS disable
  }    

  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_B0 SPI port
*******************************************************************************/
unsigned char CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UCB0TXIFG));                // Wait for TXBUF ready
  UCB0TXBUF = strobe;                       // Send strobe
  // Strobe addr is now being TX'ed
  while (UCB0STAT & UCBUSY);                // Wait for TX to complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return 0;
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCI_B0 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);

   if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(IFG2&UCB0TXIFG));              // Wait for TXBUF ready
  UCB0TXBUF = RF_SRES;                    // Send strobe
  // Strobe addr is now being TX'ed
  while (UCB0STAT & UCBUSY);              // Wait for TX to complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

#endif
// End of USCIB0

//******************************************************************************
// If USCIB1 is used
//******************************************************************************
#elif CC_RF_SER_INTF == CC_SER_INTF_USCIB1

//******************************************************************************
// Support for 5xx USCI_B1
//******************************************************************************
#ifdef MSP430_F5xx

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB1 SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
   if (RF_IDx==RF_SLOTB) {
      CC_CSBn_PxOUT |= CC_CSBn_PIN;
      CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
      CC_CSAn_PxOUT |= CC_CSAn_PIN;
      CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  UCB1CTL1 |= UCSWRST;                      // **Disable USCI state machine**
  UCB1CTL0 |= UCMST+UCCKPH+UCMSB+UCSYNC;    // 3-pin, 8-bit SPI master
  UCB1CTL1 |= UCSSEL_2;                     // SMCLK
  UCB1BR0 = 0x02;                           // UCLK/2
  UCB1BR1 = 0;
  CC_SPI_USCIB1_PxSEL |= CC_SPI_USCIB1_SOMI
                         | CC_SPI_USCIB1_UCLK;
  CC_SPI_USCIB1_PxSEL_SIMO |= CC_SPI_USCIB1_SIMO;
                                            // SPI option select
  CC_SPI_USCIB1_PxDIR |= CC_SPI_USCIB1_UCLK;
  CC_SPI_USCIB1_PxDIR_SIMO |=  CC_SPI_USCIB1_SIMO;
                                            // SPI TXD out direction
  UCB1CTL1 &= ~UCSWRST;                     // **Initialize USCI state machine**
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB1 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB1TXBUF = addr;                         // Send address
  while (!(UCB1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB1TXBUF = value;                        // Send data
  while (UCB1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB1 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB1TXBUF = addr | WRITE_BURST; // Send address
  for (i = 0; i < count; i++)
  {
    while (!(UCB1IFG&UCTXIFG));             // Wait for TXBUF ready
    UCB1TXBUF = buffer[i];                  // Send data
  }
  while (UCB1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB1 SPI port
*******************************************************************************/
char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB1TXBUF = (addr | READ_SINGLE);// Send address
  while (!(UCB1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB1TXBUF = 0;                            // Dummy write so we can read data
  while (UCB1STAT & UCBUSY);                // Wait for TX complete
  x = UCB1RXBUF;                            // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB1 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
 while (!(UCB1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB1TXBUF = (addr | READ_BURST);// Send address
  while (UCB1STAT & UCBUSY);                // Wait for TX complete
  UCB1TXBUF = 0;                            // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  UCB1IFG &= ~UCRXIFG;                      // Clear flag
  while (!(UCB1IFG&UCRXIFG));               // Wait for end of addr byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    UCB1TXBUF = 0;                          //Initiate next data RX, meanwhile..
    buffer[i] = UCB1RXBUF;                  // Store data from last data RX
    while (!(UCB1IFG&UCRXIFG));             // Wait for RX to finish
  }
  buffer[count-1] = UCB1RXBUF;              // Store last RX byte in buffer
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB1 SPI port
*******************************************************************************/
char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB1TXBUF = (addr | READ_BURST);// Send address
  while (!(UCB1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB1TXBUF = 0;                            // Dummy write so we can read data
  while (UCB1STAT & UCBUSY);                // Wait for TX complete
  status = UCB1RXBUF;                       // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB1 SPI port
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB1TXBUF = strobe;                       // Send strobe
  // Strobe addr is now being TX'ed
  while (UCB1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr". See Sec. 27.1 of CC1100 datasheet
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCI_B0 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB1IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB1TXBUF = SRES;               // Send strobe
  // Strobe addr is now being TX'ed
  while (UCB1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

// End of support for 5xx USCI_B1
#else

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCIB1 SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
   if (RF_IDx==RF_SLOTB) {
      CC_CSBn_PxOUT |= CC_CSBn_PIN;
      CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
      CC_CSAn_PxOUT |= CC_CSAn_PIN;
      CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  UCB1CTL1 |= UCSWRST;                      // **Disable USCI state machine**
  UCB1CTL0 |= UCMST+UCCKPL+UCMSB+UCSYNC;    // 3-pin, 8-bit SPI master
  UCB1CTL1 |= UCSSEL_2;                     // SMCLK
  UCB1BR0 = 0x02;                           // UCLK/2
  UCB1BR1 = 0;
  CC_SPI_USCIB1_PxSEL |= CC_SPI_USCIB1_SIMO
                         | CC_SPI_USCIB1_SOMI
                         | CC_SPI_USCIB1_UCLK;
                                            // SPI option select
  CC_SPI_USCIB1_PxDIR |= CC_SPI_USCIB1_SIMO | CC_SPI_USCIB1_UCLK;
                                            // SPI TXD out direction
  UCB1CTL1 &= ~UCSWRST;                     // **Initialize USCI state machine**
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCIB1 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UC1IFG&UCB1TXIFG));              // Wait for TXBUF ready
  UCB1TXBUF = addr;                         // Send address
  while (!(UC1IFG&UCB1TXIFG));              // Wait for TXBUF ready
  UCB1TXBUF = value;                        // Send data
  while (UCB1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCIB1 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UC1IFG&UCB1TXIFG));              // Wait for TXBUF ready
  UCB1TXBUF = addr | WRITE_BURST; // Send address
  for (i = 0; i < count; i++)
  {
    while (!(UC1IFG&UCB1TXIFG));            // Wait for TXBUF ready
    UCB1TXBUF = buffer[i];                  // Send data
  }
  while (UCB1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCIB1 SPI port
*******************************************************************************/
char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UC1IFG&UCB1TXIFG));              // Wait for TXBUF ready
  UCB1TXBUF = (addr | READ_SINGLE);// Send address
  while (!(UC1IFG&UCB1TXIFG));              // Wait for TXBUF ready
  UCB1TXBUF = 0;                            // Dummy write so we can read data
  while (UCB1STAT & UCBUSY);                // Wait for TX complete
  x = UCB1RXBUF;                            // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCIB1 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UC1IFG&UCB1TXIFG));              // Wait for TXBUF ready
  UCB1TXBUF = (addr | READ_BURST);// Send address
  while (UCB1STAT & UCBUSY);                // Wait for TX complete
  UCB1TXBUF = 0;                            // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  UC1IFG &= ~UCB1RXIFG;                     // Clear flag
  while (!(UC1IFG&UCB1RXIFG));              // Wait for end of 1st data byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    UCB1TXBUF = 0;                          //Initiate next data RX, meanwhile..
    buffer[i] = UCB1RXBUF;                  // Store data from last data RX
    while (!(UC1IFG&UCB1RXIFG));            // Wait for RX to finish
  }
  buffer[count-1] = UCB1RXBUF;              // Store last RX byte in buffer
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCIB1 SPI port
*******************************************************************************/
char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UC1IFG&UCB1TXIFG));              // Wait for TXBUF ready
  UCB1TXBUF = (addr | READ_BURST);// Send address
  while (!(UC1IFG&UCB1TXIFG));              // Wait for TXBUF ready
  UCB1TXBUF = 0;                            // Dummy write so we can read data
  while (UCB1STAT & UCBUSY);                // Wait for TX complete
  status = UCB1RXBUF;                       // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCIB1 SPI port
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UC1IFG&UCB1TXIFG));              // Wait for TXBUF ready
  UCB1TXBUF = strobe;                       // Send strobe
  // Strobe addr is now being TX'ed
  while (UCB1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr". See Sec. 27.1 of CC1100 datasheet
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F2xx USCIB1 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UC1IFG&UCB1TXIFG));              // Wait for TXBUF ready
  UCB1TXBUF = SRES;               // Send strobe
  // Strobe addr is now being TX'ed
  while (UCB1STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

// End of USCIB1
#endif

/*******************************************************************************
* If USCIB2 is used
*   |-- If 5xx
*         |-- Use 5xx Init
*******************************************************************************/
#elif CC_RF_SER_INTF == CC_SER_INTF_USCIB2
/******************************************************************************
* Support for 5xx USCI_B2
*******************************************************************************/
#ifdef MSP430_F5xx

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB2 SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
   if (RF_IDx==RF_SLOTB) {
      CC_CSBn_PxOUT |= CC_CSBn_PIN;
      CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
      CC_CSAn_PxOUT |= CC_CSAn_PIN;
      CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  UCB2CTL1 |= UCSWRST;                      // **Disable USCI state machine**
  UCB2CTL0 |= UCMST+UCCKPH+UCMSB+UCSYNC;    // 3-pin, 8-bit SPI master
  UCB2CTL1 |= UCSSEL_2;                     // SMCLK
  UCB2BR0 = 0x02;                           // UCLK/2
  UCB2BR1 = 0;
  CC_SPI_USCIB2_PxSEL |= CC_SPI_USCIB2_SOMI
                         | CC_SPI_USCIB2_UCLK
                         | CC_SPI_USCIB2_SIMO;
                                            // SPI option select
  CC_SPI_USCIB2_PxDIR |= CC_SPI_USCIB2_UCLK | CC_SPI_USCIB2_SIMO;
                                            // SPI TXD out direction
  UCB2CTL1 &= ~UCSWRST;                     // **Initialize USCI state machine**
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB2 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB2TXBUF = addr;                         // Send address
  while (!(UCB2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB2TXBUF = value;                        // Send data
  while (UCB2STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB2 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB2TXBUF = addr | WRITE_BURST; // Send address
  for (i = 0; i < count; i++)
  {
    while (!(UCB2IFG&UCTXIFG));             // Wait for TXBUF ready
    UCB2TXBUF = buffer[i];                  // Send data
  }
  while (UCB2STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB2 SPI port
*******************************************************************************/
char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB2TXBUF = (addr | READ_SINGLE);// Send address
  while (!(UCB2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB2TXBUF = 0;                            // Dummy write so we can read data
  while (UCB2STAT & UCBUSY);                // Wait for TX complete
  x = UCB2RXBUF;                            // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB2 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB2TXBUF = (addr | READ_BURST);// Send address
  while (UCB2STAT & UCBUSY);                // Wait for TX complete
  UCB2TXBUF = 0;                            // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  UCB2IFG &= ~UCRXIFG;                      // Clear flag
  while (!(UCB2IFG&UCRXIFG));               // Wait for end of addr byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    UCB2TXBUF = 0;                          //Initiate next data RX, meanwhile..
    buffer[i] = UCB2RXBUF;                  // Store data from last data RX
    while (!(UCB2IFG&UCRXIFG));             // Wait for RX to finish
  }
  buffer[count-1] = UCB2RXBUF;              // Store last RX byte in buffer
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB2 SPI port
*******************************************************************************/
char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB2TXBUF = (addr | READ_BURST);// Send address
  while (!(UCB2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB2TXBUF = 0;                            // Dummy write so we can read data
  while (UCB2STAT & UCBUSY);                // Wait for TX complete
  status = UCB2RXBUF;                       // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB2 SPI port
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB2TXBUF = strobe;                       // Send strobe
  // Strobe addr is now being TX'ed
  while (UCB2STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr". See Sec. 27.1 of CC1100 datasheet
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB2 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB2IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB2TXBUF = SRES;               // Send strobe
  // Strobe addr is now being TX'ed
  while (UCB2STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/* End of support for 5xx USCI_B2 */
#endif


/*******************************************************************************
* If USCIB3 is used
*   |-- If 5xx
*         |-- Use 5xx Init
*******************************************************************************/

#elif CC_RF_SER_INTF == CC_SER_INTF_USCIB3

/*******************************************************************************
* Support for 5xx USCI_B3
*******************************************************************************/

#ifdef MSP430_F5xx

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB3 SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
   if (RF_IDx==RF_SLOTB) {
      CC_CSBn_PxOUT |= CC_CSBn_PIN;
      CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
      CC_CSAn_PxOUT |= CC_CSAn_PIN;
      CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  UCB3CTL1 |= UCSWRST;                      // **Disable USCI state machine**
  UCB3CTL0 |= UCMST+UCCKPH+UCMSB+UCSYNC;    // 3-pin, 8-bit SPI master
  UCB3CTL1 |= UCSSEL_2;                     // SMCLK
  UCB3BR0 = 0x02;                           // UCLK/2
  UCB3BR1 = 0;
  CC_SPI_USCIB3_PxSEL |= CC_SPI_USCIB3_SOMI
                         | CC_SPI_USCIB3_UCLK
                         | CC_SPI_USCIB3_SIMO;
                                            // SPI option select
  CC_SPI_USCIB3_PxDIR |= CC_SPI_USCIB3_UCLK | CC_SPI_USCIB3_SIMO;
                                            // SPI TXD out direction
  UCB3CTL1 &= ~UCSWRST;                     // **Initialize USCI state machine**
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB3 SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB3TXBUF = addr;                         // Send address
  while (!(UCB3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB3TXBUF = value;                        // Send data
  while (UCB3STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB3 SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB3TXBUF = addr | WRITE_BURST; // Send address
  for (i = 0; i < count; i++)
  {
    while (!(UCB3IFG&UCTXIFG));             // Wait for TXBUF ready
    UCB3TXBUF = buffer[i];                  // Send data
  }
  while (UCB3STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB3 SPI port
*******************************************************************************/
char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB3TXBUF = (addr | READ_SINGLE);// Send address
  while (!(UCB3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB3TXBUF = 0;                            // Dummy write so we can read data
  while (UCB3STAT & UCBUSY);                // Wait for TX complete
  x = UCB3RXBUF;                            // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB3 SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB3TXBUF = (addr | READ_BURST);// Send address
  while (UCB3STAT & UCBUSY);                // Wait for TX complete
  UCB3TXBUF = 0;                            // Dummy write to read 1st data byte
  // Addr byte is now being TX'ed, with dummy byte to follow immediately after
  UCB3IFG &= ~UCRXIFG;                      // Clear flag
  while (!(UCB3IFG&UCRXIFG));               // Wait for end of addr byte TX
  // First data byte now in RXBUF
  for (i = 0; i < (count-1); i++)
  {
    UCB3TXBUF = 0;                          //Initiate next data RX, meanwhile..
    buffer[i] = UCB3RXBUF;                  // Store data from last data RX
    while (!(UCB3IFG&UCRXIFG));             // Wait for RX to finish
  }
  buffer[count-1] = UCB3RXBUF;              // Store last RX byte in buffer
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB3 SPI port
*******************************************************************************/
char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB3TXBUF = (addr | READ_BURST);// Send address
  while (!(UCB3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB3TXBUF = 0;                            // Dummy write so we can read data
  while (UCB3STAT & UCBUSY);                // Wait for TX complete
  status = UCB3RXBUF;                       // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB3 SPI port
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB3TXBUF = strobe;                       // Send strobe
  // Strobe addr is now being TX'ed
  while (UCB3STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr". See Sec. 27.1 of CC1100 datasheet
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCIB3 SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (!(UCB3IFG&UCTXIFG));               // Wait for TXBUF ready
  UCB3TXBUF = SRES;               // Send strobe
  // Strobe addr is now being TX'ed
  while (UCB3STAT & UCBUSY);                // Wait for TX complete
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

//******************************************************************************
//  End of support for 5xx USCI_B3
//******************************************************************************

#endif

//******************************************************************************
// If USI is used
//******************************************************************************

#elif CC_RF_SER_INTF == CC_SER_INTF_USI

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCSI SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
   if (RF_IDx==RF_SLOTB) {
      CC_CSBn_PxOUT |= CC_CSBn_PIN;
      CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
      CC_CSAn_PxOUT |= CC_CSAn_PIN;
      CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  USICTL0 |= USIPE7 +  USIPE6 + USIPE5 + USIMST + USIOE;// Port, SPI master
  USICKCTL = USISSEL_2 + USICKPL;           // SCLK = SMCLK
  USICTL0 &= ~USISWRST;                     // USI released for operation

  USISRL = 0x00;                            // Ensure SDO low instead of high,
  USICNT = 1;                               // to avoid conflict with CCxxxx
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCSI SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (CC_SPI_USI_PxIN&CC_SPI_USI_SOMI);// Wait for CCxxxx ready
  USISRL = addr;                            // Load address
  USICNT = 8;                               // Send it
  while (!(USICTL1&USIIFG));                // Wait for TX to finish
  USISRL = value;                           // Load value
  USICNT = 8;                               // Send it
  while (!(USICTL1&USIIFG));                // Wait for TX to finish
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCSI SPI port
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (CC_SPI_USI_PxIN&CC_SPI_USI_SOMI);// Wait for CCxxxx ready
  USISRL = addr | WRITE_BURST;    // Load address
  USICNT = 8;                               // Send it
  while (!(USICTL1&USIIFG));                // Wait for TX to finish
  for (i = 0; i < count; i++)
  {
    USISRL = buffer[i];                     // Load data
    USICNT = 8;                             // Send it
    while (!(USICTL1&USIIFG));              // Wait for TX to finish
  }
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCSI SPI port
*******************************************************************************/
char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (CC_SPI_USI_PxIN&CC_SPI_USI_SOMI);// Wait for CCxxxx ready
  USISRL = addr | READ_SINGLE;    // Load address
  USICNT = 8;                               // Send it
  while (!(USICTL1&USIIFG));                // Wait for TX to finish
  USICNT = 8;                               // Dummy write so we can read data
  while (!(USICTL1&USIIFG));                // Wait for RX to finish
  x = USISRL;                               // Store data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCSI SPI port
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (CC_SPI_USI_PxIN&CC_SPI_USI_SOMI);// Wait for CCxxxx ready
  USISRL = addr | READ_BURST;     // Load address
  USICNT = 8;                               // Send it
  while (!(USICTL1&USIIFG));                // Wait for TX to finish
  for (i = 0; i < count; i++)
  {
    USICNT = 8;                             // Dummy write so we can read data
    while (!(USICTL1&USIIFG));              // Wait for RX to finish
    buffer[i] = USISRL;                     // Store data
  }
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCSI SPI port
*******************************************************************************/
char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (CC_SPI_USI_PxIN&CC_SPI_USI_SOMI);// Wait for CCxxxx ready
  USISRL = addr | READ_BURST;     // Load address
  USICNT = 8;                               // Send it
  while (!(USICTL1&USIIFG));                // Wait for TX to finish
  USICNT = 8;                               // Dummy write so we can read data
  while (!(USICTL1&USIIFG));                // Wait for RX to finish
  status = USISRL;                               // Store data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCSI SPI port
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (CC_SPI_USI_PxIN&CC_SPI_USI_SOMI);// Wait for CCxxxx ready
  USISRL = strobe;                          // Load strobe
  USICNT = 8;                               // Send it
  while (!(USICTL1&USIIFG));                // Wait for TX to finish
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr". See Sec. 27.1 of CC1100 datasheet
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx USCSI SPI port
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (CC_SPI_USI_PxIN&CC_SPI_USI_SOMI);
  USISRL = SRES;
  USICNT = 8;
  while (!(USICTL1&USIIFG));
  while (CC_SPI_USI_PxIN&CC_SPI_USI_SOMI);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

// End of USI

/*******************************************************************************
* If Bit Bang is used
*******************************************************************************/
#elif CC_RF_SER_INTF == CC_SER_INTF_BITBANG

void CC_SPI_bitbang_out(unsigned char value);
unsigned char CC_SPI_bitbang_in();

/*******************************************************************************
*  void CC_SPISetup(char RF_IDx)
*
*  DESCRIPTION:
*  Configures the assigned interface to function as a SPI port and
*  initializes it. 
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx BITBANG SPI port
*******************************************************************************/
void CC_SPISetup(char RF_IDx) {
   if (RF_IDx==RF_SLOTB) {
      CC_CSBn_PxOUT |= CC_CSBn_PIN;
      CC_CSBn_PxDIR |= CC_CSBn_PIN;         // /CS disable
  } else {
      CC_CSAn_PxOUT |= CC_CSAn_PIN;
      CC_CSAn_PxDIR |= CC_CSAn_PIN;         // /CS disable
  }    

  // Config bitbang pins
  CC_SPI_BITBANG_PxOUT |= CC_SPI_BITBANG_SIMO;
  CC_SPI_BITBANG_PxOUT &= ~CC_SPI_BITBANG_UCLK;
  CC_SPI_BITBANG_PxDIR |= CC_SPI_BITBANG_SIMO | CC_SPI_BITBANG_UCLK;
}

// Output eight-bit value using selected bit-bang pins
void CC_SPI_bitbang_out(unsigned char value) {
  char x;

  for(x=8;x>0;x--)
  {
    if(value & 0x80)                                     // If bit is high...
      CC_SPI_BITBANG_PxOUT |= CC_SPI_BITBANG_SIMO; // Set SIMO high...
    else
      CC_SPI_BITBANG_PxOUT &= ~CC_SPI_BITBANG_SIMO;// Set SIMO low...
    value = value << 1;                                  // Rotate bits

    CC_SPI_BITBANG_PxOUT &= ~CC_SPI_BITBANG_UCLK;  // Set clock low
    CC_SPI_BITBANG_PxOUT |= CC_SPI_BITBANG_UCLK;   // Set clock high
  }
  CC_SPI_BITBANG_PxOUT &= ~CC_SPI_BITBANG_UCLK;  // Set clock low
}

// Input eight-bit value using selected bit-bang pins
unsigned char CC_SPI_bitbang_in() {
  char x=0,shift=0;
  int y;

  // Determine how many bit positions SOMI is from least-significant bit
  x = CC_SPI_BITBANG_SOMI;
  while(x>1)
  {
    shift++;
    x = x >> 1;
  };

  x = 0;
  for(y=8;y>0;y--)
  {
    CC_SPI_BITBANG_PxOUT &= ~CC_SPI_BITBANG_UCLK;// Set clock low
    CC_SPI_BITBANG_PxOUT |= CC_SPI_BITBANG_UCLK;// Set clock high

    x = x << 1;                             // Make room for next bit
    x = x | ((CC_SPI_BITBANG_PxIN & CC_SPI_BITBANG_SOMI) >> shift);
  }                                         // Store next bit
  CC_SPI_BITBANG_PxOUT &= ~CC_SPI_BITBANG_UCLK; // Set clock low
  return(x);
}

/*******************************************************************************
*  void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx)
*
*  DESCRIPTION:
*  Writes "value" to a single configuration register at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430F5xx BITBANG SPI port
*******************************************************************************/
void CC_SPIWriteReg(unsigned char addr, unsigned char value, char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (CC_SPI_BITBANG_PxIN&CC_SPI_BITBANG_SOMI); // Wait CCxxxx ready
  CC_SPI_bitbang_out(addr);            // Send address
  CC_SPI_bitbang_out(value);           // Send data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Writes values to multiple configuration registers, the first register being
*  at address "addr".  First data byte is at "buffer", and both addr and
*  buffer are incremented sequentially (within the CCxxxx and MSP430,
*  respectively) until "count" writes have been performed.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430Fxxx BITBANG
*******************************************************************************/
void CC_SPIWriteBurstReg(unsigned char addr, unsigned char *buffer, \
  unsigned char count, char RF_IDx) {
  unsigned int i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (CC_SPI_BITBANG_PxIN&CC_SPI_BITBANG_SOMI); // Wait CCxxxx ready
    CC_SPI_bitbang_out(addr | WRITE_BURST);   // Send address
    for (i = 0; i < count; i++)
      CC_SPI_bitbang_out(buffer[i]);     // Send data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadReg(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Reads a single configuration register at address "addr" and returns the
*  value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430Fxxx BITBANG
*******************************************************************************/
char CC_SPIReadReg(unsigned char addr, char RF_IDx) {
  char x;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_SPI_bitbang_out(addr | READ_SINGLE);//Send address
  x = CC_SPI_bitbang_in();               // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return x;
}

/*******************************************************************************
*  void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
*  unsigned char count, char RF_IDx)
*
*  DESCRIPTION:
*  Reads multiple configuration registers, the first register being at address
*  "addr".  Values read are deposited sequentially starting at address
*  "buffer", until "count" registers have been read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430Fxxx BITBANG
*******************************************************************************/
void CC_SPIReadBurstReg(unsigned char addr, unsigned char *buffer,\
  unsigned char count, char RF_IDx) {
  char i;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (CC_SPI_BITBANG_PxIN&CC_SPI_BITBANG_SOMI); // Wait CCxxxx ready
  CC_SPI_bitbang_out(addr | READ_BURST);    // Send address
  for (i = 0; i < count; i++)
    buffer[i] = CC_SPI_bitbang_in();     // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  char CC_SPIReadStatus(unsigned char addr, char RF_IDx)
*
*  DESCRIPTION:
*  Special read function for reading status registers.  Reads status register
*  at register "addr" and returns the value read.
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430Fxxx BITBANG
*******************************************************************************/
char CC_SPIReadStatus(unsigned char addr, char RF_IDx) {
  char status;

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (CC_SPI_BITBANG_PxIN & CC_SPI_BITBANG_SOMI); // Wait CCxxxx ready
  CC_SPI_bitbang_out(addr | READ_BURST);      // Send address
  status = CC_SPI_bitbang_in();               // Read data
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  return status;
}

/*******************************************************************************
*  void CC_SPIStrobe(unsigned char strobe, char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr".
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430Fxxx BITBANG
*******************************************************************************/
void CC_SPIStrobe(unsigned char strobe, char RF_IDx) {
if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (CC_SPI_BITBANG_PxIN&CC_SPI_BITBANG_SOMI);// Wait for CCxxxx ready
  CC_SPI_bitbang_out(strobe);            // Send strobe
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

/*******************************************************************************
*  void CC_PowerupReset(char RF_IDx)
*
*  DESCRIPTION:
*  Special write function for writing to command strobe registers.  Writes
*  to the strobe at address "addr". See Sec. 27.1 of CC1100 datasheet
*  Added support for diversity using dual SLOT Csn aware, October 2010
* 
* SUPPORT:
* MSP430Fxxx BITBANG
*******************************************************************************/
void CC_PowerupReset(char RF_IDx) {
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  CC_Wait(3000);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
  CC_Wait(4500);

  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT &= ~CC_CSBn_PIN;        // /CS enable
  } else {
    CC_CSAn_PxOUT &= ~CC_CSAn_PIN;        // /CS enable
  }    
  while (CC_SPI_BITBANG_PxIN&CC_SPI_BITBANG_SOMI);
  CC_SPI_bitbang_out(SRES);
  while (CC_SPI_BITBANG_PxIN&CC_SPI_BITBANG_SOMI);
  if (RF_IDx==RF_SLOTB) {
    CC_CSBn_PxOUT |= CC_CSBn_PIN;         // /CS disable
  } else {
    CC_CSAn_PxOUT |= CC_CSAn_PIN;         // /CS disable
  }    
}

#endif

/* ****************************************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR REPRESENTATIONS, 
* EITHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING ANY IMPLIED WARRANTIES OF 
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY 
* OR COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET POSSESSION, AND 
* NON-INFRINGEMENT OF ANY THIRD PARTY INTELLECTUAL PROPERTY RIGHTS WITH REGARD
* TO THE PROGRAM OR YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL OR 
* INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY THEORY OF LIABILITY AND WHETHER OR 
* NOT TI HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY 
* OUT OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. EXCLUDED 
* DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF REMOVAL OR REINSTALLATION, 
* COMPUTER TIME, LABOR COSTS, LOSS OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS,
* OR LOSS OF USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S AGGREGATE 
* LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF YOUR USE OF THE PROGRAM 
* EXCEED FIVE HUNDRED DOLLARS (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted by Texas 
* Instruments is distributed as "freeware".  You may, only under TI's copyright
* in the Program, use and modify the Program without any charge or restriction.  
* You may distribute to third parties, provided that you transfer a copy of this
* license to the third party and the third party agrees to these terms by its 
* first use of the Program. You must reproduce the copyright notice and any 
* other legend of ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains copyrighted material, 
* trade secrets and other TI proprietary information and is protected by 
* copyright laws, international copyright treaties, and trade secret laws, as
* well as other intellectual property laws.  To protect TI's rights in the 
* Program, you agree not to decompile, reverse engineer, disassemble or 
* otherwise translate any object code versions of the Program to a human-
* readable form.  You agree that in no event will you alter, remove or destroy
* any copyright notice included in the Program.  TI reserves all rights not 
* specifically granted under this license. Except as specifically provided 
* herein, nothing in this agreement shall be construed as conferring by 
* implication, estoppel, or otherwise, upon you, any license or other right
* under any TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
*******************************************************************************/