/*******************************************************************************
* Description:  This file contains functions that interface with CC1101 and 
* CC2500 series devices. 
*
* Version :  1.00, March 2011
* Comments: Initial Release Version
* Author  : Thomas Almholt, Texas Instruments
********************************************************************************
* Compiled with IAR Embedded Workbench v5.10.4.50168
*******************************************************************************/
#include "msp430.h"
#include "CCx_hal.h"
#include "CC112x_def.h"
#include "CC112x_spi.h"

/*******************************************************************************
* Globals used inside the physical layer
*******************************************************************************/

/*******************************************************************************
* Initialize the radio hardware
* This particular setting sets up the Radio for 100kB/s GFSK modulation
******************************************************************************/
int radio_init_100kbs(char RF_IDx) {
  
  // Configure SPI ports for each radio
  CC_SPISetup(RF_IDx);                    // Initialize SPI port
  
  // Perform a cold start of the CC radio in RF_CC430
  CC_PowerupReset(RF_IDx);          // Reset CCxxxx
  __delay_cycles(100000); 
  
  // Write register settings
  /* RF settings SoC CC1125 PG1.1 Silicon */
  
  CC_SPIWriteReg(IOCFG0           ,     0x06, RF_IDx);       // GPIO3 IO pin configuration register.
  CC_SPIWriteReg(IOCFG2           ,     0x06, RF_IDx);       // GPIO2 IO pin configuration register.
  CC_SPIWriteReg(SYNC3            ,     0xD3, RF_IDx);       // Sync word programming (31:24).
  CC_SPIWriteReg(SYNC2            ,     0x91, RF_IDx);       // Sync word programming (23:16).
  CC_SPIWriteReg(SYNC1            ,     0xD3, RF_IDx);       // Sync word programming (15:8).
  CC_SPIWriteReg(SYNC0            ,     0x91, RF_IDx);       // Sync word programming (7:0).
  CC_SPIWriteReg(SYNC_CFG1        ,     0x08, RF_IDx);       // 
  CC_SPIWriteReg(SYNC_CFG0        ,     0x17, RF_IDx);       // 11 bit sync word (AC3)
  CC_SPIWriteReg(DEVIATION_M      ,     0x81, RF_IDx);       // 
  CC_SPIWriteReg(MODCFG_DEV_E     ,     0x0E, RF_IDx);       // 
  CC_SPIWriteReg(DCFILT_CFG       ,     0x04, RF_IDx);       // 
  CC_SPIWriteReg(PREAMBLE_CFG1    ,     0x18, RF_IDx);       // use only 0.5byte preamble
  CC_SPIWriteReg(PREAMBLE_CFG0    ,     0x2A, RF_IDx);       // 
  CC_SPIWriteReg(FREQ_IF_CFG      ,     0x00, RF_IDx);       // 
  CC_SPIWriteReg(IQIC             ,     0x00, RF_IDx);       // IQIC configuration
  CC_SPIWriteReg(CHAN_BW          ,     0x01, RF_IDx);       // 
  CC_SPIWriteReg(MDMCFG1          ,     0x46, RF_IDx);       // 
  CC_SPIWriteReg(MDMCFG0          ,     0x0A, RF_IDx);       // 
  CC_SPIWriteReg(DRATE2           ,     0xA9, RF_IDx);       // 
  CC_SPIWriteReg(DRATE1           ,     0x99, RF_IDx);       // 
  CC_SPIWriteReg(DRATE0           ,     0x9A, RF_IDx);       // 
  CC_SPIWriteReg(AGC_REF          ,     0x3C, RF_IDx);       // 
  CC_SPIWriteReg(AGC_CS_THR       ,     0xEC, RF_IDx);       // (0x10 by oslo)
  CC_SPIWriteReg(AGC_GAIN_ADJUST  ,     0x00, RF_IDx);       // 
  CC_SPIWriteReg(AGC_CFG3         ,     0x83, RF_IDx);       // 
  CC_SPIWriteReg(AGC_CFG2         ,     0x60, RF_IDx);       // 
  CC_SPIWriteReg(AGC_CFG1         ,     0xA9, RF_IDx);       // 
  CC_SPIWriteReg(AGC_CFG0         ,     0xC0, RF_IDx);       // 
  CC_SPIWriteReg(FIFO_CFG         ,     0x00, RF_IDx);       // 
  CC_SPIWriteReg(DEV_ADDR         ,     0x00, RF_IDx);       // 
  CC_SPIWriteReg(SETTLING_CFG     ,     0x0B, RF_IDx);       //
  CC_SPIWriteReg(FS_CFG           ,     0x12, RF_IDx);       // 
  CC_SPIWriteReg(WOR_CFG1         ,     0x08, RF_IDx);       // 
  CC_SPIWriteReg(WOR_CFG0         ,     0x21, RF_IDx);       //
  CC_SPIWriteReg(WOR_EVENT0_MSB   ,     0x70, RF_IDx);       //
  CC_SPIWriteReg(WOR_EVENT0_LSB   ,     0x00, RF_IDx);       // 
  CC_SPIWriteReg(PKT_CFG2         ,     0x04, RF_IDx);       // 
  CC_SPIWriteReg(PKT_CFG1         ,     0x05, RF_IDx);       // 
  CC_SPIWriteReg(PKT_CFG0         ,     0x00, RF_IDx);       // 
  CC_SPIWriteReg(RFEND_CFG1       ,     0x0F, RF_IDx);       // 
  CC_SPIWriteReg(RFEND_CFG0       ,     0x01, RF_IDx);       // 
  CC_SPIWriteReg(PA_CFG2          ,     0x77, RF_IDx);       // 
  CC_SPIWriteReg(PA_CFG1          ,     0x56, RF_IDx);       // 
  CC_SPIWriteReg(PA_CFG0          ,     0x79, RF_IDx);       // 
  CC_SPIWriteReg(PKT_LEN          ,     0xFF, RF_IDx);       // 
  
  CC_SPIWriteExtReg(IF_MIX_CFG       ,     0x00, RF_IDx);       // IF_MIX configuration
  CC_SPIWriteExtReg(FREQOFF_CFG      ,     0x20, RF_IDx);       // FOC_CFG configuration,
  CC_SPIWriteExtReg(TOC_CFG          ,     0x0A, RF_IDx);       // Timing Offset Compensation
  CC_SPIWriteExtReg(MARC_SPARE       ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(ECG_CFG          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(SOFT_TX_DATA_CFG ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(EXT_CTRL         ,     0x01, RF_IDx);       // 
  CC_SPIWriteExtReg(RCCAL_FINE       ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(RCCAL_COARSE     ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(RCCAL_OFFSET     ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(FREQOFF1         ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(FREQOFF0         ,     0x00, RF_IDx);       // 

  //915MHz using PG2.0 silicon
  CC_SPIWriteExtReg(FREQ2            ,     0x72, RF_IDx);       //  PG2.0 silicon 
  CC_SPIWriteExtReg(FREQ1            ,     0x60, RF_IDx);       // 
  CC_SPIWriteExtReg(FREQ0            ,     0x00, RF_IDx);       // 

  CC_SPIWriteExtReg(IF_ADC2          ,     0x02, RF_IDx);       // Analog to digital converter, reg 2
  CC_SPIWriteExtReg(IF_ADC1          ,     0xA6, RF_IDx);       // Analog to digital converter, reg 1
  CC_SPIWriteExtReg(IF_ADC0          ,     0x04, RF_IDx);       // Analog to digital converter, reg 0
  CC_SPIWriteExtReg(FS_DIG1          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_DIG0          ,     0x5F, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_CAL3          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_CAL2          ,     0x20, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_CAL1          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_CAL0          ,     0x0E, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_CHP           ,     0x28, RF_IDx);       // Charge pump
  CC_SPIWriteExtReg(FS_DIVTWO        ,     0x02, RF_IDx);       // Divide by 2
  CC_SPIWriteExtReg(FS_DSM1          ,     0x00, RF_IDx);       // Digital synthesizer module, reg 1
  CC_SPIWriteExtReg(FS_DSM0          ,     0x33, RF_IDx);       // Digital synthesizer module, reg 0
  CC_SPIWriteExtReg(FS_DVC1          ,     0xFF, RF_IDx);       // Divider chain, reg 1
  CC_SPIWriteExtReg(FS_DVC0          ,     0x1F, RF_IDx);       // Divider chain, reg 0
  //CC_SPIWriteExtReg(FS_LBI           ,     0x00, RF_IDx);       // Local bias
  CC_SPIWriteExtReg(FS_PFD           ,     0x51, RF_IDx);       // Phase-frequency detector
  CC_SPIWriteExtReg(FS_PRE           ,     0x06, RF_IDx);       // Prescaler
  CC_SPIWriteExtReg(FS_REG_DIV_CML   ,     0x15, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_SPARE         ,     0xA6, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_VCO4          ,     0x14, RF_IDx);       // Voltage controlled oscillator, reg 4
  CC_SPIWriteExtReg(FS_VCO3          ,     0x00, RF_IDx);       // Voltage controlled oscillator, reg 3
  CC_SPIWriteExtReg(FS_VCO2          ,     0x00, RF_IDx);       // Voltage controlled oscillator, reg 2
  CC_SPIWriteExtReg(FS_VCO1          ,     0x00, RF_IDx);       // Voltage controlled oscillator, reg 1
  CC_SPIWriteExtReg(FS_VCO0          ,     0x81, RF_IDx);       // Voltage controlled oscillator, reg 0
  CC_SPIWriteExtReg(GBIAS6           ,     0x00, RF_IDx);       // Global bias, reg 6
  CC_SPIWriteExtReg(GBIAS5           ,     0x02, RF_IDx);       // Global bias, reg 5
  CC_SPIWriteExtReg(GBIAS4           ,     0x00, RF_IDx);       // Global bias, reg 4
  CC_SPIWriteExtReg(GBIAS3           ,     0x00, RF_IDx);       // Global bias, reg 3
  CC_SPIWriteExtReg(GBIAS2           ,     0x10, RF_IDx);       // Global bias, reg 2
  CC_SPIWriteExtReg(GBIAS1           ,     0x00, RF_IDx);       // Global bias, reg 1
  CC_SPIWriteExtReg(GBIAS0           ,     0x00, RF_IDx);       // Global bias, reg 0
  CC_SPIWriteExtReg(IFAMP            ,     0x01, RF_IDx);       // Intermediate frequency amplifier
  CC_SPIWriteExtReg(LNA              ,     0x01, RF_IDx);       // Low noise amplifier
  CC_SPIWriteExtReg(RXMIX            ,     0x01, RF_IDx);       // RX mixer
  CC_SPIWriteExtReg(XOSC5            ,     0x0E, RF_IDx);       // Crystal oscillator, reg 5
  CC_SPIWriteExtReg(XOSC4            ,     0xA1, RF_IDx);       // Crystal oscillator, reg 4
  CC_SPIWriteExtReg(XOSC3            ,     0x03, RF_IDx);       // Default is 0xC7
  CC_SPIWriteExtReg(XOSC2            ,     0x04, RF_IDx);       // Crystal oscillator, reg 2
  CC_SPIWriteExtReg(XOSC1            ,     0x05, RF_IDx);       // Default is 0x07
  //CC_SPIWriteExtReg(XOSC0            ,     0x00, RF_IDx);       // Crystal oscillator, reg 0
  CC_SPIWriteExtReg(ANALOG_SPARE     ,     0x30, RF_IDx);       // 
  CC_SPIWriteExtReg(IRQ0M            ,     0x00, RF_IDx);       // IRQ0 mask
  //CC_SPIWriteExtReg(IRQ0F            ,     0x00, RF_IDx);       // IRQ0 flag
  CC_SPIWriteExtReg(PA_CFG3          ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(WOR_TIME1        ,     0x00, RF_IDx);       // WOR timer status MSB
  //CC_SPIWriteExtReg(WOR_TIME0        ,     0x00, RF_IDx);       // WOR timer status LSB
  //CC_SPIWriteExtReg(WOR_CAPTURE1     ,     0x00, RF_IDx);       // WOR timer capture MSB
  //CC_SPIWriteExtReg(WOR_CAPTURE0     ,     0x00, RF_IDx);       // WOR timer capture LSB
  CC_SPIWriteExtReg(BIST             ,     0x00, RF_IDx);       // MARC BIST
  CC_SPIWriteExtReg(DCFILTOFFSET_I1  ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(DCFILTOFFSET_I0  ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(DCFILTOFFSET_Q1  ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(DCFILTOFFSET_Q0  ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(IQIE_I1          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(IQIE_I0          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(IQIE_Q1          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(IQIE_Q0          ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(RSSI1            ,     0x80, RF_IDx);       // Received signal strength indicator, reg 1
  //CC_SPIWriteExtReg(RSSI0            ,     0x00, RF_IDx);       // Received signal strength indicator, reg 0
  //CC_SPIWriteExtReg(MARCSTATE        ,     0x41, RF_IDx);       // MARC state value
  //CC_SPIWriteExtReg(LQI_VAL          ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(PQT_SYNC_ERR     ,     0xFF, RF_IDx);       // 
  //CC_SPIWriteExtReg(DEM_STATUS       ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(FREQOFF_EST1     ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(FREQOFF_EST0     ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(AGC_GAIN3        ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(AGC_GAIN2        ,     0xD1, RF_IDx);       // 
  CC_SPIWriteExtReg(AGC_GAIN1        ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(AGC_GAIN0        ,     0x3F, RF_IDx);       // 
  //CC_SPIWriteExtReg(SOFT_RX_DATA_OUT ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(SOFT_TX_DATA_IN  ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(ASK_SOFT_RX_DATA ,     0x30, RF_IDx);       // 
  CC_SPIWriteExtReg(RNDGEN           ,     0x7F, RF_IDx);       // 
  //CC_SPIWriteExtReg(MAGN2            ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(MAGN1            ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(MAGN0            ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(ANG1             ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(ANG0             ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(CHFILT_I2        ,     0x08, RF_IDx);       // 
  //CC_SPIWriteExtReg(CHFILT_I1        ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(CHFILT_I0        ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(CHFILT_Q2        ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(CHFILT_Q1        ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(CHFILT_Q0        ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(GPIO_STATUS      ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(FSCAL_CTRL       ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(PHASE_ADJUST     ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(PARTNUMBER       ,     0x58, RF_IDx);       // 
  //CC_SPIWriteExtReg(PARTVERSION      ,     0x11, RF_IDx);       // 
  CC_SPIWriteExtReg(SERIAL_STATUS    ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(RX_STATUS        ,     0x10, RF_IDx);       // 
  //CC_SPIWriteExtReg(TX_STATUS        ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(MARC_STATUS1     ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(MARC_STATUS0     ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(PA_IFAMP_TEST    ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(FSRF_TEST        ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(PRE_TEST         ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(PRE_OVR          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(ADC_TEST         ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(DVC_TEST         ,     0x0B, RF_IDx);       // 
  CC_SPIWriteExtReg(ATEST            ,     0x40, RF_IDx);       // 
  CC_SPIWriteExtReg(ATEST_LVDS       ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(ATEST_MODE       ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(XOSC_TEST1       ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(XOSC_TEST0       ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(NUM_TXBYTES      ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(NUM_RXBYTES      ,     0x00, RF_IDx);       // 
  
  return RADIO_IDLE;
}


/*******************************************************************************
* Initialize the radio hardware
* This configuration sets up the Radio for Flex Pager spec of 6.4kBaud, 4FSK
******************************************************************************/
int radio_init(char RF_IDx) {
  
  // Configure SPI ports for each radio
  CC_SPISetup(RF_IDx);                    // Initialize SPI port
  
  // Perform a cold start of the CC radio in RF_CC430
  CC_PowerupReset(RF_IDx);          // Reset CCxxxx
  __delay_cycles(100000); 
  
  // Write register settings
  /* RF settings SoC CC1125 PG2.0 Silicon */
  
  CC_SPIWriteReg(IOCFG0           ,     0x06, RF_IDx);       // GPIO3 IO pin configuration register.
  CC_SPIWriteReg(IOCFG2           ,     0x06, RF_IDx);       // GPIO2 IO pin configuration register.
  CC_SPIWriteReg(SYNC3            ,     0xD3, RF_IDx);       // Sync word programming (31:24).
  CC_SPIWriteReg(SYNC2            ,     0x91, RF_IDx);       // Sync word programming (23:16).
  CC_SPIWriteReg(SYNC1            ,     0xD3, RF_IDx);       // Sync word programming (15:8).
  CC_SPIWriteReg(SYNC0            ,     0x91, RF_IDx);       // Sync word programming (7:0).
  CC_SPIWriteReg(SYNC_CFG1        ,     0x08, RF_IDx);       // 
  CC_SPIWriteReg(SYNC_CFG0        ,     0x17, RF_IDx);       // 11 bit sync word (AC3)
  CC_SPIWriteReg(DEVIATION_M      ,     0x3A, RF_IDx);       // 
  CC_SPIWriteReg(MODCFG_DEV_E     ,     0x2B, RF_IDx);       // 
  CC_SPIWriteReg(DCFILT_CFG       ,     0x56, RF_IDx);       // 
  CC_SPIWriteReg(PREAMBLE_CFG1    ,     0x18, RF_IDx);       // use only 0.5byte preamble
  CC_SPIWriteReg(PREAMBLE_CFG0    ,     0x2A, RF_IDx);       // 
  CC_SPIWriteReg(FREQ_IF_CFG      ,     0x3D, RF_IDx);       // 
  CC_SPIWriteReg(IQIC             ,     0xCE, RF_IDx);       // IQIC configuration
  CC_SPIWriteReg(CHAN_BW          ,     0x10, RF_IDx);       // 
  CC_SPIWriteReg(MDMCFG1          ,     0x46, RF_IDx);       // 
  CC_SPIWriteReg(MDMCFG0          ,     0x05, RF_IDx);       // 
  CC_SPIWriteReg(DRATE2           ,     0x5A, RF_IDx);       // 
  CC_SPIWriteReg(DRATE1           ,     0x36, RF_IDx);       // 
  CC_SPIWriteReg(DRATE0           ,     0xE3, RF_IDx);       // 
  CC_SPIWriteReg(AGC_REF          ,     0x20, RF_IDx);       // 
  CC_SPIWriteReg(AGC_CS_THR       ,     0xEA, RF_IDx);       //
  CC_SPIWriteReg(AGC_GAIN_ADJUST  ,     0x00, RF_IDx);       // 
  CC_SPIWriteReg(AGC_CFG3         ,     0x91, RF_IDx);       // 
  CC_SPIWriteReg(AGC_CFG2         ,     0x20, RF_IDx);       // 
  CC_SPIWriteReg(AGC_CFG1         ,     0x2D, RF_IDx);       // 
  CC_SPIWriteReg(AGC_CFG0         ,     0x5F, RF_IDx);       // 
  CC_SPIWriteReg(FIFO_CFG         ,     0x00, RF_IDx);       // 
  CC_SPIWriteReg(DEV_ADDR         ,     0x00, RF_IDx);       // 
  CC_SPIWriteReg(SETTLING_CFG     ,     0x0B, RF_IDx);       //
  CC_SPIWriteReg(FS_CFG           ,     0x12, RF_IDx);       // 
  CC_SPIWriteReg(WOR_CFG1         ,     0x08, RF_IDx);       // 
  CC_SPIWriteReg(WOR_CFG0         ,     0x21, RF_IDx);       //
  CC_SPIWriteReg(WOR_EVENT0_MSB   ,     0x00, RF_IDx);       //
  CC_SPIWriteReg(WOR_EVENT0_LSB   ,     0x00, RF_IDx);       // 
  CC_SPIWriteReg(PKT_CFG2         ,     0x04, RF_IDx);       // 
  CC_SPIWriteReg(PKT_CFG1         ,     0x05, RF_IDx);       // 
  CC_SPIWriteReg(PKT_CFG0         ,     0x00, RF_IDx);       // 
  CC_SPIWriteReg(RFEND_CFG1       ,     0x0F, RF_IDx);       // 
  CC_SPIWriteReg(RFEND_CFG0       ,     0x00, RF_IDx);       // 
  CC_SPIWriteReg(PA_CFG2          ,     0x7F, RF_IDx);       // 
  CC_SPIWriteReg(PA_CFG1          ,     0x56, RF_IDx);       // 
  CC_SPIWriteReg(PA_CFG0          ,     0x7C, RF_IDx);       // 
  CC_SPIWriteReg(PKT_LEN          ,     0x08, RF_IDx);       // 
  
  CC_SPIWriteExtReg(IF_MIX_CFG       ,     0x04, RF_IDx);       // IF_MIX configuration
  CC_SPIWriteExtReg(FREQOFF_CFG      ,     0x24, RF_IDx);       // FOC_CFG configuration,
  CC_SPIWriteExtReg(TOC_CFG          ,     0x0B, RF_IDx);       // Timing Offset Compensation
  CC_SPIWriteExtReg(MARC_SPARE       ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(ECG_CFG          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(SOFT_TX_DATA_CFG ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(EXT_CTRL         ,     0x01, RF_IDx);       // 
  CC_SPIWriteExtReg(RCCAL_FINE       ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(RCCAL_COARSE     ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(RCCAL_OFFSET     ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(FREQOFF1         ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(FREQOFF0         ,     0x00, RF_IDx);       // 

  //915MHz using PG2.0 silicon
  CC_SPIWriteExtReg(FREQ2            ,     0x74, RF_IDx);       //  PG2.0 silicon 
  CC_SPIWriteExtReg(FREQ1            ,     0x40, RF_IDx);       // 
  CC_SPIWriteExtReg(FREQ0            ,     0x32, RF_IDx);       // 

  CC_SPIWriteExtReg(IF_ADC2          ,     0x02, RF_IDx);       // Analog to digital converter, reg 2
  CC_SPIWriteExtReg(IF_ADC1          ,     0xA6, RF_IDx);       // Analog to digital converter, reg 1
  CC_SPIWriteExtReg(IF_ADC0          ,     0x04, RF_IDx);       // Analog to digital converter, reg 0
  CC_SPIWriteExtReg(FS_DIG1          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_DIG0          ,     0x5F, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_CAL3          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_CAL2          ,     0x20, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_CAL1          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_CAL0          ,     0x0E, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_CHP           ,     0x28, RF_IDx);       // Charge pump
  CC_SPIWriteExtReg(FS_DIVTWO        ,     0x02, RF_IDx);       // Divide by 2
  CC_SPIWriteExtReg(FS_DSM1          ,     0x00, RF_IDx);       // Digital synthesizer module, reg 1
  CC_SPIWriteExtReg(FS_DSM0          ,     0x33, RF_IDx);       // Digital synthesizer module, reg 0
  CC_SPIWriteExtReg(FS_DVC1          ,     0xFF, RF_IDx);       // Divider chain, reg 1
  CC_SPIWriteExtReg(FS_DVC0          ,     0x1F, RF_IDx);       // Divider chain, reg 0
  //CC_SPIWriteExtReg(FS_LBI           ,     0x00, RF_IDx);       // Local bias
  CC_SPIWriteExtReg(FS_PFD           ,     0x51, RF_IDx);       // Phase-frequency detector
  CC_SPIWriteExtReg(FS_PRE           ,     0x06, RF_IDx);       // Prescaler
  CC_SPIWriteExtReg(FS_REG_DIV_CML   ,     0x15, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_SPARE         ,     0xA6, RF_IDx);       // 
  CC_SPIWriteExtReg(FS_VCO4          ,     0x14, RF_IDx);       // Voltage controlled oscillator, reg 4
  CC_SPIWriteExtReg(FS_VCO3          ,     0x00, RF_IDx);       // Voltage controlled oscillator, reg 3
  CC_SPIWriteExtReg(FS_VCO2          ,     0x00, RF_IDx);       // Voltage controlled oscillator, reg 2
  CC_SPIWriteExtReg(FS_VCO1          ,     0x00, RF_IDx);       // Voltage controlled oscillator, reg 1
  CC_SPIWriteExtReg(FS_VCO0          ,     0x81, RF_IDx);       // Voltage controlled oscillator, reg 0
  CC_SPIWriteExtReg(GBIAS6           ,     0x00, RF_IDx);       // Global bias, reg 6
  CC_SPIWriteExtReg(GBIAS5           ,     0x02, RF_IDx);       // Global bias, reg 5
  CC_SPIWriteExtReg(GBIAS4           ,     0x00, RF_IDx);       // Global bias, reg 4
  CC_SPIWriteExtReg(GBIAS3           ,     0x00, RF_IDx);       // Global bias, reg 3
  CC_SPIWriteExtReg(GBIAS2           ,     0x10, RF_IDx);       // Global bias, reg 2
  CC_SPIWriteExtReg(GBIAS1           ,     0x00, RF_IDx);       // Global bias, reg 1
  CC_SPIWriteExtReg(GBIAS0           ,     0x00, RF_IDx);       // Global bias, reg 0
  CC_SPIWriteExtReg(IFAMP            ,     0x01, RF_IDx);       // Intermediate frequency amplifier
  CC_SPIWriteExtReg(LNA              ,     0x01, RF_IDx);       // Low noise amplifier
  CC_SPIWriteExtReg(RXMIX            ,     0x01, RF_IDx);       // RX mixer
  CC_SPIWriteExtReg(XOSC5            ,     0x0E, RF_IDx);       // Crystal oscillator, reg 5
  CC_SPIWriteExtReg(XOSC4            ,     0xA1, RF_IDx);       // Crystal oscillator, reg 4
  CC_SPIWriteExtReg(XOSC3            ,     0xC7, RF_IDx);       // Default is 0xC7
  CC_SPIWriteExtReg(XOSC2            ,     0x04, RF_IDx);       // Crystal oscillator, reg 2
  CC_SPIWriteExtReg(XOSC1            ,     0x07, RF_IDx);       // Default is 0x07
  //CC_SPIWriteExtReg(XOSC0            ,     0x00, RF_IDx);       // Crystal oscillator, reg 0
  CC_SPIWriteExtReg(ANALOG_SPARE     ,     0x30, RF_IDx);       // 
  CC_SPIWriteExtReg(IRQ0M            ,     0x00, RF_IDx);       // IRQ0 mask
  //CC_SPIWriteExtReg(IRQ0F            ,     0x00, RF_IDx);       // IRQ0 flag
  CC_SPIWriteExtReg(PA_CFG3          ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(WOR_TIME1        ,     0x00, RF_IDx);       // WOR timer status MSB
  //CC_SPIWriteExtReg(WOR_TIME0        ,     0x00, RF_IDx);       // WOR timer status LSB
  //CC_SPIWriteExtReg(WOR_CAPTURE1     ,     0x00, RF_IDx);       // WOR timer capture MSB
  //CC_SPIWriteExtReg(WOR_CAPTURE0     ,     0x00, RF_IDx);       // WOR timer capture LSB
  CC_SPIWriteExtReg(BIST             ,     0x00, RF_IDx);       // MARC BIST
  CC_SPIWriteExtReg(DCFILTOFFSET_I1  ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(DCFILTOFFSET_I0  ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(DCFILTOFFSET_Q1  ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(DCFILTOFFSET_Q0  ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(IQIE_I1          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(IQIE_I0          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(IQIE_Q1          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(IQIE_Q0          ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(RSSI1            ,     0x80, RF_IDx);       // Received signal strength indicator, reg 1
  //CC_SPIWriteExtReg(RSSI0            ,     0x00, RF_IDx);       // Received signal strength indicator, reg 0
  //CC_SPIWriteExtReg(MARCSTATE        ,     0x41, RF_IDx);       // MARC state value
  //CC_SPIWriteExtReg(LQI_VAL          ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(PQT_SYNC_ERR     ,     0xFF, RF_IDx);       // 
  //CC_SPIWriteExtReg(DEM_STATUS       ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(FREQOFF_EST1     ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(FREQOFF_EST0     ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(AGC_GAIN3        ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(AGC_GAIN2        ,     0xD1, RF_IDx);       // 
  CC_SPIWriteExtReg(AGC_GAIN1        ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(AGC_GAIN0        ,     0x3F, RF_IDx);       // 
  //CC_SPIWriteExtReg(SOFT_RX_DATA_OUT ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(SOFT_TX_DATA_IN  ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(ASK_SOFT_RX_DATA ,     0x30, RF_IDx);       // 
  CC_SPIWriteExtReg(RNDGEN           ,     0x7F, RF_IDx);       // 
  //CC_SPIWriteExtReg(MAGN2            ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(MAGN1            ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(MAGN0            ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(ANG1             ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(ANG0             ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(CHFILT_I2        ,     0x08, RF_IDx);       // 
  //CC_SPIWriteExtReg(CHFILT_I1        ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(CHFILT_I0        ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(CHFILT_Q2        ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(CHFILT_Q1        ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(CHFILT_Q0        ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(GPIO_STATUS      ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(FSCAL_CTRL       ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(PHASE_ADJUST     ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(PARTNUMBER       ,     0x58, RF_IDx);       // 
  //CC_SPIWriteExtReg(PARTVERSION      ,     0x11, RF_IDx);       // 
  CC_SPIWriteExtReg(SERIAL_STATUS    ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(RX_STATUS        ,     0x10, RF_IDx);       // 
  //CC_SPIWriteExtReg(TX_STATUS        ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(MARC_STATUS1     ,     0x00, RF_IDx);       // 
  //CC_SPIWriteExtReg(MARC_STATUS0     ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(PA_IFAMP_TEST    ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(FSRF_TEST        ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(PRE_TEST         ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(PRE_OVR          ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(ADC_TEST         ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(DVC_TEST         ,     0x0B, RF_IDx);       // 
  CC_SPIWriteExtReg(ATEST            ,     0x40, RF_IDx);       // 
  CC_SPIWriteExtReg(ATEST_LVDS       ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(ATEST_MODE       ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(XOSC_TEST1       ,     0x31, RF_IDx);       // 
  CC_SPIWriteExtReg(XOSC_TEST0       ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(NUM_TXBYTES      ,     0x00, RF_IDx);       // 
  CC_SPIWriteExtReg(NUM_RXBYTES      ,     0x00, RF_IDx);       // 
  
  return RADIO_IDLE;
}

/*******************************************************************************
* Prepare the radio with a packet to be sent
* Fill the FIFO with data, but do not initiate TX mode
******************************************************************************/
int radio_prepare(unsigned char *payload, unsigned short payload_len, char RF_IDx) {
  CC_SPIWriteBurstReg(TXFIFO, payload, payload_len, RF_IDx);          // Write TX data
  return RADIO_TX_PACKET_RDY;
}

/* Send the packet that has previously been prepared (used for exact timing)*/
int radio_transmit(char RF_IDx) {
   CC_SPIStrobe(RF_STX, RF_IDx);               // Change state to TX, initiating
  return(RADIO_TX_ACTIVE);
}

/* Send the packet that has previously been prepared (used for exact timing)*/
int radio_receive_on(char RF_IDx) {
  /* AGC Erata : AGC does not reset during SRX, work around is to manually do it*/
   //CC_SPIWriteExtReg(AGC_GAIN2, 0xD1, RF_IDx);       // 
   //CC_SPIWriteExtReg(AGC_GAIN1, 0x00, RF_IDx);       // 
   //CC_SPIWriteExtReg(AGC_GAIN0, 0x3F, RF_IDx);       // 
   
   CC_SPIStrobe(RF_SRX, RF_IDx);               // Change state to TX, initiating
  return(RADIO_RX_ACTIVE);
}

/* Prepare & transmit a packet in same call (slightly worse timing jitter) */
int radio_send(unsigned char *payload, unsigned short payload_len, char RF_IDx) {
   CC_SPIWriteBurstReg(TXFIFO, payload, payload_len, RF_IDx); // Write TX data
   CC_SPIStrobe(RF_STX, RF_IDx);               // Change state to TX, initiating
   return(RADIO_TX_ACTIVE);
}

/* Read a received packet into a buffer */
int radio_read(unsigned char *buf, unsigned short *buf_len, char RF_IDx) {
    char status;
    char pktLen;
    
    if ((pktLen = CC_SPIReadExtReg(NUM_RXBYTES, RF_IDx))) {
    if (pktLen <= *buf_len) {                       // If pktLen size <= rxBuffer
      CC_SPIReadBurstReg(RXFIFO, buf, pktLen, RF_IDx); // Pull data
      *buf_len = pktLen - 2;   
      status  = buf[*buf_len+LQI_RX] & CRC_OK;
      return (status);
    } else {                                         // Return CRC_OK bit
      *buf_len = pktLen;                             // Return the large size
      CC_SPIStrobe(RF_SFRX, RF_IDx);                 // Flush RXFIFO
      return RADIO_GENERAL_ERROR;                    // Error
    }
  } else {
      return RADIO_GENERAL_ERROR;                    // Error
  }
}

/* Perform a Clear-Channel Assessment (CCA) to find out if channel is clear */
int radio_channel_clear(char RF_IDx) {
  int status;
  status =CC_SPIReadExtReg(PQT_SYNC_ERR, RF_IDx);
  if (status == 0 ) {
    return(RADIO_CHANNEL_IS_CLR);
  } else {
    return(RADIO_CHANNEL_NOT_CLR);
  }
}

/* Wait for radio to become idle (currently receiving or transmitting) */
int radio_wait_for_idle(char RF_IDx) {
  if(RF_IDx==RF_SLOTA) {
    while (!(CC_GDO0_PxIN&CC_GDO0_PIN));
                                            // Wait GDO0 to go hi -> sync TX'ed
    while (CC_GDO0_PxIN&CC_GDO0_PIN);
                                            // Wait GDO0 to clear -> end of pkt
    CC_GDO0_PxIFG &= ~CC_GDO0_PIN;          // After pkt TX, this flag is set.
  } else {
    while (!(CC_GDO2_PxIN&CC_GDO2_PIN));
                                            // Wait GDO0 to go hi -> sync TX'ed
    while (CC_GDO2_PxIN&CC_GDO2_PIN);
                                            // Wait GDO0 to clear -> end of pkt
    CC_GDO2_PxIFG &= ~CC_GDO2_PIN;          // After pkt TX, this flag is set.
  }
  return(RADIO_IDLE);
}

/* Wait for radio to become idle (currently receiving or transmitting) */
int radio_is_busy(char RF_IDx) {
  if(RF_IDx==RF_SLOTA) {
    while (!(CC_GDO0_PxIN&CC_GDO0_PIN));
                                            // Wait GDO0 to go hi -> sync TX'ed
    while (CC_GDO0_PxIN&CC_GDO0_PIN);
                                            // Wait GDO0 to clear -> end of pkt
    CC_GDO0_PxIFG &= ~CC_GDO0_PIN;          // After pkt TX, this flag is set.
  } else {
    while (!(CC_GDO2_PxIN&CC_GDO2_PIN));
                                            // Wait GDO0 to go hi -> sync TX'ed
    while (CC_GDO2_PxIN&CC_GDO2_PIN);
                                            // Wait GDO0 to clear -> end of pkt
    CC_GDO2_PxIFG &= ~CC_GDO2_PIN;          // After pkt TX, this flag is set.
  }
  return(RADIO_IDLE);
}

/* Check if the radio driver has just received a packet */
int radio_pending_packet(char RF_IDx) {
    return (CC_SPIReadExtReg(NUM_RXBYTES,  RF_IDx));
}

/* Change channel of radio, using predefined parameters, this is not supported
*  in the CC112x family of device and will need to be implemented elsewhere   */
int radio_set_channel(unsigned char rf_channel, char RF_IDx) {
//    CC_SPIWriteReg(CHANNR, rf_channel, RF_IDx);
    return(rf_channel);
}

/* Idle the radio, used when leaving low power modes (below)*/
int radio_idle(char RF_IDx) {
  CC_SPIStrobe(RF_SIDLE, RF_IDx);
  CC_SPIStrobe(RF_SFRX, RF_IDx);               // Flush the RX FIFO's
  CC_SPIStrobe(RF_SFTX, RF_IDx);               // Flush the TX FIFO's  
  return(RADIO_IDLE);
}

/* Put the radio into sleep mode */
int radio_sleep(char RF_IDx) { 
  CC_SPIStrobe(RF_SPWD, RF_IDx);
  return RADIO_SLEEP;
}
 
/* Put the radio into sleep mode */
int radio_wakeup(char RF_IDx) { 
   CC_SPIStrobe(RF_SIDLE, RF_IDx);
   return RADIO_IDLE;
}

/* Put the radio into sleep mode */
int radio_calibrate_on(char RF_IDx) { 
  CC_SPIStrobe(RF_SCAL, RF_IDx);
  return RADIO_IDLE;
}



/*******************************************************************************
* @brief  ISR for Timer A0
* 
* @param  none
* 
* @return none
*******************************************************************************/

#ifdef MSP430_F5xx
  #pragma vector=TIMER0_A0_VECTOR     /* MSP430F5x */
  __interrupt void TIMER0_A0_ISR(void) {
    __bic_SR_register_on_exit(LPM3_bits);    
  }
#else
  #pragma vector=TIMERA0_VECTOR      /* MSP430F2x */
  __interrupt void TIMERA0_ISR(void) {
    __bic_SR_register_on_exit(LPM3_bits);   
  }
#endif

#pragma vector=PORT1_VECTOR
__interrupt void Radio_ISR(void) {
  
  if(CC_GDO2_PxIFG & CC_GDO2_PIN) {
    //__bic_SR_register_on_exit(LPM3_bits);    // Clear LPM0 bits from 0(SR)     
    // no operation.    
  }
  if(CC_GDO0_PxIFG & CC_GDO0_PIN) {
    rf_end_packet = 1;
    __bic_SR_register_on_exit(LPM3_bits);      // Clear LPM0 bits from 0(SR)     
  }
  CC_GDO0_PxIFG &= ~CC_GDO0_PIN;
  CC_GDO2_PxIFG &= ~CC_GDO2_PIN;
}




/* ****************************************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR REPRESENTATIONS, 
* EITHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING ANY IMPLIED WARRANTIES OF 
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY 
* OR COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET POSSESSION, AND 
* NON-INFRINGEMENT OF ANY THIRD PARTY INTELLECTUAL PROPERTY RIGHTS WITH REGARD
* TO THE PROGRAM OR YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL OR 
* INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY THEORY OF LIABILITY AND WHETHER OR 
* NOT TI HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY 
* OUT OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. EXCLUDED 
* DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF REMOVAL OR REINSTALLATION, 
* COMPUTER TIME, LABOR COSTS, LOSS OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS,
* OR LOSS OF USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S AGGREGATE 
* LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF YOUR USE OF THE PROGRAM 
* EXCEED FIVE HUNDRED DOLLARS (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted by Texas 
* Instruments is distributed as "freeware".  You may, only under TI's copyright
* in the Program, use and modify the Program without any charge or restriction.  
* You may distribute to third parties, provided that you transfer a copy of this
* license to the third party and the third party agrees to these terms by its 
* first use of the Program. You must reproduce the copyright notice and any 
* other legend of ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains copyrighted material, 
* trade secrets and other TI proprietary information and is protected by 
* copyright laws, international copyright treaties, and trade secret laws, as
* well as other intellectual property laws.  To protect TI's rights in the 
* Program, you agree not to decompile, reverse engineer, disassemble or 
* otherwise translate any object code versions of the Program to a human-
* readable form.  You agree that in no event will you alter, remove or destroy
* any copyright notice included in the Program.  TI reserves all rights not 
* specifically granted under this license. Except as specifically provided 
* herein, nothing in this agreement shall be construed as conferring by 
* implication, estoppel, or otherwise, upon you, any license or other right
* under any TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
*******************************************************************************/
