/*******************************************************************************
* Description:  This file contains functions that interface with CC1101 and 
* CC2500 series devices. 
*
* Version :  1.00, March 2011
* Comments: Initial Release Version
* Author  : Thomas Almholt, Texas Instruments
********************************************************************************
* Compiled with IAR Embedded Workbench v5.10.4.50168
*******************************************************************************/
#include "msp430.h"
#include "CCx_hal.h"
#include "CCxx00_def.h"
#include "CCxx00_spi.h"

/*******************************************************************************
* Globals used inside the physical layer
*******************************************************************************/
unsigned char paTable[1];    // PATABLE (11 dBm output power)
unsigned char rf_end_packet = 0;

/*******************************************************************************
* Initialize the radio hardware
* This particular setting sets up the Radio for 100kB/s GFSK modulation
******************************************************************************/
int radio_init(char RF_IDx) {
  
    // Configure SPI ports for each radio
    CC_SPISetup(RF_IDx);                    // Initialize SPI port
  
    // Perform a cold start of the CC radio in RF_CC430
    CC_PowerupReset(RF_IDx);          // Reset CCxxxx
    __delay_cycles(100000); 
    
    /* RF settings SoC */
    //# Deviation = 20.629883 
    //# Base frequency = 902.499847 
    //# Carrier frequency = 902.499847 
    //# Channel number = 0 
    //# Modulated = true 
    //# Modulation format = GFSK 
    //# Manchester enable = false 
    //# Sync word qualifier mode = 30/32 sync word bits detected 
    //# Preamble count = 4 
    //# Channel spacing = 99.975586 
    //# Data rate = 38.3835 
    //# RX filter BW = 101.562500 
    //# Data format = Normal mode 
    //# CRC enable = true 
    //# Whitening = true 
    //# Device address = 0 
    //# Address config = No address check 
    //# CRC autoflush = false 
    //# PA ramping = false 
    //# TX power = 11 
    CC_SPIWriteReg(IOCFG2,   0x06, RF_IDx);    //gdo2 output pin configuration
    CC_SPIWriteReg(IOCFG1,   0x34, RF_IDx);    //gdo1 output pin configuration
    CC_SPIWriteReg(IOCFG0,   0x2F, RF_IDx);    //gdo0 output pin configuration
    CC_SPIWriteReg(FIFOTHR,  0x47, RF_IDx);    //rx fifo and tx fifo thresholds
    CC_SPIWriteReg(SYNC1,    0xD3, RF_IDx);    //sync word, high byte
    CC_SPIWriteReg(SYNC0,    0x91, RF_IDx);    //sync word, low byte
    CC_SPIWriteReg(PKTLEN,   0x0A, RF_IDx);    //packet length
    CC_SPIWriteReg(PKTCTRL1, 0x04, RF_IDx);    //packet automation control
    CC_SPIWriteReg(PKTCTRL0, 0x44, RF_IDx);    //packet automation control
    CC_SPIWriteReg(ADDR,     0x00, RF_IDx);    //device address
    CC_SPIWriteReg(CHANNR,   0x00, RF_IDx);    //channel number
    CC_SPIWriteReg(FSCTRL1,  0x06, RF_IDx);    //frequency synthesizer control
    CC_SPIWriteReg(FSCTRL0,  0x00, RF_IDx);    //frequency synthesizer control
    CC_SPIWriteReg(FREQ2,    0x22, RF_IDx);    //frequency control word, high byte
    CC_SPIWriteReg(FREQ1,    0xB6, RF_IDx);    //frequency control word, middle byte
    CC_SPIWriteReg(FREQ0,    0x27, RF_IDx);    //frequency control word, low byte
    CC_SPIWriteReg(MDMCFG4,  0xCA, RF_IDx);    //modem configuration
    CC_SPIWriteReg(MDMCFG3,  0x83, RF_IDx);    //modem configuration
    CC_SPIWriteReg(MDMCFG2,  0x13, RF_IDx);    //modem configuration
    CC_SPIWriteReg(MDMCFG1,  0x21, RF_IDx);    //modem configuration
    CC_SPIWriteReg(MDMCFG0,  0xF8, RF_IDx);    //modem configuration
    CC_SPIWriteReg(DEVIATN,  0x35, RF_IDx);    //modem deviation setting
    CC_SPIWriteReg(MCSM2,    0x07, RF_IDx);    //main radio control state machine configuration
    CC_SPIWriteReg(MCSM1,    0x30, RF_IDx);    //main radio control state machine configuration
    CC_SPIWriteReg(MCSM0,    0x18, RF_IDx);    //main radio control state machine configuration
    CC_SPIWriteReg(FOCCFG,   0x16, RF_IDx);    //frequency offset compensation configuration
    CC_SPIWriteReg(BSCFG,    0x6C, RF_IDx);    //bit synchronization configuration
    CC_SPIWriteReg(AGCCTRL2, 0x43, RF_IDx);    //agc control
    CC_SPIWriteReg(AGCCTRL1, 0x40, RF_IDx);    //agc control
    CC_SPIWriteReg(AGCCTRL0, 0x91, RF_IDx);    //agc control
    CC_SPIWriteReg(FREND1,   0x56, RF_IDx);    //front end rx configuration
    CC_SPIWriteReg(FREND0,   0x10, RF_IDx);    //front end tx configuration
    CC_SPIWriteReg(FSCAL3,   0xE9, RF_IDx);    //frequency synthesizer calibration
    CC_SPIWriteReg(FSCAL2,   0x2A, RF_IDx);    //frequency synthesizer calibration
    CC_SPIWriteReg(FSCAL1,   0x00, RF_IDx);    //frequency synthesizer calibration
    CC_SPIWriteReg(FSCAL0,   0x1F, RF_IDx);    //frequency synthesizer calibration
    CC_SPIWriteReg(FSTEST,   0x59, RF_IDx);    //frequency synthesizer calibration control
    CC_SPIWriteReg(TEST2,    0x81, RF_IDx);    //various test settings
    CC_SPIWriteReg(TEST1,    0x35, RF_IDx);    //various test settings
    CC_SPIWriteReg(TEST0,    0x09, RF_IDx);    //various test settings

    paTable[0] = 0x8D;                         // PATABLE (0 dBm output power)
    CC_SPIWriteBurstReg(PATABLE, paTable, 1, RF_IDx);   //Write PATABLE
    
  return 0;
}
/*******************************************************************************
* Prepare the radio with a packet to be sent
* Fill the FIFO with data, but do not initiate TX mode
******************************************************************************/
int radio_prepare(unsigned char *payload, unsigned short payload_len, char RF_IDx) {
  CC_SPIWriteBurstReg(TXFIFO, payload, payload_len, RF_IDx);          // Write TX data
  return 0;
}

/* Send the packet that has previously been prepared (used for exact timing)*/
int radio_transmit(char RF_IDx) {
#ifdef CCx_FEM_INCLUDED
  hal_fem_pa_on();
#endif
   CC_SPIStrobe(RF_STX, RF_IDx);               // Change state to TX, initiating
  return(0);
}

/* Send the packet that has previously been prepared (used for exact timing)*/
int radio_receive_on(char RF_IDx) {
#ifdef CCx_FEM_INCLUDED
  hal_fem_lna_on();
#endif
   CC_SPIStrobe(RF_SRX, RF_IDx);             // Change state to TX, initiating
   __delay_cycles(1000);                     // 1 ms delay for letting RX settle
  return(0);
}

/* Prepare & transmit a packet in same call (slightly worse timing jitter) */
int radio_send(unsigned char *payload, unsigned short payload_len, char RF_IDx) {
#ifdef CCx_FEM_INCLUDED
   hal_fem_pa_on();
#endif
   CC_SPIWriteBurstReg(TXFIFO, payload, payload_len, RF_IDx); // Write TX data
   CC_SPIStrobe(RF_STX, RF_IDx);               // Change state to TX, initiating
   return(0);
}

/* Read a received packet into a buffer */
/*******************************************************************************
*  DESCRIPTION:
*  The packet length should not exceed the RXFIFO size.  To use
*  this function, APPEND_STATUS in the PKTCTRL1 register must be enabled.  It
*  is assumed that the function is called after it is known that a packet has
*  been received; for example, in response to GDO0 going low when it is
*  configured to output packet reception status.
*
*  The RXBYTES register is first read to ensure there are bytes in the FIFO.
*  This is done because the GDO signal will go high even if the FIFO is flushed
*  due to address filtering, CRC filtering, or packet length filtering.
*
*  ARGUMENTS:
*      char *rxBuffer
*          Pointer to the buffer where the incoming data should be stored
*      char *length
*          Pointer to a variable containing the size of the buffer where the
*          incoming data should be stored. After this function returns, that
*          variable holds the packet length.
*
*  RETURN VALUE:
*      char
*          0x80:  CRC OK
*          0x00:  CRC NOT OK (or no pkt was put in the RXFIFO due to filtering)
*******************************************************************************/
int radio_read(unsigned char *buf, unsigned short *buf_len, char RF_IDx) {
    char status;
    char pktLen;
    
    if ((pktLen = CC_SPIReadStatus(RXBYTES, RF_IDx) & NUM_RXBYTES)) {
    if (pktLen <= *buf_len) {                       // If pktLen size <= rxBuffer
      CC_SPIReadBurstReg(RXFIFO, buf, pktLen, RF_IDx); // Pull data
      *buf_len = pktLen - 2;   
      status  = buf[*buf_len+LQI_RX] & CRC_OK;
      return (status);
    } else {                                         // Return CRC_OK bit
      *buf_len = pktLen;                             // Return the large size
      CC_SPIStrobe(RF_SFRX, RF_IDx);                 // Flush RXFIFO
      return 0;                                      // Error
    }
  } else {
      return 0;                                      // Error
  }
}

/* Perform a Clear-Channel Assessment (CCA) to find out if channel is clear */
int radio_channel_clear(char RF_IDx) {
  int status;
  status = CC_SPIReadStatus(PKTSTATUS, RF_IDx) & 0x40;
  return(status);
}

/*******************************************************************************
* Description:  Wait for end of packet interupt to happen.
* Wait for radio to become idle (currently receiving or transmitting)
* Comments:
*      timeout is controlled by TimerA running at 8MHz
*      64000 = 128ms, 32000 = 64ms, 16000 = 32ms
*      0 = no timeout.
* Return:
*      timer value:  Interupt happened based on end_of_packet interupt
*      max_hold   :  Watch dog timeout happend, no end of packet in time
*
* Thomas Almholt, Texas Instruments Inc., 2011.
******************************************************************************/
int radio_wait_for_idle(unsigned short max_hold, char RF_IDx) {

  unsigned int status;
  
  rf_end_packet = 0;  // initialize global variable for use in this function
  
  if(max_hold > 0) {
    /********  Enable the Timer (F5438)******************************************/
#ifdef MSP430_F5xx
    TA0CCR0  = max_hold-1;                       // 32=1ms @ 32768
    TA0CCTL0 = CCIE;                             // CCR0 interrupt enabled  
    TA0CTL   = TASSEL_1 + MC_3 + ID_0;           // SMCLK, up/down mode, clk div/8
    /********  Enable the Timer (2274)******************************************/
#else
    TACCR0  = max_hold-1;                       // 32=1ms @ 32768Hz
    TACCTL0 |= CCIE;                            // CCR0 interrupt enabled  
    TACTL   = TASSEL_1 + MC_3 + ID_3;           // SMCLK, up/down mode, clk div/8
#endif
  } else {
    status = 0;
  }
  // BIT9 is the Sync recieved/sent and end-of-packet.
  RF1AIES |=  BIT9;        // edge select, high to low (end-of-packet)
  RF1AIFG &= ~BIT9;        // Clear pending interrupts
  RF1AIE  |=  BIT9;        // Enable RX end-of-packet interrupt

  //HAL_LED1_OFF();                 // indicate low power RF+MCU mode
  _BIS_SR(LPM3_bits + GIE);         // Enter LPM0
  //HAL_LED1_ON();                  // indicate normal mode
  
  /********  Setup the GDO ports to not interupts ****************************/
  RF1AIE &= ~BIT9;         // Disable RX end-of-packet interrupt
  
  /* Get timer values, however if we did not get a packet in time use 0      */
  /********  Disable the Timer (F5438) *****************************************/
#ifdef MSP430_F5xx  
  status = TA0R+1;
  TA0CTL = TACLR;                                // clear the timer registers
  TA0CCTL0 &= ~CCIE;                             // CCR0 interrupt disabled  
#else
  status = TAR+1;
  TACTL = TACLR;                                    // clear the timer registers
  TACCTL0 &= ~CCIE;                                 // CCR0 interrupt disabled  
#endif

  if(rf_end_packet == 0) {
    status = max_hold;
  }

#ifdef CCx_FEM_INCLUDED
    hal_fem_shutdown();
#endif
  
  return status;
}

/* Wait for radio to become idle (currently receiving or transmitting) */
int radio_is_busy(char RF_IDx) {
  if(RF_IDx==RF_SLOTA) {
    while (!(CC_GDO0_PxIN&CC_GDO0_PIN));
                                            // Wait GDO0 to go hi -> sync TX'ed
    while (CC_GDO0_PxIN&CC_GDO0_PIN);
                                            // Wait GDO0 to clear -> end of pkt
    CC_GDO0_PxIFG &= ~CC_GDO0_PIN;          // After pkt TX, this flag is set.
  } else {
    while (!(CC_GDO2_PxIN&CC_GDO2_PIN));
                                            // Wait GDO0 to go hi -> sync TX'ed
    while (CC_GDO2_PxIN&CC_GDO2_PIN);
                                            // Wait GDO0 to clear -> end of pkt
    CC_GDO2_PxIFG &= ~CC_GDO2_PIN;          // After pkt TX, this flag is set.
  }
  return(0);
}


/* Check if the radio driver has just received a packet */
int radio_pending_packet(char RF_IDx);

/* Change channel of radio, using predefined paramters */
int radio_set_channel(unsigned char rf_channel, char RF_IDx) {
    CC_SPIWriteReg(CHANNR, rf_channel, RF_IDx);
    return(rf_channel);
}

/* Idle the radio, used when leaving low power modes (below)*/
int radio_idle(char RF_IDx) {
#ifdef CCx_FEM_INCLUDED
  hal_fem_shutdown();
#endif
  CC_SPIStrobe(RF_SIDLE, RF_IDx);
  CC_SPIStrobe(RF_SFRX, RF_IDx);               // Flush the RX FIFO's
  CC_SPIStrobe(RF_SFTX, RF_IDx);               // Flush the TX FIFO's  
  return(0);
}

/* Put the radio into sleep mode */
/* Put the radio into sleep mode */
int radio_sleep(char RF_IDx) { 
  CC_SPIStrobe(RF_SPWD, RF_IDx);
  return RADIO_SLEEP;
}

/* Wake the radio from sleep mode */
int radio_wakeup(char RF_IDx);

/* Force PLL calibration, used enabling manual calibration for ultra low power */
int radio_calibrate_on(char RF_IDx);




/*******************************************************************************
* @brief  ISR for Timer A0
* 
* @param  none
* 
* @return none
*******************************************************************************/

#ifdef MSP430_F5xx
  #pragma vector=TIMER0_A0_VECTOR     /* MSP430F5x */
  __interrupt void TIMER0_A0_ISR(void) {
    __bic_SR_register_on_exit(LPM3_bits);    
  }
#else
  #pragma vector=TIMERA0_VECTOR      /* MSP430F2x */
  __interrupt void TIMERA0_ISR(void) {
    __bic_SR_register_on_exit(LPM3_bits);   
  }
#endif



/*******************************************************************************
* @brief  Port 1 interupt service routine
* 
* @param  none
* 
* @return none
*
*******************************************************************************/
#pragma vector=CC1101_VECTOR
__interrupt void Radio_ISR(void) {
 
  switch(__even_in_range(RF1AIV,32)) {
      case  RF1AIV_NONE: break;    // no interupt
      case  RF1AIV_RFIFG0: break;  // Based on GDO0 signal - programmable using IOCFG0      
      case  RF1AIV_RFIFG1: break;  // Based on GDO1 signal - programmable using IOCFG1
      case  RF1AIV_RFIFG2: break;  // Based on GDO2 signal - programmable using IOCFG2
      case  RF1AIV_RFIFG3: break;  // RX FIFO filled or above the RX FIFO threshold.
      case  RF1AIV_RFIFG4: break;  // RX FIFO filled or above the RX FIFO threshold or end of packet is reached
      case  RF1AIV_RFIFG5: break;  // TX FIFO filled or above the TX FIFO threshold.
      case  RF1AIV_RFIFG6: break;  // Positive edge: TX FIFO full. Negative edge: TX FIFO below TX FIFO threshold
      case  RF1AIV_RFIFG7: break;  // RX FIFO overflowed. Negative edge: RX FIFO flushed. (Equal to GDOx_CFG=4)
      case  RF1AIV_RFIFG8: break;  // Positive edge: TX FIFO underflowed. Negative edge: TX FIFO flushed. (Equal to GDOx_CFG=5)
      case  RF1AIV_RFIFG9:         // Positive edge: Sync word sent or received. Negative edge: End of packet
        rf_end_packet = 1;
        break;                     
      case  RF1AIV_RFIFG10: break; // Positive edge: Packet received with CRC OK.
      case  RF1AIV_RFIFG11: break; // Positive edge: Preamble quality reached (PQI)
      case  RF1AIV_RFIFG12: break; // CCA_MODE setting
      case  RF1AIV_RFIFG13: break; // Positive edge: Carrier sense. RSSI level is above threshold. Negative edge: RSSI level is below threshold. (Equal to GDOx_CFG=14)
      case  RF1AIV_RFIFG14: break; // Positive edge: WOR event 0 Negative edge: WOR event 0 + 1 ACLK. (Equal to GDOx_CFG=36)
      case  RF1AIV_RFIFG15: break; // WOR event 1 Negative edge: RF oscillator stable or next WOR event0 triggered. (Equal to GDOx_CFG=37)
      default: break; 
    }
    __bic_SR_register_on_exit(LPM3_bits);         // Clear LPM0 bits from 0(SR)     
}





/* ****************************************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR REPRESENTATIONS, 
* EITHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING ANY IMPLIED WARRANTIES OF 
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY 
* OR COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET POSSESSION, AND 
* NON-INFRINGEMENT OF ANY THIRD PARTY INTELLECTUAL PROPERTY RIGHTS WITH REGARD
* TO THE PROGRAM OR YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL OR 
* INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY THEORY OF LIABILITY AND WHETHER OR 
* NOT TI HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY 
* OUT OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. EXCLUDED 
* DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF REMOVAL OR REINSTALLATION, 
* COMPUTER TIME, LABOR COSTS, LOSS OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS,
* OR LOSS OF USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S AGGREGATE 
* LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF YOUR USE OF THE PROGRAM 
* EXCEED FIVE HUNDRED DOLLARS (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted by Texas 
* Instruments is distributed as "freeware".  You may, only under TI's copyright
* in the Program, use and modify the Program without any charge or restriction.  
* You may distribute to third parties, provided that you transfer a copy of this
* license to the third party and the third party agrees to these terms by its 
* first use of the Program. You must reproduce the copyright notice and any 
* other legend of ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains copyrighted material, 
* trade secrets and other TI proprietary information and is protected by 
* copyright laws, international copyright treaties, and trade secret laws, as
* well as other intellectual property laws.  To protect TI's rights in the 
* Program, you agree not to decompile, reverse engineer, disassemble or 
* otherwise translate any object code versions of the Program to a human-
* readable form.  You agree that in no event will you alter, remove or destroy
* any copyright notice included in the Program.  TI reserves all rights not 
* specifically granted under this license. Except as specifically provided 
* herein, nothing in this agreement shall be construed as conferring by 
* implication, estoppel, or otherwise, upon you, any license or other right
* under any TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
*******************************************************************************/


