/*******************************************************************************
* Description:  This file contains functions that configure the CC1100/2500
* device
*
* Demo application for MSP430/CCxx00 interface code library v1.0
*
* T.Almholt, Texas Instruments, Inc.
* IAR Embedded Workbench v5.10
********************************************************************************
* Version:  1.2
* Comments: Added support for antenna diversity thru the use of multiple radios
*           Added support for collision detection, and various help functions.
*           Changed comments for better readability
* Author  : Thomas Almholt, Texas Instruments, October 2010
*
* Version:  1.1
* Comments: Fixed several bugs where it is stuck in a infinite while loop
*           Added support for 5xx
*
* Version:  1.00
* Comments: Initial Release Version
*******************************************************************************/

#include "stdio.h"
#include "msp430.h"
#include "hal_usb.h"
#include "CCx_hal.h"
#include "CC112x_def.h"
#include "CC112x_spi.h"


/*******************************************************************************
* @brief  int debug_cc_state(void)
* This is used to debug the states of the CCxx device as is either 
* transmits or recieves a packet. Its a very good tool to see what is going on
* in the radio and why we-
.did not see a packet. 
* @param  none
* 
* @return none
*******************************************************************************/
void debug_cc_state(unsigned char RF_IDx){
  unsigned char rssi_table[4096];
  unsigned short ii;
  
  for(ii=0;ii<4096;ii++) {
    rssi_table[ii] = CC_SPIReadExtReg(NUM_TXBYTES,  RF_IDx) & 0x1F;
    __delay_cycles(1); 
    //CC_SPIWriteReg(CHANNR,  ii, RF_SLOTA);          // Update a register
  } 
  rssi_table[0] = rssi_table[254]+1;
}

/*******************************************************************************
* @brief  rf_set_freq (in kHz)
*  sets the rf center frequency based on a frequency value to the nearest 1KHz
*   f_c = (f_xosc/2^16) * FREQ (control word)
*   FREQ = (f_c * 2^16)/f_osc;
* @param  int freq, char RF_IDx
* 
* @return none
*******************************************************************************/
char rf_set_freq(unsigned long freq, unsigned char RF_IDx) {
//    unsigned long freq_word, freq_byte;
//    
//   freq_word = freq*1000;
//   freq_word = (unsigned long)(freq_word / 396.7285156);   // 26000000/65536
// 
//   freq_byte = (freq_word>>16) & 0xFF;
//   CC_SPIWriteExtReg(FREQ2, freq_byte, RF_IDx);        // Write TX data
//   freq_byte = (freq_word>>8)  & 0xFF;
//   CC_SPIWriteExtReg(FREQ1, freq_byte, RF_IDx);        // Write TX data
//   freq_byte = freq_word       & 0xFF;
//   CC_SPIWriteExtReg(FREQ0, freq_byte, RF_IDx);        // Write TX data
return(0);
}

/******************************************************************************
* set_freq_dev Set the output power of the CC430 by looking up in table
*
*		freq_dev_m							
* freq_dev_e	0	1	2	3	4	5	6	7
* 0	1	1586	1785	1983	2182	2380	2578	2777	2975
* 1	2	3173	3570	3967	4364	4760	5157	5554	5950
* 2	4	6347	7141	7934	8728	9521	10314	11108   11901
* 3	8	12695	14282	15869	17456	19042	20629	22216	23803
* 4	16	25390	28564	31738	34912	38085	41259	44433	47607
* 5	32	50781	57128	63476	69824	76171	82519	88867	95214
* 6	64	101562	114257	126953	139648	152343	165039	177734	190429
* 7	128	203125	228515	253906	279296	304687	330078	355468	380859
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
int set_freq_dev(int freq_dev, unsigned char RF_IDx) {
//  unsigned char freq_dev_byte;
//  const unsigned int rx_table[64] = \
//    {2,	  2,	2,	2,	2,	3,	3,	3,
//    3,	  4,	4,	4,	5,	5,	6,	6,
//    6,	  7,	8,	9,	10,	10,	11,	12,
//    13,	  14,	16,	17,	19,	21,	22,	24,
//    25,	  29,	32,	35,	38,	41,	44,	48,  
//    51,	  57,	63,	70,	76,	83,	89,	95,
//    102,  114,	127,	140,	152,	165,	178,	190,
//    203,  229,	254,	279,	305,	330,	355,	381};
//  unsigned char ee, mm;
//  int best_fit;
//  
//  for(ee=0; ee<7;ee++){
//    for(mm=0; mm<7;mm++){
//      freq_dev_byte = (ee<<3)+mm;
//      best_fit = rx_table[freq_dev_byte];
//      if(freq_dev <= best_fit ){
//        freq_dev_byte = (ee<<4)+mm;
//        //CC_SPIWriteReg(DEVIATION_M, freq_dev_byte , RF_IDx);
//        mm = 7;                                          // terminate for loop
//        ee = 7;                                          // terminate for loop
//      }
//    }
//  }
//  if(freq_dev > 381) {
//    freq_dev_byte = 0x77;
//    best_fit = 381;
//    //CC_SPIWriteReg(DEVIATION_M, freq_dev_byte , RF_IDx);
//  }
//  return best_fit;
    return(freq_dev);
}

/******************************************************************************
* set_rx_bw: Set the output power of the CC430 by looking up in table
*
* 	RX filter (use all 4 rows)			
* 	0	1	2	3 (mm=3)
*   0	812500	650000	541666	464285 (ee = 0)
*   1	406250	325000	270833	232142 (ee = 1)
*   2	203125	162500	135416	116071 (ee = 2)
*   3	101562	81250	67708	58035  (ee = 3)

* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
*  {58, 68, 81, 102, 116, 135, 162, 203, 232, 271, 325, 406, 464, 542, 650, 813};
******************************************************************************/
int set_rx_bw(int rx_bw, unsigned char RF_IDx) {
//  const unsigned int rx_table[16] = \
//    {813, 650, 542, 464, 406, 325, 271, 232, 203, 162, 135, 116, 102, 81, 68, 58};
//  unsigned char ee, mm, cc_chan_bw;
//  int best_fit;
//
//  for(ee=0; ee<4;ee++){
//    for(mm=0; mm<4;mm++){
//      best_fit = rx_table[(ee<<2)+mm];
//      if(rx_bw >= best_fit ){
//        cc_chan_bw = (ee<<6) + (mm<<4);
//        //CC_SPIWriteReg(CHAN_BW, cc_chan_bw , RF_IDx);
//         mm = 4;                                          // terminate for loop
//         ee = 4;                                          // terminate for loop
//      }
//    }
//  }
//  return best_fit;
  return rx_bw;
}

/******************************************************************************
* set_rx_bw: Set the output power of the CC430 by looking up in table
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
int set_ch_spc(int ch_spc, unsigned char RF_IDx){
//  long ch_spc_long;
//  unsigned int chan_e = 0;
//  unsigned int chan_m = 0;
//  int final_ch_spc;
//  unsigned char cc_MDMCFG1 = 0;
//
//  ch_spc_long = ch_spc;
//  ch_spc_long = ch_spc_long * 1000;
//  
//
//  chan_e = 0;
//  chan_m = (unsigned int)(ch_spc_long/99.18212891);
//  if(chan_m>256) {
//    chan_m = chan_m - 256;
//  } else {
//    chan_m = 0;
//  }
//  final_ch_spc = (int)((chan_m+256) * 0.09918212891);
//
//  if( ch_spc_long > 50781 ) {
//    chan_e = 1;
//    chan_m = (unsigned char)(ch_spc_long/198.3642578) - 256;
//    chan_m = chan_m & 0xFF;
//    final_ch_spc = (int)(((int)chan_m+256) * 0.1983642578);
//  }
//  if( ch_spc_long > 101562 ) {
//    chan_e = 2;
//    chan_m = (unsigned char)(ch_spc_long/396.7285156) - 256;
//    chan_m = chan_m & 0xFF;
//    final_ch_spc = (int)(((int)chan_m+256) * 0.3967285156);
//  }
//  if( ch_spc_long > 203125 ) {
//    chan_e = 3;
//    chan_m = (unsigned char)(ch_spc_long/793.4570313) - 256;
//    chan_m = chan_m & 0xFF;
//    final_ch_spc = (int)(((int)chan_m+256) * 0.7934570313);    
//  }
//
//   cc_MDMCFG1 = CC_SPIReadReg(MDMCFG1, RF_IDx);
//   cc_MDMCFG1 = (cc_MDMCFG1 & 0xFC) + chan_e;
//   CC_SPIWriteReg(MDMCFG1, cc_MDMCFG1 , RF_IDx);
//   CC_SPIWriteReg(MDMCFG0, chan_m , RF_IDx);
//   return final_ch_spc;
  return (ch_spc);
}


/******************************************************************************
* set_tx_pwr: Set the output power of the CC430 by looking up in table
*
*   PWR 868M I(mA) 915M I(mA)                  PWR 868M I(mA) 915M I(mA)
*   -30 0x03 12.0  0x03 11.9                   -20 0x17 12.6  0x0E 12.5
*   -15 0x1D 13.3  0x1E 13.3                   -10 0x26 14.5  0x27 14.8
*    -6 0x37 16.4  0x38 17.0                     0 0x50 16.8  0x8E 17.2
*     5 0x86 19.9  0x84 20.2                     7 0xCD 25.8  0xCC 25.7
*    10 0xC5 30.0  0xC3 30.7                 12/11 0xC0 34.2  0xC0 33.4
*
*   PWR 2.5G I(mA)
*  <55 0x00 8.4    30 0x50 9.9    28 0x44 9.7    -26 0xC0 10.2
*   24 0x84 10.1   22 0x81 10.0   20 0x46 10.1   18 0x93 11.7
*   16 0x55 10.8   14 0x8D 12.2   12 0xC6 11.1   10 0x97 12.2
*    8 0x6E 14.1    6 0x7F 15.0    4 0xA9 16.2    2 0xBB 17.7
*     0 0xFE 21.2    +1 0xFF 21.5
* 
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
signed char set_tx_pwr(signed char tx_pwr, char cc_device_id, unsigned char RF_IDx){
//  char ee;//  signed char actual_pwr;
//  const unsigned char paTable_CC1101[10] = 
//  {0x03, 0x0E, 0x1E, 0x27, 0x38, 0x8E, 0x84, 0xCC, 0xC3, 0xC0};
//  const signed char paOut_CC1101[10]     = \
//    {-30,  -20,  -15,  -10,   -6,    0,    5,    7,   10,   12};
//  const unsigned char paTable_CC2500[18] = \
//    {0x00, 0x50, 0x44, 0xC0, 0x84, 0x81, 0x46, 0x93, \
//      0x55, 0x8D, 0xC6, 0x97, 0x6E, 0x7F, 0xA9, 0xBB, 0xFE, 0xFF};
//  const signed char paOut_CC2500[18]     = \
//    { -55,  -30,  -28,  -26,  -24,  -22,  -20,  -18,\
//      -16,  -14,  -12,  -10,   -8,   -6,   -4,   -2,    0,    1};
//  
//  switch(cc_device_id) {
//  case DEV_CC2500:                    // CC2500 power table
//    for(ee=0; ee<18;ee++){
//      actual_pwr = paOut_CC2500[ee];
//      if(tx_pwr <= actual_pwr){
//        //CC_SPIWriteReg(PATABLE, paTable_CC2500[ee], RF_IDx);   //Write PATABLE
//        ee=18;                                             // terminate for loop
//      }
//    }
//    break;
//  case DEV_CC1100:                  // CC1100 power table
//    for(ee=0; ee<10;ee++){
//      actual_pwr = paOut_CC1101[ee];
//      if(tx_pwr <= actual_pwr){
//        //CC_SPIWriteReg(PATABLE, paTable_CC1101[ee], RF_IDx);     //Write PATABLE
//        ee=10;                                             // terminate for loop
//      }
//    }
//    break;
//  case DEV_CC1101:                 // CC1101 power table
//  for(ee=0; ee<10;ee++){
//      actual_pwr = paOut_CC1101[ee];
//      if(tx_pwr <= actual_pwr){
//        //CC_SPIWriteReg(PATABLE, paTable_CC1101[ee], RF_IDx);     //Write PATABLE
//        ee=10;                                             // terminate for loop
//      }
//    }
//  case DEV_CC430x:                 // CC1101 power table
//  for(ee=0; ee<10;ee++){
//      actual_pwr = paOut_CC1101[ee];
//      if(tx_pwr <= actual_pwr){
//        //CC_SPIWriteReg(PATABLE, paTable_CC1101[ee], RF_IDx);     //Write PATABLE
//        ee=10;                                             // terminate for loop
//      }
//    }
//
//    break;
//  default:
//    actual_pwr = -100;
//    break;
//  }
//  return actual_pwr;
  return tx_pwr;
}

/******************************************************************************
* set_tx_unmodulated_test_mode
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void set_tx_unmodulated_test_mode(unsigned char RF_IDx) {
  CC_SPIWriteExtReg(PA_CFG2,          0x3F, RF_IDx);  // infinite packet length mode
  CC_SPIWriteExtReg(SOFT_TX_DATA_CFG, 0x01, RF_IDx);  // infinite packet length mode
}

/******************************************************************************
* set_tx_modulated_test_mode
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void set_tx_modulated_test_mode(unsigned char RF_IDx) {
  CC_SPIWriteReg(PKT_CFG2         ,     0x05,RF_IDx);       // 
  CC_SPIWriteReg(PKT_CFG1         ,     0x00,RF_IDx);       // 
  CC_SPIWriteReg(PKT_CFG0         ,     0x20,RF_IDx);       // 
}

/******************************************************************************
* Get and calculate the RSSI from unsigned char value from Radio
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
int get_rssi(unsigned char RF_IDx) {
  int rssi;
  char cc_rssi;
  
  cc_rssi = CC_SPIReadExtReg(RSSI1, RF_IDx);  // no delay, due to printing

  if (cc_rssi >= 128) {
    rssi = 128 - cc_rssi;
  } else {
    rssi = cc_rssi;
  }
  return rssi; 
}


/******************************************************************************
* get_device_id 
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
char get_device_id(unsigned char RF_IDx) {
  char ret_partnum;
  char ret_version;
  ret_partnum = CC_SPIReadExtReg(PARTNUMBER, RF_IDx);      // Read the PARTNUM status
  ret_version = CC_SPIReadExtReg(PARTVERSION, RF_IDx);      // Read the Version status
  
  switch (ret_partnum) {
  case 0:
    if(ret_version == 0x04) {
      return DEV_CC1101;
    } 
    if(ret_version == 0x06) {
      return DEV_CC430x;
    }    
    if(ret_version == 0x00) {
      return DEV_CC1100;
    }
  case 128:
    if(ret_version == 0x03) {
      return DEV_CC2500;
    }
  case 0x58:    
    if(ret_version == 0x11) {
      return DEV_CC1125;
    }
 case 0x48:    
   if(ret_version == 0x10) {
     return DEV_CC1120;
   }

  default:
    break;
  }
  
  return DEV_UNKNOWN;
}

/******************************************************************************
* set_rf_packet_length
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
unsigned char set_rf_packet_length(unsigned char length, unsigned char RF_IDx) {
   CC_SPIWriteReg(PKT_LEN,  length, RF_IDx);          // Write to register
   CC_SPIWriteReg(PKT_CFG0,  0x00, RF_IDx);           // Write to register
    return (0);
}

/* ****************************************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR REPRESENTATIONS, 
* EITHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING ANY IMPLIED WARRANTIES OF 
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY 
* OR COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET POSSESSION, AND 
* NON-INFRINGEMENT OF ANY THIRD PARTY INTELLECTUAL PROPERTY RIGHTS WITH REGARD
* TO THE PROGRAM OR YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL OR 
* INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY THEORY OF LIABILITY AND WHETHER OR 
* NOT TI HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY 
* OUT OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. EXCLUDED 
* DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF REMOVAL OR REINSTALLATION, 
* COMPUTER TIME, LABOR COSTS, LOSS OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS,
* OR LOSS OF USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S AGGREGATE 
* LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF YOUR USE OF THE PROGRAM 
* EXCEED FIVE HUNDRED DOLLARS (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted by Texas 
* Instruments is distributed as "freeware".  You may, only under TI's copyright
* in the Program, use and modify the Program without any charge or restriction.  
* You may distribute to third parties, provided that you transfer a copy of this
* license to the third party and the third party agrees to these terms by its 
* first use of the Program. You must reproduce the copyright notice and any 
* other legend of ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains copyrighted material, 
* trade secrets and other TI proprietary information and is protected by 
* copyright laws, international copyright treaties, and trade secret laws, as
* well as other intellectual property laws.  To protect TI's rights in the 
* Program, you agree not to decompile, reverse engineer, disassemble or 
* otherwise translate any object code versions of the Program to a human-
* readable form.  You agree that in no event will you alter, remove or destroy
* any copyright notice included in the Program.  TI reserves all rights not 
* specifically granted under this license. Except as specifically provided 
* herein, nothing in this agreement shall be construed as conferring by 
* implication, estoppel, or otherwise, upon you, any license or other right
* under any TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
*******************************************************************************/
