//#############################################################################
//
// FILE:   cla_ex4_pwm_control.c
//
// TITLE:  Controlling PWM output using CLA
//
//! \addtogroup driver_example_list
//! <h1> Controlling PWM output using CLA </h1>
//!
//! This example showcases how to update PWM signal output using CLA.
//! EPWM1 is configured to generate complementary signals on both of its
//! channels of fixed frequency 100 KHz. EPWM4 is configured to trigger
//! a periodic CLA control task of frequency 10 KHz. The CLA task implements
//! a very simple logic to vary the duty of the EPWM1 outputs by increasing it
//! by 0.1 in every iteration and maintaining it in the range of 0.1-0.9. For
//! actual use-cases, the control logic could be modified to much more complex
//! depending upon the application. The other CLA task (CLA task 8) is
//! triggered by software at beginning to initialize the CLA global variables
//!
//! \b External \b Connections \n
//!  - Observe GPIO0 (EPWM1A) on oscilloscope
//!  - Observe GPIO1 (EPWM1B) on oscilloscope
//!
//! \b Watch \b Variables \n
//!  - duty
//!
//
//#############################################################################
// $TI Release: F28004x Support Library v1.10.00.00 $
// $Release Date: Tue May 26 17:06:03 IST 2020 $
// $Copyright:
// Copyright (C) 2020 Texas Instruments Incorporated - http://www.ti.com/
//
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions 
// are met:
// 
//   Redistributions of source code must retain the above copyright 
//   notice, this list of conditions and the following disclaimer.
// 
//   Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the 
//   documentation and/or other materials provided with the   
//   distribution.
// 
//   Neither the name of Texas Instruments Incorporated nor the names of
//   its contributors may be used to endorse or promote products derived
//   from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// $
//#############################################################################

//=================================Included Files===========================//
#include <cla_ex4_pwm_control_shared.h>
#include "driverlib.h"
#include "device.h"
#include "F28x_Project.h"
#include "DCLCLA.h"
#include "X45_control.h"
#include "math.h"
//###########################################################################//

// Linker Defined variables
extern uint32_t Cla1ProgRunStart, Cla1ProgLoadStart, Cla1ProgLoadSize;
extern uint32_t Cla1ConstRunStart, Cla1ConstLoadStart, Cla1ConstLoadSize;

// Function Prototypes
//
void initEPWM(void);
void initCLA(void);
void initTIMER(void);
void computeNotchFltrCoeff(float32_t Fs, float32_t notch_freq,float32_t c1, float32_t c2);//Fs, ISR Frequency
volatile unsigned int tt1,cycle_;

//#pragma DATA_SECTION(testtest2,"CpuToCla1MsgRAM");

//vBUS set point

//=============data mapped on CPU TO CLA MSGRAM,CPU read&write================//
#pragma SET_DATA_SECTION("CpuToCla1MsgRAM")
volatile float32_t vBusRef, vBusRefSlewed;//voltage loop set point
volatile float32_t vbus_sensed,ac_sensed,ibus_sensed;//value=sample_sum/sample_num *DC_PU_SCALE_FACTOR
volatile float32_t ton_final,ton_final_pre,toff_final_pre,toff_final,perid_final_pre,perid_final;
volatile float32_t vout_vin_pre=1.0f,vout_vin=1.0f;
volatile float32_t testtest2;//for test
volatile float32_t Notch_a1,Notch_a2,Notch_b0,Notch_b1,Notch_b2;//Notch compute result
enum enum_BuildLevel  buildLevel;
volatile float32_t threshold_PZC1,threshold_PZC2,threshold_NZC1,threshold_NZC2;
#pragma SET_DATA_SECTION()
//############################################################################//

//===============data mapped on CLA TO CPU MSGRAM,CPU read_only===============//
#pragma SET_DATA_SECTION("Cla1ToCpuMsgRAM")
volatile float32_t acSine,acSinePrev,acCosine,acTheta,acFo,accRrossover,acMAX,acRMS;//AC state
volatile uint16_t ac_sign_filtered,state_slew,noswitching,softtransition;
volatile float32_t gv_out,gi_out,ton_cal_adjusted;//voltage&current loop output,ton adjusted value
volatile uint16_t ton_zvs_adj_en,toff_zvs_adj_en;//ton,toff adjust enable
volatile DCL_PI_CLA gi, gv;//voltage&current loop calculation
volatile float32_t testtest;//for test
volatile enum enum_boardStatus boardStatus;
volatile enum enum_boardState boardState;
volatile enum enum_pwmSwState crossoverState;
volatile enum enum_interleavingState interleavingState;
volatile float32_t bus_cur_ref,bus_cur_ref_inst, bus_cur_ref_inst_prev;//current loop set point
#pragma SET_DATA_SECTION()
//############################################################################//

//===============ACL,ACN,VBUS,IBUS DATA collection============================//
volatile float32_t vbus_sample_sum,ac_sample_sum,ibus_sample_sum,sample_num;
volatile float32_t vbus_sensed_tempOLD,vbus_sensed_temp,ac_sensed_tempOLD,ac_sensed_temp,ibus_sensed_temp;
volatile uint16_t dead_band_en;
volatile float32_t DB_FED,DB_RED;
//############################################################################//
//for Notch compute
float32_t Notch_temp1,Notch_temp2,Notch_wn2,Notch_Ts;
volatile float32_t rad_gen=0,ac_gen;


//**********************************Main**************************************//
void main(void)
{
    //
    // Initialize device clock and peripherals
    //
    Device_init();
    //
    // Disable pin locks and enable internal pullups.
    //
    Device_initGPIO();
    //
    // GPIO0 is set to EPWM1A
    // GPIO1 is set to EPWM1B
    //
    GPIO_setMasterCore(0, GPIO_CORE_CPU1);
    GPIO_setPadConfig(0,GPIO_PIN_TYPE_STD);
    GPIO_setPinConfig(GPIO_0_EPWM1A);
    GPIO_setMasterCore(1, GPIO_CORE_CPU1);
    GPIO_setPadConfig(1,GPIO_PIN_TYPE_STD);
    GPIO_setPinConfig(GPIO_1_EPWM1B);

    GPIO_setDirectionMode(34,GPIO_DIR_MODE_OUT );
    GPIO_setMasterCore(34,GPIO_CORE_CPU1_CLA1);

    GPIO_setDirectionMode(23,GPIO_DIR_MODE_OUT );
    GPIO_setMasterCore(23,GPIO_CORE_CPU1_CLA1);

    // Initialize PIE and clear PIE registers. Disables CPU interrupts.
    //
    Interrupt_initModule();

    //
    // Initialize the PIE vector table with pointers to the shell Interrupt
    // Service Routines (ISR).
    //
    Interrupt_initVectorTable();

    //
    // Disable sync(Freeze clock to PWM as well)
    //
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    //
    // Initialize EPWM module
    //
    initEPWM();

    //
    // Setup the CLA
    //
    initCLA();

    initTIMER();

    //
    // Enable global interrupts.
    //
    EINT;

    //
    // Enable sync and clock to PWM
    //
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
    computeNotchFltrCoeff(10000.0f,100.0f,0.25f,0.00001f);
    CLA_forceTasks(CLA1_BASE, CLA_TASKFLAG_2);
    CLA_forceTasks(CLA1_BASE, CLA_TASKFLAG_3);

    while(1)
    {
//        duty+=0.1f;
    }
}
void globalVariablesInit(void)
{
    threshold_PZC1=PWM_SWITCHING_PZC1;
    threshold_PZC2=PWM_SWITCHING_PZC2;
    threshold_NZC1=PWM_SWITCHING_NZC1;
    threshold_NZC2=PWM_SWITCHING_NZC2;
}
void computeNotchFltrCoeff(float32_t Fs, float32_t notch_freq,float32_t c1, float32_t c2)
{
    Notch_Ts=1.0f/Fs;
    Notch_wn2=2*Fs*tanf(notch_freq*PI_VALUE*Notch_Ts);
    Notch_temp1=4*Fs*Fs + 4*  Notch_wn2 * c2 * Fs + Notch_wn2*Notch_wn2;
    Notch_temp2=1/ ( 4*Fs*Fs + 4* Notch_wn2 * c1 * Fs + Notch_wn2*Notch_wn2);
    Notch_b0 = Notch_temp1* Notch_temp2;
    Notch_b1 = (-8*Fs*Fs + 2* Notch_wn2* Notch_wn2)* Notch_temp2;
    Notch_b2 = (4*Fs*Fs-4*Notch_wn2*c2*Fs+Notch_wn2*Notch_wn2)*Notch_temp2;
    Notch_a1 = (-8*Fs*Fs + 2* Notch_wn2* Notch_wn2)*Notch_temp2;
    Notch_a2 = (4*Fs*Fs-4*Notch_wn2*c1*Fs+Notch_wn2*Notch_wn2)*Notch_temp2;
}

__interrupt void cpuTimer0ISR()
{
//__interrupt
    DINT;
    PieCtrlRegs.PIEIFR1.bit.INTx7=0;//clear timer0 interrupt
    PieCtrlRegs.PIEACK.bit.ACK1=1;//clean group1 regs

    PieCtrlRegs.PIEACK.all=0x0001;
//    tt1=EPwm1Regs.TBCTR;
    rad_gen+=PI_VALUE*2*50/50000;
    rad_gen=rad_gen>2*PI_VALUE?(rad_gen-2*PI_VALUE):rad_gen;
    ac_gen=sinf(rad_gen);
    ac_sensed=ac_gen*0.8f;
//    cycle_=EPwm1Regs.TBCTR-tt1;
    while(0)//(EPwm1Regs.TBCTR-tt1)<400)
    {
        NOP;
    }
    CLA_forceTasks(CLA1_BASE, CLA_TASKFLAG_3);
    EINT;
}

void initTIMER(void)//initialize timer to generate interrupt
{
    // timer0 200khz,trip ADCSOC.   timer1 400hz&timer2 10hz,for state machine
    EALLOW;
    CpuTimer0Regs.TIM.all=2000;
    CpuTimer0Regs.PRD.all=2000;
    CpuTimer0Regs.TCR.bit.TIE=1;//enable CPUTEMER interrupt
    CpuTimer0Regs.TCR.bit.TSS=0;//0:timer is running,1:timer is stopped
    //CpuTimer0Regs.TCR.bit.TRB=0;//0:ignored,1:TIM is loaded with PRD
    CpuTimer0Regs.TCR.bit.FREE=0;//0:continues to run,1:SOFT bit controls the emulation behavior
    //CpuTimer0Regs.TCR.bit.SOFT=1;//0:timer halts next rime TIM decrement,1:halts when TIM decremented to zero
    //CpuTimer0Regs.TPR.bit.TDDR=0;//Every (TDDRH:TDDR + 1) timer clock source cycles, the timer counter register (TIMH:TIM) decrements by one.
    //CpuTimer0Regs.TPRH.bit.TDDRH=0;//
    PieVectTable.TIMER0_INT = &cpuTimer0ISR;
    PieCtrlRegs.PIEIER1.bit.INTx7 = 1;
    EDIS;
    EINT;
    IER=1;
}

//
// EPWM Initialization
// EPWM1 : generates output of frequency EPWM1_FREQ (100 KHz)
// EPWM4 : Triggers CLA task at frequency EPWM4_FREQ (10 KHz)
//
void initEPWM(void)
{
    //
    // Set up EPWM1 to
    // - run on a base clock of SYSCLK
    // - have a period of EPWM1_PERIOD
    // - run in count up-down mode
    //
    EPWM_setClockPrescaler(EPWM1_BASE, EPWM_CLOCK_DIVIDER_1,
                           EPWM_HSCLOCK_DIVIDER_1);
    EPWM_setTimeBasePeriod(EPWM1_BASE, 65530);
    EPWM_setCounterCompareValue(EPWM1_BASE, EPWM_COUNTER_COMPARE_A, EPWM1_PERIOD/10U);
    EPWM_setTimeBaseCounterMode(EPWM1_BASE, EPWM_COUNTER_MODE_UP);
    EPWM_setTimeBaseCounter(EPWM1_BASE, 0U);

    //
    // Configuring action-qualifiers for EPWM1 to generate symmetric
    // and complementary outputs on channel A and B
    //
    EPWM_setActionQualifierAction(EPWM1_BASE, EPWM_AQ_OUTPUT_A,
                         EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(EPWM1_BASE, EPWM_AQ_OUTPUT_A,
                             EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
    EPWM_setActionQualifierAction(EPWM1_BASE, EPWM_AQ_OUTPUT_B,
                             EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(EPWM1_BASE, EPWM_AQ_OUTPUT_B,
                             EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);

    //
    // Enabling Counter Compare shadow mode
    //
    EPWM_setCounterCompareShadowLoadMode(EPWM1_BASE, EPWM_COUNTER_COMPARE_A, EPWM_COMP_LOAD_ON_SYNC_CNTR_ZERO);

    //
    // Set up EPWM4 to
    // - run on a base clock of SYSCLK
    // - have a period of EPWM4_PERIOD
    // - run in count up mode
    EPWM_setClockPrescaler(EPWM4_BASE, EPWM_CLOCK_DIVIDER_1,
                           EPWM_HSCLOCK_DIVIDER_1);
    EPWM_setTimeBasePeriod(EPWM4_BASE, EPWM4_PERIOD);
    EPWM_setTimeBaseCounterMode(EPWM4_BASE, EPWM_COUNTER_MODE_UP);
    EPWM_setTimeBaseCounter(EPWM4_BASE, 0U);

    //
    // Enabling EPWM4 interrupt at TBCTR = 0 to trigger
    // CLA task
    //
    EPWM_setInterruptSource(EPWM4_BASE, EPWM_INT_TBCTR_ZERO);
    EPWM_enableInterrupt(EPWM4_BASE);
    EPWM_setInterruptEventCount(EPWM4_BASE, 1U);

    //
    // EPWM 1 and 4 should run freely in emulation mode
    //
    EPWM_setEmulationMode(EPWM1_BASE, EPWM_EMULATION_FREE_RUN);
    EPWM_setEmulationMode(EPWM4_BASE, EPWM_EMULATION_FREE_RUN);
}

//
// CLA Initialization
//
// Description: This function will
// - copy over code and const from flash to CLA program and data ram
//   respectively
// - Initialize the task vectors (MVECTx)
// - setup each task's trigger
// - enable each individual task
// - map program and data spaces to the CLA
// - run any one-time initialization task
// Please note that the CLA can only run code and access data that is in RAM.
// the user must assign constants (tables) to FLASH, and copy them over to
// RAM at run-time. They must be copied to a RAM that lies in the address space
// of the CLA, and prior to giving the CLA control over that space
//
void initCLA(void)
{
    //
    // Copy the program and constants from FLASH to RAM before configuring
    // the CLA
    //
#if defined(_FLASH)
    memcpy((uint32_t *)&Cla1ProgRunStart, (uint32_t *)&Cla1ProgLoadStart,
        (uint32_t)&Cla1ProgLoadSize );
    memcpy((uint32_t *)&Cla1ConstRunStart, (uint32_t *)&Cla1ConstLoadStart,
        (uint32_t)&Cla1ConstLoadSize );
#endif //defined(_FLASH)

    //
    // CLA Program will reside in RAMLS2,3,4,5,6 and data in RAMLS1
    //
   /* MemCfg_setCLAMemType(MEMCFG_SECT_LS2, MEMCFG_CLA_MEM_PROGRAM);
    MemCfg_setCLAMemType(MEMCFG_SECT_LS3, MEMCFG_CLA_MEM_PROGRAM);
    MemCfg_setCLAMemType(MEMCFG_SECT_LS4, MEMCFG_CLA_MEM_PROGRAM);
    MemCfg_setCLAMemType(MEMCFG_SECT_LS5, MEMCFG_CLA_MEM_PROGRAM);
    MemCfg_setCLAMemType(MEMCFG_SECT_LS6, MEMCFG_CLA_MEM_PROGRAM);
    MemCfg_setCLAMemType(MEMCFG_SECT_LS1, MEMCFG_CLA_MEM_DATA);
    MemCfg_setLSRAMMasterSel(MEMCFG_SECT_LS0, MEMCFG_LSRAMMASTER_CPU_CLA1);
    MemCfg_setLSRAMMasterSel(MEMCFG_SECT_LS1, MEMCFG_LSRAMMASTER_CPU_CLA1);
    MemCfg_setLSRAMMasterSel(MEMCFG_SECT_LS2, MEMCFG_LSRAMMASTER_CPU_CLA1);
    MemCfg_setLSRAMMasterSel(MEMCFG_SECT_LS3, MEMCFG_LSRAMMASTER_CPU_CLA1);
    MemCfg_setLSRAMMasterSel(MEMCFG_SECT_LS4, MEMCFG_LSRAMMASTER_CPU_CLA1);
    MemCfg_setLSRAMMasterSel(MEMCFG_SECT_LS5, MEMCFG_LSRAMMASTER_CPU_CLA1);
    MemCfg_setLSRAMMasterSel(MEMCFG_SECT_LS6, MEMCFG_LSRAMMASTER_CPU_CLA1);
    */
    MemCfg_setCLAMemType(MEMCFG_SECT_LS0, MEMCFG_CLA_MEM_PROGRAM);
    MemCfg_setCLAMemType(MEMCFG_SECT_LS1, MEMCFG_CLA_MEM_DATA);
    MemCfg_setLSRAMMasterSel(MEMCFG_SECT_LS0, MEMCFG_LSRAMMASTER_CPU_CLA1);
    MemCfg_setLSRAMMasterSel(MEMCFG_SECT_LS1, MEMCFG_LSRAMMASTER_CPU_CLA1);

#if defined(_FLASH)
    //
    // In Flash config, constants are loaded in Flash and then copied to LS3.
    // CLA reads the constants from LS3.
    //
    MemCfg_setCLAMemType(MEMCFG_SECT_LS3, MEMCFG_CLA_MEM_DATA);
    MemCfg_setLSRAMMasterSel(MEMCFG_SECT_LS3, MEMCFG_LSRAMMASTER_CPU_CLA1);
#endif //defined(_FLASH)

//
// Suppressing #770-D conversion from pointer to smaller integer
// The CLA address range is 16 bits so the addresses passed to the MVECT
// registers will be in the lower 64KW address space. Turn the warning
// back on after the MVECTs are assigned addresses
//
#pragma diag_suppress=770

    //
    // Assign the task vectors and set the triggers for task 1
    // and 8
    //
    CLA_mapTaskVector(CLA1_BASE, CLA_MVECT_1, (uint16_t)&Cla1Task1);
    CLA_mapTaskVector(CLA1_BASE, CLA_MVECT_2, (uint16_t)&Cla1Task2);
    CLA_mapTaskVector(CLA1_BASE, CLA_MVECT_3, (uint16_t)&Cla1Task3);
    CLA_mapTaskVector(CLA1_BASE, CLA_MVECT_4, (uint16_t)&Cla1Task4);
    CLA_mapTaskVector(CLA1_BASE, CLA_MVECT_5, (uint16_t)&Cla1Task5);
//    CLA_mapTaskVector(CLA1_BASE, CLA_MVECT_8, (uint16_t)&Cla1Task8);
//    CLA_setTriggerSource(CLA_TASK_1, CLA_TRIGGER_EPWM4INT);
    CLA_setTriggerSource(CLA_TASK_8, CLA_TRIGGER_TINT0);
    // Enable Tasks 1,2,3,4,5,8
    CLA_enableTasks(CLA1_BASE, (CLA_TASKFLAG_1| CLA_TASKFLAG_2 |CLA_TASKFLAG_3|CLA_TASKFLAG_4|CLA_TASKFLAG_5));// | CLA_TASKFLAG_8));
    EALLOW;
    Cla1Regs._MCTLBGRND.bit.TRIGEN = 1U;
    Cla1Regs._MVECTBGRND = (uint16_t)&Cla1Task8;
//    DmaClaSrcSelRegs.CLA1TASKSRCSEL2.bit.TASK8 = 0U; //Software
#pragma diag_warning=770
    Cla1Regs._MCTLBGRND.bit.BGEN = 1U;

//    Cla1Regs._MCTLBGRND.bit.BGSTART = 1U;
    EDIS;
    // Force task 8, the one time initialization task
    CLA_forceTasks(CLA1_BASE, CLA_TASKFLAG_1);
//    CLA_enableBackgroundTask(CLA1_BASE);
//    CLA_startBackgroundTask(CLA1_BASE);
}
