//#############################################################################
//
//  FILE:  rampgen.h
//
//  TITLE: Ramp Generator Module
//
//#############################################################################
// $TI Release: Utilities Library v1.01.00.00 $
// $Release Date: Mon Jul 29 16:04:10 CDT 2019 $
// $Copyright:
// Copyright (C) 2019 Texas Instruments Incorporated - http://www.ti.com/
//
// ALL RIGHTS RESERVED
// $
//#############################################################################

#ifndef RAMPGEN_H
#define RAMPGEN_H

#ifdef __cplusplus
extern "C" {
#endif

//*****************************************************************************
//
//! \addtogroup RAMPGEN
//! @{
//
//*****************************************************************************

//
// Included Files
//
#include <stdint.h>
#ifndef __TMS320C28XX_CLA__
#include <math.h>
#else
#include <CLAmath.h>
#endif

//#############################################################################
//
// Macro Definitions
//
//#############################################################################
#ifndef C2000_IEEE754_TYPES
#define C2000_IEEE754_TYPES
#ifdef __TI_EABI__
typedef float         float32_t;
typedef double        float64_t;
#else // TI COFF
typedef float         float32_t;
typedef long double   float64_t;
#endif // __TI_EABI__
#endif // C2000_IEEE754_TYPES

//
// Typedefs
//

//! \brief          Defines the Ramp Signal Generator (RAMPGEN) structure
//!
//! \details        The rampgen can be used to generate a ramp signal 
//!                 of a desired frequency
//!
typedef volatile struct {
    float32_t  freq;        // Input: Ramp frequency (pu)
    float32_t  stepAngleMax;// Parameter: Maximum step angle (pu)
    float32_t  out;         // Output: Ramp signal (pu)
} RAMPGEN;

//! \brief      resets RAMPGEN internal storage data and outputs
//! \param v    a pointer to a RAMPGEN structure
//!
static inline void RAMPGEN_reset(RAMPGEN *v)
{
    v->out=0;
}

//! \brief      configures RAMPGEN module
//! \param v    a pointer to a RAMPGEN structure
//! \param isrFrequency Frequency at which the RAMGEN is called in ISR
//! \param rampFrequency Frequency of the RAMP generated by RAMPGEN module
//!
static inline void RAMPGEN_config(RAMPGEN *v,
                                 float32_t isrFrequency,
                                 float32_t rampFrequency)
{
    v->freq=rampFrequency;
    v->stepAngleMax=((float32_t)1.0)/isrFrequency;
}

//! \brief      Run RAMPGEN module
//! \param v    a pointer to a RAMPGEN structure
//!
static inline void RAMPGEN_run(RAMPGEN *v)
{
    //
    // Compute the angle rate
    //
    v->out += (v->stepAngleMax*v->freq);
    
    //
    // Saturate the angle rate within (0,1)
    //
    if (v->out>(1.0))
    {
        v->out -= (1.0);
    }
}

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************
#ifdef __cplusplus
}
#endif // extern "C"


#endif // end of  _RAMPGEN_H_ definition

//
// End of File
//

