//#############################################################################
// FILE:            hv1phdcac_board.c
//
// System Name:     TIDM_HV_1PH_DCAC R3
//
// Target:			F2837xD
//
// Author:			Manish Bhardwaj
//
// Description:	    This file consists of board related initialization, this 
//                  file is used to make the main file more readable
//
//#############################################################################
// $TI Release: HV_1PH_DCAC v3.00.04.00 $
// $Release Date: Thu Jan 26 15:57:53 CST 2023 $
// $Copyright:
// Copyright (C) 2023 Texas Instruments Incorporated - http://www.ti.com/
//
// ALL RIGHTS RESERVED
// $
//#############################################################################

//
// Included Files
//
#include "hv1phdcac_board.h"

/*  This routine sets up the basic device configuration such as initializing PLL
    copying code from FLASH to RAM */
//TODO device_setup()
void setupDevice(void)
{
	//
	// Initialize device clock and peripherals
	//
	Device_init();

	//
	// Disable pin locks and enable internal pull-ups.
	//
	Device_initGPIO();

	//
	// Initialize PIE and clear PIE registers. Disables CPU interrupts.
	//
	Interrupt_initModule();

	//
	// Initialize the PIE vector table with pointers to the shell Interrupt
	// Service Routines (ISR).
	//
	Interrupt_initVectorTable();


	// initialize CPU timers

	// Initialize timer period to maximum
	CPUTimer_setPeriod(CPUTIMER0_BASE, DEVICE_SYSCLK_FREQ/100 ); // 100Hz - > 10ms
	// Initialize timer period to maximum
	CPUTimer_setPeriod(CPUTIMER1_BASE, DEVICE_SYSCLK_FREQ/50 ); // 50Hz -> 20ms
	// Initialize timer period to maximum
	CPUTimer_setPeriod(CPUTIMER2_BASE, DEVICE_SYSCLK_FREQ/10 ); // 10Hz -> 100ms
	// Initialize pre-scale counter to divide by 1 (SYSCLKOUT)
	CPUTimer_setPreScaler(CPUTIMER0_BASE, 0);
	// Initialize pre-scale counter to divide by 1 (SYSCLKOUT)
	CPUTimer_setPreScaler(CPUTIMER1_BASE, 0);
	// Initialize pre-scale counter to divide by 1 (SYSCLKOUT)
	CPUTimer_setPreScaler(CPUTIMER2_BASE, 0);
    // Make sure timer is stopped
    CPUTimer_stopTimer(CPUTIMER0_BASE);
    CPUTimer_stopTimer(CPUTIMER1_BASE);
    CPUTimer_stopTimer(CPUTIMER2_BASE);
    CPUTimer_setEmulationMode(CPUTIMER0_BASE,
                              CPUTIMER_EMULATIONMODE_STOPAFTERNEXTDECREMENT);
    CPUTimer_setEmulationMode(CPUTIMER1_BASE,
                              CPUTIMER_EMULATIONMODE_STOPAFTERNEXTDECREMENT);
    CPUTimer_setEmulationMode(CPUTIMER2_BASE,
                              CPUTIMER_EMULATIONMODE_STOPAFTERNEXTDECREMENT);
    // Reload all counter register with period value
    CPUTimer_reloadTimerCounter(CPUTIMER0_BASE);
    CPUTimer_reloadTimerCounter(CPUTIMER1_BASE);
    CPUTimer_reloadTimerCounter(CPUTIMER2_BASE);

    CPUTimer_resumeTimer(CPUTIMER0_BASE);
    CPUTimer_resumeTimer(CPUTIMER1_BASE);
    CPUTimer_resumeTimer(CPUTIMER2_BASE);
}

//TODO setupADC()
void setupADC(void)
{
	//Write ADC configurations and power up the ADC for both ADC A
	uint16_t i;

	//write configurations for ADC-A

	ADC_setPrescaler(ADCA_BASE, ADC_CLK_DIV_4_0); //set ADCCLK divider to /4
	ADC_setMode(ADCA_BASE, ADC_RESOLUTION_12BIT, ADC_MODE_SINGLE_ENDED);

	//Set pulse positions to late
	ADC_setInterruptPulseMode(ADCA_BASE, ADC_PULSE_END_OF_CONV);

	//power up the ADC
	ADC_enableConverter(ADCA_BASE);

	//write configurations for ADC-B

    ADC_setPrescaler(ADCB_BASE, ADC_CLK_DIV_4_0); //set ADCCLK divider to /4
    ADC_setMode(ADCB_BASE, ADC_RESOLUTION_12BIT, ADC_MODE_SINGLE_ENDED);

    //Set pulse positions to late
    ADC_setInterruptPulseMode(ADCB_BASE, ADC_PULSE_END_OF_CONV);

    //power up the ADC
    ADC_enableConverter(ADCB_BASE);

    //write configurations for ADC-C
    // External REFERENCE must be provided
    ADC_setPrescaler(ADCC_BASE, ADC_CLK_DIV_4_0);
    ADC_setMode(ADCC_BASE, ADC_RESOLUTION_12BIT, ADC_MODE_SINGLE_ENDED);

    //Set pulse positions to late
    ADC_setInterruptPulseMode(ADCC_BASE, ADC_PULSE_END_OF_CONV);

    //power up the ADC
    ADC_enableConverter(ADCC_BASE);

    //write configurations for ADC-D
    // External REFERENCE must be provided
    ADC_setPrescaler(ADCD_BASE, ADC_CLK_DIV_4_0);
    ADC_setMode(ADCD_BASE, ADC_RESOLUTION_12BIT, ADC_MODE_SINGLE_ENDED);

    //Set pulse positions to late
    ADC_setInterruptPulseMode(ADCD_BASE, ADC_PULSE_END_OF_CONV);

    //power up the ADC
    ADC_enableConverter(ADCD_BASE);

	//delay for > 1ms to allow ADC time to power up
	for(i = 0; i < 1000; i++){
	asm("   RPT#255 || NOP");
	}

    ADC_setupSOC(II_ADC_BASE, II_1_ADC_SOC_NO, II_ADC_TRIG_SOURCE,
                 II_ADC_PIN, ACQPS_SYS_CLKS);
    ADC_setupSOC(II_ADC_BASE, II_2_ADC_SOC_NO, II_ADC_TRIG_SOURCE,
                 II_ADC_PIN, ACQPS_SYS_CLKS);
    ADC_setupSOC(II_ADC_BASE, II_3_ADC_SOC_NO, II_ADC_TRIG_SOURCE,
                 II_ADC_PIN, ACQPS_SYS_CLKS);
    ADC_setupSOC(II_ADC_BASE, II_4_ADC_SOC_NO, II_ADC_TRIG_SOURCE,
                 II_ADC_PIN, ACQPS_SYS_CLKS);
    ADC_setupSOC(VO_ADC_BASE, VO_ADC_SOC_NO, VO_ADC_TRIG_SOURCE,
                 VO_ADC_PIN, ACQPS_SYS_CLKS);
    ADC_setupSOC(VBUS_ADC_BASE, VBUS_ADC_SOC_NO, VBUS_ADC_TRIG_SOURCE,
                 VBUS_ADC_PIN, ACQPS_SYS_CLKS);
    ADC_setupSOC(IREF_ADC_BASE, IREF_ADC_SOC_NO, IREF_ADC_TRIG_SOURCE,
                 IREF_ADC_PIN, ACQPS_SYS_CLKS);
}

//TODO setupSDFM()
void setupSDFM(uint16_t HLT, uint16_t LLT, uint16_t PWM_ticks, 
               uint16_t PWM_ticks_in_sdfm_osr, uint16_t SD_clk_ecap_sys_ticks, 
               uint16_t sdfmosr)
{

	// setup CAP to generate CLK for SDFM
	configureECAPforSDFMClk(SD_clk_ecap_sys_ticks);

	// setup PWM 11 for syncing up the SD filter windows with motor control PWMs
	configurePWMsyncforSDFM(PWM_ticks,PWM_ticks_in_sdfm_osr);

	//
	// Create a subroutine to encompass all the repeated lines below (***investigate code size, speed, etc.)
	// Updated sdfm base register namings, and filter names
	//

	// Setup GPIO for SD
	//SD Filter 1 - Data 1
	GPIO_setPadConfig(GPIO_IF_SDFM_DATA, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(GPIO_IF_SDFM_DATA,GPIO_QUAL_SYNC);
    GPIO_setDirectionMode(GPIO_IF_SDFM_DATA, GPIO_DIR_MODE_IN);
    GPIO_setPinConfig(GPIO_IF_SDFM_DATA_PIN_CONFIG);

    //SD Filter 1 - Clock 1
    GPIO_setPadConfig(GPIO_IF_SDFM_CLOCK, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(GPIO_IF_SDFM_CLOCK,GPIO_QUAL_SYNC);
    GPIO_setDirectionMode(GPIO_IF_SDFM_CLOCK, GPIO_DIR_MODE_IN);
    GPIO_setPinConfig(GPIO_IF_SDFM_CLOCK_PIN_CONFIG);

	//SD Filter 2 - Data 2
    GPIO_setPadConfig(GPIO_IO_SDFM_DATA, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(GPIO_IO_SDFM_DATA,GPIO_QUAL_SYNC);
    GPIO_setDirectionMode(GPIO_IO_SDFM_DATA, GPIO_DIR_MODE_IN);
    GPIO_setPinConfig(GPIO_IO_SDFM_DATA_PIN_CONFIG);

    //SD Filter 2 - Clock 2
    GPIO_setPadConfig(GPIO_IO_SDFM_CLOCK, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(GPIO_IO_SDFM_CLOCK, GPIO_QUAL_SYNC);
    GPIO_setDirectionMode(GPIO_IO_SDFM_CLOCK, GPIO_DIR_MODE_IN);
    GPIO_setPinConfig(GPIO_IO_SDFM_CLOCK_PIN_CONFIG);

	//SD Filter 3 - Data 3
    GPIO_setPadConfig(GPIO_VO_SDFM_DATA, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(GPIO_VO_SDFM_DATA,GPIO_QUAL_SYNC);
    GPIO_setDirectionMode(GPIO_VO_SDFM_DATA, GPIO_DIR_MODE_IN);
    GPIO_setPinConfig(GPIO_VO_SDFM_DATA_PIN_CONFIG);

    //SD Filter 3 - Clock 3
    GPIO_setPadConfig(GPIO_VO_SDFM_CLOCK, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(GPIO_VO_SDFM_CLOCK, GPIO_QUAL_SYNC);
    GPIO_setDirectionMode(GPIO_VO_SDFM_CLOCK, GPIO_DIR_MODE_IN);
    GPIO_setPinConfig(GPIO_VO_SDFM_CLOCK_PIN_CONFIG);

	//SD Filter 4 - Data 4 (bus voltage)
    GPIO_setPadConfig(GPIO_VBUS_SDFM_DATA, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(GPIO_VBUS_SDFM_DATA,GPIO_QUAL_SYNC);
    GPIO_setDirectionMode(GPIO_VBUS_SDFM_DATA, GPIO_DIR_MODE_IN);
    GPIO_setPinConfig(GPIO_VBUS_SDFM_DATA_PIN_CONFIG);

    //SD Filter 4 - Clock 4
    GPIO_setPadConfig(GPIO_VBUS_SDFM_CLOCK, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(GPIO_VBUS_SDFM_CLOCK, GPIO_QUAL_SYNC);
    GPIO_setDirectionMode(GPIO_VBUS_SDFM_CLOCK, GPIO_DIR_MODE_IN);
    GPIO_setPinConfig(GPIO_VBUS_SDFM_CLOCK_PIN_CONFIG);


	/*******************************************************/
	/* Input Control Module */
	/*******************************************************/
	//Configure Input Control Mode: Modulator Clock rate = Modulator data rate
    SDFM_setupModulatorClock(IF_SDFM_BASE, IF_SDFM_FILTER,
                             SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(IO_SDFM_BASE, IO_SDFM_FILTER,
                             SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(VO_SDFM_BASE, VO_SDFM_FILTER,
                             SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(VBUS_SDFM_BASE, VBUS_SDFM_FILTER,
                             SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);

	SDFM_enableFilter(IF_SDFM_BASE, IF_SDFM_FILTER);
	SDFM_enableFilter(IO_SDFM_BASE, IO_SDFM_FILTER);
	SDFM_enableFilter(VO_SDFM_BASE, VO_SDFM_FILTER);
	SDFM_enableFilter(VBUS_SDFM_BASE, VBUS_SDFM_FILTER);

	/*******************************************************/
	/* Sinc filter Module */
	/*******************************************************/
	//Configure Data filter modules filter type, OSR value and enable / disable data filter
	// 16 bit data representation is chosen for OSR 128 using Sinc3, from the table in the TRM
	// the max value represented for OSR 128 using sinc 3 is +/-2097152 i.e. 2^21
	// the max value represented for OSR 64 using sinc 3 is +/-262144 i.e. 2^18
	// to represent this in 16 bit format where the first bit is sign shift by 6 bits
	SDFM_setFilterType(IF_SDFM_BASE, IF_SDFM_FILTER, SDFM_FILTER_SINC_3);
	SDFM_setFilterType(IO_SDFM_BASE, IO_SDFM_FILTER, SDFM_FILTER_SINC_3);
	SDFM_setFilterType(VO_SDFM_BASE, VO_SDFM_FILTER, SDFM_FILTER_SINC_3);
	SDFM_setFilterType(VBUS_SDFM_BASE, VBUS_SDFM_FILTER, SDFM_FILTER_SINC_3);

	SDFM_setOutputDataFormat(IF_SDFM_BASE, IF_SDFM_FILTER, SDFM_DATA_FORMAT_32_BIT);
	SDFM_setOutputDataFormat(IO_SDFM_BASE, IO_SDFM_FILTER, SDFM_DATA_FORMAT_32_BIT);
	SDFM_setOutputDataFormat(VO_SDFM_BASE, VO_SDFM_FILTER, SDFM_DATA_FORMAT_32_BIT);
	SDFM_setOutputDataFormat(VBUS_SDFM_BASE, VBUS_SDFM_FILTER, SDFM_DATA_FORMAT_32_BIT);

    SDFM_setFilterOverSamplingRatio(IF_SDFM_BASE, IF_SDFM_FILTER, sdfmosr-1);
    SDFM_setFilterOverSamplingRatio(IO_SDFM_BASE, IO_SDFM_FILTER, sdfmosr-1);
    SDFM_setFilterOverSamplingRatio(VO_SDFM_BASE, VO_SDFM_FILTER, sdfmosr-1);
    SDFM_setFilterOverSamplingRatio(VBUS_SDFM_BASE, VBUS_SDFM_FILTER, sdfmosr-1);

	/*******************************************************/
	/* Comparator Module */
	/*******************************************************/
	//Comparator HLT and LLT
	//Configure Comparator module's comparator filter type and comparator's OSR value, higher threshold, lower threshold

	SDFM_setComparatorFilterType(IF_SDFM_BASE,IF_SDFM_FILTER,SDFM_FILTER_SINC_3);
	SDFM_setCompFilterOverSamplingRatio(IF_SDFM_BASE,IF_SDFM_FILTER,31);
	SDFM_setCompFilterHighThreshold(IF_SDFM_BASE,IF_SDFM_FILTER,HLT);
	SDFM_setCompFilterLowThreshold(IF_SDFM_BASE,IF_SDFM_FILTER,LLT);

	SDFM_setComparatorFilterType(IO_SDFM_BASE,IO_SDFM_FILTER,SDFM_FILTER_SINC_3);
	SDFM_setCompFilterOverSamplingRatio(IO_SDFM_BASE,IO_SDFM_FILTER,31);
	SDFM_setCompFilterHighThreshold(IO_SDFM_BASE,IO_SDFM_FILTER,24576);
	SDFM_setCompFilterLowThreshold(IO_SDFM_BASE,IO_SDFM_FILTER,8192);

	SDFM_setComparatorFilterType(VO_SDFM_BASE,VO_SDFM_FILTER,SDFM_FILTER_SINC_3);
	SDFM_setCompFilterOverSamplingRatio(VO_SDFM_BASE,VO_SDFM_FILTER,31);
	SDFM_setCompFilterHighThreshold(VO_SDFM_BASE,VO_SDFM_FILTER,HLT);
	SDFM_setCompFilterLowThreshold(VO_SDFM_BASE,VO_SDFM_FILTER,LLT);

	SDFM_setComparatorFilterType(VBUS_SDFM_BASE,VBUS_SDFM_FILTER,SDFM_FILTER_SINC_3);
	SDFM_setCompFilterOverSamplingRatio(VBUS_SDFM_BASE,VBUS_SDFM_FILTER,31);
	SDFM_setCompFilterHighThreshold(VBUS_SDFM_BASE,VBUS_SDFM_FILTER,HLT);
	SDFM_setCompFilterLowThreshold(VBUS_SDFM_BASE,VBUS_SDFM_FILTER,LLT);

	/*******************************************************/
	/* EXTERNAL RESET  */
	/*******************************************************/
	//PWM11.CMPC, PWM11.CMPD, PWM12.CMPC and PWM12.CMPD signals cannot synchronize the filters. This option is not being used in this example.
	 SDFM_enableExternalReset(IF_SDFM_BASE,IF_SDFM_FILTER);
	 SDFM_enableExternalReset(IO_SDFM_BASE,IO_SDFM_FILTER);
	 SDFM_enableExternalReset(VO_SDFM_BASE,VO_SDFM_FILTER);
	 SDFM_enableExternalReset(VBUS_SDFM_BASE,VBUS_SDFM_FILTER);

	/*******************************************************/
	/* Enable interrupts */
	/*******************************************************/
	//Following SDFM interrupts can be enabled / disabled using this function.
	//	Enable / disable comparator high threshold
	//  Enable / disable comparator low threshold
	//  Enable / disable modulator clock failure
	//  Enable / disable filter acknowledge
	SDFM_enableInterrupt(IF_SDFM_BASE, IF_SDFM_FILTER,SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT);
	SDFM_enableInterrupt(IO_SDFM_BASE, IO_SDFM_FILTER,SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT);
	SDFM_enableInterrupt(VO_SDFM_BASE, VO_SDFM_FILTER,SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT);
	SDFM_enableInterrupt(VBUS_SDFM_BASE, VBUS_SDFM_FILTER,SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT);

	SDFM_enableInterrupt(IF_SDFM_BASE, IF_SDFM_FILTER,SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT);
	SDFM_enableInterrupt(IO_SDFM_BASE, IO_SDFM_FILTER,SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT);
	SDFM_enableInterrupt(VO_SDFM_BASE, VO_SDFM_FILTER,SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT);
	SDFM_enableInterrupt(VBUS_SDFM_BASE, VBUS_SDFM_FILTER,SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT);

	// Enable master filter bit of the SDFM module 1
	SDFM_enableMasterFilter(IF_SDFM_BASE);
	SDFM_enableMasterInterrupt(IF_SDFM_BASE);
	SDFM_enableMasterFilter(IO_SDFM_BASE);
	SDFM_enableMasterInterrupt(IO_SDFM_BASE);
	SDFM_enableMasterFilter(VO_SDFM_BASE);
	SDFM_enableMasterInterrupt(VO_SDFM_BASE);
	SDFM_enableMasterFilter(VBUS_SDFM_BASE);
	SDFM_enableMasterInterrupt(VBUS_SDFM_BASE);
}


//TODO configureECAPforSDFMClk()
void configureECAPforSDFMClk(uint16_t SD_clk_ecap_sys_ticks)
{
	//Use CAP as source for the SD Clock
	// PWM3A -> OUTPUTXBAR3 -> GPIO4
	// PWM4A -> OUTPUTXBAR4 -> GPIO6


	// Select ECAP1.OUT on Mux0
	XBAR_setOutputMuxConfig(XBAR_OUTPUT3, XBAR_OUT_MUX00_ECAP1_OUT);
	// Enable MUX0 for ECAP1.OUT
	XBAR_enableOutputMux(XBAR_OUTPUT3, XBAR_MUX00);

	// Select ECAP1.OUT on Mux0
	XBAR_setOutputMuxConfig(XBAR_OUTPUT4, XBAR_OUT_MUX00_ECAP1_OUT);
	// Enable MUX0 for ECAP1.OUT
	XBAR_enableOutputMux(XBAR_OUTPUT4, XBAR_MUX00);


    GPIO_setDirectionMode(GPIO_XBAR1,GPIO_DIR_MODE_OUT);
    GPIO_setQualificationMode(GPIO_XBAR1,GPIO_QUAL_ASYNC);
    GPIO_setPinConfig(GPIO_XBAR1_PIN_CONFIG);


    GPIO_setDirectionMode(GPIO_XBAR2,GPIO_DIR_MODE_OUT);
    GPIO_setQualificationMode(GPIO_XBAR2,GPIO_QUAL_ASYNC);
    GPIO_setPinConfig(GPIO_XBAR2_PIN_CONFIG);


	// Setup APWM mode on CAP1, set period and compare registers
	ECAP_enableAPWMMode(SDFM_CLK_ECAP1_BASE);	// Enable APWM mode
	ECAP_setAPWMPeriod(SDFM_CLK_ECAP1_BASE, SD_clk_ecap_sys_ticks-1);
	ECAP_setAPWMCompare(SDFM_CLK_ECAP1_BASE, SD_clk_ecap_sys_ticks>>1);

	// set next duty cycle to 50%
	ECAP_setAPWMShadowPeriod(SDFM_CLK_ECAP1_BASE,SD_clk_ecap_sys_ticks-1);
	ECAP_setAPWMShadowCompare(SDFM_CLK_ECAP1_BASE,SD_clk_ecap_sys_ticks>>1);

	ECAP_clearInterrupt(SDFM_CLK_ECAP1_BASE,0xFF);
	ECAP_clearGlobalInterrupt(SDFM_CLK_ECAP1_BASE);

	// Start counters
	ECAP_startCounter(SDFM_CLK_ECAP1_BASE);

}

//TODO configurePWMsyncforSDFM()
void configurePWMsyncforSDFM(uint16_t PWM_ticks, uint16_t PWM_ticks_in_sdfm_osr)
{
    configurePWM1chUpCnt(EPWM11_BASE,PWM_ticks);

	EPWM_setSyncOutPulseMode(EPWM3_BASE,EPWM_SYNC_OUT_PULSE_ON_SOFTWARE);
	EPWM_setCountModeAfterSync(EPWM3_BASE, EPWM_COUNT_MODE_UP_AFTER_SYNC);
	EPWM_enablePhaseShiftLoad(EPWM3_BASE);

	EPWM_setPhaseShift(EPWM3_BASE,2);

	EPWM_setSyncOutPulseMode(EPWM4_BASE,EPWM_SYNC_OUT_PULSE_ON_SOFTWARE);
	EPWM_setSyncOutPulseMode(EPWM5_BASE,EPWM_SYNC_OUT_PULSE_ON_SOFTWARE);

	EPWM_setSyncOutPulseMode(EPWM10_BASE,EPWM_SYNC_OUT_PULSE_ON_SOFTWARE);

	EPWM_setPhaseShift(EPWM11_BASE,2);
	EPWM_enablePhaseShiftLoad(EPWM11_BASE);
	EPWM_setCountModeAfterSync(EPWM11_BASE, EPWM_COUNT_MODE_UP_AFTER_SYNC);

	EPWM_setCounterCompareValue(EPWM11_BASE, EPWM_COUNTER_COMPARE_C,
	                            ((EPWM_getTimeBasePeriod(EPWM11_BASE))>>1));

	EPWM_setCounterCompareValue(EPWM11_BASE, EPWM_COUNTER_COMPARE_D,
	                                    ((EPWM_getTimeBasePeriod(EPWM11_BASE))>>1));

	EPWM_setCounterCompareValue(EPWM11_BASE, EPWM_COUNTER_COMPARE_A,
	                                    ((EPWM_getTimeBasePeriod(EPWM11_BASE))>>1) +( PWM_ticks_in_sdfm_osr *3 +10));
}


//TODO configurePWM1chUpDwnCnt()
void configurePWM1chUpDwnCnt(uint32_t base1, int16_t period, int16_t mode, int16_t phase, int16_t db_red, int16_t db_fed,int16_t phase_dir)
{
	// Time Base SubModule Registers
	EPWM_setPeriodLoadMode(base1,EPWM_PERIOD_SHADOW_LOAD);
	EPWM_setTimeBasePeriod(base1,period>>1);
	EPWM_setTimeBaseCounter(base1,0);
	EPWM_setPhaseShift(base1,0);
	EPWM_setTimeBaseCounterMode(base1,EPWM_COUNTER_MODE_UP_DOWN);
	EPWM_setClockPrescaler(base1,EPWM_CLOCK_DIVIDER_1,EPWM_HSCLOCK_DIVIDER_1);

	if(mode == 1) // config as a Master
	{
		EPWM_disablePhaseShiftLoad(base1);
		EPWM_setSyncOutPulseMode(base1,EPWM_SYNC_OUT_PULSE_ON_COUNTER_ZERO);
	}
	if(mode == 0) // config as a Slave (Note: Phase+2 value used to compensate for logic delay)
	{
		EPWM_enablePhaseShiftLoad(base1);

		if(phase_dir>0)
		{
			EPWM_setPhaseShift(base1,phase);
			EPWM_setCountModeAfterSync(base1, EPWM_COUNT_MODE_UP_AFTER_SYNC);
		}
		else
		{
			EPWM_setPhaseShift(base1,phase);
			EPWM_setCountModeAfterSync(base1, EPWM_COUNT_MODE_DOWN_AFTER_SYNC);
		}


		EPWM_setSyncOutPulseMode(base1,EPWM_SYNC_OUT_PULSE_ON_SOFTWARE);

	}

	// Counter Compare Submodule Registers
	EPWM_setCounterCompareValue(base1,EPWM_COUNTER_COMPARE_A,0); // set duty 0% initially
	EPWM_setCounterCompareShadowLoadMode(base1,EPWM_COUNTER_COMPARE_A,EPWM_COMP_LOAD_ON_CNTR_ZERO_PERIOD);


	// Action Qualifier SubModule Registers
	// HWREGH(base1 + EPWM_O_AQCTLA) =0 ;  ***Delete after test
	EPWM_setActionQualifierAction(base1, EPWM_AQ_OUTPUT_A ,
							EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
	EPWM_setActionQualifierAction(base1, EPWM_AQ_OUTPUT_A ,
								EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);

	// Active high complementary PWMs - Set up the deadband
	EPWM_setDeadBandCounterClock(base1,EPWM_DB_COUNTER_CLOCK_FULL_CYCLE);
	EPWM_setRisingEdgeDelayCount(base1,db_red);
	EPWM_setFallingEdgeDelayCount(base1,db_fed);
	EPWM_setDeadBandDelayMode(base1,EPWM_DB_RED,true);
	EPWM_setDeadBandDelayMode(base1,EPWM_DB_FED,true);
	EPWM_setRisingEdgeDeadBandDelayInput(base1,EPWM_DB_INPUT_EPWMA);
	EPWM_setFallingEdgeDeadBandDelayInput(base1,EPWM_DB_INPUT_EPWMA);
	EPWM_setDeadBandDelayPolarity(base1,EPWM_DB_FED,EPWM_DB_POLARITY_ACTIVE_LOW);
	EPWM_setDeadBandDelayPolarity(base1,EPWM_DB_RED,EPWM_DB_POLARITY_ACTIVE_HIGH);

}

//TODO configurePWM1chUpCnt()
void configurePWM1chUpCnt(uint32_t base1, uint16_t period)
{
	// Time Base SubModule Registers
	EPWM_setPeriodLoadMode(base1,EPWM_PERIOD_DIRECT_LOAD);
	EPWM_setTimeBasePeriod(base1,period-1);
	EPWM_setTimeBaseCounter(base1,0);
	EPWM_setPhaseShift(base1,0);
	EPWM_setTimeBaseCounterMode(base1,EPWM_COUNTER_MODE_UP);
	EPWM_setClockPrescaler(base1,EPWM_CLOCK_DIVIDER_1,EPWM_HSCLOCK_DIVIDER_1);

	// configure PWM 1 as master and Phase 2 as slaves and let it pass the sync in pulse from PWM1
	EPWM_disablePhaseShiftLoad(base1);
	EPWM_setSyncOutPulseMode(base1,EPWM_SYNC_OUT_PULSE_ON_COUNTER_ZERO);

	// Counter Compare Submodule Registers
	EPWM_setCounterCompareValue(base1,EPWM_COUNTER_COMPARE_A,0); // set duty 0% initially
	EPWM_setCounterCompareShadowLoadMode(base1,EPWM_COUNTER_COMPARE_A,EPWM_COMP_LOAD_ON_CNTR_ZERO);

	// Action Qualifier SubModule Registers
	// HWREGH(base1 + EPWM_O_AQCTLA) =0 ;
	EPWM_setActionQualifierAction(base1, EPWM_AQ_OUTPUT_A ,
							EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
	EPWM_setActionQualifierAction(base1, EPWM_AQ_OUTPUT_A ,
								EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
}

//TODO setupCMPSS
void setupCMPSS(uint32_t base1, float32_t current_limit, float32_t current_max_sense )
{
    //Enable CMPSS1
    CMPSS_enableModule(base1);

    //Use VDDA as the reference for comparator DACs
    CMPSS_configDAC(base1, CMPSS_DACVAL_SYSCLK | CMPSS_DACREF_VDDA | CMPSS_DACSRC_SHDW);

    //Set DAC to H~75% and L ~25% values
    CMPSS_setDACValueHigh(base1, 2048+(int16_t)((float32_t)current_limit*(float32_t)2048.0/(float32_t)current_max_sense));
    CMPSS_setDACValueLow(base1, 2048-(int16_t)((float32_t)current_limit*(float32_t)2048.0/(float32_t)current_max_sense));

    // Make sure the asynchronous path compare high and low event
    // does not go to the OR gate with latched digital filter output
    // hence no additional parameter CMPSS_OR_ASYNC_OUT_W_FILT  is passed
    // comparator oputput is "not" inverted for high compare event
    CMPSS_configHighComparator(base1, CMPSS_INSRC_DAC );
    // Comparator output is inverted for for low compare event
    CMPSS_configLowComparator(base1, CMPSS_INSRC_DAC | CMPSS_INV_INVERTED);

    CMPSS_configFilterHigh(base1,2,10,7);
    CMPSS_configFilterLow(base1,2,10,7);

    //Reset filter logic & start filtering
    CMPSS_initFilterHigh(base1);
    CMPSS_initFilterLow(base1);

    // Configure CTRIPOUT path
    CMPSS_configOutputsHigh(base1, CMPSS_TRIP_FILTER|CMPSS_TRIP_FILTER);
    CMPSS_configOutputsLow(base1, CMPSS_TRIP_FILTER|CMPSS_TRIP_FILTER);

    //Comparator hysteresis control , set to 2x typical value
    CMPSS_setHysteresis(base1, 2);

    // Clear the latched comparator events
    CMPSS_clearFilterLatchHigh(base1);
    CMPSS_clearFilterLatchLow(base1);
}

//TODO setupBoardProtection()
void setupBoardProtection(uint16_t current_sense, float32_t current_limit, float32_t current_max_sense)
{
	if(current_sense==1)
	{
		setupCMPSS(II_SENSE_CMPSS_BASE, current_limit, current_max_sense );

        // Set source for XBAR TRIP 4 from the CMPSS
        XBAR_setEPWMMuxConfig(XBAR_TRIP4, II_SENSE_XBAR_MUX_VAL);

        // Enable Mux to generate TRIP4
        XBAR_enableEPWMMux(XBAR_TRIP4, II_SENSE_XBAR_MUX);
	}
	else if(current_sense==2)
	{
        //// NOTE: USE SDFM_configComparator function
		// SDFM Filter 2 is used to sense the output current
		// max range is 16Amps, for a current trip of 8 Amps use
		// HLT= (32768/2)+ (16384/16)*8
		// HLT= (32768/2)- (16384/16)*8
		SDFM_setComparatorFilterType(IO_SDFM_BASE,IO_SDFM_FILTER,SDFM_FILTER_SINC_3);
		SDFM_setCompFilterOverSamplingRatio(IO_SDFM_BASE,IO_SDFM_FILTER,31);
		SDFM_setCompFilterHighThreshold(IO_SDFM_BASE,IO_SDFM_FILTER,((32768/2)+(int16_t)((float32_t)current_limit*(float32_t)16384/(float32_t)current_max_sense)));
		SDFM_setCompFilterLowThreshold(IO_SDFM_BASE,IO_SDFM_FILTER,((32768/2)-(int16_t)((float32_t)current_limit*(float32_t)16384/(float32_t)current_max_sense)));

		SDFM_clearInterruptFlag(IO_SDFM_BASE,0x8000FFFF);

		XBAR_setEPWMMuxConfig(XBAR_TRIP4,IO_SDFM_SENSE_XBAR_MUX_VAL); // select the 2nd option of MUx 18

        // Enables mux value of IO to be passed to X-BAR trip signal to generate TRIP
		XBAR_enableEPWMMux(XBAR_TRIP4, IO_SDFM_SENSE_XBAR_MUX);

	}

	//Trip 4 is the input to the DCAHCOMPSEL
	EPWM_selectDigitalCompareTripInput(INV_PWM1_BASE,EPWM_DC_TRIP_TRIPIN4, EPWM_DC_TYPE_DCAH);
	EPWM_setTripZoneDigitalCompareEventCondition(INV_PWM1_BASE,EPWM_TZ_DC_OUTPUT_A1,EPWM_TZ_EVENT_DCXH_HIGH);
	EPWM_setDigitalCompareEventSource(INV_PWM1_BASE,EPWM_DC_MODULE_A,EPWM_DC_EVENT_1,EPWM_DC_EVENT_SOURCE_ORIG_SIGNAL);
	EPWM_setDigitalCompareEventSyncMode(INV_PWM1_BASE,EPWM_DC_MODULE_A ,EPWM_DC_EVENT_1,EPWM_DC_EVENT_INPUT_SYNCED);
	EPWM_enableTripZoneSignals(INV_PWM1_BASE,EPWM_TZ_SIGNAL_DCAEVT1);

	EPWM_selectDigitalCompareTripInput(INV_PWM2_BASE,EPWM_DC_TRIP_TRIPIN4, EPWM_DC_TYPE_DCAH);
	EPWM_setTripZoneDigitalCompareEventCondition(INV_PWM2_BASE,EPWM_TZ_DC_OUTPUT_A1,EPWM_TZ_EVENT_DCXH_HIGH);
	EPWM_setDigitalCompareEventSource(INV_PWM2_BASE,EPWM_DC_MODULE_A,EPWM_DC_EVENT_1,EPWM_DC_EVENT_SOURCE_ORIG_SIGNAL);
	EPWM_setDigitalCompareEventSyncMode(INV_PWM2_BASE,EPWM_DC_MODULE_A ,EPWM_DC_EVENT_1,EPWM_DC_EVENT_INPUT_SYNCED);
	EPWM_enableTripZoneSignals(INV_PWM2_BASE,EPWM_TZ_SIGNAL_DCAEVT1);

    EPWM_selectDigitalCompareTripInput(INV_PWM1_BASE,EPWM_DC_TRIP_TRIPIN5, EPWM_DC_TYPE_DCBH);
    EPWM_setTripZoneDigitalCompareEventCondition(INV_PWM1_BASE,EPWM_TZ_DC_OUTPUT_B1,EPWM_TZ_EVENT_DCXH_HIGH);
    EPWM_setDigitalCompareEventSource(INV_PWM1_BASE,EPWM_DC_MODULE_B,EPWM_DC_EVENT_1,EPWM_DC_EVENT_SOURCE_ORIG_SIGNAL);
    EPWM_setDigitalCompareEventSyncMode(INV_PWM1_BASE,EPWM_DC_MODULE_B ,EPWM_DC_EVENT_1,EPWM_DC_EVENT_INPUT_SYNCED);
    EPWM_enableTripZoneSignals(INV_PWM1_BASE,EPWM_TZ_SIGNAL_DCBEVT1);

	EPWM_selectDigitalCompareTripInput(INV_PWM2_BASE,EPWM_DC_TRIP_TRIPIN5, EPWM_DC_TYPE_DCBH);
    EPWM_setTripZoneDigitalCompareEventCondition(INV_PWM2_BASE,EPWM_TZ_DC_OUTPUT_B1,EPWM_TZ_EVENT_DCXH_HIGH);
    EPWM_setDigitalCompareEventSource(INV_PWM2_BASE,EPWM_DC_MODULE_B,EPWM_DC_EVENT_1,EPWM_DC_EVENT_SOURCE_ORIG_SIGNAL);
    EPWM_setDigitalCompareEventSyncMode(INV_PWM2_BASE,EPWM_DC_MODULE_B ,EPWM_DC_EVENT_1,EPWM_DC_EVENT_INPUT_SYNCED);
    EPWM_enableTripZoneSignals(INV_PWM2_BASE,EPWM_TZ_SIGNAL_DCBEVT1);

	EPWM_enableTripZoneSignals(INV_PWM1_BASE,EPWM_TZ_SIGNAL_CBC6); // Emulator Stop
	EPWM_enableTripZoneSignals(INV_PWM2_BASE,EPWM_TZ_SIGNAL_CBC6); // Emulator Stop

	// What do we want the OST/CBC events to do?
	// TZA events can force EPWMxA
	// TZB events can force EPWMxB

	EPWM_setTripZoneAction(INV_PWM1_BASE, EPWM_TZ_ACTION_EVENT_TZA,EPWM_TZ_ACTION_LOW);
	EPWM_setTripZoneAction(INV_PWM1_BASE, EPWM_TZ_ACTION_EVENT_TZB,EPWM_TZ_ACTION_LOW);
	EPWM_setTripZoneAction(INV_PWM2_BASE, EPWM_TZ_ACTION_EVENT_TZA,EPWM_TZ_ACTION_LOW);
	EPWM_setTripZoneAction(INV_PWM2_BASE, EPWM_TZ_ACTION_EVENT_TZB,EPWM_TZ_ACTION_LOW);

    // Clear any spurious trip
    EPWM_clearTripZoneFlag(INV_PWM1_BASE,(EPWM_TZ_FLAG_OST | 
                                          EPWM_TZ_FLAG_DCAEVT1| 
                                          EPWM_TZ_FLAG_DCBEVT1));
    EPWM_clearTripZoneFlag(INV_PWM2_BASE,(EPWM_TZ_FLAG_OST | 
                                          EPWM_TZ_FLAG_DCAEVT1| 
                                          EPWM_TZ_FLAG_DCBEVT1));

	EPWM_forceTripZoneEvent(INV_PWM1_BASE,EPWM_TZ_FORCE_EVENT_OST);
	EPWM_forceTripZoneEvent(INV_PWM2_BASE,EPWM_TZ_FORCE_EVENT_OST);

//************************** End of Prot. Conf. ***************************//
}

//TODO calibrateOffset()
void calibrateOffset(volatile float32_t *II_offset, float32_t k1, float32_t k2)
{
	int16_t offsetCalCounter=0;

    //
    // Select SOC from counter at ctr =PRD
    //
	EPWM_setADCTriggerSource(INV_ISR_TRIG_PWM_BASE,EPWM_SOC_A,EPWM_SOC_TBCTR_U_CMPB);
    
    //
    // set duty 0% initially
    //
	EPWM_setCounterCompareValue(INV_ISR_TRIG_PWM_BASE,EPWM_COUNTER_COMPARE_B,EPWM_getTimeBasePeriod(INV_ISR_TRIG_PWM_BASE)-(ACQPS_SYS_CLKS <<1)); // set duty 0% initially

    //
    // Generate pulse on 1st event
    //
	EPWM_setADCTriggerEventPrescale(INV_ISR_TRIG_PWM_BASE,EPWM_SOC_A,1);
    
    //
    // Enable SOC on A group
    //
	EPWM_enableADCTrigger(INV_ISR_TRIG_PWM_BASE,EPWM_SOC_A);

    //
    // PWM1 INT is used to trigger the ISR
    // INT on Period event
    //
	EPWM_setInterruptSource(INV_ISR_TRIG_PWM_BASE,EPWM_INT_TBCTR_D_CMPB);
    
    //
    // Enable INT
    //
	EPWM_enableInterrupt(INV_ISR_TRIG_PWM_BASE);
    
    //
    // Generate INT on every event
    //
	EPWM_setInterruptEventCount(INV_ISR_TRIG_PWM_BASE,1);


	EPWM_clearEventTriggerInterruptFlag(INV_ISR_TRIG_PWM_BASE);


	offsetCalCounter=0;
	while(offsetCalCounter<25000)
	{
		if(EPWM_getEventTriggerInterruptStatus(INV_ISR_TRIG_PWM_BASE)==1)
		{
			if(offsetCalCounter>1000)
			{
                //
                // offset of the inductor current sense
                //
                (*II_offset) = k1* (*II_offset) + 
                     k2*(II_FB1+II_FB2+II_FB3+II_FB4)*0.25f*ADC_PU_SCALE_FACTOR;
			}
			EPWM_clearEventTriggerInterruptFlag(INV_ISR_TRIG_PWM_BASE);
			offsetCalCounter++;
		}
	}
}

//TODO disablePWMCLKCounting
void disablePWMCLKCounting(void)
{
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
}

//TODO enablePWMCLKCounting
void enablePWMCLKCounting(void)
{
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
}

//TODO setupLEDGPIO()
void setupLEDGPIO(void)
{
	GPIO_setDirectionMode(GPIO_LED1,GPIO_DIR_MODE_OUT);
	GPIO_setDirectionMode(GPIO_LED2,GPIO_DIR_MODE_OUT);
	GPIO_setQualificationMode(GPIO_LED1,GPIO_QUAL_ASYNC);
	GPIO_setQualificationMode(GPIO_LED2,GPIO_QUAL_ASYNC);
	GPIO_setPinConfig(GPIO_LED1_PIN_CONFIG);
	GPIO_setPinConfig(GPIO_LED2_PIN_CONFIG);
}

//TODO toggleLED()
void toggleLED(void)
{
    static uint16_t ledCnt1=0;

	if(ledCnt1==0)
    {
	    GPIO_togglePin(GPIO_LED2);
	    ledCnt1=50;
    }
	else
	{
	    ledCnt1--;
	}
}

//TODO setupProfilingGPIO()
void setupProfilingGPIO(void)
{
	GPIO_setDirectionMode(GPIO_PROFILING1,GPIO_DIR_MODE_OUT);
	GPIO_setDirectionMode(GPIO_PROFILING2,GPIO_DIR_MODE_OUT);
	GPIO_setQualificationMode(GPIO_PROFILING1,GPIO_QUAL_ASYNC);
	GPIO_setQualificationMode(GPIO_PROFILING2,GPIO_QUAL_ASYNC);
	GPIO_setPinConfig(GPIO_PROFILING1_PIN_CONFIG);
	GPIO_setPinConfig(GPIO_PROFILING2_PIN_CONFIG);
}

//TODO setupRelayGPIO();
void setupRelayGPIO(void)
{
	GPIO_setDirectionMode(GPIO_RELAY,GPIO_DIR_MODE_OUT);
	GPIO_setQualificationMode(GPIO_RELAY,GPIO_QUAL_ASYNC);

	GPIO_setPinConfig(GPIO_RELAY_PIN_CONFIG);

}

//TODO setupSCIconnectionForSFRA()
void setupSCIconnectionForSFRA()
{
    //
    // Use Gpio 28 and 29 for the SCI comms through JTAG for SFRA
    //
    GPIO_setPinConfig(SFRA_GUI_SCIRX_GPIO_PIN_CONFIG);
    GPIO_setPinConfig(SFRA_GUI_SCITX_GPIO_PIN_CONFIG);
    GPIO_setDirectionMode(SFRA_GUI_SCIRX_GPIO,GPIO_DIR_MODE_IN);
    GPIO_setDirectionMode(SFRA_GUI_SCITX_GPIO,GPIO_DIR_MODE_OUT);
    GPIO_setQualificationMode(SFRA_GUI_SCIRX_GPIO,GPIO_QUAL_ASYNC);
    GPIO_setQualificationMode(SFRA_GUI_SCITX_GPIO,GPIO_QUAL_ASYNC);
    EDIS;

    //
    // Initialize SCIA and its FIFO.
    //
    SCI_performSoftwareReset(SFRA_GUI_SCI_BASE);

    //
    // 50000000 is the LSPCLK or the Clock used for the SCI Module
    // 57600 is the Baudrate desired of the SCI module
	//
    SCI_setConfig(SFRA_GUI_SCI_BASE, 50000000, 57600,
                  (SCI_CONFIG_WLEN_8 |
                   SCI_CONFIG_STOP_ONE |
                   SCI_CONFIG_PAR_NONE));

    //
    // No loopback
    //
    SCI_disableLoopback(SFRA_GUI_SCI_BASE);

    SCI_resetChannels(SFRA_GUI_SCI_BASE);
    SCI_resetRxFIFO(SFRA_GUI_SCI_BASE);
    SCI_resetTxFIFO(SFRA_GUI_SCI_BASE);
    SCI_clearInterruptStatus(SFRA_GUI_SCI_BASE, SCI_INT_TXFF | SCI_INT_RXFF);
    SCI_enableFIFO(SFRA_GUI_SCI_BASE);
    SCI_enableModule(SFRA_GUI_SCI_BASE);
    SCI_performSoftwareReset(SFRA_GUI_SCI_BASE);

    SCI_enableInterrupt(SFRA_GUI_SCI_BASE, SCI_INT_RXRDY_BRKDT | SCI_INT_TXRDY);

    //
    // Disable RX ERR, SLEEP, TXWAKE
    //
    SCI_clearInterruptStatus(SFRA_GUI_SCI_BASE, SCI_INT_TXRDY | SCI_INT_RXRDY_BRKDT );
}

//TODO setupInverterPWM
void setupInverterPWM(uint32_t base1, uint32_t base2, uint16_t pwm_period_ticks,
                      uint16_t pwm_dbred_ticks, uint16_t pwm_dbfed_ticks)
{
    //
    // PWM clock on F2837x is divided by 2
    // ClkCfgRegs.PERCLKDIVSEL.bit.EPWMCLKDIV=1
    // Deadband needs to be 0.5us => 10ns*50=500ns
    // Time Base SubModule Registers
    //
	EPWM_setPeriodLoadMode(base1,EPWM_PERIOD_SHADOW_LOAD);
	EPWM_setTimeBasePeriod(base1,pwm_period_ticks >>1);
	EPWM_setTimeBaseCounter(base1,0);
	EPWM_setPhaseShift(base1,0);
	EPWM_setTimeBaseCounterMode(base1,EPWM_COUNTER_MODE_UP_DOWN);
	EPWM_setClockPrescaler(base1,EPWM_CLOCK_DIVIDER_1,EPWM_HSCLOCK_DIVIDER_1);

	EPWM_setPeriodLoadMode(base2,EPWM_PERIOD_SHADOW_LOAD);
	EPWM_setTimeBasePeriod(base2,pwm_period_ticks >>1);
	EPWM_setTimeBaseCounter(base2,0);
	EPWM_setPhaseShift(base2,0);
	EPWM_setTimeBaseCounterMode(base2,EPWM_COUNTER_MODE_UP_DOWN);
	EPWM_setClockPrescaler(base2,EPWM_CLOCK_DIVIDER_1,EPWM_HSCLOCK_DIVIDER_1);

    //
    // Counter Compare Submodule Registers
    //
	EPWM_setCounterCompareValue(base1,EPWM_COUNTER_COMPARE_A,0); // set duty 0% initially
	EPWM_setCounterCompareShadowLoadMode(base1,EPWM_COUNTER_COMPARE_A,EPWM_COMP_LOAD_ON_CNTR_ZERO);

	EPWM_setCounterCompareValue(base2,EPWM_COUNTER_COMPARE_A,0); // set duty 0% initially
    
    //
    // set as immediate mode
    //
	EPWM_disableCounterCompareShadowLoadMode(base2,EPWM_COUNTER_COMPARE_A);

    //
    // Action Qualifier SubModule Registers
	//
    EPWM_setActionQualifierAction(base1, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_HIGH,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(base1, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_TOGGLE,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
    EPWM_setActionQualifierAction(base1, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_LOW,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
	// to start don't configure the PWM to do anything
	HWREGH(base1 + EPWM_O_AQCTLA) =0 ;
	HWREGH(base2 + EPWM_O_AQCTLA) =0 ;

	// CTR = CMPA@Down , clear
	EPWM_setActionQualifierAction(base2, EPWM_AQ_OUTPUT_A ,
					EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);

	// Active high complementary PWMs - Set up the deadband
	EPWM_setDeadBandCounterClock(base1,EPWM_DB_COUNTER_CLOCK_FULL_CYCLE);
	EPWM_setRisingEdgeDelayCount(base1,pwm_dbred_ticks);
	EPWM_setFallingEdgeDelayCount(base1,pwm_dbred_ticks);
	EPWM_setDeadBandDelayMode(base1,EPWM_DB_RED,true);
	EPWM_setDeadBandDelayMode(base1,EPWM_DB_FED,true);
	EPWM_setRisingEdgeDeadBandDelayInput(base1,EPWM_DB_INPUT_EPWMA);
	EPWM_setFallingEdgeDeadBandDelayInput(base1,EPWM_DB_INPUT_EPWMA);
	EPWM_setDeadBandDelayPolarity(base1,EPWM_DB_FED,EPWM_DB_POLARITY_ACTIVE_LOW);
	EPWM_setDeadBandDelayPolarity(base1,EPWM_DB_RED,EPWM_DB_POLARITY_ACTIVE_HIGH);

	EPWM_setDeadBandCounterClock(base2,EPWM_DB_COUNTER_CLOCK_FULL_CYCLE);
	EPWM_setRisingEdgeDelayCount(base2,pwm_dbred_ticks);
	EPWM_setFallingEdgeDelayCount(base2,pwm_dbred_ticks);
	EPWM_setDeadBandDelayMode(base2,EPWM_DB_RED,true);
	EPWM_setDeadBandDelayMode(base2,EPWM_DB_FED,true);
	EPWM_setRisingEdgeDeadBandDelayInput(base2,EPWM_DB_INPUT_EPWMA);
	EPWM_setFallingEdgeDeadBandDelayInput(base2,EPWM_DB_INPUT_EPWMA);
	EPWM_setDeadBandDelayPolarity(base2,EPWM_DB_FED,EPWM_DB_POLARITY_ACTIVE_LOW);
	EPWM_setDeadBandDelayPolarity(base2,EPWM_DB_RED,EPWM_DB_POLARITY_ACTIVE_HIGH);

    //
    // configure PWM 1 as master and Phase 2 as slaves and let it 
    // pass the sync in pulse from PWM1
    //
	EPWM_disablePhaseShiftLoad(base1);
	EPWM_setSyncOutPulseMode(base1,EPWM_SYNC_OUT_PULSE_ON_COUNTER_ZERO);
	EPWM_enablePhaseShiftLoad(base2);
	EPWM_setSyncOutPulseMode(base2,EPWM_SYNC_OUT_PULSE_ON_SOFTWARE);
	EPWM_setPhaseShift(base2,2);
	EPWM_setCountModeAfterSync(base2, EPWM_COUNT_MODE_UP_AFTER_SYNC);

	GPIO_setDirectionMode(INV_PWM1_H_GPIO,GPIO_DIR_MODE_OUT);
	GPIO_setPadConfig(INV_PWM1_H_GPIO,GPIO_PIN_TYPE_STD); // disable pull up
	GPIO_setPinConfig(INV_PWM1_H_GPIO_PIN_CONFIG);

	GPIO_setDirectionMode(INV_PWM1_L_GPIO,GPIO_DIR_MODE_OUT);
	GPIO_setPadConfig(INV_PWM1_L_GPIO,GPIO_PIN_TYPE_STD); // disable pull up
	GPIO_setPinConfig(INV_PWM1_L_GPIO_PIN_CONFIG);

	GPIO_setDirectionMode(INV_PWM2_H_GPIO,GPIO_DIR_MODE_OUT);
	GPIO_setPadConfig(INV_PWM2_H_GPIO,GPIO_PIN_TYPE_STD); // disable pull up
	GPIO_setPinConfig(INV_PWM2_H_GPIO_PIN_CONFIG);

	GPIO_setDirectionMode(INV_PWM2_L_GPIO,GPIO_DIR_MODE_OUT);
	GPIO_setPadConfig(INV_PWM2_L_GPIO,GPIO_PIN_TYPE_STD); // disable pull up
	GPIO_setPinConfig(INV_PWM2_L_GPIO_PIN_CONFIG);
}
