//#############################################################################
// FILE:            hv1phdcac_board.h
//
// System Name:     TIDM_HV_1PH_DCAC R3
//
// Target:			F2837xD
//
// Author:			Manish Bhardwaj
//
// Description:	This file consists of common variables for a particular hardware
//              board. Like variables and functions to read current and voltage 
//              signals on the board functions to setup the basic peripherals 
//              of the board. These common set is independent of the control 
//              method used and hence helps in eliminating duplication of this 
//              code when implementing different control schemes on the same 
//              design. This file must be settings independent, an settings 
//              dependent code should reside in the parent solution project.
//
//#############################################################################
// $TI Release: HV_1PH_DCAC v3.00.04.00 $
// $Release Date: Thu Jan 26 15:57:53 CST 2023 $
// $Copyright:
// Copyright (C) 2023 Texas Instruments Incorporated - http://www.ti.com/
//
// ALL RIGHTS RESERVED
// $
//#############################################################################

#ifndef HV1PHDCAC_H
#define HV1PHDCAC_H

#ifdef __cplusplus

extern "C" {
#endif


//*****************************************************************************
// Include Files
//*****************************************************************************

// Include files for device support, F2837x in this case
#include "driverlib.h"
#include "device.h"
#include "DCLF32.h"

#ifdef GCI
#include "gridconnectedinvlclfltr_settings.h"
#else
#include "voltagesourceinvlcfltr_settings.h"
#endif
//*****************************************************************************
//defines
//*****************************************************************************

#define MATH_EMAVG2_F_MACRO(In, Out, Mul)	\
	Out = (_IQmpy((In - Out), Mul) + Out);	\

#define OFFSET_DC_BUS_MEAS 0.01

// Timer definitions based on System Clock
// 150 MHz devices
#define		mSec0_5		0.5*SYSTEM_FREQUENCY*1000		// 0.5 mS
#define		mSec1		1*SYSTEM_FREQUENCY*1000		// 1.0 mS
#define		mSec2		2.0*SYSTEM_FREQUENCY*1000		// 2.0 mS
#define		mSec5		5*SYSTEM_FREQUENCY*1000		// 5.0 mS
#define		mSec7_5		7.5*SYSTEM_FREQUENCY*1000		// 7.5 mS
#define		mSec10		10*SYSTEM_FREQUENCY*1000		// 10 mS
#define		mSec20		20*SYSTEM_FREQUENCY*1000		// 20 mS
#define		mSec50		50*SYSTEM_FREQUENCY*1000		// 50 mS
#define		mSec100		100*SYSTEM_FREQUENCY*1000		// 100 mS
#define		mSec500		500*SYSTEM_FREQUENCY*1000	// 500 mS
#define		mSec1000	1000*SYSTEM_FREQUENCY*1000	// 1000 mS

#ifndef TRUE
#define FALSE 0
#define TRUE  1
#endif

#define II_FB1 ADC_readResult(II_ADCRESULT_BASE, II_1_ADC_SOC_NO)
#define II_FB2 ADC_readResult(II_ADCRESULT_BASE, II_2_ADC_SOC_NO)
#define II_FB3 ADC_readResult(II_ADCRESULT_BASE, II_3_ADC_SOC_NO)
#define II_FB4 ADC_readResult(II_ADCRESULT_BASE, II_4_ADC_SOC_NO)

#define VAC_FB  ADC_readResult(VO_ADCRESULT_BASE, VO_ADC_SOC_NO)
#define VBUS_FB ADC_readResult(VBUS_ADCRESULT_BASE, VBUS_ADC_SOC_NO)
#define VREF165_FB ADC_readResult(IREF_ADCRESULT_BASE, IREF_ADC_SOC_NO)

#define HistorySize 8
//#define DLOG_SIZE   200

#define ADC_PU_SCALE_FACTOR  (float32_t)(0.000244140625)
#define HI_RES_SHIFT (float32_t)(65536.0)
#define ADC_PU_PPB_SCALE_FACTOR 0.000488281250 //1/2^11
#define SD_PU_SCALE_FACTOR  0.000030517578125 // 1/2^15
#define SD32_PU_SCALE_FACTOR  (float32_t) ( 1.0f / ((float32_t)SDFM_OSR* \
                                      (float32_t)SDFM_OSR*(float32_t)SDFM_OSR))

//definitions for selecting DACH reference
#define REFERENCE_VDDA     0

//definitions for COMPH input selection
#define NEGIN_DAC          0
#define NEGIN_PIN          1

//definitions for CTRIPH/CTRIPOUTH output selection
#define CTRIP_ASYNCH       0
#define CTRIP_SYNCH        1
#define CTRIP_FILTER       2
#define CTRIP_LATCH        3

// ADC Configuration
#define REFERENCE_INTERNAL 0 //internal reference (12-bit only)
#define REFERENCE_EXTERNAL 1 //external reference
//definitions for selecting ADC resolution
#define RESOLUTION_12BIT 0 //12-bit resolution
#define RESOLUTION_16BIT 1 //16-bit resolution (not supported for all variants)
//definitions for selecting ADC signal mode
#define SIGNAL_SINGLE 0 //single-ended channel conversions (12-bit mode only)
#define SIGNAL_DIFFERENTIAL 1 //differential pair channel conversions

#define ACQPS_SYS_CLKS     32
#define REFERENCE_VDAC     0
#define REFERENCE_VREF     1

#define GET_TASK_A_TIMER_OVERFLOW_STATUS    CPUTimer_getTimerOverflowStatus(CPUTIMER0_BASE)
#define CLEAR_TASK_A_TIMER_OVERFLOW_FLAG    CPUTimer_clearOverflowFlag(CPUTIMER0_BASE)

#define GET_TASK_B_TIMER_OVERFLOW_STATUS    CPUTimer_getTimerOverflowStatus(CPUTIMER1_BASE)
#define CLEAR_TASK_B_TIMER_OVERFLOW_FLAG    CPUTimer_clearOverflowFlag(CPUTIMER1_BASE)

#define GET_TASK_C_TIMER_OVERFLOW_STATUS    CPUTimer_getTimerOverflowStatus(CPUTIMER2_BASE)
#define CLEAR_TASK_C_TIMER_OVERFLOW_FLAG    CPUTimer_clearOverflowFlag(CPUTIMER2_BASE)

#define HWREG32(x) (*((volatile uint32_t *)(x)))
#define HWREG16(x) (*((volatile uint16_t *)(x)))

//*****************************************************************************
// the function prototypes
//*****************************************************************************

void setupDevice(void);
void setupInverterPWM(uint32_t base1,uint32_t base2, uint16_t pwm_period_ticks,
                      uint16_t pwm_dbred_ticks, uint16_t pwm_dbfed_ticks);
void setupADC(void);
void setupSDFM(uint16_t HLT, uint16_t LLT, uint16_t PWM_ticks,
               uint16_t PWM_ticks_in_sdfm_osr, uint16_t SD_clk_ecap_sys_ticks,
               uint16_t sdfmosr);
void configureECAPforSDFMClk(uint16_t SD_clk_ecap_sys_ticks);
void configurePWMsyncforSDFM(uint16_t PWM_ticks, uint16_t PWM_ticks_in_sdfm_osr);
void configurePWM1chUpDwnCnt(uint32_t base1, int16_t period, int16_t mode,
                             int16_t phase, int16_t db_red, int16_t db_fed,
                             int16_t phase_dir);
void configurePWM1chUpCnt(uint32_t base1, uint16_t period);
void setupBoardProtection(uint16_t current_sense, float32_t current_limit,
                          float32_t current_max_sense);
void calibrateOffset(volatile float32_t *invIiOffset, float32_t k1,
                     float32_t k2);
void setupProfilingGPIO(void);
void setupRelayGPIO(void);
void toggleLED(void);
void setupLEDGPIO(void);
void disablePWMCLKCounting(void);
void enablePWMCLKCounting(void);
void setupSCIconnectionForSFRA(void);
void updateBoardStatus(void);

void DeviceInit(void);
void MemCopy(void);

interrupt void inverterISR(void);
interrupt void onekHzISR(void);

// Inline functions

//TODO updateInverterPWM()
static inline void updateInverterPWM(uint32_t base1, uint32_t base2,
                                     float32_t duty)
{
    uint16_t invDuty;

    invDuty= ((float32_t)(INV_PWM_PERIOD/2.0f)) * (1.0f-fabsf(duty));

    if(invDuty==(EPWM_getTimeBasePeriod(base1)))
    {
        invDuty=invDuty-1;
    }

    EPWM_setCounterCompareValue(base1,EPWM_COUNTER_COMPARE_A,invDuty);
    EPWM_setCounterCompareValue(base2,EPWM_COUNTER_COMPARE_A,1);

    // Used for debugging purposes
    // wait for the PWM to start counting down
    // make sure the PWM is counting down
	// if(EPWM_getTimeBaseCounterDirection(base1)==0)
	// {
    if(duty>=0)
    {
        // CTR = CMPA@UP , set to 1
        EPWM_setActionQualifierAction(base1, EPWM_AQ_OUTPUT_A ,
                EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
        // CTR = CMPA@Down , toggle
        EPWM_setActionQualifierAction(base1, EPWM_AQ_OUTPUT_A ,
                EPWM_AQ_OUTPUT_TOGGLE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
        // CTR=0, clear to 0
        EPWM_setActionQualifierAction(base1, EPWM_AQ_OUTPUT_A ,
                EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
        // CTR = CMPA@Down , clear
        EPWM_setActionQualifierAction(base2, EPWM_AQ_OUTPUT_A ,
                EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
    }
    else
    {
        // CTR = CMPA@UP , clear to 0
        EPWM_setActionQualifierAction(base1, EPWM_AQ_OUTPUT_A ,
                EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
        // CTR = CMPA@Down , toggle
        EPWM_setActionQualifierAction(base1, EPWM_AQ_OUTPUT_A ,
                EPWM_AQ_OUTPUT_TOGGLE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
        // CTR=0, set to 1
        EPWM_setActionQualifierAction(base1, EPWM_AQ_OUTPUT_A ,
                EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
        // CTR = CMPA@Down , set
        EPWM_setActionQualifierAction(base2, EPWM_AQ_OUTPUT_A ,
                EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
    }
	// }
}

//TODO readCurrVolADCSignals()
static inline void readCurrVolADCSignals(volatile float32_t *offset,
                                         volatile float32_t *VBUS_inst,
                                         volatile float32_t *VBUS_offset,
                                         volatile float32_t *VO_inst,
                                         volatile float32_t *II_inst,
                                         volatile float32_t *II_offset)
{
    (*offset) = ((float32_t) (VREF165_FB))*ADC_PU_SCALE_FACTOR;
    (*VBUS_inst) = ((float32_t)(VBUS_FB))*ADC_PU_SCALE_FACTOR+(*VBUS_offset);

    (*VO_inst)= (((float32_t)(VAC_FB))*ADC_PU_SCALE_FACTOR - 0.5f)*2.0f;
    (*II_inst) = (((float32_t) ((II_FB1 + II_FB2 + II_FB3 + II_FB4))* 
                                 ADC_PU_SCALE_FACTOR*0.25f - (*II_offset)))*2.0f;
}

//TODO readCurrVolSDFMSignals()
static inline void readCurrVolSDFMSignals(volatile float32_t *IF_inst,
                                          volatile float32_t *IO_inst,
                                          volatile float32_t *II_inst,
                                          volatile float32_t *VO_inst,
                                          volatile float32_t *VBUS_inst)
{
    (*IF_inst)=(float32_t)(((int32_t) SDFM_getFilterData(IF_SDFM_BASE, IF_SDFM_FILTER))
                                      * SD32_PU_SCALE_FACTOR);
    (*IO_inst)=(float32_t)(((int32_t) SDFM_getFilterData(IO_SDFM_BASE, IO_SDFM_FILTER)) 
                                      * SD32_PU_SCALE_FACTOR);

    (*IO_inst)=-1*(*IO_inst);
    (*II_inst) = (*IO_inst) - (*IF_inst);

    (*VO_inst)=(float32_t)((int32_t) SDFM_getFilterData(VO_SDFM_BASE, VO_SDFM_FILTER)) * SD32_PU_SCALE_FACTOR;
    (*VBUS_inst)=(float32_t)((int32_t) SDFM_getFilterData(VBUS_SDFM_BASE, VBUS_SDFM_FILTER)) * SD32_PU_SCALE_FACTOR; // 0.000000476837158203125);
}

//TODO closeRelay
static inline void closeRelay(void)
{
	GPIO_writePin(GPIO_RELAY,1);
}

//TODO openRelay
static inline void openRelay(void)
{
	GPIO_writePin(GPIO_RELAY,0);
}

//TODO setProfilingGPIO
static inline void setProfilingGPIO(void)
{
	GPIO_writePin(GPIO_PROFILING1,1);
}

//TODO resetProfilingGPIO
static inline void resetProfilingGPIO(void)
{
	GPIO_writePin(GPIO_PROFILING1,0);
}

//TODO setProfilingGPIO
static inline void setProfilingGPIO2(void)
{
	GPIO_writePin(GPIO_PROFILING2,1);
}

//TODO resetProfilingGPIO
static inline void resetProfilingGPIO2(void)
{
	GPIO_writePin(GPIO_PROFILING2,0);
}

//TODO clearInterrupt
static inline void clearInterrupt(void)
{
#if SENSING_OPTION ==SDFM_BASED_SENSING
    EPWM_clearEventTriggerInterruptFlag(INV_ISR_TRIG_PWM_BASE);
#elif SENSING_OPTION ==ADC_BASED_SENSING
    EPWM_clearEventTriggerInterruptFlag(INV_ISR_TRIG_PWM_BASE);
#endif
    Interrupt_clearACKGroup(INV_ISR_PIE_GROUP_NO);
}

static inline void setupInterrupt(void)
{
#if SENSING_OPTION ==SDFM_BASED_SENSING
    //PWM11 INT is used to trigger the ISR
    Interrupt_register(INV_ISR, &inverterISR);
    EPWM_setInterruptSource(INV_ISR_TRIG_PWM_BASE,EPWM_INT_TBCTR_U_CMPA); // INT on PRD event
    EPWM_setInterruptEventCount(INV_ISR_TRIG_PWM_BASE,CNTRL_ISR_FREQ_RATIO);
    EPWM_enableInterrupt(INV_ISR_TRIG_PWM_BASE);
    clearInterrupt();

    EPWM_clearEventTriggerInterruptFlag(INV_ISR_TRIG_PWM_BASE);

    Interrupt_enable(INV_ISR);

#else
    EPWM_clearEventTriggerInterruptFlag(INV_ISR_TRIG_PWM_BASE);

    //PWM11 INT is used to trigger the ISR
    Interrupt_register(INV_ISR, &inverterISR);

    EPWM_setInterruptSource(INV_ISR_TRIG_PWM_BASE,EPWM_INT_TBCTR_PERIOD); // INT on PRD event
    EPWM_setInterruptEventCount(INV_ISR_TRIG_PWM_BASE,CNTRL_ISR_FREQ_RATIO);
    EPWM_enableInterrupt(INV_ISR_TRIG_PWM_BASE);

    clearInterrupt();

    EPWM_clearEventTriggerInterruptFlag(INV_ISR_TRIG_PWM_BASE);
    Interrupt_enable(INV_ISR);
#endif

#ifdef GCI
    // Initialize timer period to maximum
    CPUTimer_setPeriod(ONE_KHZ_ISR_TRIG_CPUTIMER_BASE, DEVICE_SYSCLK_FREQ/10 ); // 10Hz -> 100ms
    CPUTimer_setPeriod(ONE_KHZ_ISR_TRIG_CPUTIMER_BASE, DEVICE_SYSCLK_FREQ/10 ); // 10Hz -> 100ms
    CPUTimer_setPreScaler(ONE_KHZ_ISR_TRIG_CPUTIMER_BASE, 0);
    CPUTimer_stopTimer(ONE_KHZ_ISR_TRIG_CPUTIMER_BASE);
    CPUTimer_setEmulationMode(ONE_KHZ_ISR_TRIG_CPUTIMER_BASE,CPUTIMER_EMULATIONMODE_STOPAFTERNEXTDECREMENT);

    CPUTimer_reloadTimerCounter(ONE_KHZ_ISR_TRIG_CPUTIMER_BASE);
    CPUTimer_resumeTimer(ONE_KHZ_ISR_TRIG_CPUTIMER_BASE);
    Interrupt_register(ONE_KHZ_CPUTIMER_ISR, &onekHzISR);  // 1kHz interrupt from CPU timer 2

    Interrupt_enable(ONE_KHZ_CPUTIMER_ISR);
#endif
    EINT;  // Enable Global interrupt INTM
    ERTM;  // Enable Global realtime interrupt DBGM
}

#ifdef __cplusplus
}
#endif                                  /* extern "C" */


#endif
