//#############################################################################
// FILE:            voltagesourceinvlcfltr.c
//
// System Name:     Voltage source inverter
//
// Author:          Manish Bhardwaj, 2/3/2016
//
// Description: Project implements output voltage control i.e. a voltage source
//              inverter.
//
//              Current and voltages on the design are sensed using the SDFM
//              or ADC selectable through the powerSUITE page.
//
//              The Inverter ISR rate is 20Khz and is triggered by EPWM1 when
//              ADC sensing is used, Of by EPWM 11 when SD reading is used ,
//              this is because EPWM11 provides periodic sync resets to the
//              SDFM module and after each reset the SDFM data is ready after
//              3 OSRs and needs to be read immediately.
//
//              following is the description of other files that are used by
//              this system
//              <solution name>.c -> This file, which has the main.c of the
//                                   project
//              <solution name>.h -> This is the main header file for the
//                                   project
//              <solution name>_settings.h -> powerSUITE generated settings
//              <boad name>_board.c -> This is the file that is used to keep
//                                     common functions related to the board
//                                     hardware like setting up the PWM, ADC,
//                                     SDFM, reading data and common variables
//                                     liked current, voltage etc.
//
//#############################################################################
// $TI Release: HV_1PH_DCAC v3.00.04.00 $
// $Release Date: Thu Jan 26 15:57:53 CST 2023 $
// $Copyright:
// Copyright (C) 2023 Texas Instruments Incorporated - http://www.ti.com/
//
// ALL RIGHTS RESERVED
// $
//#############################################################################

#include "voltagesourceinvlcfltr.h"

//*********************** USER Variables ************************************//
// Global variables used in this system specific to this control method
// common variables are kpet in <system_name>_board.c
//***************************************************************************//

// RAMP to generate forced angle when grid is not present
RAMPGEN rgen1;

//
// DF22
DCL_DF22 gi = DF22_DEFAULTS;
DCL_PI pie = PI_DEFAULTS;
float32_t gi_out;

//
// DF22
DCL_DF22 gv_pr1 = DF22_DEFAULTS;
DCL_DF22 gv_r3 = DF22_DEFAULTS;
DCL_DF22 gv_r5 = DF22_DEFAULTS;
DCL_DF22 gv_r7 = DF22_DEFAULTS;
DCL_DF22 gv_r9 = DF22_DEFAULTS;
DCL_DF22 gv_r11 = DF22_DEFAULTS;
DCL_DF22 gv_lead_lag = DF22_DEFAULTS;
float32_t gv_out;
float32_t gv_lead_lag_out;

//
// for tuning PR and R controllers
float32_t kpV_1H;
float32_t kiV_1H, kiV_3H, kiV_5H, kiV_7H;
float32_t woV_1H, woV_3H, woV_5H, woV_7H;
float32_t wrcV_1H, wrcV_3H, wrcV_5H, wrcV_7H;

uint16_t guiInvStart;
uint16_t guiInvStop;
int16_t slewState;

enum enum_boardState {
    boardState_InverterOFF = 0,
    boardState_CheckDCBus = 1,
    boardState_RlyConnect = 2,
    boardState_InverterON = 3,
    boardState_TripCondition = 4,
};
enum enum_boardState boardState = boardState_InverterOFF;

//
// Stand Alone Flash Image Instrumentation
// GPIO toggles for different states
int16_t ledBlinkCnt1, ledBlinkCnt2;
int16_t timer1;

//--------------------------------- SFRA Related Variables -------------------
float32_t plantMagVect[SFRA_FREQ_LENGTH];
float32_t plantPhaseVect[SFRA_FREQ_LENGTH];
float32_t olMagVect[SFRA_FREQ_LENGTH];
float32_t olPhaseVect[SFRA_FREQ_LENGTH];
float32_t freqVect[SFRA_FREQ_LENGTH];

SFRA_F32 sfra1;

//
// extern to access tables in ROM
extern long FPUsinTable[];

float32_t adc_noise;
float32_t H_calc;
float32_t error_bound;
float32_t inj_calculated;
//---------------------- State Machine Related --------------------------------

int16_t vTimer0[4];         // Virtual Timers slaved off CPU Timer 0 (A events)
int16_t vTimer1[4];         // Virtual Timers slaved off CPU Timer 1 (B events)
int16_t vTimer2[4];         // Virtual Timers slaved off CPU Timer 2 (C events)

//
// Variable declarations for state machine
void (*Alpha_State_Ptr)(void);  // Base States pointer
void (*A_Task_Ptr)(void);       // State pointer A branch
void (*B_Task_Ptr)(void);       // State pointer B branch
void (*C_Task_Ptr)(void);       // State pointer C branch

//-----------------------------------------------------------------------------
enum enum_BuildLevel {
    BuildLevel1_OpenLoop_AC = 0,
    BuildLevel1_OpenLoop_DC = 1,
    BuildLevel2_CurrentLoop_AC = 2,
    BuildLevel2_CurrentLoop_DC = 3,
    BuildLevel3_VoltageLoop_AC = 4,
    BuildLevel3_VoltageLoop_DC = 5,
};
enum enum_BuildLevel BuildInfo = BuildLevel1_OpenLoop_AC ;

//
// TODO Inverter Control Variables
// Inverter Variables
//NOTE: Io & Ig are used interchangeably,
//NOTE: Vo and Vg are used Interchangeably
// Measurement Variables
// Inverter Side Inductor Current Measurement
float32_t invIiInst;
// Output Voltage measurement
float32_t invVoInst;
// Inverter Side Inductor Current Measurement Offset
volatile float32_t invIiOffset;
// Cap Current SD measurement
volatile float32_t invIfInst;
// Cap Current SD measurement offset
volatile float32_t invIfSDOffset;
// Output current measurement
float32_t invIoInst;
// Output current measurement
volatile float32_t invIoSDOffset;
// Output Voltage measurement Offset
// Vo Offset May not be applicable in real apps as Vo may be present always
volatile float32_t invVoOffset,invVoSDOffset;
// Inv DC Bus Voltage Measurement
volatile float32_t invVbusInst;
// Inv DC Bus Voltage Measurement Offset
volatile float32_t invVbusOffset,invVbusSDOffset;
// Inv Filtered DC bus measurement
volatile float32_t invVbusAvg;
// Inv DC Bus measurement by SD Offset
// Reference variables
// Inverter modulation index, used for open loop check
volatile float32_t invModIndex;
// Peak value of the Ii, Io current set point
volatile float32_t invIiRef, invIoRef;
// Peak value of the Vo  set point
volatile float32_t invVoRef;
// Instantaneous Ii or Io reference = invIiRef* angle
float32_t invIiRefInst,invIoRefInst;
// Instantaneous Vo reference = invVoRef* angle
volatile float32_t invVoRefInst;
// DC bus reference
volatile float32_t invVbusRef;
// Voltage Error
volatile float32_t voltage_error;

// Display Values
float32_t guiVbus;
float32_t guiACFreq;
float32_t guiPrms;
float32_t guiIrms;
volatile float32_t guiVrms;
float32_t guiVavg;
float32_t guiVo;
float32_t guiIi;
float32_t guiIo;
float32_t guiPowerFactor;
float32_t guiVA;
float32_t guiFreqAvg;
float32_t guiVema;

//
// Variables used to calibrate measurement offsets
//Offset filter coefficient K1: 0.05/(T+0.05);
float32_t k1 = 0.998;
//Offset filter coefficient K2: T/(T+0.05)
float32_t k2 = 0.001999;
int16_t offsetCalCounter;
float32_t offset165;

//
// Duty variables for inverter (per unit and compare register values)
// per unit duty value
float32_t invDutyPU,invDutyPUPrev;
//
// duty value written into the PWM register
int16_t invDuty,invDutyCompl;
//
// PWM TBPRD Half
volatile float32_t halfPWMPeriod;

//
// Flags for clearing trips and closing the loops and the Relay
int16_t closeILoopInv,closeVLoopInv, clearInvTrip, rlyConnect;

//
// Flags for detecting ZCD
float32_t invSine, invSinePrev;
int16_t zeroCrossDetectFlag;

volatile int16_t tempChange = 0;
volatile int16_t period;

//
// Sine analyzer block for RMS Volt, Curr and Power measurements
POWER_MEAS_SINE_ANALYZER sine_mains;

//
// datalogger
DLOG_4CH dLog1;
float32_t dBuff1[200], dBuff2[200], dBuff3[200], dBuff4[200];
float32_t dVal1, dVal2, dVal3, dVal4;

//
// adc static cal
int16_t *adc_cal;

int ledCnt1 = 0;

//
// SD Trip Level
// 32 OSR and SINC3 is used for the comparator trip
// Max value is 32768
// As the SD is used to sense positive and negative values offset is 16384
uint16_t  HLT = 0x7FFF;
uint16_t  LLT = 0x0000;

int16_t i;

enum enum_boardStatus boardStatus = boardStatus_Idle;

//
// Note that the watchdog is disabled in codestartbranch.asm
// for this project. This is to prevent it from expiring while
// c_init routine initializes the global variables before
// reaching the main()
//
void main(void)
{
    //
    // This routine sets up the basic device configuration such as initializing
    // PLL copying code from FLASH to RAM,
    // this routine will also initialize the CPU timers that are used in the
    // background task for this system
    setupDevice();

    //
    // Tasks State-machine init
    Alpha_State_Ptr = &A0;
    A_Task_Ptr = &A1;
    B_Task_Ptr = &B1;
    C_Task_Ptr = &C1;

    //
    // Stop all PWM mode clock
    disablePWMCLKCounting();

    setupInverterPWM(INV_PWM1_BASE,INV_PWM2_BASE,
                     INV_PWM_PERIOD,INV_DEADBAND_PWM_COUNT,
                     INV_DEADBAND_PWM_COUNT);

#if SENSING_OPTION == ADC_BASED_SENSING
    //
    // power up ADC on the device
    setupADC();

#elif SENSING_OPTION == SDFM_BASED_SENSING
    //
    // setup Sigma Delta, configure ECAP to generate the clock,
    // configure GPIO mux for SDFM pins etc. OSR and SINC order is fixed,
    // if one needs to change it the routine below needs to be modified
    setupSDFM(HLT, LLT, INV_PWM_PERIOD,PWM_CLK_IN_SDFM_OSR, SD_CLK_COUNT,
              SDFM_OSR);
#endif

    //
    // Profiling GPIO and led GPIOs
    setupProfilingGPIO();

    //
    // configure lED GPIO
    setupLEDGPIO();

    //
    // configure Relay GPIO
    setupRelayGPIO();

    //
    // Control Variable specific to the voltage source inverter Initialization

    //
    // RAMPGEN
    RAMPGEN_reset(&rgen1);
    rgen1.freq = (float32_t)(AC_FREQ);
    rgen1.stepAngleMax = (float32_t)(1.0/INV_PWM_SWITCHING_FREQUENCY);

    kpV_1H = KPV_1H;
    kiV_1H = KIV_1H;
    wrcV_1H= WRCV_1H;
    woV_1H = 2.0*PI_VALUE*AC_FREQ;
    computeDF22_PRcontrollerCoeff(&gv_pr1, kpV_1H,kiV_1H,woV_1H,
                                   ISR_CONTROL_FREQUENCY,wrcV_1H);

    kiV_3H = KIV_3H;
    wrcV_3H = WRCV_3H;
    woV_3H = 2.0*PI_VALUE*AC_FREQ*3;
    computeDF22_PRcontrollerCoeff(&gv_r3, 0,kiV_3H,woV_3H,
                                   ISR_CONTROL_FREQUENCY,wrcV_3H);

    kiV_5H = KIV_5H;
    wrcV_5H = WRCV_5H;
    woV_5H = 2.0*PI_VALUE*AC_FREQ*5;
    computeDF22_PRcontrollerCoeff(&gv_r5, 0,kiV_5H,woV_5H,
                                   ISR_CONTROL_FREQUENCY,wrcV_5H);

    kiV_7H = KIV_7H;
    wrcV_7H = WRCV_7H;
    woV_7H = 2.0*PI_VALUE*AC_FREQ*7;
    computeDF22_PRcontrollerCoeff(&gv_r7, 0,kiV_7H,woV_7H,
                                   ISR_CONTROL_FREQUENCY,wrcV_7H);

    //
    // DF 22 coefficients
    gi.a1=(float32_t)(GI_A1);
    gi.a2=(float32_t)(GI_A2);
    gi.b0=(float32_t)(GI_B0);
    gi.b1=(float32_t)(GI_B1);
    gi.b2=(float32_t)(GI_B2);
    gi.x1 = 0;
    gi.x2 = 0;

    //
    // PI coefficients
    pie.Kp= (float32_t) (PI_KP);
    pie.Ki= (float32_t) (PI_KI);

    gv_lead_lag.a1=(float32_t)(GV_LEAD_LAG_A1);
    gv_lead_lag.a2=(float32_t)(GV_LEAD_LAG_A2);
    gv_lead_lag.b0=(float32_t)(GV_LEAD_LAG_B0);
    gv_lead_lag.b1=(float32_t)(GV_LEAD_LAG_B1);
    gv_lead_lag.b2=(float32_t)(GV_LEAD_LAG_B2);
    gv_lead_lag.x1 = 0;
    gv_lead_lag.x2 = 0;

    setupSFRA();
    adc_noise = 0.0005;
    H_calc = 0;
    error_bound = 0.2;
    inj_calculated = 0;

    //
    // sine analyzer initialization
    //
    POWER_MEAS_SINE_ANALYZER_reset(&sine_mains);
    sine_mains.sampleFreq = (float32_t)(ISR_CONTROL_FREQUENCY);
    sine_mains.threshold = (float32_t)(0.25);
    sine_mains.nSamplesMax = ISR_CONTROL_FREQUENCY/UNIVERSAL_GRID_MIN_FREQ;
    sine_mains.nSamplesMin = ISR_CONTROL_FREQUENCY/UNIVERSAL_GRID_MAX_FREQ;
    sine_mains.emaFilterMultiplier = 2.0f/ISR_CONTROL_FREQUENCY;

    guiInvStart = 0;
    guiInvStop = 0;

    //
    // Initialize global variables generic to the board like ones used
    // to read current values etc
    initGlobalVariables();

    //
    // Enable PWM Clocks
    enablePWMCLKCounting();

    //
    // Offset Calibration Routine
#if SENSING_OPTION ==ADC_BASED_SENSING
    calibrateOffset(&invIiOffset, k1, k2);
#endif

    //
    // clear any spurious flags in the SDFM module
    // setup protection and trips for the board
    setupBoardProtection(SENSING_OPTION, I_TRIP_LIMIT,I_MAX_SENSE);

    //
    // ISR Mapping
    setupInterrupt();

    // IDLE loop. Just sit and loop forever, periodically will branch into
    // A0-A3, B0-B3, C0-C3 tasks
    // Frequency of this branching is set in setupDevice routine:
    for(;;)  //infinite loop
    {
        //
        // State machine entry & exit point
        //
        (*Alpha_State_Ptr)();   // jump to an Alpha state (A0,B0,...)
    }
} //END MAIN CODE

//=============================================================================
//  STATE-MACHINE SEQUENCING AND SYNCRONIZATION FOR SLOW BACKGROUND TASKS
//=============================================================================

//--------------------------------- SFRAMEWORK --------------------------------
void A0(void)
{
    //
    // loop rate synchronizer for A-tasks
    if(GET_TASK_A_TIMER_OVERFLOW_STATUS == 1)
    {
        CLEAR_TASK_A_TIMER_OVERFLOW_FLAG;   // clear flag

        //-----------------------------------------------------------
        (*A_Task_Ptr)();        // jump to an A Task (A1,A2,A3,...)
        //-----------------------------------------------------------

        vTimer0[0]++;           // virtual timer 0, instance 0 (spare)
    }

    Alpha_State_Ptr = &B0;      // Comment out to allow only A tasks
}

void B0(void)
{
    // loop rate synchronizer for B-tasks
    if(GET_TASK_B_TIMER_OVERFLOW_STATUS == 1)
    {
        CLEAR_TASK_B_TIMER_OVERFLOW_FLAG;               // clear flag

        //-----------------------------------------------------------
        (*B_Task_Ptr)();        // jump to a B Task (B1,B2,B3,...)
        //-----------------------------------------------------------
        vTimer1[0]++;           // virtual timer 1, instance 0 (spare)
    }

    Alpha_State_Ptr = &C0;      // Allow C state tasks
}

void C0(void)
{
    // loop rate synchronizer for C-tasks
    if(GET_TASK_C_TIMER_OVERFLOW_STATUS == 1)
    {
        CLEAR_TASK_C_TIMER_OVERFLOW_FLAG;               // clear flag

        //-----------------------------------------------------------
        (*C_Task_Ptr)();        // jump to a C Task (C1,C2,C3,...)
        //-----------------------------------------------------------
        vTimer2[0]++;           //virtual timer 2, instance 0 (spare)
    }

    Alpha_State_Ptr = &A0;  // Back to State A0
}


//=============================================================================
//  A - TASKS (executed in every 1 msec)
//=============================================================================
//--------------------------------------------------------
void A1(void) // SPARE (not used)
//--------------------------------------------------------
{
#if(INCR_BUILD==1)||(INCR_BUILD==2)||(INCR_BUILD==3)
    if(rlyConnect==1)
    {
        closeRelay();
    }
    else
    {
        openRelay();
        invVoRef = 0;
        invIiRef = 0;
    }
#endif

    SFRA_GUI_runSerialHostComms(&sfra1);
    SFRA_F32_runBackgroundTask(&sfra1);

    //-------------------
    //the next time CpuTimer0 'counter' reaches Period value go to A2
    A_Task_Ptr = &A1;
    //-------------------
}

//=============================================================================
//  B - TASKS (executed in every 5 msec)
//=============================================================================

//----------------------------------- USER ------------------------------------

//----------------------------------------
void B1(void)
//----------------------------------------
{
    updateBoardStatus();
#if INCR_BUILD==1
    #if DC_CHECK==1
        BuildInfo = BuildLevel1_OpenLoop_DC;
    #else
        BuildInfo = BuildLevel1_OpenLoop_AC;
    #endif
#elif INCR_BUILD==2
    #if DC_CHECK==1
        BuildInfo = BuildLevel2_CurrentLoop_DC;
    #else
        BuildInfo = BuildLevel2_CurrentLoop_AC;
    #endif
#elif INCR_BUILD==3
    #if DC_CHECK==1
        BuildInfo = BuildLevel3_VoltageLoop_DC;
    #else
        BuildInfo = BuildLevel3_VoltageLoop_AC;
    #endif
#endif


    //
    // the next time CpuTimer1 'counter' reaches Period value go to B2
    B_Task_Ptr = &B2;
}

//----------------------------------------
void B2(void) //  SPARE
//----------------------------------------
{
    switch(boardState)
    {
        case boardState_InverterOFF:
            if (guiInvStart==1)
                {
                    guiInvStart = 0;
                    boardState = boardState_CheckDCBus;
                }
            break;
        case boardState_CheckDCBus:
            // assuming a ~10% margin on the DC Bus vs Typical AC Output Max
            // Only start the inverter if the DC Bus is higher than that
            if(guiVbus>(float32_t)1.55*VAC_TYPICAL && guiVbus<VAC_MAX_SENSE)
            {
                boardState = boardState_RlyConnect;
                rlyConnect = 1;
                slewState = 20;
            }
            break;
        case boardState_RlyConnect:
            slewState--;
            if(slewState==0)
            {
                boardState = boardState_InverterON;
                invVoRef = 0;
                clearInvTrip = 1;
            }
            break;
        case boardState_InverterON:
            if(invVoRef<((float32_t)VAC_TYPICAL*(float32_t)1.414/(float32_t)VAC_MAX_SENSE))
            {
                invVoRef = invVoRef+0.005f;
            }
            if(guiInvStop==1)
                {
                    guiInvStop = 0;
                    boardState = boardState_InverterOFF;
                    invVoRef = 0;
                    rlyConnect = 0;
                    closeILoopInv = 0;
                }
            if(boardStatus==boardStatus_OverCurrentTrip ||
               boardStatus== boardStatus_EmulatorStopTrip)
            {
                boardState= boardState_TripCondition;
            }
            break;
        case boardState_TripCondition: break;
    }

    //-----------------
    //the next time CpuTimer1 'counter' reaches Period value go to B3
    B_Task_Ptr = &B1;
    //-----------------
}

//=============================================================================
//  C - TASKS (executed in every 50 msec)
//=============================================================================

//--------------------------------- USER --------------------------------------

//----------------------------------------
void C1(void)
//----------------------------------------
{

    toggleLED();

    //-----------------
    //the next time CpuTimer2 'counter' reaches Period value go to C2
    C_Task_Ptr = &C1;
    //-----------------

}

//TODO Inverter ISR Code
interrupt void inverterISR(void)
{
    setProfilingGPIO();

    //
    // Read inverter Current and Voltage Measurements
    //
#if SENSING_OPTION ==ADC_BASED_SENSING
    readCurrVolADCSignals(&offset165, &invVbusInst, &invVbusOffset,
                          &invVoInst, &invIiInst, &invIiOffset);
#elif SENSING_OPTION ==SDFM_BASED_SENSING
    readCurrVolSDFMSignals(&invIfInst, &invIoInst, &invIiInst,
                           &invVoInst, &invVbusInst);
#endif

    //
    // Generate the RAMP angle
    //
    RAMPGEN_run(&rgen1);
    
    //
    // Use the angle value to compute the sine value
    //
    invSine = sinf((rgen1.out)*6.283185307f);

    detectZeroCrossingClearPWMTrip(INV_PWM1_BASE,INV_PWM2_BASE);

    //
    // TODO BUILD 1
    //-------------------------------------------------------------------------
    #if (INCR_BUILD == 1 ) // Open Loop Check
    //-------------------------------------------------------------------------
        #if (DC_CHECK==1)
            invDutyPU = SFRA_F32_inject(invModIndex);
        #else
            invDutyPU = SFRA_F32_inject(invSine * invModIndex);
        #endif
    #endif // (INCR_BUILD == 1)
    //TODO BUILD 2

    //
    // Closed Current Loop Inverter with forced/internal sine angle
    #if (INCR_BUILD == 2)

        #if (DC_CHECK==1)
            invIiRefInst = invIiRef;
        #else
            invIiRefInst = invIiRef*invSine;
        #endif

        if(closeILoopInv==1)
        {
//          gi_out= DCL_runDF22_C1(&gi, (SFRA_F32_inject(invIiRefInst) - invIiInst));
            gi_out= DCL_runPI_C4(&pie, SFRA_F32_inject(invIiRefInst), invIiInst);

            invDutyPU= (gi_out+invVoInst)/invVbusInst;
            invDutyPU= (invDutyPU>(float32_t)(0.98))?(float32_t)(0.98):invDutyPU;
            invDutyPU= (invDutyPU<(float32_t)(-0.98))?(float32_t)(-0.98):invDutyPU;
        }
        else
        {
            invDutyPU = 0;
        }

    #endif
    //
    // TODO BUILD 3
    //-------------------------------------------------------------------------
    #if(INCR_BUILD ==3)
    //-------------------------------------------------------------------------
        #if(DC_CHECK==1)
            invVoRefInst = invVoRef;
        #else
            invVoRefInst = invSine * invVoRef;
        #endif

        if(closeILoopInv==1)
        {
            #if SFRA_TYPE==1 && DC_CHECK == 1
                invVoRefInst = SFRA_F32_inject(invVoRef);
            #elif SFRA_TYPE==1 && DC_CHECK == 0
                invVoRefInst = SFRA_F32_inject(invVoRef*invSine);
            #endif

            voltage_error= invVoRefInst - invVoInst;
            gv_out= DCL_runDF22_C1(&gv_pr1, voltage_error)
                    + DCL_runDF22_C1(&gv_r3, voltage_error)
                    + DCL_runDF22_C1(&gv_r5, voltage_error)
                    + DCL_runDF22_C1(&gv_r7, voltage_error);

            gv_lead_lag_out= DCL_runDF22_C1(&gv_lead_lag, gv_out);

            #if SFRA_TYPE==2
                invIiRefInst = SFRA_F32_inject(gv_lead_lag_out);
            #else
                invIiRefInst=(gv_lead_lag_out);
            #endif

            gi_out= DCL_runDF22_C1(&gi, (invIiRefInst - invIiInst));
//          gi_out= DCL_runPI_C4(&pie, SFRA_F32_inject(invIiRefInst), invIiInst);

            invDutyPU= (gi_out+invVoInst)/invVbusInst;
            invDutyPU= (invDutyPU>(float32_t)(0.98))?(float32_t)(0.98):invDutyPU;
            invDutyPU= (invDutyPU<(float32_t)(-0.98))?(float32_t)(-0.98):invDutyPU;
        }
        else
        {
            invDutyPU = 0;
        }

    #endif

    //-------------------------------------------------------------------------
    // PWM Driver for the inverter
    //-------------------------------------------------------------------------

    updateInverterPWM(INV_PWM1_BASE,INV_PWM2_BASE, invDutyPU);

    invDutyPUPrev = invDutyPU;

    #if (SFRA_TYPE == 1 && INCR_BUILD==3)   //Running FRA on Voltage
            SFRA_F32_collect(&invIiRefInst,&invVoInst);
    #else
        #if(INCR_BUILD==1)
            SFRA_F32_collect(&invDutyPU,&invIiInst);
        #elif(INCR_BUILD ==2 || INCR_BUILD ==3)
            SFRA_F32_collect(&gi_out,&invIiInst);
        #endif
    #endif

    // ------------------------------------------------------------------------
    //    Connect inputs of the Datalogger module
    // ------------------------------------------------------------------------
    //TODO DLOG
    #if INCR_BUILD==1
        // check output voltage and inverter current meas.
        dVal1 = invVoInst;
        dVal2 = invIiInst;
#if SENSING_OPTION ==1
        dVal3 = guiVo;
#else
        dVal3 = invIoInst;
#endif
        dVal4 = guiIo;
    #elif INCR_BUILD==2
        // check inverter current meas and reference.
        dVal1 = invIiInst;
        dVal2 = invIiRefInst;
        dVal3 = invVoInst;
        dVal4 = guiIi;
    #else
        // check inverter voltage meas and reference
        dVal1 = invVoInst;
        dVal2 = invVoRefInst;
        dVal3 = invIiInst;
        dVal4 = invSine;
    #endif
    DLOG_4CH_run(&dLog1);

    guiVbus = invVbusInst*VAC_MAX_SENSE;
    guiVo = invVoInst*VDCBUS_MAX_SENSE;
    guiIi = invIiInst*I_MAX_SENSE;
#if SENSING_OPTION ==SDFM_BASED_SENSING
    guiIo = invIoInst*I_MAX_SENSE;
#endif


    //
    // Calculate RMS input voltage and input frequency
#if SENSING_OPTION ==ADC_BASED_SENSING
    sine_mains.i = invIiInst;
#elif SENSING_OPTION ==SDFM_BASED_SENSING
    sine_mains.i = invIoInst;
#endif
    sine_mains.v = invVoInst;
    POWER_MEAS_SINE_ANALYZER_run(&sine_mains);
    guiIrms = sine_mains.iRms*I_MAX_SENSE;
    guiVrms = sine_mains.vRms*VAC_MAX_SENSE;
    guiPrms = sine_mains.pRms*VAC_MAX_SENSE*I_MAX_SENSE;
    guiPowerFactor = sine_mains.powerFactor;
    guiVA = sine_mains.vaRms*VAC_MAX_SENSE*I_MAX_SENSE;
    guiFreqAvg = sine_mains.acFreqAvg;
    guiVema = sine_mains.vEma*VAC_MAX_SENSE;

    clearInterrupt();

    resetProfilingGPIO();

}// MainISR Ends Here

//TODO setupSFRA
void setupSFRA(void)
{
    //
    // SFRA Related
    //SFRA Object Initialization
    SFRA_F32_reset(&sfra1);

    //
    // Assign array location to Pointers in the SFRA object
    SFRA_F32_config(&sfra1,
                    SFRA_ISR_FREQ,      //Specify the SFRA ISR Frequency
                    SFRA_AMPLITUDE,     //Specify the injection amplitude
                    SFRA_FREQ_LENGTH,   //Specify the length of SFRA
                    SFRA_FREQ_START,    //Specify start Freq of SFRA analysis
                    FREQ_STEP_MULTIPLY,
                    plantMagVect,
                    plantPhaseVect,
                    olMagVect,
                    olPhaseVect,
                    NULL,
                    NULL,
                    freqVect,
                    1);

    SFRA_F32_resetFreqRespArray(&sfra1);

    i = 0;
    #if SFRA_TYPE==1
    sfra1.freqVect[i] = SFRA_FREQ_START;
    sfra1.freqVect[i++] = 5;
    sfra1.freqVect[i++] = 10;
    sfra1.freqVect[i++] = 20;
    sfra1.freqVect[i++] = 30;
    sfra1.freqVect[i++] = 40;
    sfra1.freqVect[i++] = 50;
    sfra1.freqVect[i++] = 55;
    sfra1.freqVect[i++] = 56;
    sfra1.freqVect[i++] = 57;
    sfra1.freqVect[i++] = 58;
    sfra1.freqVect[i++] = 58.5;
    sfra1.freqVect[i++] = 58.75;
    sfra1.freqVect[i++] = 59;
    sfra1.freqVect[i++] = 59.2;
    sfra1.freqVect[i++] = 59.4;
    sfra1.freqVect[i++] = 59.6;
    sfra1.freqVect[i++] = 59.8;
    sfra1.freqVect[i++] = 60;
    sfra1.freqVect[i++] = 60.2;
    sfra1.freqVect[i++] = 60.4;
    sfra1.freqVect[i++] = 60.6;
    sfra1.freqVect[i++] = 60.8;
    sfra1.freqVect[i++] = 61;
    sfra1.freqVect[i++] = 61.25;
    sfra1.freqVect[i++] = 61.5;
    sfra1.freqVect[i++] = 62;
    sfra1.freqVect[i++] = 65;
    sfra1.freqVect[i++] = 70;
    sfra1.freqVect[i++] = 90;
    sfra1.freqVect[i++] = 100;
    sfra1.freqVect[i++] = 110;
    sfra1.freqVect[i++] = 120;
    sfra1.freqVect[i++] = 140;
    sfra1.freqVect[i++] = 160;
    sfra1.freqVect[i++] = 170;
    sfra1.freqVect[i++] = 175;
    sfra1.freqVect[i++] = 178;
    sfra1.freqVect[i++] = 178.5;
    sfra1.freqVect[i++] = 179;
    sfra1.freqVect[i++] = 179.2;
    sfra1.freqVect[i++] = 179.4;
    sfra1.freqVect[i++] = 179.6;
    sfra1.freqVect[i++] = 179.8;
    sfra1.freqVect[i++] = 180;
    sfra1.freqVect[i++] = 180.2;
    sfra1.freqVect[i++] = 180.4;
    sfra1.freqVect[i++] = 180.6;
    sfra1.freqVect[i++] = 180.8;
    sfra1.freqVect[i++] = 181;
    sfra1.freqVect[i++] = 182;
    sfra1.freqVect[i++] = 185;
    sfra1.freqVect[i++] = 190;
    sfra1.freqVect[i++] = 200;
    sfra1.freqVect[i++] = 220;
    sfra1.freqVect[i++] = 250;
    sfra1.freqVect[i++] = 280;
    sfra1.freqVect[i++] = 290;
    sfra1.freqVect[i++] = 295;
    sfra1.freqVect[i++] = 298;
    sfra1.freqVect[i++] = 298.5;
    sfra1.freqVect[i++] = 299;
    sfra1.freqVect[i++] = 299.4;
    sfra1.freqVect[i++] = 299.6;
    sfra1.freqVect[i++] = 299.8;
    sfra1.freqVect[i++] = 300;
    sfra1.freqVect[i++] = 300.2;
    sfra1.freqVect[i++] = 300.4;
    sfra1.freqVect[i++] = 300.6;
    sfra1.freqVect[i++] = 301;
    sfra1.freqVect[i++] = 305;
    sfra1.freqVect[i++] = 310;
    sfra1.freqVect[i++] = 330;
    sfra1.freqVect[i++] = 360;
    sfra1.freqVect[i++] = 380;
    sfra1.freqVect[i++] = 390;
    sfra1.freqVect[i++] = 410;
    sfra1.freqVect[i++] = 415;
    sfra1.freqVect[i++] = 418;
    sfra1.freqVect[i++] = 418.5;
    sfra1.freqVect[i++] = 419;
    sfra1.freqVect[i++] = 419.4;
    sfra1.freqVect[i++] = 419.6;
    sfra1.freqVect[i++] = 419.8;
    sfra1.freqVect[i++] = 420;
    sfra1.freqVect[i++] = 420.2;
    sfra1.freqVect[i++] = 420.4;
    sfra1.freqVect[i++] = 420.8;
    sfra1.freqVect[i++] = 421.2;
    sfra1.freqVect[i++] = 422;
    sfra1.freqVect[i++] = 425;
    sfra1.freqVect[i++] = 430;
    sfra1.freqVect[i++] = 450;
    sfra1.freqVect[i++] = 500;
    sfra1.freqVect[i++] = 600;
    sfra1.freqVect[i++] = 700;
    sfra1.freqVect[i++] = 800;
    sfra1.freqVect[i++] = 900;
    sfra1.freqVect[i++] = 1000;
    sfra1.freqVect[i++] = 1100;
    sfra1.freqVect[i++] = 1200;

    #else
    SFRA_F32_initFreqArrayWithLogSteps(&sfra1,
                                       SFRA_FREQ_START,
                                       FREQ_STEP_MULTIPLY); //Specify the Freq. Step
    /*
    sfra1.freqVect[i++] = 5;
    sfra1.freqVect[i++] = 20;
    sfra1.freqVect[i++] = 40;
    sfra1.freqVect[i++] = 55;
    sfra1.freqVect[i++] = 57;
    sfra1.freqVect[i++] = 58;
    sfra1.freqVect[i++] = 59;
    sfra1.freqVect[i++] = 59.2; //10
    sfra1.freqVect[i++] = 59.4;
    sfra1.freqVect[i++] = 59.6;
    sfra1.freqVect[i++] = 59.8;
    sfra1.freqVect[i++] = 60;
    sfra1.freqVect[i++] = 60.2;
    sfra1.freqVect[i++] = 60.4;
    sfra1.freqVect[i++] = 60.6;
    sfra1.freqVect[i++] = 60.8;
    sfra1.freqVect[i++] = 61;
    sfra1.freqVect[i++] = 64;
    sfra1.freqVect[i++] = 70;
    sfra1.freqVect[i++] = 100;
    sfra1.freqVect[i++] = 130;
    sfra1.freqVect[i++] = 170;
    sfra1.freqVect[i++] = 175;
    sfra1.freqVect[i++] = 177;
    sfra1.freqVect[i++] = 178;
    sfra1.freqVect[i++] = 179;
    sfra1.freqVect[i++] = 179.2;
    sfra1.freqVect[i++] = 179.4;
    sfra1.freqVect[i++] = 179.6;
    sfra1.freqVect[i++] = 179.8;
    sfra1.freqVect[i++] = 180;
    sfra1.freqVect[i++] = 180.2;
    sfra1.freqVect[i++] = 180.4;
    sfra1.freqVect[i++] = 180.6;
    sfra1.freqVect[i++] = 180.8;
    sfra1.freqVect[i++] = 181;
    sfra1.freqVect[i++] = 182;
    sfra1.freqVect[i++] = 185;
    sfra1.freqVect[i++] = 200;
    sfra1.freqVect[i++] = 240;
    sfra1.freqVect[i++] = 280;
    sfra1.freqVect[i++] = 290;
    sfra1.freqVect[i++] = 295;
    sfra1.freqVect[i++] = 297;
    sfra1.freqVect[i++] = 298;
    sfra1.freqVect[i++] = 299;
    sfra1.freqVect[i++] = 299.2;
    sfra1.freqVect[i++] = 299.4;
    sfra1.freqVect[i++] = 299.6;
    sfra1.freqVect[i++] = 299.8;
    sfra1.freqVect[i++] = 300;
    sfra1.freqVect[i++] = 300.2;
    sfra1.freqVect[i++] = 300.4;
    sfra1.freqVect[i++] = 300.6;
    sfra1.freqVect[i++] = 300.8;
    sfra1.freqVect[i++] = 301;
    sfra1.freqVect[i++] = 305;
    sfra1.freqVect[i++] = 310;
    sfra1.freqVect[i++] = 330;
    sfra1.freqVect[i++] = 360;
    sfra1.freqVect[i++] = 400;
    sfra1.freqVect[i++] = 410;
    sfra1.freqVect[i++] = 415;
    sfra1.freqVect[i++] = 417;
    sfra1.freqVect[i++] = 418;
    sfra1.freqVect[i++] = 419;
    sfra1.freqVect[i++] = 419.2;
    sfra1.freqVect[i++] = 419.4;
    sfra1.freqVect[i++] = 419.6;
    sfra1.freqVect[i++] = 419.8;
    sfra1.freqVect[i++] = 420;
    sfra1.freqVect[i++] = 420.2;
    sfra1.freqVect[i++] = 420.4;
    sfra1.freqVect[i++] = 420.6;
    sfra1.freqVect[i++] = 420.8;
    sfra1.freqVect[i++] = 421;
    sfra1.freqVect[i++] = 425;
    sfra1.freqVect[i++] = 440;
    sfra1.freqVect[i++] = 490;
    sfra1.freqVect[i++] = 530;
    sfra1.freqVect[i++] = 535;
    sfra1.freqVect[i++] = 537;
    sfra1.freqVect[i++] = 538;
    sfra1.freqVect[i++] = 539;
    sfra1.freqVect[i++] = 539.2;
    sfra1.freqVect[i++] = 539.4;
    sfra1.freqVect[i++] = 539.6;
    sfra1.freqVect[i++] = 539.8;
    sfra1.freqVect[i++] = 540;
    sfra1.freqVect[i++] = 540.2;
    sfra1.freqVect[i++] = 540.4;
    sfra1.freqVect[i++] = 540.6;
    sfra1.freqVect[i++] = 540.8;
    sfra1.freqVect[i++] = 545;
    sfra1.freqVect[i++] = 580;
    sfra1.freqVect[i++] = 630;
    sfra1.freqVect[i++] = 700;
    sfra1.freqVect[i++] = 800;
    sfra1.freqVect[i++] = 900;
    sfra1.freqVect[i++] = 1000;
    */
    #endif

    SFRA_GUI_config(SFRA_GUI_SCI_BASE,
                    SCI_VBUS_CLK,
                    SFRA_GUI_SCI_BAUDRATE,
                    SFRA_GUI_SCIRX_GPIO,
                    SFRA_GUI_SCIRX_GPIO_PIN_CONFIG,
                    SFRA_GUI_SCITX_GPIO,
                    SFRA_GUI_SCITX_GPIO_PIN_CONFIG,
                    SFRA_GUI_LED_INDICATOR,
                    SFRA_GUI_LED_GPIO,
                    SFRA_GUI_LED_GPIO_PIN_CONFIG,
                    &sfra1,
                    1);

}

//TODO initGlobalVariables()
void initGlobalVariables(void)
{
    // Gui Variables
    guiVbus = 0;
    guiACFreq = 0;
    guiPrms = 0;
    guiIrms = 0;
    guiVrms = 0;
    guiVavg = 0;
    guiFreqAvg = 0;
    guiVema = 0;

    dLog1.inputPtr1 = &dVal1;
    dLog1.inputPtr2 = &dVal2;
    dLog1.inputPtr3 = &dVal3;  // spll.Out logged in buff3
    dLog1.inputPtr4 = &dVal4;  // ramp_sin logged in buff4
    dLog1.outputPtr1 = dBuff1;
    dLog1.outputPtr2 = dBuff2;
    dLog1.outputPtr3 = dBuff3;
    dLog1.outputPtr4 = dBuff4;
    dLog1.prevValue = (float32_t)(0);
    dLog1.trigValue = (float32_t)(0.1);
    dLog1.status = 1;
    dLog1.preScalar = 5;
    dLog1.skipCount = 0;
    dLog1.size = 200;
    dLog1.count = 0;

    //
    // Variable initialization
    invIiInst = 0;
    invIiOffset = 0;

    invIfInst = 0;
    invIfSDOffset = 0;
    invIoInst = 0;
    invIoSDOffset = 0;
    invVoInst = 0;
    invVoOffset = 0.5f;
    invVoSDOffset = 0;
    invVbusInst = 0;
    invVbusOffset = 0;
    invVbusAvg = 0;
    invVbusSDOffset = 0;
    invModIndex = 0;
    invIiRef = 0;
    invIoRef = 0;
    invIoRefInst = 0;
    invVoRef = 0;
    invIiRefInst = 0;
    invVoRefInst = 0;
    invVbusRef = 0;
    offsetCalCounter = 0;
    invDutyPU = 0;
    invDuty = 0;
    halfPWMPeriod = 0;


    closeILoopInv = 0;
    closeVLoopInv = 0;
    clearInvTrip = 0;
    rlyConnect = 0;
    invSine = 0;
    invSinePrev = 0;
    zeroCrossDetectFlag = 0;
}

// TODO computeDF22_PIcontrollerCoeff()
void computeDF22_PIcontrollerCoeff(DCL_DF22 *v, float32_t gain, float32_t z0,
                                   float32_t fsw)
{
    float32_t temp1;
    temp1 = gain/(2*fsw);

    v->b0=(2.0f*fsw+z0)*temp1;
    v->b1=(-2.0f*fsw+z0)*temp1;
    v->b2 = 0;
    v->a1=-1.0f;
    v->a2 = 0;
    v->x1 = 0;
    v->x2 = 0;
}

//TODO computeDF22_PRcontrollerCoeff(
void computeDF22_PRcontrollerCoeff(DCL_DF22 *v, float32_t kp, float32_t ki,
                                   float32_t wo, float32_t fs, float32_t wrc )
{
    float32_t temp1, temp2, wo_adjusted;
    wo_adjusted = 2.0f*fs*tanf(wo/(2.0f*fs));

    temp1 = 4.0f*fs*fs+wo_adjusted*wo_adjusted+4.0f*fs*wrc;
    temp2 = 4.0f*ki*wrc*fs/temp1;
    v->b0 = temp2;
    v->b1 = 0;
    v->b2=-temp2;
    v->a1=((-8.0f*fs*fs+2.0f*wo_adjusted*wo_adjusted)/temp1);
    v->a2=((temp1-8.0f*fs*wrc)/temp1);
    v->x1 = 0;
    v->x2 = 0;

    if(kp!=0)
    {
        v->b0+=kp;
        v->b1+=kp*v->a1;
        v->b2+=kp*v->a2;
    }

    v->a1=(v->a1);
    v->a2=(v->a2);
}

//TODO updateBoardStatus()
void updateBoardStatus(void)
{
    if(EPWM_getTripZoneFlagStatus(INV_PWM1_BASE)==0 &&
       EPWM_getTripZoneFlagStatus(INV_PWM2_BASE)==0)
    {
        boardStatus = boardStatus_NoFault;
    }
    else
    {
        if((EPWM_getTripZoneFlagStatus(INV_PWM1_BASE)&EPWM_TZ_INTERRUPT_DCAEVT1)!=0 ||
           (EPWM_getTripZoneFlagStatus(INV_PWM2_BASE)&EPWM_TZ_INTERRUPT_DCAEVT1)!=0)
        {
            boardStatus = boardStatus_OverCurrentTrip;

            if(closeILoopInv==1)
            {
                rlyConnect = 0;
                openRelay();
                invVoRef = 0;
                invIiRef = 0;
            }

        }
        else if((EPWM_getTripZoneFlagStatus(INV_PWM1_BASE)&EPWM_TZ_INTERRUPT_CBC)!=0 ||
                (EPWM_getTripZoneFlagStatus(INV_PWM2_BASE)&EPWM_TZ_INTERRUPT_CBC)!=0)
        {
            boardStatus = boardStatus_EmulatorStopTrip;
            rlyConnect = 0;
            openRelay();
            invVoRef = 0;
            invIiRef = 0;
        }
        else if ((EPWM_getTripZoneFlagStatus(INV_PWM1_BASE)& EPWM_TZ_INTERRUPT_OST)!=0 &&
                 (EPWM_getTripZoneFlagStatus(INV_PWM2_BASE)&EPWM_TZ_INTERRUPT_OST)!=0 )
        {
            boardStatus = boardStatus_Idle;
        }
    }
}

//===========================================================================
// No more.
//===========================================================================
