#include <board.h>
#include <pio/pio.h>
#include <spi/spi.h>

#include "tsc2046.h"

#define ADS_CTRL_PD0              (1 << 0)        // PD0
#define ADS_CTRL_PD1              (1 << 1)        // PD1
#define ADS_CTRL_DFR              (1 << 2)        // SER/DFR
#define ADS_CTRL_EIGHT_BITS_MOD   (1 << 3)        // Mode
#define ADS_CTRL_START            (1 << 7)        // Start Bit
#define ADS_CTRL_SWITCH_SHIFT     4               // Address setting

// Get X position command
#define CMD_Y_POSITION ((1 << ADS_CTRL_SWITCH_SHIFT) | ADS_CTRL_START | ADS_CTRL_PD0 | ADS_CTRL_PD1)

// Get Y position command
#define CMD_X_POSITION ((5 << ADS_CTRL_SWITCH_SHIFT) | ADS_CTRL_START | ADS_CTRL_PD0 | ADS_CTRL_PD1)

// Enable penIRQ
#define CMD_ENABLE_PENIRQ  ((1 << ADS_CTRL_SWITCH_SHIFT) | ADS_CTRL_START)

#define AT91C_TOUCHSCREEN_TIMEOUT        5000000

#define DELAY_BEFORE_SPCK          (200 << 16) // 2us min (tCSS) <=> 200/100 000 000 = 2us
#define DELAY_BETWEEN_CONS_COM     (0xf << 24) // 5us min (tCSH) <=> (32 * 15) / (100 000 000) = 5us
  
/// Pins used by SPI
static const Pin pinsSPI[]  = {BOARD_TSC_SPI_PINS, BOARD_TSC_NPCS_PIN};

/// Touch screen BUSY pin
static const Pin pinBusy[] = {PIN_TCS_BUSY};

/// Touch screen CS pin
static const Pin pinNss[]  = {BOARD_TSC_NPCS_PIN};

static unsigned int SendCommand(unsigned char bCmd)
{
    int i;
    AT91S_SPI *spi = BOARD_TSC_SPI_BASE;
    unsigned int  uResult = 0;
    //unsigned int  uTimeout = 0;
    // (volatile declaration needed for code optimisation by compiler)
    volatile unsigned char bufferRX[3];
    volatile unsigned char bufferTX[3];

    bufferRX[0] = 0;
    bufferRX[1] = 0;
    bufferRX[2] = 0;

    bufferTX[0] = bCmd;
    bufferTX[1] = 0;
    bufferTX[2] = 0;

    PIO_Clear(pinNss);
    // Send command
    i = 0;
    while ((spi->SPI_SR & AT91C_SPI_TXEMPTY) == 0);
    spi->SPI_TDR = bufferTX[i] | SPI_PCS(BOARD_TSC_NPCS);
    while ((spi->SPI_SR & AT91C_SPI_TDRE) == 0);

    while (PIO_Get(pinBusy) == 1);
    
    while ((spi->SPI_SR & AT91C_SPI_RDRF) == 0);
    bufferRX[i] = spi->SPI_RDR & 0xFFFF;

    // Read data
    for (i = 1; i < 3; i++) 
    {
        while ((spi->SPI_SR & AT91C_SPI_TXEMPTY) == 0);
        spi->SPI_TDR = bufferTX[i] | SPI_PCS(BOARD_TSC_NPCS);
        while ((spi->SPI_SR & AT91C_SPI_TDRE) == 0);

        while ((spi->SPI_SR & AT91C_SPI_RDRF) == 0);
        bufferRX[i] = spi->SPI_RDR & 0xFFFF;
    }
    PIO_Set(pinNss);

    uResult = ((unsigned int)bufferRX[1] << 8) | (unsigned int)bufferRX[2];
    uResult = uResult >> 4;

    return uResult;
}

void tsc2046_get_position(unsigned int* px_pos, unsigned int* py_pos)
{
    // Get X position
    *px_pos = SendCommand(CMD_X_POSITION);
    
    // Get Y position
    *py_pos = SendCommand(CMD_Y_POSITION);
    
    // Switch to full power mode
    SendCommand(CMD_ENABLE_PENIRQ);
}

void tsc2046_init(void)
{
    volatile unsigned int uDummy;
      
    // Configure pins
    PIO_Configure(pinsSPI, PIO_LISTSIZE(pinsSPI));

    PIO_Set(pinNss);
       
    PIO_Configure(pinBusy, PIO_LISTSIZE(pinBusy));

    SPI_Configure(AT91C_BASE_SPI0,
                  AT91C_ID_SPI0,
                  AT91C_SPI_MSTR | AT91C_SPI_MODFDIS | SPI_PCS(BOARD_TSC_NPCS) // Value of the SPI configuration register.
    );

    SPI_ConfigureNPCS(AT91C_BASE_SPI0, BOARD_TSC_NPCS,
                      AT91C_SPI_NCPHA | (AT91C_SPI_DLYBS & DELAY_BEFORE_SPCK) |
                      (AT91C_SPI_DLYBCT & DELAY_BETWEEN_CONS_COM) | (0xC8 << 8) );

    SPI_Enable(AT91C_BASE_SPI0);

    for (uDummy=0; uDummy<100000; uDummy++);

    uDummy = AT91C_BASE_SPI0->SPI_SR;
    uDummy = AT91C_BASE_SPI0->SPI_RDR;

    SendCommand(CMD_ENABLE_PENIRQ);
}

void tsc2046_reset(void)
{
    // Disable SPI 0
    SPI_Disable(AT91C_BASE_SPI0);
}

