#include <board.h>
#include <irq/irq.h>
#include <pio/pio.h>
#include <pio/pio_it.h>
#include <systick/systick.h>
#include <string.h>
#include <dbgu/dbgu.h>
#include <stdio.h>
#include <string.h>

#include "system.h"
#include "tsd.h"
#include "tsc2046.h"
#include "uart.h"

/// Delay for pushbutton debouncing (in PIT_PERIOD time-base).
#define DEBOUNCE_TIME       6 // PIT_PERIOD * 6 = 60 ms

/// pen state
typedef enum {
    STATE_PEN_RELEASED  = 0,
    STATE_PEN_PRESSED   = 1,
    STATE_PEN_DEBOUNCE  = 2
} e_pen_state;

/// Pins used by Interrupt Signal for Touch Screen Controller
static const Pin pinPenIRQ  = PIN_TCS_IRQ;

/// Global timestamp in milliseconds since start of application.
static volatile unsigned int timestamp = 0;

/// last time when the pen is pressed on the touchscreen
static volatile unsigned int timePress = 0;

/// last time when the pen is released
static volatile unsigned int timeRelease = 0;

/// pen state
static volatile e_pen_state penState = STATE_PEN_RELEASED;

/// Touch screen initiallized flag 
static unsigned int tsInitFlag = 0;

uint16 finger_pos_x[1];
uint16 finger_pos_y[1];

void TSD_PenPressed(uint32_t x, uint32_t y);
void TSD_PenMoved(uint32_t x, uint32_t y);
void TSD_PenReleased(uint32_t x, uint32_t y);

void TSD_GetRawMeasurement(unsigned int *pData);

void send_finger_position(uint32_t x, uint32_t y)
{          
    uint8 len;
    
    finger_pos_x[0] = x;
    finger_pos_y[0] = y;
    
    len = encode_finger_position_frame();
    send_frame(len);
}

void TSD_PenPressed(uint32_t x, uint32_t y)
{
#if (ENABLE_DEBUG_UART_GUI & ENABLE_DEBUG_UART_GUI_FINGER_POS)
    send_finger_position(x,y);
#endif
    
#if (ENABLE_DEBUG_UART & ENABLE_DEBUG_UART_FINGER_POS)   
    printf("Pen pressed at  (%04u, %04u)\n\r", x, y);
#endif         
}

void TSD_PenMoved(uint32_t x, uint32_t y)
{
#if (ENABLE_DEBUG_UART_GUI & ENABLE_DEBUG_UART_GUI_FINGER_POS)
    send_finger_position(x,y);
#endif 
    
#if (ENABLE_DEBUG_UART & ENABLE_DEBUG_UART_FINGER_POS)
    printf("Pen moved at    (%04u, %04u)\n\r", x, y);
#endif
}

void TSD_PenReleased(uint32_t x, uint32_t y)
{
#if (ENABLE_DEBUG_UART_GUI & ENABLE_DEBUG_UART_GUI_FINGER_POS)
    send_finger_position(x,y);
#endif
    
#if (ENABLE_DEBUG_UART & ENABLE_DEBUG_UART_FINGER_POS)
    printf("Pen released at (%04u, %04u)\n\r", x, y);
#endif
}

//------------------------------------------------------------------------------
/// Timer handler for touch screen. Increments the timestamp counter.
/// Determine the state "Pen Pressed" or "Pen Released". To change state,
/// the penIRQ has to keep the same value during DEBOUNCE_TIME.
//------------------------------------------------------------------------------
void TSD_TimerHandler(void)
{
    unsigned int data[2];
    static unsigned int point[2];

    if (!tsInitFlag) return;

    timestamp++;
    
    // Get the current position of the pen if penIRQ has low value (pen pressed)
    if (PIO_Get(&pinPenIRQ) == 0) 
    {
        // Get the current position of the pressed pen
        TSD_GetRawMeasurement(data);
        point[0] = data[0];
        point[1] = data[1];

        // call the callback function
        if(penState == STATE_PEN_PRESSED) 
        {
           TSD_PenMoved(point[0], point[1]);
        }
    }

    // Determine the pen state
    if (PIO_Get(&pinPenIRQ) == 0) 
    {
        // reinit the last time when release
        timeRelease = timestamp;

        if(penState == STATE_PEN_DEBOUNCE) 
        {
            if( (timestamp - timePress) > DEBOUNCE_TIME) 
            {
                // pen is pressed during an enough time : the state change
                penState = STATE_PEN_PRESSED;
                
                // call the callback function
                TSD_PenPressed(point[0], point[1]);
            }
        }
    }
    else 
    {
        // reinit the last time when release
        timePress = timestamp;

        if(penState == STATE_PEN_DEBOUNCE) 
        {
            if( (timestamp - timeRelease) > DEBOUNCE_TIME) 
            {
                // pen is released during an enough time : the state change
                penState = STATE_PEN_RELEASED;
                
                // call the callback function
                TSD_PenReleased(point[0], point[1]);
            }
        }
    }
}

static void ISR_PenIRQ(void)
{
  // Check if the pen has been pressed
  if (!PIO_Get(&pinPenIRQ)) 
  {
    if(penState == STATE_PEN_RELEASED) 
    {
      timePress = timestamp;
      penState = STATE_PEN_DEBOUNCE;
    }
  }
  else 
  {
    if(penState == STATE_PEN_PRESSED) 
    {
      timeRelease = timestamp;
      penState = STATE_PEN_DEBOUNCE;
    }
  }
}

void ConfigurePenIRQ(void)
{
  // Configure pios
  PIO_Configure(&pinPenIRQ, PIO_LISTSIZE(pinPenIRQ));
  
  // Initialize interrupts
  PIO_InitializeInterrupts(0);
  
  PIO_ConfigureIt(&pinPenIRQ, (void (*)(const Pin *)) ISR_PenIRQ);
  
  // Enable the interrupt
  PIO_EnableIt(&pinPenIRQ);
}

void TSD_GetRawMeasurement(unsigned int *pData)
{
  // Get the current position of the pressed pen
  tsc2046_get_position(&pData[0], &pData[1]);
}

void TSD_Initialize(void)
{
  tsc2046_init();
  
  ConfigurePenIRQ();
  
  tsInitFlag = 1;
}

void TSD_Reset(void)
{
  // Disable SPI 0
  tsc2046_reset();
  
  // Disable the interrupt
  PIO_DisableIt(&pinPenIRQ);
}
