//*****************************************************************************
//
// Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
//
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//    Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
//
//    Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the
//    distribution.
//
//    Neither the name of Texas Instruments Incorporated nor the names of
//    its contributors may be used to endorse or promote products derived
//    from this software without specific prior written permission.
//
//  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
//  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
//  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
//  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
//  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
//  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
//  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//*****************************************************************************

//*****************************************************************************
//
// Application Name     - Timer Count Capture
// Application Overview - This application showcases Timer's count capture 
//                        feature to measure frequency of an external signal.
// Application Details  -
// http://processors.wiki.ti.com/index.php/CC32xx_Timer_Count_Capture_Application
// or
// docs\examples\CC32xx_Timer_Count_Capture_Application.pdf
//
//*****************************************************************************

// Driverlib includes
#include "hw_types.h"
#include "hw_ints.h"
#include "hw_memmap.h"
#include "interrupt.h"
#include "prcm.h"
#include "gpio.h"
#include "utils.h"
#include "timer.h"
#include "rom.h"
#include "rom_map.h"
#include "pin.h"

// Common interface includes
#include "uart_if.h"
#include "pinmux.h"


#define APPLICATION_VERSION     "1.1.1"
#define APP_NAME        "Timer Count Capture"
#define TIMER_FREQ      80000000

//*****************************************************************************
//                 GLOBAL VARIABLES -- Start
//*****************************************************************************
static unsigned long g_ulSamples[3];
static unsigned long g_ulFreq;

#if defined(ccs) || defined(gcc)
extern void (* const g_pfnVectors[])(void);
#endif
#if defined(ewarm)
extern uVectorEntry __vector_table;
#endif
//*****************************************************************************
//                 GLOBAL VARIABLES -- End
//*****************************************************************************


//*****************************************************************************
//
//! Timer interrupt handler
//
//*****************************************************************************
static void TimerIntHandler()
{
    //
    // Clear the interrupt
    //
    MAP_TimerIntClear(TIMERA2_BASE,TIMER_CAPA_EVENT);

    //
    // Get the samples and compute the frequency
    //
//    g_ulSamples[0] = g_ulSamples[1];       //һβļֵֵg_ulSamples[0]
//    g_ulSamples[1] = MAP_TimerValueGet(TIMERA2_BASE,TIMER_A);   //²ֵ
//    g_ulFreq = (TIMER_FREQ/(g_ulSamples[0] - g_ulSamples[1]));
    
    g_ulSamples[g_ulFreq++] = MAP_TimerValueGet(TIMERA2_BASE,TIMER_A); 
}


//*****************************************************************************
//
//! Application startup display on UART
//!
//! \param  none
//!
//! \return none
//!
//*****************************************************************************
void
DisplayBanner(char * AppName)
{

    Report("\n\n\n\r");
    Report("\t\t *************************************************\n\r");
    Report("\t\t\t  CC3200 %s Application       \n\r", AppName);
    Report("\t\t *************************************************\n\r");
    Report("\n\n\n\r");
}

//*****************************************************************************
//
//! Board Initialization & Configuration
//!
//! \param  None
//!
//! \return None
//
//*****************************************************************************
void
BoardInit(void)
{
/* In case of TI-RTOS vector table is initialize by OS itself */
#ifndef USE_TIRTOS
  //
  // Set vector table base
  //
#if defined(ccs) || defined(gcc)
    MAP_IntVTableBaseSet((unsigned long)&g_pfnVectors[0]);
#endif
#if defined(ewarm)
    MAP_IntVTableBaseSet((unsigned long)&__vector_table);
#endif
#endif
    //
    // Enable Processor
    //
    MAP_IntMasterEnable();
    MAP_IntEnable(FAULT_SYSTICK);

    PRCMCC3200MCUInit();
}

void SetupTimerPWMMode(unsigned long ulBase, unsigned long ulTimer,
                       unsigned long ulConfig, unsigned char ucInvert)
{
    //
    // Set GPT - Configured Timer in PWM mode.
    //
    MAP_TimerConfigure(ulBase,ulConfig);
    MAP_TimerPrescaleSet(ulBase,ulTimer,0);
    
    //
    // Inverting the timer output if required
    //
    MAP_TimerControlLevel(ulBase,ulTimer,ucInvert);   //ߵ͵ƽ 1=ture-͵ƽ  0=False-ߵƽ
                                                      //LoadSet=MatchSetֵʱPWMĵƽǸߵƽǵ͵ƽ
    //                                                1=ture-͵ƽ  LoadSetMatchSet ʱPWM͵ƽ ---Green/Yellow LED
    //                                                               LoadSet<MatchSet PWMߵƽ
    //                                                0=False-ߵƽ
    //                                                               LoadSetMatchSet ʱPWMߵƽ ---RED LED
    //                                                               LoadSet<MatchSet PWM͵ƽ
    //
    
                                                                                                                         
}

void InitPWMModules()
{
    //
    // Initialization of timers to generate PWM output
    //
    MAP_PRCMPeripheralClkEnable(PRCM_TIMERA2, PRCM_RUN_MODE_CLK);
    MAP_PRCMPeripheralClkEnable(PRCM_TIMERA3, PRCM_RUN_MODE_CLK);

    //
    // TIMERA2 (TIMER B) as RED of RGB light. GPIO 9 --> PWM_5
    //
    SetupTimerPWMMode(TIMERA2_BASE, TIMER_B,(TIMER_CFG_SPLIT_PAIR | TIMER_CFG_B_PWM), 0);
    //
    // TIMERA3 (TIMER B--󣡣) as YELLOW of RGB light. GPIO 10 --> PWM_6
    //
//    SetupTimerPWMMode(TIMERA3_BASE, TIMER_A, (TIMER_CFG_SPLIT_PAIR | TIMER_CFG_A_PWM | TIMER_CFG_B_PWM), 1);
//  ͬPWM_6--Pin1   
    SetupTimerPWMMode(TIMERA3_BASE, TIMER_A, (TIMER_CFG_SPLIT_PAIR | TIMER_CFG_A_PWM ), 1);     //дͰλϢ
    //
    // TIMERA3 (TIMER A--󣡣) as GREEN of RGB light. GPIO 11 --> PWM_7
//    SetupTimerPWMMode(TIMERA3_BASE, TIMER_B, (TIMER_CFG_SPLIT_PAIR | TIMER_CFG_A_PWM | TIMER_CFG_B_PWM), 1);
    
// ޸-(1)PinMux-PWM -->PIN_02 for TimerPWM7  GT_PWM07
//      (2)ͨPWM_OUT7--TRMTable 9-1. Available CCP Pins and PWM Outputs/Signals Pins ҵӦĶʱ --- Timer3->Timer B
//      (3)TIMERA3_BASE  TIMER_B  TIMER_CFG_SPLIT_PAIR/TIMER_CFG_B_PWM ĬPWMƽ ---OK
// ע⣺PWM_OUT6PWM_OUT7ͬʱʹõĻһTIMER_BдTIMER_CFG_A_PWMãὫTIMER_A
// TIMER_CFG_A_PWM          0x0000000A
// TIMER_CFG_B_PWM          0x00000A00  ---յ£
//    HWREG(ulBase + TIMER_O_TAMR) = ulConfig & 255;         //ȡͰλ
//    HWREG(ulBase + TIMER_O_TBMR) = (ulConfig >> 8) & 255;  //ȡͰλǰλ    ---ԼӸж
    SetupTimerPWMMode(TIMERA3_BASE, TIMER_B, (TIMER_CFG_SPLIT_PAIR | TIMER_CFG_A_PWM | TIMER_CFG_B_PWM), 1);  //Ͱλ --TIMER_B --OK


}

//*****************************************************************************
//
//! Main  Function
//
//*****************************************************************************
int main()
{
    //
    // Initialize Board configurations
    //
    BoardInit();

    //
    // Pinmux for UART
    //
    PinMuxConfig();
    
        
    //
    // Enable pull down
    //
    //    MAP_PinConfigSet(PIN_05,PIN_TYPE_STD_PD,PIN_STRENGTH_6MA);

    //
    // Configuring UART
    //
    InitTerm();

    //
    // Display Application Banner
    //
    DisplayBanner(APP_NAME);
    //
    // Register timer interrupt hander
    //
    MAP_TimerIntRegister(TIMERA2_BASE,TIMER_A,TimerIntHandler); //ʱA¼
    //
    // Enable capture event interrupt
    //
    MAP_TimerIntEnable(TIMERA2_BASE,TIMER_CAPA_EVENT);   
    //
    // Configure the timer in edge count mode
//      (1)PinMux-PWM -->PIN_04 for GT_CCP04
//      (2)ͨGT_CCP04--TRMTable 9-1. Available CCP Pins and PWM Outputs/Signals Pins ҵӦĶʱ --- Timer2->Timer A
    MAP_TimerConfigure(TIMERA2_BASE, (TIMER_CFG_SPLIT_PAIR | TIMER_CFG_A_CAP_TIME | TIMER_CFG_B_PWM));
    MAP_TimerConfigure(TIMERA3_BASE, (TIMER_CFG_SPLIT_PAIR | TIMER_CFG_A_PWM | TIMER_CFG_B_PWM));
    
    //öʱ2/3ԤƵΪ200  ƵʱΪ80MHz/200=400kHz
    TimerPrescaleSet(TIMERA2_BASE,TIMER_BOTH,199);
    TimerPrescaleSet(TIMERA3_BASE,TIMER_BOTH,199);
    //
    // Set the detection edge
    // xxؽв
    MAP_TimerControlEvent(TIMERA2_BASE,TIMER_A,TIMER_EVENT_BOTH_EDGES); //˫

    //
    // Set the reload value
    //
    MAP_TimerLoadSet(TIMERA2_BASE,TIMER_A,65535);
    // PWMƵΪ
    MAP_TimerLoadSet(TIMERA2_BASE,TIMER_B,60000);  //6000/400kHz=150mS
    MAP_TimerLoadSet(TIMERA3_BASE,TIMER_A,40000);  //100ms
    MAP_TimerLoadSet(TIMERA3_BASE,TIMER_B,20000);  //50ms
    // 50% duty
    MAP_TimerMatchSet(TIMERA2_BASE,TIMER_B,30000); //75ms
    MAP_TimerMatchSet(TIMERA3_BASE,TIMER_A,20000);
    MAP_TimerMatchSet(TIMERA3_BASE,TIMER_B,10000);

    //
    // Enable Timer
    //
    MAP_TimerEnable(TIMERA2_BASE,TIMER_A);
    MAP_TimerEnable(TIMERA2_BASE,TIMER_B);
    MAP_TimerEnable(TIMERA3_BASE,TIMER_A);
    MAP_TimerEnable(TIMERA3_BASE,TIMER_B);
  
    while(1)
    {
        if(g_ulFreq>=3)
        {
             MAP_TimerIntDisable(TIMERA2_BASE,TIMER_A); //ֹʱ ٽ뱣
          
             Report("Period: %03dms",(65536 + g_ulSamples[0]-g_ulSamples[2])/400);
             Report("Width: %03dms",(65536 + g_ulSamples[0]-g_ulSamples[1])/400);
             g_ulFreq=0;
             MAP_UtilsDelay(80000000/5);
             MAP_TimerIntEnable(TIMERA2_BASE,TIMER_A);  
        }
      
      
        //
        // Report the calculate frequency
        //
//        Report("Frequency : %d Hz\n\n\r",g_ulFreq);
//
//        //
//        // Delay loop
//        //
//        MAP_UtilsDelay(80000000/5);
    }
}
