/*
 *
 *   Copyright (C) 2016 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission of
 *   Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been supplied,
 *   and under no circumstances can it be used with non-TI connectivity device.
 *
 */

#ifndef __RADIOTOOL_H__
#define __RADIOTOOL_H__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#include <ti/drivers/net/wifi/simplelink.h>

#ifdef RADIO_TOOL_WINDOWS_DLL
#define RTAPI __declspec(dllexport) _i32
#else
#define RTAPI _i32
#endif


#define MSEC_SLEEP(x)\
								if((x * 1000) >= 1000000)\
								{\
									sleep(x /1000);\
								}\
								else\
								{\
									usleep(1000*x);\
								}\


#define RADIO_CMD_BUFF_SIZE_MAX 100
#define DEV_VER_LENGTH          136

#define CW_LOW_TONE             (-25)
#define CW_HIGH_TONE            (25)
#define CW_STOP                 (128)

/* Radio Tool Error Codes */
#define RADIO_TOOL_ERROR_STARTING_RADIO_TOOL        (-101)
#define RADIO_TOOL_ERROR_STOPPING_RADIO_TOOL        (-102)
#define RADIO_TOOL_ERROR_GETTING_MAC_ADDR           (-110)
#define RADIO_TOOL_ERROR_GETTING_DEV_VERSION        (-111)

#define RADIO_TOOL_ERROR_TX_TYPE_UNKNOWN            (-100)
#define RADIO_TOOL_ERROR_TX_CREATING_RAW_SOCKET     (-101)
#define RADIO_TOOL_ERROR_TX_FULL_SIZE_DATA          (-102)

#define RADIO_TOOL_ERROR_RX_CREATING_RAW_SOCKET     (-103)

#define RADIO_TOOL_ERROR_GETTING_RX_STATS           (-111)

#define RADIO_TOOL_TX_POWER_LEVEL_MIN               (0)
#define RADIO_TOOL_TX_POWER_LEVEL_MAX               (15)
#define RADIO_TOOL_TX_CW_TONE_MIN                   (-25)
#define RADIO_TOOL_TX_CW_TONE_MAX                   (25)
#define RADIO_TOOL_TX_CHANNEL_MIN                   (1)
#define RADIO_TOOL_TX_CHANNEL_MAC                   (13)
#define RADIO_TOOL_TX_PREAMBLE_MIN                  (0)
#define RADIO_TOOL_TX_PREAMBLE_MAX                  (1)
#define RADIO_TOOL_TX_SIZE_MIN                      (24)
#define RADIO_TOOL_TX_SIZE_MAX                      (1400)
#define RADIO_TOOL_TX_DELAY_MIN                     (100)
#define RADIO_TOOL_TX_DELAY_MAX                     (1000000)
#define RADIO_TOOL_TX_AMOUNT_MIN                    (0)
#define RADIO_TOOL_TX_AMOUNT_MAX                    (1000000)
#define RADIO_TOOL_TX_CCA_THRESHOLD_MIN             (0)
#define RADIO_TOOL_TX_CCA_THRESHOLD_MAX             SL_TX_INHIBIT_THRESHOLD_MAX


typedef enum
{
    RADIO_START_TX      = 0x1,
    RADIO_STOP_TX       = 0x2,
    RADIO_START_RX      = 0x3,
    RADIO_GET_STATS     = 0x4,
    RADIO_STOP_RX       = 0x5,
    MAX_RADIO_COMMANDS  = 0xff
}RadioCommands_e;

/* Radio Tool Data Structures */
typedef enum
{
    RADIO_TX_CONTINUOUS = 1,
    RADIO_TX_PACKETIZED = 2,
    RADIO_TX_CW         = 3,
    MAX_RADIO_TX_MODE   = 0xff
}RadioTxMode_e;

typedef enum
{
    PATTERN_ALL_0       = 0,
    PATTERN_ALL_1       = 1,
    PATTERN_INCREMENTAL = 2,
    PATTERN_DECREMENTAL = 3,
    PATTERN_PN9         = 4,
    PATTERN_PN15        = 5,
    PATTERN_PN23        = 6,
    MAX_NUM_PATTERN     = 0xFF
}RadioDataPattern_e;

typedef enum
{
    TX_START_COMMAND_OP     = 0x71,
    TX_STOP_COMMAND_OP      = 0x72,
    RX_START_COMMAND_OP     = 0x73,
    RX_GET_STATS_COMMAND_OP = 0x74,
    RX_STOP_COMMAND_OP      = 0x75,
    RX_GET_STATS_RESPOSE_OP = 0xE1,
    ACK_RESPONSE_OP         = 0xE2
}RadioOpcode_e;

void RadioToolCreate();

void RadioToolDestroy();

RTAPI RadioToolCommand(_u8 aucOpCode, _u8 *apucPayload);

/* */
RTAPI RadioStartTX(RadioTxMode_e eTxMode,
    _i8                ePowerLevel_Tone,
    _u8                eChannel,
    SlWlanRateIndex_e  eRate,
    _u8                ePreamble,
    RadioDataPattern_e eDataPattern,
    _u16               eSize,
    _u32               eDelay,
    _u32               eAmount,
    _u8                eOverrideCCA,
    SlTxInhibitThreshold_e eCCAThreshold,
    _u8                eEnableACKs,
    _u8                *epDstMac);

/*!
 *  \brief      Stops TX
 *  \param[in]  eTxMode - the TX mode the device was in. Needs to be
 *              identical to the eTxMode specified in RadioStartTX()
 *  \return     0 on success, -ve otherwise
 */
RTAPI RadioStopTX(RadioTxMode_e eTxMode);

/*!
 *  \brief      Starts RX
 *  \param[in]  eChannel - channel to scan
 *  \return     0 on success, -ve otherwise
 */
RTAPI RadioStartRX(_u8 eChannel);

/*!
 *  \brief      Stops RX
 *  \param      None
 *  \return     0 on success, -ve otherwise
 */
RTAPI RadioStopRX();

/*!
 *  \brief      Gets RX Statistics
 *  \param[out] pRxStats - RX Statistics, see SlWlanGetRxStatResponse_t
 *  \return     0 on success, -ve otherwise
 */
RTAPI RadioGetStats(_u8 *pRxStats);

/*!
 *  \brief      Gets the MAC address of the device
 *  \param[out] pMacAddress
 *  \return     0 on success, -ve otherwise
 */
RTAPI RadioGetMacAddr(_u8 *pMacAddress);

/*!
 *  \brief      Gets device version information
 *  \param[out] pDevVersion - Device version details, see SlDeviceVersion_t
 *  \param[out] pHDLength - length of the pHDVersion text
 *  \param[out] pHDVersion - Host Driver version
 *  \return     0 on success, -ve otherwise
 */
RTAPI RadioGetDeviceVersion(_u8 *pDevVersion, _u16 *pHDLength, _u8 *pHDVersion);

#ifdef  __cplusplus
}
#endif // __cplusplus
#endif  // __RADIOTOOL_H__
