/******************************************************************************
*
*   user.h - CC31xx/CC32xx Host Driver Implementation
*
*   Copyright (C) 2016 Texas Instruments Incorporated
*
*   All rights reserved. Property of Texas Instruments Incorporated.
*   Restricted rights to use, duplicate or disclose this code are
*   granted through contract.
*
*   The program may not be used without the written permission of
*   Texas Instruments Incorporated or against the terms and conditions
*   stipulated in the agreement under which this program has been supplied,
*   and under no circumstances can it be used with non-TI connectivity device.
*
******************************************************************************/

#ifndef __USER_H__
#define __USER_H__


#ifdef  __cplusplus
extern "C" {
#endif
/*!

    \addtogroup Porting
    @{

*/

/*!
 ******************************************************************************

    \defgroup       porting_user_include        User Include Files

    This section IS NOT REQUIRED in case user provided primitives are handled
    in makefiles or project configurations (IDE)

    PORTING ACTION:
        - Include all required header files for the definition of:
            -# Transport layer library API (e.g. SPI, UART)
            -# OS primitives definitions (e.g. Task spawn, Semaphores)
            -# Memory management primitives (e.g. alloc, free)

 ******************************************************************************
 */

#include "slstudio.h"
#include "slosi.h"
#include <stdio.h>

//#include "board.h"
/*
 * In order to "install" external SimpleLink library one should follow the next steps:
 * 1. Include the external library API header file (i.e. #include "IOT.h")
 * 2. Define each one of the desired external lib with one (out of 5) of the following:
 *    #define SL_EXT_LIB_1 <Your external lib name>
      #define SL_EXT_LIB_2 <Your external lib name>
      #define SL_EXT_LIB_3 <Your external lib name>
      #define SL_EXT_LIB_4 <Your external lib name>
      #define SL_EXT_LIB_5 <Your external lib name>

       i.e.
      #define SL_EXT_LIB_1 IOT
      #define SL_EXT_LIB_2 WAC

   3. Verify the corresponding event handlers are existing in the external lib API header file.
 *
 * */




typedef unsigned int       _SlFd_t;
typedef short (*SL_P_EVENT_HANDLER)(void* pValue);


#define SL_DRIVER_TIMEOUT_SHORT        (30000) /* msec units */
#define SL_DRIVER_TIMEOUT_LONG         (60000) /* msec units */
#define SYNC_PATTERN_TIMEOUT_IN_MSEC   (1000) /* the sync pattern timeout in milliseconds units */

#define SL_TIMESTAMP_TICKS_IN_10_MILLISECONDS     (_u32)(30205)
#define SL_TIMESTAMP_MAX_VALUE                    0xFFFFFFFF /* 32-bit timer counter */

/*!
 ******************************************************************************

    \defgroup       porting_capabilities        Capability Set Definition

    This section IS NOT REQUIRED in case one of the following pre defined
    capabilities set is in use:
    - SL_TINY
    - SL_SMALL
    - SL_FULL

    PORTING ACTION:
        - Define one of the pre-defined capabilities set or uncomment the
          relevant definitions below to select the required capabilities

    @{

 *******************************************************************************
 */




/*!
    \def        MAX_CONCURRENT_ACTIONS

    \brief      Defines the maximum number of concurrent action in the system
                Min:1 , Max: 32

                Actions which has async events as return, can be

    \sa

    \note       In case there are not enough resources for the actions needed in the system,
                error is received: POOL_IS_EMPTY
                one option is to increase MAX_CONCURRENT_ACTIONS
                (improves performance but results in memory consumption)
                Other option is to call the API later (decrease performance)

    \warning    In case of setting to one, recommend to use non-blocking recv\recvfrom to allow
                multiple socket recv
*/
#ifndef SL_TINY_EXT
#define MAX_CONCURRENT_ACTIONS 18
#else
#define MAX_CONCURRENT_ACTIONS 1
#endif
/*!
    \def        CPU_FREQ_IN_MHZ
    \brief      Defines CPU frequency for Host side, for better accuracy of busy loops, if any
    \sa
    \note

    \warning    If not set the default CPU frequency is set to 200MHz
                This option will be deprecated in future release
*/
/*
#define CPU_FREQ_IN_MHZ        25
*/


/*!
    \def        SL_INC_ARG_CHECK

    \brief      Defines whether the SimpleLink driver perform argument check
                or not

                When defined, the SimpleLink driver perform argument check on
                function call. Removing this define could reduce some code
                size and improve slightly the performances but may impact in
                unpredictable behaviour in case of invalid arguments

    \sa

    \note       belongs to \ref porting_sec

    \warning    Removing argument check may cause unpredictable behaviour in
                case of invalid arguments.
                In this case the user is responsible to argument validity
                (for example all handlers must not be NULL)
*/
#define SL_INC_ARG_CHECK


/*!
    \def        SL_INC_STD_BSD_API_NAMING

    \brief      Defines whether SimpleLink driver should expose standard BSD
                APIs or not

                When defined, the SimpleLink driver in addition to its alternative
                BSD APIs expose also standard BSD APIs.
                Standard BSD API includes the following functions:
                socket , close , accept , bind , listen , connect , select ,
                setsockopt  , getsockopt , recv , recvfrom , write , send , sendto ,
                gethostbyname

    \sa

    \note       belongs to \ref porting_sec

    \warning
*/
/*
#define SL_INC_STD_BSD_API_NAMING
*/

/*!
    \brief      Defines whether to include extended API in SimpleLink driver
                or not

                When defined, the SimpleLink driver will include also all
                extended API of the included packages

    \sa         ext_api

    \note       belongs to \ref porting_sec

    \warning
*/
#define SL_INC_EXT_API

/*!
    \brief      Defines whether to include WLAN package in SimpleLink driver
                or not

                When defined, the SimpleLink driver will include also
                the WLAN package

    \sa

    \note       belongs to \ref porting_sec

    \warning
*/
#define SL_INC_WLAN_PKG

/*!
    \brief      Defines whether to include SOCKET package in SimpleLink
                driver or not

                When defined, the SimpleLink driver will include also
                the SOCKET package

    \sa

    \note       belongs to \ref porting_sec

    \warning
*/
#define SL_INC_SOCKET_PKG

/*!
    \brief      Defines whether to include NET_APP package in SimpleLink
                driver or not

                When defined, the SimpleLink driver will include also the
                NET_APP package

    \sa

    \note       belongs to \ref porting_sec

    \warning
*/
#define SL_INC_NET_APP_PKG

/*!
    \brief      Defines whether to include NET_CFG package in SimpleLink
                driver or not

                When defined, the SimpleLink driver will include also
                the NET_CFG package

    \sa

    \note       belongs to \ref porting_sec

    \warning
*/
#define SL_INC_NET_CFG_PKG

/*!
    \brief      Defines whether to include NVMEM package in SimpleLink
                driver or not

                When defined, the SimpleLink driver will include also the
                NVMEM package

    \sa

    \note       belongs to \ref porting_sec

    \warning
*/
#define SL_INC_NVMEM_PKG
/*!
    \brief      Defines whether to include NVMEM extended package in SimpleLink
                driver or not

                When defined, the SimpleLink driver will include also the
                NVMEM extended package

    \sa

    \note       belongs to \ref nvmem_ext

    \warning
*/
#define SL_INC_NVMEM_EXT_PKG

/*!
    \brief      Defines whether to include socket server side APIs
                in SimpleLink driver or not

                When defined, the SimpleLink driver will include also socket
                server side APIs

    \sa         server_side

    \note

    \warning
*/

#define SL_INC_SOCK_SERVER_SIDE_API

/*!
    \brief      Defines whether to include socket client side APIs in SimpleLink
                driver or not

                When defined, the SimpleLink driver will include also socket
                client side APIs

    \sa         client_side

    \note       belongs to \ref porting_sec

    \warning
*/
#define SL_INC_SOCK_CLIENT_SIDE_API

/*!
    \brief      Defines whether to include socket receive APIs in SimpleLink
                driver or not

                When defined, the SimpleLink driver will include also socket
                receive side APIs

    \sa         recv_api

    \note       belongs to \ref porting_sec

    \warning
*/
#define SL_INC_SOCK_RECV_API

/*!
    \brief      Defines whether to include socket send APIs in SimpleLink
                driver or not

                When defined, the SimpleLink driver will include also socket
                send side APIs

    \sa         send_api

    \note       belongs to \ref porting_sec

    \warning
*/
#define SL_INC_SOCK_SEND_API

/*!

 Close the Doxygen group.
 @}

 */


/*!
 ******************************************************************************

    \defgroup   porting_enable_device       Device Enable/Disable IO

    The enable/disable API provide mechanism to enable/disable the network processor


    PORTING ACTION:
        - None
    @{

 ******************************************************************************
 */
/*!
    \brief      Preamble to the enabling the Network Processor.
                        Placeholder to implement any pre-process operations
                        before enabling networking operations.

    \sa         sl_DeviceEnable

    \note       belongs to \ref ported_sec

*/
#define sl_DeviceEnablePreamble()

/*!
    \brief      Enable the Network Processor

    \sa         sl_DeviceDisable

    \note       belongs to \ref porting_sec

*/
#define sl_DeviceEnable()           SlStudio_EnableDevice()

/*!
    \brief      Disable the Network Processor

    \sa         sl_DeviceEnable

    \note       belongs to \ref porting_sec
*/
#define sl_DeviceDisable()          SlStudio_DisableDevice()

/*!

 Close the Doxygen group.
 @}

 */

/*!
 ******************************************************************************

    \defgroup   porting_interface         Hardware Transport Interface

    The simple link device can work with different transport interfaces
    (namely,SPI or UART). Texas Instruments provides single driver
    that can work with all these types. This section binds the
    physical transport interface with the SimpleLink driver


    \note       Correct and efficient implementation of this driver is critical
                for the performances of the SimpleLink device on this platform.


    PORTING ACTION:
        - None

    @{

 ******************************************************************************
 */

#ifndef SL_IF_TYPE_UART
#define _SlFd_t                 SpiHandle_t
#else
#define _SlFd_t                 UartHandle_t
#endif

/*!
    \brief      Opens an interface communication port to be used for communicating
                with a SimpleLink device

                Given an interface name and option flags, this function opens
                the communication port and creates a file descriptor.
                This file descriptor is used afterwards to read and write
                data from and to this specific communication channel.
                The speed, clock polarity, clock phase, chip select and all other
                specific attributes of the channel are all should be set to hardcoded
                in this function.

    \param      ifName  -   points to the interface name/path. The interface name is an
                            optional attributes that the simple link driver receives
                            on opening the driver (sl_Start).
                            In systems that the spi channel is not implemented as
                            part of the OS device drivers, this parameter could be NULL.

    \param      flags   -   optional flags parameters for future use

    \return     upon successful completion, the function shall open the channel
                and return a non-negative integer representing the file descriptor.
                Otherwise, -1 shall be returned

    \sa         sl_IfClose , sl_IfRead , sl_IfWrite

    \note       The prototype of the function is as follow:
                    Fd_t xxx_IfOpen(char* pIfName , unsigned long flags);

    \note       belongs to \ref porting_sec

    \warning
*/
#ifndef SL_IF_TYPE_UART
#define sl_IfOpen                           SlStudio_SpiOpen
#else
#define sl_IfOpen                           SlStudio_UartOpen
#endif

/*!
    \brief      Closes an opened interface communication port

    \param      fd  -   file descriptor of opened communication channel

    \return     upon successful completion, the function shall return 0.
                Otherwise, -1 shall be returned

    \sa         sl_IfOpen , sl_IfRead , sl_IfWrite

    \note       The prototype of the function is as follow:
                    int xxx_IfClose(Fd_t Fd);

    \note       belongs to \ref porting_sec

    \warning
*/
#ifndef SL_IF_TYPE_UART
#define sl_IfClose                          SlStudio_SpiClose
#else
#define sl_IfClose                          SlStudio_UartClose
#endif

/*!
    \brief      Attempts to read up to len bytes from an opened communication channel
                into a buffer starting at pBuff.

    \param      fd      -   file descriptor of an opened communication channel

    \param      pBuff   -   pointer to the first location of a buffer that contains enough
                            space for all expected data

    \param      len     -   number of bytes to read from the communication channel

    \return     upon successful completion, the function shall return the number of read bytes.
                Otherwise, 0 shall be returned

    \sa         sl_IfClose , sl_IfOpen , sl_IfWrite


    \note       The prototype of the function is as follow:
                    int xxx_IfRead(Fd_t Fd , char* pBuff , int Len);

    \note       belongs to \ref porting_sec

    \warning
*/
#ifndef SL_IF_TYPE_UART
#define sl_IfRead                           SlStudio_SpiRead
#else
#define sl_IfRead                           SlStudio_UartRead
#endif

/*!
    \brief attempts to write up to len bytes to the SPI channel

    \param      fd      -   file descriptor of an opened communication channel

    \param      pBuff   -   pointer to the first location of a buffer that contains
                            the data to send over the communication channel

    \param      len     -   number of bytes to write to the communication channel

    \return     upon successful completion, the function shall return the number of sent bytes.
                otherwise, 0 shall be returned

    \sa         sl_IfClose , sl_IfOpen , sl_IfRead

    \note       This function could be implemented as zero copy and return only upon successful completion
                of writing the whole buffer, but in cases that memory allocation is not too tight, the
                function could copy the data to internal buffer, return back and complete the write in
                parallel to other activities as long as the other SPI activities would be blocked until
                the entire buffer write would be completed

               The prototype of the function is as follow:
                    int xxx_IfWrite(Fd_t Fd , char* pBuff , int Len);

    \note       belongs to \ref porting_sec

    \warning
*/
#ifndef SL_IF_TYPE_UART
#define sl_IfWrite                          SlStudio_SpiWrite
#else
#define sl_IfWrite                          SlStudio_UartWrite
#endif

/*!
    \brief      register an interrupt handler routine for the host IRQ

    \param      InterruptHdl    -   pointer to interrupt handler routine

    \param      pValue          -   pointer to a memory structure that is passed
                                    to the interrupt handler.

    \return     upon successful registration, the function shall return 0.
                Otherwise, -1 shall be returned

    \sa

    \note       If there is already registered interrupt handler, the function
                should overwrite the old handler with the new one

    \note       If the handler is a null pointer, the function should un-register the
                interrupt handler, and the interrupts can be disabled.

    \note       belongs to \ref porting_sec

    \warning
*/
#define sl_IfRegIntHdlr(InterruptHdl , pValue)          SlStudio_RegisterInterruptHandler(InterruptHdl , pValue)
/*!
    \brief      Masks the Host IRQ

    \sa     sl_IfUnMaskIntHdlr



    \note       belongs to \ref porting_sec

    \warning
*/
#ifndef SL_IF_TYPE_UART
#define sl_IfMaskIntHdlr()
#else
#define sl_IfMaskIntHdlr                            SlStudio_UartIRQMask
#endif

/*!
    \brief      Unmasks the Host IRQ

    \sa     sl_IfMaskIntHdlr



    \note       belongs to \ref porting_sec

    \warning
*/
#ifndef SL_IF_TYPE_UART
#define sl_IfUnMaskIntHdlr()
#else
#define sl_IfUnMaskIntHdlr                          SlStudio_UartIRQUnMask
#endif

/*!
    \brief      Write Handers for statistics debug on write

    \param      interface handler   -   pointer to interrupt handler routine


    \return     no return value

    \sa

    \note       An optional hooks for monitoring before and after write info

    \note       belongs to \ref porting_sec

    \warning
*/
#define SL_START_WRITE_STAT

#ifdef SL_START_WRITE_STAT
#define sl_IfStartWriteSequence                         SlStudio_SpiStartWriteSeq
#define sl_IfEndWriteSequence                           SlStudio_SpiEndWriteSeq
#endif

/*!
    \brief      Get the timer counter value (timestamp).
                The timer must count from zero to its MAX value.

    \param      None.


    \return     Returns 32-bit timer counter value (ticks unit)

    \sa

    \note

    \note       belongs to \ref porting_sec

    \warning
*/
#ifndef SL_TINY_EXT
#undef slcb_GetTimestamp
/* #define slcb_GetTimestamp                           SlStudio_GetTimestamp */
#endif

/*!

 Close the Doxygen group.
 @}

 */

/*!
 ******************************************************************************

    \defgroup   porting_mem_mgm             Memory Management

    This section declare in which memory management model the SimpleLink driver
    will run:
        -# Static
        -# Dynamic

    This section IS NOT REQUIRED in case Static model is selected.

    The default memory model is Static

    PORTING ACTION:
        - If dynamic model is selected, define the alloc and free functions.

    @{

 *****************************************************************************
 */

/*!
    \brief      Defines whether the SimpleLink driver is working in dynamic
                memory model or not

                When defined, the SimpleLink driver use dynamic allocations
                if dynamic allocation is selected malloc and free functions
                must be retrieved

    \sa

    \note       belongs to \ref porting_sec

    \warning
*/
/*
#define SL_MEMORY_MGMT_DYNAMIC
*/

#ifdef SL_MEMORY_MGMT_DYNAMIC

/*!
    \brief

    \sa

    \note           belongs to \ref porting_sec

    \warning
*/
#define sl_Malloc(Size)                                 malloc(Size)

/*!
    \brief

    \sa

    \note           belongs to \ref porting_sec

    \warning
*/
#define sl_Free(pMem)                                   free(pMem)

#endif

/*!

 Close the Doxygen group.
 @}

 */

/*!
 ******************************************************************************

    \defgroup   porting_os          Operating System (OS)

    The simple link driver can run on multi-threaded environment as well
    as non-os environment (main loop)

    This section IS NOT REQUIRED in case you are working on non-os environment.

    If you choose to work in multi-threaded environment under any operating system
    you will have to provide some basic adaptation routines to allow the driver
    to protect access to resources from different threads (locking object) and
    to allow synchronization between threads (sync objects).

    PORTING ACTION:
        -# Uncomment SL_PLATFORM_MULTI_THREADED define
        -# Bind locking object routines
        -# Bind synchronization object routines
        -# Optional - Bind spawn thread routine

    @{

 ******************************************************************************
 */


#define SL_PLATFORM_MULTI_THREADED

#ifdef SL_PLATFORM_MULTI_THREADED

/*!
    \brief
    \sa
    \note           belongs to \ref porting_sec
    \warning
*/
#define SL_OS_RET_CODE_OK                       ((int)OSI_OK)

/*!
    \brief
    \sa
    \note           belongs to \ref porting_sec
    \warning
*/
#define SL_OS_WAIT_FOREVER                      ((OsiTime_t)OSI_WAIT_FOREVER)

/*!
    \brief
    \sa
    \note           belongs to \ref porting_sec
    \warning
*/
#define SL_OS_NO_WAIT                           ((OsiTime_t)OSI_NO_WAIT)

/*!
    \brief type definition for a time value

    \note   On each porting or platform the type could be whatever is needed - integer, pointer to structure etc.

    \note       belongs to \ref porting_sec
*/
#define _SlTime_t               OsiTime_t

/*!
    \brief  type definition for a sync object container

    Sync object is object used to synchronize between two threads or thread and interrupt handler.
    One thread is waiting on the object and the other thread send a signal, which then
    release the waiting thread.
    The signal must be able to be sent from interrupt context.
    This object is generally implemented by binary semaphore or events.

    \note   On each porting or platform the type could be whatever is needed - integer, structure etc.

    \note       belongs to \ref porting_sec
*/
#define _SlSyncObj_t            OsiSyncObj_t


/*!
    \brief  This function creates a sync object

    The sync object is used for synchronization between different thread or ISR and
    a thread.

    \param  pSyncObj    -   pointer to the sync object control block

    \return upon successful creation the function should return 0
            Otherwise, a negative value indicating the error code shall be returned

    \note       belongs to \ref porting_sec
    \warning
*/
#define sl_SyncObjCreate(pSyncObj,pName)            osi_SyncObjCreate(pSyncObj)


/*!
    \brief  This function deletes a sync object

    \param  pSyncObj    -   pointer to the sync object control block

    \return upon successful deletion the function should return 0
            Otherwise, a negative value indicating the error code shall be returned
    \note       belongs to \ref porting_sec
    \warning
*/
#define sl_SyncObjDelete(pSyncObj)                  osi_SyncObjDelete(pSyncObj)


/*!
    \brief      This function generates a sync signal for the object.

    All suspended threads waiting on this sync object are resumed

    \param      pSyncObj    -   pointer to the sync object control block

    \return     upon successful signalling the function should return 0
                Otherwise, a negative value indicating the error code shall be returned
    \note       the function could be called from ISR context
    \warning
*/
#define sl_SyncObjSignal(pSyncObj)                  osi_SyncObjSignal(pSyncObj)

/*!
    \brief      This function generates a sync signal for the object from Interrupt

    This is for RTOS that should signal from IRQ using a dedicated API

    \param      pSyncObj    -   pointer to the sync object control block

    \return     upon successful signalling the function should return 0
                Otherwise, a negative value indicating the error code shall be returned
    \note       the function could be called from ISR context
    \warning
*/
#define sl_SyncObjSignalFromIRQ(pSyncObj)           osi_SyncObjSignal(pSyncObj)
/*!
    \brief  This function waits for a sync signal of the specific sync object

    \param  pSyncObj    -   pointer to the sync object control block
    \param  Timeout     -   numeric value specifies the maximum number of mSec to
                            stay suspended while waiting for the sync signal
                            Currently, the simple link driver uses only two values:
                                - OSI_WAIT_FOREVER
                                - OSI_NO_WAIT

    \return upon successful reception of the signal within the timeout window return 0
            Otherwise, a negative value indicating the error code shall be returned
    \note       belongs to \ref porting_sec
    \warning
*/
#define sl_SyncObjWait(pSyncObj,Timeout)            osi_SyncObjWait(pSyncObj,Timeout)

/*!
    \brief  type definition for a locking object container

    Locking object are used to protect a resource from mutual accesses of two or more threads.
    The locking object should support reentrant locks by a signal thread.
    This object is generally implemented by mutex semaphore

    \note   On each porting or platform the type could be whatever is needed - integer, structure etc.
    \note       belongs to \ref porting_sec
*/
#define _SlLockObj_t            OsiLockObj_t

/*!
    \brief  This function creates a locking object.

    The locking object is used for protecting a shared resources between different
    threads.

    \param  pLockObj    -   pointer to the locking object control block

    \return upon successful creation the function should return 0
            Otherwise, a negative value indicating the error code shall be returned
    \note       belongs to \ref porting_sec
    \warning
*/
#define sl_LockObjCreate(pLockObj,pName)            osi_LockObjCreate(pLockObj, pName)

/*!
    \brief  This function deletes a locking object.

    \param  pLockObj    -   pointer to the locking object control block

    \return upon successful deletion the function should return 0
            Otherwise, a negative value indicating the error code shall be returned
    \note       belongs to \ref porting_sec
    \warning
*/
#define sl_LockObjDelete(pLockObj)                  osi_LockObjDelete(pLockObj)

/*!
    \brief  This function locks a locking object.

    All other threads that call this function before this thread calls
    the osi_LockObjUnlock would be suspended

    \param  pLockObj    -   pointer to the locking object control block
    \param  Timeout     -   numeric value specifies the maximum number of mSec to
                            stay suspended while waiting for the locking object
                            Currently, the simple link driver uses only two values:
                                - OSI_WAIT_FOREVER
                                - OSI_NO_WAIT


    \return upon successful reception of the locking object the function should return 0
            Otherwise, a negative value indicating the error code shall be returned
    \note       belongs to \ref porting_sec
    \warning
*/
#define sl_LockObjLock(pLockObj,Timeout)            osi_LockObjLock(pLockObj,Timeout)

/*!
    \brief  This function unlock a locking object.

    \param  pLockObj    -   pointer to the locking object control block

    \return upon successful unlocking the function should return 0
            Otherwise, a negative value indicating the error code shall be returned
    \note       belongs to \ref porting_sec
    \warning
*/
#define sl_LockObjUnlock(pLockObj)                  osi_LockObjUnlock(pLockObj)

#endif
/*!
    \brief  This function call the pEntry callback from a different context

    \param  pEntry      -   pointer to the entry callback function

    \param  pValue      -   pointer to any type of memory structure that would be
                            passed to pEntry callback from the execution thread.

    \param  flags       -   execution flags - reserved for future usage

    \return upon successful registration of the spawn the function should return 0
            (the function is not blocked till the end of the execution of the function
            and could be returned before the execution is actually completed)
            Otherwise, a negative value indicating the error code shall be returned
    \note       belongs to \ref porting_sec
    \warning
*/
#define SL_PLATFORM_EXTERNAL_SPAWN

#ifdef SL_PLATFORM_EXTERNAL_SPAWN
#define sl_Spawn(pEntry,pValue,flags)               osi_Spawn(pEntry,pValue,flags)
#endif

/*!

 Close the Doxygen group.
 @}

 */


/*!
 ******************************************************************************

    \defgroup       porting_events      Event Handlers

    This section includes the asynchronous event handlers routines

    PORTING ACTION:
        -Uncomment the required handler and define your routine as the value
        of this handler

    @{

 ******************************************************************************
 */

/*!
    \brief      Fatal Error async event for inspecting fatal error events.
                This event handles events/errors reported from the device/host
                driver
    \sa

    \note       belongs to \ref porting_sec

    \warning
*/

#define slcb_DeviceFatalErrorEvtHdlr            SimpleLinkFatalErrorEvtHdlr

/*!
    \brief      General async event for inspecting general events.
                This event handles events/errors reported from the device/host
                driver
    \sa

    \note       belongs to \ref porting_sec

    \warning
*/

#define slcb_DeviceGeneralEvtHdlr               SimpleLinkGeneralEventHandler


/*!
    \brief          An event handler for WLAN connection or disconnection
                    indication. This event handles async WLAN events.

                    Possible events are:
                    SL_WLAN_EVENT_CONNECT - indicates WLAN is connected
                    SL_WLAN_EVENT_DISCONNECT - indicates WLAN is disconnected
    \sa

    \note           belongs to \ref porting_sec

    \warning
*/

#define slcb_WlanEvtHdlr                        SimpleLinkWlanEventHandler


/*!
    \brief          An event handler for IP address asynchronous event.
                    Usually accepted after new WLAN connection.
                    This event handles networking events.

                    Possible events are:
                    SL_NETAPP_IPV4_ACQUIRED - IP address was acquired
                    (DHCP or Static)

    \sa

    \note           belongs to \ref porting_sec

    \warning
*/

#define slcb_NetAppEvtHdlr                      SimpleLinkNetAppEventHandler

/*!
    \brief          A callback for HTTP server events.
                    Possible events are:
                    SL_NETAPP_EVENT_HTTP_TOKEN_GET -  NWP requests to get the
                    value of a specific token

                    SL_NETAPP_EVENT_HTTP_TOKEN_POST - NWP post to the host a
                    new value for a specific token

    \param          pServerEvent - Contains the relevant event information
                   (SL_NETAPP_HTTPGETTOKENVALUE or
                   SL_NETAPP_HTTPPOSTTOKENVALUE)

    \param          pServerResponse - Should be filled by the user with the
                    relevant response information
                    (i.e SL_NETAPP_HTTPSETTOKENVALUE as a response to
                    SL_NETAPP_HTTPGETTOKENVALUE event)

    \sa

    \note           belongs to \ref porting_sec

    \warning
*/

#define slcb_HttpServerEvtHdlr                  SimpleLinkHttpServerCallback


/*!
    \brief          A handler for handling Netapp requests.
                    Netapp request types:
                    For HTTP server: GET / POST (future: PUT / DELETE)

    \param

    \param

    \sa

    \note           belongs to \ref porting_sec

    \warning
*/
#define slcb_NetAppRequestHdlr                  SimpleLinkNetAppRequestHandler

/*!
    \brief          A handler for freeing the memory of the NetApp response.

    \param

    \param

    \sa

    \note           belongs to \ref porting_sec

    \warning
*/
/*#define sl_NetAppMemFree_cb                   NetAppMemFree_cb*/

/*!
    \brief

    \sa

    \note           belongs to \ref porting_sec

    \warning
*/
#define slcb_SockEvtHdlr                        SimpleLinkSockEventHandler



/*!

 Close the Doxygen group.
 @}

 */


#ifdef  __cplusplus
}
#endif /* __cplusplus */

#endif /* __USER_H__ */

