/*
 *
 *   Copyright (C) 2016 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission of
 *   Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been supplied,
 *   and under no circumstances can it be used with non-TI connectivity device.
 *
 */


#ifndef __RADIOTOOL_H__
#define	__RADIOTOOL_H__

#ifdef	__cplusplus
extern "C"
{
#endif

#include "BaseTsd.h"

#define RTAPI __declspec(dllexport) INT32 

#define SL_BSSID_LENGTH			6
#define NUM_OF_RATE_INDEXES		20
#define SIZE_OF_RSSI_HISTOGRAM	6
#define CW_LOW_TONE				-25
#define CW_HIGH_TONE			25

#define RADIO_TOOL_ERROR_DELIVER_TX_STOP		(-201)
#define RADIO_TOOL_ERROR_DELIVER_TX_START		(-202)
#define RADIO_TOOL_ERROR_DELIVER_RX_STOP		(-203)
#define RADIO_TOOL_ERROR_DELIVER_RX_START		(-204)

#define RADIO_TOOL_ERROR_DELIVER_RX_GET_STATS	(-205)
#define RADIO_TOOL_ERROR_OPCODE_RX_GET_STATS	(-206)
#define RADIO_TOOL_ERROR_RX_GET_STATS			(-207)

#define RADIO_TOOL_ERROR_DELIVER_GET_MAC_ADDR	(-208)
#define RADIO_TOOL_ERROR_OPCODE_GET_MAC_ADDR	(-209)
#define RADIO_TOOL_ERROR_GET_MAC_ADDR			(-210)

#define RADIO_TOOL_ERROR_DELIVER_GET_DEV_VER	(-211)
#define RADIO_TOOL_ERROR_OPCODE_GET_DEV_VER		(-212)
#define RADIO_TOOL_ERROR_GET_DEV_VER			(-213)

/*****************************************************************************
 Data Structures
 *****************************************************************************/
typedef enum
{
    RADIO_TX_CONTINUOUS = 1,
    RADIO_TX_PACKETIZED = 2,
    RADIO_TX_CW			= 3,
    MAX_RADIO_TX_MODE	= 0xff
}RadioTxMode_e;

typedef enum
{
    PATTERN_ALL_0		= 0,
    ALL_1_PATTERN		= 1,
    INCREMENTAL_PATTERN	= 2,
    DECREMENTAL_PATTERN	= 3,
    PN9_PATTERN			= 4,
    PN15_PATTERN		= 5,
    PN23_PATTERN		= 6,
    MAX_NUM_PATTERN		= 0xff
}RadioDataPattern_e;

typedef enum
{
	TX_START_COMMAND_OP			= 0x71,
	TX_STOP_COMMAND_OP			= 0x72,
	RX_START_COMMAND_OP			= 0x73,
	RX_GET_STATS_COMMAND_OP		= 0x74,
	RX_STOP_COMMAND_OP			= 0x75,
	GET_MAC_ADDR_COMMAND_OP		= 0x76,
	GET_DEV_VER_COMMAND_OP		= 0x77,

	RX_GET_STATS_RESPONSE_OP	= 0xE1,

	GET_MAC_ADDR_RESPONSE_OP 	= 0xE3,
	GET_DEV_VER_RESPONSE_OP 	= 0xE4,
}RadioOpcode_e;

typedef enum
{
	OK_CODE				= 0x00,
	BAD_COM_CHKSUM_CODE	= 0xF1,
	UNKNOWN_OPCODE_CODE = 0xF2,
	INVALID_PARAM_CODE	= 0xF3
}RadioErrorCode_e;

typedef struct
{
	UINT16	AckLength;
	UINT8	Checksum;
	UINT8	Opcode;
	UINT8	ErrorCode;
}CommandAckResponse_s;

/*****************************************************************************/
/* These are copied structure from Simplelink API.
 * Updates need to be made for every release.
 */
typedef enum
{
  SL_TX_INHIBIT_THRESHOLD_MIN = 1,
  SL_TX_INHIBIT_THRESHOLD_LOW = 2,
  SL_TX_INHIBIT_THRESHOLD_DEFAULT = 3,
  SL_TX_INHIBIT_THRESHOLD_MED = 4,
  SL_TX_INHIBIT_THRESHOLD_HIGH = 5,
  SL_TX_INHIBIT_THRESHOLD_MAX = 6
} SlTxInhibitThreshold_e;


typedef enum
{
    SL_WLAN_RATE_1M         = 1,
    SL_WLAN_RATE_2M         = 2,
    SL_WLAN_RATE_5_5M       = 3,
    SL_WLAN_RATE_11M        = 4,
    SL_WLAN_RATE_6M         = 6,
    SL_WLAN_RATE_9M         = 7,
    SL_WLAN_RATE_12M        = 8,
    SL_WLAN_RATE_18M        = 9,
    SL_WLAN_RATE_24M        = 10,
    SL_WLAN_RATE_36M        = 11,
    SL_WLAN_RATE_48M        = 12,
    SL_WLAN_RATE_54M        = 13,
    SL_WLAN_RATE_MCS_0      = 14,
    SL_WLAN_RATE_MCS_1      = 15,
    SL_WLAN_RATE_MCS_2      = 16,
    SL_WLAN_RATE_MCS_3      = 17,
    SL_WLAN_RATE_MCS_4      = 18,
    SL_WLAN_RATE_MCS_5      = 19,
    SL_WLAN_RATE_MCS_6      = 20,
    SL_WLAN_RATE_MCS_7      = 21,
    SL_WLAN_MAX_NUM_RATES   = 0xFF
}SlWlanRateIndex_e;

#define SL_WLAN_NUM_OF_RATE_INDEXES             (20)
#define SL_WLAN_SIZE_OF_RSSI_HISTOGRAM          (6)

typedef struct
{
    UINT32  ReceivedValidPacketsNumber;							/* sum of the packets that been received OK (include filtered) */
    UINT32   ReceivedFcsErrorPacketsNumber;						/* sum of the packets that been dropped due to FCS error */
    UINT32   ReceivedAddressMismatchPacketsNumber;					/* sum of the packets that been received but filtered out by one of the HW filters */
    INT16  AvarageDataCtrlRssi;									/* average RSSI for all valid data packets received */
    INT16  AvarageMgMntRssi;										/* average RSSI for all valid management packets received */
    UINT16   RateHistogram[SL_WLAN_NUM_OF_RATE_INDEXES];           /* rate histogram for all valid packets received */
    UINT16   RssiHistogram[SL_WLAN_SIZE_OF_RSSI_HISTOGRAM];        /* RSSI histogram from -40 until -87 (all below and above\n RSSI will appear in the first and last cells */
    UINT32   StartTimeStamp;										/* the time stamp started collecting the statistics in uSec */
    UINT32   GetTimeStamp;											/* the time stamp called the get statistics command */
}SlWlanGetRxStatResponse_t;

typedef struct
{
    UINT32                ChipId;
    UINT8                 FwVersion[4];
    UINT8                 PhyVersion[4];
    UINT8                 NwpVersion[4];
    UINT16                RomVersion;
    UINT16                Padding;
}SlDeviceVersion_t;
/*****************************************************************************/

/*!
 *  \brief		Connects to SL device
 *  \param[in]	comPort  - The COM port number of the device
 *  \param[in]	baudRate - The baud rate for the COM port
 *  \return		0 on success, -ve otherwise
 */
RTAPI RadioToolOpen(unsigned char comPort, unsigned long baudRate);

/*!
 *  \brief		Disconnects from SL device
 *  \param		None
 *  \return		0 on success, -ve otherwise
 */
RTAPI RadioToolClose();

/*!
 *  \brief		Starts TX

 *  \return
 */
RTAPI RadioStartTX(RadioTxMode_e eTxMode,
	INT8                ePowerLevel_Tone,
	UINT8               eChannel,
	SlWlanRateIndex_e   eRate,
	UINT8               ePreamble,
	RadioDataPattern_e  eDataPattern,
	UINT16              eSize,
	UINT32              eDelay,
	UINT32              eAmount,
	UINT8               eOverrideCCA,
	SlTxInhibitThreshold_e eCCAThreshold,
	UINT8               eEnableACKs,
    UINT8               *epDstMac);

/*!
 *  \brief		Stops TX
 *  \param[in]	eTxMode - the TX mode the device was in. Needs to be
 *				identical to the eTxMode specified in RadioStartTX()
 *  \return		0 on success, -ve otherwise
 */
RTAPI RadioStopTX(RadioTxMode_e eTxMode);

/*!
 *  \brief		Starts RX
 *  \param[in]	eChannel - channel to scan
 *  \return		0 on success, -ve otherwise
 */
RTAPI RadioStartRX(UINT8 eChannel);

/*!
 *  \brief		Stops RX
 *  \param		None
 *  \return		0 on success, -ve otherwise
 */
RTAPI RadioStopRX();

/*!
 *  \brief		Gets RX Statistics
 *  \param[out]	pRxStats - RX Statistics, see SlWlanGetRxStatResponse_t
 *  \return		0 on success, -ve otherwise
 */
RTAPI RadioGetStats(UINT8 *stats);

/*!
 *  \brief      Gets the MAC address of the device
 *  \param[out] pMacAddress
 *  \return     0 on success, -ve otherwise
 */
RTAPI RadioGetMacAddr(UINT8 *pMacAddress);

/*!
 *  \brief		Gets device version information
 *  \param[out]	pDevVersion - Device version details, see SlDeviceVersion_t
 *  \param[out]	pHDLength - length of the pHDVersion text
 *  \param[out]	pHDVersion - Host Driver version
 *  \return		0 on success, -ve otherwise
 */
RTAPI RadioGetDeviceVersion(UINT8 *pDevVersion, UINT16 *pHDLength, UINT8 *pHDVersion);

#ifdef	__cplusplus
}
#endif // __cplusplus
#endif	// __RADIOTOOL_H__
