/*
 *   Copyright (C) 2015 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission of
 *   Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been supplied,
 *   and under no circumstances can it be used with non-TI connectivity device.
 *   
 */
/******************************************************************************
* 	cc_pal.c
*
*	simplelink abstraction file for CC3200
******************************************************************************/

//Simplelink includes
#include <simplelink.h>
#include <cc_pal.h>

//Driverlib includes
#include <hw_ints.h>
#include <hw_udma.h>
#include <hw_types.h>
#include <pin.h>
#include <hw_memmap.h>
#include <hw_mcspi.h>
#include <hw_common_reg.h>
#include <rom.h>
#include <rom_map.h>
#include <spi.h>
#include <prcm.h>
#include <rom.h>
#include <rom_map.h>
#include <hw_ints.h>
#include <interrupt.h>
#include <udma.h>
#include <utils.h>
#include <ti/drivers/SPI.h>
#include "protocol.h"
#include "driver.h"

//OSLib includes
#if defined(SL_PLATFORM_MULTI_THREADED)
#include <osi.h>
#endif


#define REG_INT_MASK_SET                0x400F7088
#define REG_INT_MASK_CLR                0x400F708C
#define APPS_SOFT_RESET_REG             0x4402D000
#define OCP_SHARED_MAC_RESET_REG        0x4402E168
#define ROM_VERSION_ADDR                0x00000400

#define DMA_BUFF_SIZE_MIN               100
#define MAX_NUM_CH	                    64	//32*2 entries
#define MAX_DMA_RECV_TRANSACTION_SIZE   4096

//SPI rate determined by the ROM version
#define ROM_VER_PG1_21                  1
#define ROM_VER_PG1_32                  2
#define ROM_VER_PG1_33                  3
#define SPI_RATE_13M 		            13000000
#define SPI_RATE_20M                    20000000
#define SPI_RATE_10M                    10000000

#define UNUSED(x) 						x=x

//Structure definition to determine the ROM version
typedef struct
{
    unsigned short ucMajorVerNum;
    unsigned short ucMinorVerNum;
    unsigned short ucSubMinorVerNum;
    unsigned short ucDay;
    unsigned short ucMonth;
    unsigned short ucYear;
}tROMVersion;

//
// GLOBAL VARIABLES -- Start
//
volatile Fd_t g_SpiFd =0;
P_EVENT_HANDLER g_pHostIntHdl  = NULL;
unsigned long g_ucDinDout[20];
unsigned char g_ucDMAEnabled = 0;

#if defined(SL_PLATFORM_MULTI_THREADED)
OsiMsgQ_t DMAMsgQ;
char g_cDummy[4];
#else //SL_PLATFORM_MULTI_THREADED
volatile char g_cDummy;
#endif //SL_PLATFORM_MULTI_THREADED

//
// GLOBAL VARIABLES -- End
//

//****************************************************************************
//                      LOCAL FUNCTION DEFINITIONS
//****************************************************************************


#ifdef slcb_GetTimestamp
extern void _SlDrvStartMeasureTimeout(_SlTimeoutParams_t *pTimeoutInfo, _u32 TimeoutInMsec);
extern _u8 _SlDrvIsTimeoutExpired(_SlTimeoutParams_t *pTimeoutInfo);

void __sleep(_u16 DurationInMsec)
{
    _SlTimeoutParams_t      TimeoutInfo={0};

    _SlDrvStartMeasureTimeout(&TimeoutInfo, DurationInMsec);
    while(!_SlDrvIsTimeoutExpired(&TimeoutInfo));
}
#endif



/*!
    \brief  DMA SPI interrupt handler

	\param			None

	\return			None

	\note			This function
       	   	   	   	   1. Invoked when SPI Transaction Completes
    \warning
*/
void DmaSpiSwIntHandler()
{
	MAP_SPIIntClear(LSPI_BASE,SPI_INT_EOW);
	MAP_SPICSDisable(LSPI_BASE);

#if defined(SL_PLATFORM_MULTI_THREADED)
    osi_MsgQWrite(&DMAMsgQ,g_cDummy,OSI_NO_WAIT);
#else
    g_cDummy = 0x1;
#endif
}

/*!
    \brief attempts to read up to len bytes from SPI channel into a buffer starting at pBuff.

	\param			pBuff		- 	points to first location to start writing the data

	\param			len			-	number of bytes to read from the SPI channel

	\return			upon successful completion, the function shall return Read Size.
					Otherwise, -1 shall be returned

    \sa             spi_Read_CPU , spi_Write_CPU
	\note
    \warning
*/
int spi_Read_CPU(unsigned char *pBuff, int len)
{
    unsigned long ulCnt;
    unsigned long ulStatusReg;
    unsigned long *ulDataIn;
    unsigned long ulTxReg;
    unsigned long ulRxReg;

    MAP_SPIEnable(LSPI_BASE);
    MAP_SPICSEnable(LSPI_BASE);

    //
    // Initialize local variable.
    //
    ulDataIn = (unsigned long *)pBuff;
    ulCnt = (len + 3) >> 2;
    ulStatusReg = LSPI_BASE+MCSPI_O_CH0STAT;
    ulTxReg = LSPI_BASE + MCSPI_O_TX0;
    ulRxReg = LSPI_BASE + MCSPI_O_RX0;

    //
    // Reading loop
    //
    while(ulCnt--)
    {
          while(!( HWREG(ulStatusReg)& MCSPI_CH0STAT_TXS ));
          HWREG(ulTxReg) = 0xFFFFFFFF;
          while(!( HWREG(ulStatusReg)& MCSPI_CH0STAT_RXS ));
          *ulDataIn = HWREG(ulRxReg);
          ulDataIn++;
    }

    MAP_SPICSDisable(LSPI_BASE);

    return len;
}

/*!
    \brief attempts to write up to len bytes to the SPI channel

	\param			pBuff		- 	points to first location to start getting the data from

	\param			len			-	number of bytes to write to the SPI channel

	\return			upon successful completion, the function shall return write size.
					Otherwise, -1 shall be returned

    \sa             spi_Read_CPU , spi_Write_CPU
	\note			This function could be implemented as zero copy and return only upon successful completion
					of writing the whole buffer, but in cases that memory allocation is not too tight, the
					function could copy the data to internal buffer, return back and complete the write in
					parallel to other activities as long as the other SPI activities would be blocked untill
					the entire buffer write would be completed
    \warning
*/
int spi_Write_CPU(unsigned char *pBuff, int len)
{


    unsigned long ulCnt;
    unsigned long ulStatusReg;
    unsigned long *ulDataOut;
    unsigned long ulDataIn;
    unsigned long ulTxReg;
    unsigned long ulRxReg;

    MAP_SPIEnable(LSPI_BASE);
    MAP_SPICSEnable(LSPI_BASE);

    //
    // Initialize local variable.
    //
    ulDataOut = (unsigned long *)pBuff;
    ulCnt = (len +3 ) >> 2;
    ulStatusReg = LSPI_BASE+MCSPI_O_CH0STAT;
    ulTxReg = LSPI_BASE + MCSPI_O_TX0;
    ulRxReg = LSPI_BASE + MCSPI_O_RX0;

    //
    // Writing Loop
    //
    while(ulCnt--)
    {
          while(!( HWREG(ulStatusReg)& MCSPI_CH0STAT_TXS ));
          HWREG(ulTxReg) = *ulDataOut;
          while(!( HWREG(ulStatusReg)& MCSPI_CH0STAT_RXS ));
          ulDataIn = HWREG(ulRxReg);
          ulDataOut++;
    }

    MAP_SPICSDisable(LSPI_BASE);

    UNUSED(ulDataIn);
    return len;

}

/*!
    \brief open spi communication port to be used for communicating with a SimpleLink device

	Given an interface name and option flags, this function opens the spi communication port
	and creates a file descriptor. This file descriptor can be used afterwards to read and
	write data from and to this specific spi channel.
	The SPI speed, clock polarity, clock phase, chip select and all other attributes are all
	set to hardcoded values in this function.

	\param	 		ifName		-	points to the interface name/path. The interface name is an
									optional attributes that the simple link driver receives
									on opening the device. in systems that the spi channel is
									not implemented as part of the os device drivers, this
									parameter could be NULL.
	\param			flags		-	option flags

	\return			upon successful completion, the function shall open the spi channel and return
					a non-negative integer representing the file descriptor.
					Otherwise, -1 shall be returned

    \sa             spi_Close , spi_Read , spi_Write
	\note
    \warning
*/
Fd_t spi_Open(char *ifName, unsigned long flags)
{
	void *lspi_hndl;
	unsigned int lspi_index;
	SPI_Params SPI_Config;
	SPI_Params_init(&SPI_Config);

	/* configure the SPI settings */
	SPI_Config.transferMode = SPI_MODE_BLOCKING;
	SPI_Config.mode = SPI_MASTER;
	SPI_Config.bitRate = SPI_RATE_20M;
	SPI_Config.dataSize = 32;
	SPI_Config.frameFormat = SPI_POL0_PHA0;

	/* index of the link SPI initialisation configuration in the SPI_Config
	 * table */
	lspi_index = 0;
	lspi_hndl = SPI_open(lspi_index, &SPI_Config);
	if(NULL == lspi_hndl) {
			return -1;
	} else {
			return (Fd_t)lspi_hndl;
	}
}

/*!
    \brief closes an opened spi communication port

	\param	 		fd			-	file descriptor of an opened SPI channel

	\return			upon successful completion, the function shall return 0.
					Otherwise, -1 shall be returned

    \sa             spi_Open
	\note
    \warning
*/
int spi_Close(Fd_t fd)
{
	SPI_close((void *)fd);
    return 0;
}

/*!
    \brief closes an opened spi communication port

	\param	 		fd			-	file descriptor of an opened SPI channel

	\return			upon successful completion, the function shall return number
                    of bytes read. Otherwise, -1 shall be returned

    \sa             spi_Open
	\note
    \warning
*/
int spi_Read(Fd_t fd, unsigned char *pBuff, int len)
{    
    SPI_Transaction transact_details;
    int read_size = 0;
    
    /* check if the link SPI has been initialised successfully */
    if(fd < 0)
    {
        return -1;
    }
    
    /* check if DMA should be used for the SPI transaction */
    if(len>DMA_BUFF_SIZE_MIN && (((unsigned long)pBuff % 4) == 0))
	{
        transact_details.txBuf = NULL;
        transact_details.arg = NULL;
        while(len > 0)
        {
            
            /* DMA can transfer upto a maximum of 1024 words in one go. So, if 
               the data to be read is more than 1024 words, it will be done in 
               parts */
            /* length is received in bytes, should be specified in words for the
             * SPI driver.
             */
            if(len > MAX_DMA_RECV_TRANSACTION_SIZE)
            {
                transact_details.count = (MAX_DMA_RECV_TRANSACTION_SIZE +3)>>2;
                transact_details.rxBuf = (void*)(pBuff + read_size);
                if(SPI_transfer((SPI_Handle)fd, &transact_details))
                {
                    read_size += MAX_DMA_RECV_TRANSACTION_SIZE;
                    len = len - MAX_DMA_RECV_TRANSACTION_SIZE;
                }
                else
                {
                    return -1;
                }
                
            }
            else
            {
                transact_details.count = (len+3)>>2;
                transact_details.rxBuf = (void*)(pBuff + read_size);
                if(SPI_transfer((SPI_Handle)fd, &transact_details))
                {
                    read_size += len;
                    len = 0;
                    return read_size;
                }
                else
                {
                     return -1;
                }
            }
        }        
    }
    else
    {
        
        MAP_SPIWordCountSet(LSPI_BASE,0);
		read_size += spi_Read_CPU(pBuff,len);
    }
    return(read_size);
}

/*!
    \brief attempts to write up to len bytes to the SPI channel

	\param	 		fd			-	file descriptor of an opened SPI channel

	\param			pBuff		- 	points to first location to start getting the data from

	\param			len			-	number of bytes to write to the SPI channel

	\return			upon successful completion, the function shall return 0.
					Otherwise, -1 shall be returned

    \sa             spi_Open , spi_Read
	\note			This function could be implemented as zero copy and return only upon successful completion
					of writing the whole buffer, but in cases that memory allocation is not too tight, the
					function could copy the data to internal buffer, return back and complete the write in
					parallel to other activities as long as the other SPI activities would be blocked untill
					the entire buffer write would be completed
    \warning
*/
int spi_Write(Fd_t fd, unsigned char *pBuff, int len)
{
    SPI_Transaction transact_details;
    int write_size = 0;
    
    /* check if the link SPI has been initialised successfully */
    if(fd < 0)
    {
    	return -1;
    }

    /* check if DMA should be used for the SPI transaction */
    if(len>DMA_BUFF_SIZE_MIN && (((unsigned long)pBuff % 4) == 0))
	{
        transact_details.rxBuf = NULL;
        transact_details.arg = NULL;
        while(len > 0)
		{
			/* configure the transaction details.


			 * length is received in bytes, should be specified in words for the SPI
			 * driver.
			 */
			if(len > MAX_DMA_RECV_TRANSACTION_SIZE)
			{
				transact_details.count = (MAX_DMA_RECV_TRANSACTION_SIZE +3)>>2;
				transact_details.txBuf = (void*)(pBuff + write_size);
				if(SPI_transfer((SPI_Handle)fd, &transact_details))
				{
					write_size += MAX_DMA_RECV_TRANSACTION_SIZE;
					len = len - MAX_DMA_RECV_TRANSACTION_SIZE;
				}
				else
				{
					return -1;
				}
			}
			else
			{
				transact_details.count = (len+3)>>2;
				transact_details.txBuf = (void*)(pBuff + write_size);
				if(SPI_transfer((SPI_Handle)fd, &transact_details))
				{
					write_size += len;
					len = 0;
					return write_size;
				}
				else
				{
					 return -1;
				}
			}
        }        
    }
    else
    {
        MAP_SPIWordCountSet(LSPI_BASE,0);
		write_size += spi_Write_CPU(pBuff,len);
    }
    return(write_size);
}

/*!
    \brief register an interrupt handler for the host IRQ

	\param	 		InterruptHdl	-	pointer to interrupt handler function

	\param 			pValue			-	pointer to a memory strcuture that is passed to the interrupt handler.

	\return			upon successful registration, the function shall return 0.
					Otherwise, -1 shall be returned

    \sa
	\note			If there is already registered interrupt handler, the function should overwrite the old handler
					with the new one
    \warning
*/

int NwpRegisterInterruptHandler(P_EVENT_HANDLER InterruptHdl , void* pValue)
{

    if(InterruptHdl == NULL)
    {
		//De-register Interprocessor communication interrupt between App and NWP
		#ifdef SL_PLATFORM_MULTI_THREADED
		  osi_InterruptDeRegister(INT_NWPIC);
		#else
		  MAP_IntDisable(INT_NWPIC);
		  MAP_IntUnregister(INT_NWPIC);
		  MAP_IntPendClear(INT_NWPIC);
		#endif
    }
    else
    {
		  #ifdef SL_PLATFORM_MULTI_THREADED
			 MAP_IntPendClear(INT_NWPIC);
			 osi_InterruptRegister(INT_NWPIC, (P_OSI_INTR_ENTRY)InterruptHdl,INT_PRIORITY_LVL_1);
		  #else
			 MAP_IntRegister(INT_NWPIC, InterruptHdl);
			 MAP_IntPrioritySet(INT_NWPIC, INT_PRIORITY_LVL_1);
			 MAP_IntPendClear(INT_NWPIC);
			 MAP_IntEnable(INT_NWPIC);
		  #endif
    }

  return 0;
}


/*!
    \brief 				Masks host IRQ


    \sa             		NwpUnMaskInterrupt

    \warning
*/
void NwpMaskInterrupt()
{
	(*(unsigned long *)REG_INT_MASK_SET) = 0x1;
}


/*!
    \brief 				Unmasks host IRQ


    \sa             		NwpMaskInterrupt

    \warning
*/
void NwpUnMaskInterrupt()
{
	(*(unsigned long *)REG_INT_MASK_CLR) = 0x1;
}

#ifndef DISABLE_DEBUGGER_RECONNECT
/*!
    \brief		Preamble to the enabling the Network Processor.
                        Placeholder to implement any pre-process operations
                        before enabling networking operations.

    \sa			sl_DeviceEnable

    \note       belongs to \ref ported_sec

*/
void NwpPowerOnPreamble(void)
{

#define MAX_RETRY_COUNT         1000
    unsigned int sl_stop_ind, apps_int_sts_raw, nwp_lpds_wake_cfg;
    unsigned int retry_count;
    /* Perform the sl_stop equivalent to ensure network services
       are turned off if active */
    HWREG(0x400F70B8) = 1;   /* APPs to NWP interrupt */
    ROM_UtilsDelayDirect(1600000/5);

    retry_count = 0;
    nwp_lpds_wake_cfg = HWREG(0x4402D404);
    sl_stop_ind = HWREG(0x4402E16C);

    if((nwp_lpds_wake_cfg != 0x20) && /* Check for NWP POR condition */
            !(sl_stop_ind & 0x2))     /* Check if sl_stop was executed */
    {
        /* Loop until APPs->NWP interrupt is cleared or timeout */
        while(retry_count < MAX_RETRY_COUNT)
        {
            apps_int_sts_raw = HWREG(0x400F70C0);
            if(apps_int_sts_raw & 0x1)
            {
                ROM_UtilsDelayDirect(1600000/5);
                retry_count++;
            }
            else
            {
                break;
            }
        }
    }
    HWREG(0x400F70B0) = 1;   /* Clear APPs to NWP interrupt */
    ROM_UtilsDelayDirect(1600000/5);

    /* Stop the networking services */
    NwpPowerOff();
}

#endif
/*!
    \brief		Enable the Network Processor

    \sa			sl_DeviceDisable

    \note       belongs to \ref ported_sec

*/
void NwpPowerOn(void)
{

    //bring the 1.32 eco out of reset
    HWREG(0x4402E16C) &= 0xFFFFFFFD;

    /* Clear host IRQ indication */
    HWREG(0x400F7094) = 1;

    //NWP Wakeup
    HWREG(0x44025118) = 1;
#ifndef DISABLE_DEBUGGER_RECONNECT
    ROM_UtilsDelayDirect(16000000);
#endif

    //UnMask Host Interrupt
    NwpUnMaskInterrupt();
}


/*!
    \brief		Disable the Network Processor

    \sa			sl_DeviceEnable

    \note       belongs to \ref ported_sec
*/
void NwpPowerOff(void)
{
	//Must delay 300 usec to enable the NWP to finish all sl_stop activities
	ROM_UtilsDelayDirect(600*80/3);

	//Mask Host Interrupt
    NwpMaskInterrupt();

    //Switch to PFM Mode
    HWREG(0x4402F024) &= 0xF7FFFFFF;
    //sl_stop eco for PG1.32 devices
    HWREG(0x4402E16C) |= 0x2;

    ROM_UtilsDelayDirect(1600000);
}
