/*
 * Display.c
 *
 * This module handles the display related functions
 *
 * Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
 * ALL RIGHTS RESERVED
 *
*/

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <string.h>
#include <linux/fb.h>
#include <stdlib.h>

#include "Common.h"
#include "Display.h"

#ifndef FBIO_WAITFORVSYNC
#define FBIO_WAITFORVSYNC _IOW('F', 0x20, __u32)
#endif
#ifndef FBIO_FRAME_UPDATE_START
#define FBIO_FRAME_UPDATE_START _IOW('F', 11, int)
#endif

#define DISP_DEVICE		"/dev/fb0"
#define PRINT_INFO		0

static int DisplayFD = -1;
static unsigned DisplayWidth;
static unsigned DisplayHeight;
static unsigned DisplayLineWidth;
static int DisplayFrame;

static size_t FrameBuffSize = 0;
static uint8 *FrameBuffer;
static uint8 *FrameBufferFull;
static struct fb_fix_screeninfo FixScreenInfo;
static struct fb_var_screeninfo VarScreenInfo;

static ErrorCode_t DISP_DrawImage24(unsigned XOff, unsigned YOff, Image_t *Image,int flag);
static ErrorCode_t DISP_DrawImage32(unsigned XOff, unsigned YOff, Image_t *Image);

/**
 * Initialize the Projector module
 *
 * returns ERR_OUT_OF_RESOURCE, ERR_DEVICE_FAIL, SUCCESS
 */
ErrorCode_t DISP_Init(void)
{
	ErrorCode_t Error = SUCCESS;

	ERR_BLOCK_BEGIN
	{
		if((DisplayFD = open(DISP_DEVICE, O_RDWR)) <= 0)
			ERR_THROW_MSG(Error = ERR_DEVICE_FAIL, "Unable to open Display Device");

		if(ioctl(DisplayFD, FBIOGET_FSCREENINFO, &FixScreenInfo) < 0)
			ERR_THROW_MSG(Error = ERR_DEVICE_FAIL, "ioctl:FBIOGET_FSCREENINFO");
#if 0
		DEBUG_MSG("Fix Screen Info\n");
		DEBUG_MSG("-------------------------\n");
		DEBUG_MSG(" Line Length      = %d\n", FixScreenInfo.line_length);
		DEBUG_MSG(" Physical Address = %lx\n", FixScreenInfo.smem_start);
		DEBUG_MSG(" Buffer Length    = %d\n", FixScreenInfo.smem_len);
#endif

		if(ioctl(DisplayFD, FBIOGET_VSCREENINFO, &VarScreenInfo) < 0)
			ERR_THROW_MSG(Error = ERR_DEVICE_FAIL, "ioctl:FBIOGET_VSCREENINFO");

		/*
		VarScreenInfo.xres = 640;
		VarScreenInfo.yres = 480;
		*/
		VarScreenInfo.xres_virtual = VarScreenInfo.xres;
		VarScreenInfo.yres_virtual = VarScreenInfo.yres;
		VarScreenInfo.yres_virtual = VarScreenInfo.yres * FB_NUM_BUFFERS;

		if(ioctl(DisplayFD, FBIOPUT_VSCREENINFO, &VarScreenInfo) < 0)
			ERR_THROW_MSG(Error = ERR_DEVICE_FAIL, "ioctl:FBIOPET_VSCREENINFO");

		if(ioctl(DisplayFD, FBIOGET_VSCREENINFO, &VarScreenInfo) < 0)
			ERR_THROW_MSG(Error = ERR_DEVICE_FAIL, "ioctl:FBIOGET_VSCREENINFO");

#if PRINT_INFO == 1
		DEBUG_MSG("Var Screen Info\n");
        DEBUG_MSG(" Display Width    = %d\n", VarScreenInfo.xres);
        DEBUG_MSG(" Display Height   = %d\n", VarScreenInfo.yres);
        DEBUG_MSG(" Bits Per Pixel   = %d\n", VarScreenInfo.bits_per_pixel);
        DEBUG_MSG(" Pixel Clk        = %d\n", VarScreenInfo.pixclock);
#if 0
        DEBUG_MSG(" Display Rotation = %d\n", VarScreenInfo.rotate);
        DEBUG_MSG(" Xres Virtual     = %d\n", VarScreenInfo.xres_virtual);
        DEBUG_MSG(" Yres Virtual     = %d\n", VarScreenInfo.yres_virtual);
        DEBUG_MSG(" XOffset          = %d\n", VarScreenInfo.xoffset);
        DEBUG_MSG(" YOffset          = %d\n", VarScreenInfo.yoffset);
#endif
#endif
		
	FrameBuffSize = VarScreenInfo.xres * VarScreenInfo.yres * VarScreenInfo.bits_per_pixel/8;
	//DEBUG_MSG(" Line_Length      = %d\n", FixScreenInfo.line_length);
	DEBUG_MSG(" FrameBuffer size = %d\n", FrameBuffSize);

		FrameBufferFull = (uint8 *)mmap(0, FrameBuffSize * FB_NUM_BUFFERS, 
											(PROT_READ|PROT_WRITE), 
											MAP_SHARED, DisplayFD, 0);
		if(FrameBufferFull == MAP_FAILED)
			ERR_THROW_MSG(Error = ERR_OUT_OF_RESOURCE, "mmap() failed");

		DisplayWidth = VarScreenInfo.xres;
		DisplayHeight = VarScreenInfo.yres;
		DisplayLineWidth = FixScreenInfo.line_length;
		
#ifdef BUILD_CFG_PIX_SUBSAMPLE
		DisplayWidth /= 2;
#endif
		FrameBuffer = FrameBufferFull;
        //Initialize such that writes go to buffer0 and we are displaying buffer1
        if(FB_NUM_BUFFERS == 1)
            DisplayFrame = 0;
        else
            DisplayFrame = 1;
	}
	ERR_BLOCK_END;

	if(Error)
		DISP_Close(0);

	return Error;
}

ErrorCode_t DISP_StartFrameUpdates(struct frame_update *pUpdate)
{
	ErrorCode_t Error = SUCCESS;

	ERR_BLOCK_BEGIN
	{
    if(ioctl(DisplayFD,  FBIO_FRAME_UPDATE_START, pUpdate))
        ERR_THROW_MSG(Error = ERR_DEVICE_FAIL, "ioctl:FBIO_FRAME_UPDATE_START");
	}
	ERR_BLOCK_END;


    return Error;
}

ErrorCode_t DISP_SetDisplayBufferAtBeginning(void)
{
    DisplayFrame = FB_NUM_BUFFERS - 1;
    DISP_UpdateFrame(TRUE, TRUE);

	return SUCCESS;
}

ErrorCode_t DISP_SetWriteBufferAtBeginning(void)
{
    DisplayFrame = FB_NUM_BUFFERS - 2;
    DISP_UpdateFrame(TRUE, TRUE);

	return SUCCESS;
}

ErrorCode_t DISP_UpdateFrame(BOOL Update, BOOL Wait)
{
	if(Wait)
	{
		int Arg = 0;
		if(ioctl(DisplayFD, FBIO_WAITFORVSYNC, &Arg))
			THROW(ERR_DEVICE_FAIL);
	}

	if(Update)
	{
        DisplayFrame++;
        if(DisplayFrame == FB_NUM_BUFFERS) //Wrap around display pointer
        {
            DisplayFrame = 0;
			VarScreenInfo.yoffset = 0;
        }
        else
        {
			VarScreenInfo.yoffset = DisplayHeight * DisplayFrame;
        }

        if(DisplayFrame == FB_NUM_BUFFERS - 1) //Wrap around write pointer
        {
            FrameBuffer = FrameBufferFull;
        }
        else
        {
            FrameBuffer = FrameBufferFull + (DisplayFrame+1) * FrameBuffSize;
        }

		if(ioctl(DisplayFD, FBIOPAN_DISPLAY, &VarScreenInfo))
			THROW(ERR_DEVICE_FAIL);

	}
	return SUCCESS;
}

/**
 * Uninitialize the Projector module
 *
 * returns SUCCESS
 */
ErrorCode_t DISP_Close(int flag)
{
	if(DisplayFD > 0)
	{
		close(DisplayFD);
		DisplayFD = -1;
	}

	if(FrameBufferFull)
	{
		munmap(FrameBufferFull, FrameBuffSize * FB_NUM_BUFFERS);
		FrameBuffSize = 0;
		FrameBuffer = NULL;
	}
	return SUCCESS;
}

/**
 * Gets the width of the display
 *
 * returns width in pixel
 */
unsigned DISP_GetWidth(void)
{
	return DisplayWidth;
}

/**
 * Gets the height of the display
 *
 * returns height in pixel
 */
unsigned DISP_GetHeight(void)
{
	return DisplayHeight;
}


/**
 * Fills the projector full screen with given color
 *
 * Color - I - Color to be filled
 *				Use DISP_COLOR_XXXX macros
 *
 * returns SUCCESS
 */
ErrorCode_t DISP_FillScreen(DISP_Color_t Color)
{
	unsigned y;
	unsigned x;
	uint32 *Ptr = (uint32 *)FrameBuffer;
	unsigned Width = DisplayLineWidth/4;
	uint32 FillVal, FillVal2;
	DISP_Color_t Color2=DISP_COLOR_BLACK;

	if(FrameBuffer == NULL)
		THROW(ERR_NOT_INITIALIZED);

	switch(sizeof(DISP_Color_t))
	{
		case 1:
			FillVal = (Color << 24) | (Color << 16) | (Color << 8) | Color;
			FillVal2 = (Color2 << 24) | (Color2 << 16) | (Color2 << 8) | Color2;
			break;
		case 2:
			FillVal = (Color << 16) | Color;
			FillVal2 = (Color2 << 16) | Color2;
			break;
		case 4:
			FillVal = Color;
			FillVal2 = Color2;
			break;
		default:
			DEBUG_ERR("DISP-Unsupported pixel size");
			return FAIL;
	}

	for(y = 0; y < DisplayHeight; y++)
	{
		for(x = 0; x < Width; x++)
			Ptr[x] = FillVal;

		Ptr += Width;
	}

	for(y = DisplayHeight; y < DisplayHeight*FB_NUM_BUFFERS; y++)
	{
		for(x = 0; x < Width; x++)
			Ptr[x] = FillVal2;

		Ptr += Width;
	}

	return SUCCESS;
}

/**
 * Sets the given pixel
 *
 * X - I - Horizontal position
 * Y - I - Vertical position
 * Color - I - Color to be filled
 *				Use DISP_COLOR_XXXX macros
 *
 * returns SUCCESS
 */
ErrorCode_t DISP_SetPixel(unsigned X, unsigned Y, DISP_Color_t Color)
{
	if(FrameBuffer == NULL)
		THROW(ERR_NOT_INITIALIZED);

	if(X >= DisplayWidth && Y >= DisplayHeight)
		THROW(ERR_INVALID_PARAM);

	DISP_Color_t *Ptr = (DISP_Color_t *)(FrameBuffer + Y * DisplayLineWidth);
	Ptr[X] = Color;

	return SUCCESS;
}

/**
 * Get Pixel iterator
 *
 * X - I - Horizontal position
 * Y - I - Vertical position
 * Iter - O - Pixel Iterator
 *
 * returns SUCCESS
 */
ErrorCode_t DISP_GetPixIter(unsigned X, unsigned Y, DISP_PixIter_t *Iter)
{
	if(FrameBuffer == NULL)
		THROW(ERR_NOT_INITIALIZED);

	if(X >= DisplayWidth && Y >= DisplayHeight)
		THROW(ERR_INVALID_PARAM);

	*Iter = (DISP_Color_t *)(FrameBuffer + Y * DisplayLineWidth) + X;
	return SUCCESS;
}


/**
 * Fills one line with given color
 *
 * Color - I - Color to be filled
 *				Use DISP_COLOR_XXXX macros
 *
 * returns SUCCESS
 */
ErrorCode_t DISP_FillLine(unsigned Line, DISP_Color_t Color)
{
	unsigned x;
	uint32 *Ptr = (uint32 *)(FrameBuffer + Line * DisplayLineWidth);
	uint32 FillVal;
	unsigned Width = DisplayLineWidth/4;

	if(FrameBuffer == NULL)
		THROW(ERR_NOT_INITIALIZED);

	if(Line >= DisplayHeight)
		THROW(ERR_INVALID_PARAM);

	switch(sizeof(DISP_Color_t))
	{
		case 1:
			FillVal = (Color << 24) | (Color << 16) | (Color << 8) | Color;
			break;
		case 2:
			FillVal = (Color << 16) | Color;
			break;
		case 4:
			FillVal = Color;
			break;
		default:
			THROW_MSG(ERR_INVALID_PARAM, "DISP-Unsupported pixel size");
	}

	for(x = 0; x < Width; x++)
		Ptr[x] = FillVal;

	return SUCCESS;
}

#if 0
/**
 * Fills one line with given bit pattern
 *
 * Line - I - Line number to be filled
 * Pattern - I - Bit pattern, each pixel represents one pixel (on/off)
 * Invert - I - TRUE : Inver the pattern
 *              FALSE : Dont invert the pattern
 *
 * returns SUCCESS
 */
ErrorCode_t DISP_DrawLinePattern(unsigned Line, uint8 *Pattern, BOOL Invert)
{
	unsigned x;
	unsigned i;
	uint32 *Ptr = (uint32 *)(FrameBuffer + Line * DisplayLineWidth);
	uint32 Pat2Pix[4] = { 0x00000000, 0x0000FFFF, 0xFFFF0000, 0xFFFFFFFF };

	for(x = 0; x < DisplayWidth/2;)
	{
		uint8 Byte = *Pattern ++;

		/* Whether or not to invert the pattern */
		if(Invert)
			Byte = ~Byte;
		/* Display one byte - 8 pixles - of the pattern */
		for(i = 0; i < 4; i ++)
		{
			Ptr[x] = Pat2Pix[Byte&3];
			Byte >>= 2;
			x++;

			if(x >= DisplayWidth/2)
				break;
		}
	}

	return SUCCESS;
}
#endif


/**
 * Convert Y U V value to RGB value
 *
 * returns 32 bit RGB
 */
static uint32 YUV2RGB(int y, int u, int v)
{
   int r, g, b;
   int c = y-16, d = u - 128, e = v - 128;       

   r = (298 * c           + 409 * e + 128) >> 8;
   g = (298 * c - 100 * d - 208 * e + 128) >> 8;
   b = (298 * c + 516 * d           + 128) >> 8;

   // Even with proper conversion, some values still need clipping.

   if (r > 255) r = 255;
   if (g > 255) g = 255;
   if (b > 255) b = 255;
   if (r < 0) r = 0;
   if (g < 0) g = 0;
   if (b < 0) b = 0;

   return DISP_PACK_COLOR(r,g,b);
}


/**
 * Converts the BGGR (Bayer  pattern) image pixel to 
 * Projector pixel value
 *
 *	Image - Image to get the pixel from
 *	x - Column number
 *	y - Row number
 *
 * returns Pixel value in projector format
 */
unsigned BGGR2Pixel(Image_t *Image, int x, int y)
{
	uint32 Red = 0;
	uint32 Green = 0;
	uint32 Blue = 0;
	/*
	 * B G B G B G
	 * G R G R G R
	 * B G B G B G
	 * G R G R G R
	 * B G B G B G
	 */
	switch(((y&1)<<1) | (x&1))
	{
		case 0:
			Red   = (GetImagePixel(Image, x-1, y-1) + 
					 GetImagePixel(Image, x+1, y-1) + 
					 GetImagePixel(Image, x-1, y+1) + 
					 GetImagePixel(Image, x+1, y+1))/4;

			Green = (GetImagePixel(Image, x-1, y) + 
					 GetImagePixel(Image, x+1, y) + 
					 GetImagePixel(Image, x, y-1) + 
					 GetImagePixel(Image, x, y+1))/4; 
			Blue  =  GetImagePixel(Image, x,y);
			break;
		case 1:
			Red   = (GetImagePixel(Image, x, y-1) + 
					 GetImagePixel(Image, x, y+1))/2;
			Green =  GetImagePixel(Image, x,y);
			Blue  = (GetImagePixel(Image, x-1, y) + 
					 GetImagePixel(Image, x+1, y))/2;
			break;
		case 2:
			Red   = (GetImagePixel(Image, x-1, y) + 
					 GetImagePixel(Image, x+1, y))/2;
			Green =  GetImagePixel(Image, x,y);
			Blue  = (GetImagePixel(Image, x, y-1) + 
					 GetImagePixel(Image, x, y+1))/2;
			break;
		case 3:
			Red   =  GetImagePixel(Image, x,y);
			Green = (GetImagePixel(Image, x-1, y) + 
					 GetImagePixel(Image, x+1, y) + 
					 GetImagePixel(Image, x, y-1) + 
					 GetImagePixel(Image, x, y+1))/4; 
			Blue  = (GetImagePixel(Image, x-1, y-1) + 
					 GetImagePixel(Image, x+1, y-1) + 
					 GetImagePixel(Image, x-1, y+1) + 
					 GetImagePixel(Image, x+1, y+1))/4;
			break;
	}
	return DISP_PACK_COLOR(Red, Green, Blue);
}

/**
 * Draw the given image on the projetor screen
 *
 *	Image - Image to be written
 *
 *  returns Pixel value in projector format
 */
ErrorCode_t DISP_DrawImage(unsigned XOff, unsigned YOff, Image_t *Image,int flag)
{
	int y;
	int x;
	unsigned Width = MIN(DisplayWidth, Image->Width);
	unsigned Height = MIN(DisplayHeight, Image->Height);
	uint8 *ImgPtr = (uint8 *)Image->Buffer;
	DISP_PixIter_t Iter = { 0 };

	switch(Image->PixFormat)
	{
		case IMAGE_PIX_FORMAT_UYVY16:
			for(y = 0; y < Height; y++)
			{
				DISP_GetPixIter(XOff, y  + YOff, &Iter);
				for(x = 0; x < Width ; x++)
				{
					if(x&1)
						DISP_SET_PIX(Iter, YUV2RGB(ImgPtr[x*2+1], ImgPtr[x*2-2], ImgPtr[x*2]));
					else
						DISP_SET_PIX(Iter, YUV2RGB(ImgPtr[x*2+1], ImgPtr[x*2], ImgPtr[x*2+2]));
					Iter = DISP_NEXT_PIX(Iter);
				}
				ImgPtr += Image->LineWidth;
			}
			break;

		case IMAGE_PIX_FORMAT_SBGGR:
			for(y = 0; y < Height; y++)
			{
				DISP_GetPixIter(XOff, y + YOff, &Iter);
				for(x = 0; x < Width; x++)
				{
					DISP_SET_PIX(Iter, BGGR2Pixel(Image, x, y));
					Iter = DISP_NEXT_PIX(Iter);
				}
			}
			break;

		case IMAGE_PIX_FORMAT_GREY10:
			for(y = 0; y < Height; y++)
			{
				DISP_GetPixIter(XOff, y + YOff, &Iter);
				for(x = 0; x < Width ; x++)
				{
					uint16 Color = ((uint16 *)ImgPtr)[x];
					Color >>= 2;
					DISP_SET_PIX(Iter, DISP_PACK_COLOR(Color, Color, Color));
					Iter = DISP_NEXT_PIX(Iter);
				}
				ImgPtr += Image->LineWidth;
			}
			break;


		case IMAGE_PIX_FORMAT_GREY8:
			for(y = 0; y < Height; y++)
			{
				DISP_GetPixIter(XOff, y+YOff, &Iter);
				for(x = 0; x < Width ; x++)
				{
					uint8 Color = ImgPtr[x];
					DISP_SET_PIX(Iter, DISP_PACK_COLOR(Color, Color, Color));
					Iter = DISP_NEXT_PIX(Iter);
				}
				ImgPtr += Image->LineWidth;
			}
			break;

		case IMAGE_PIX_FORMAT_RGB24:
				if(DISP_DrawImage24(XOff, YOff, Image,flag))
				THROW(FAIL);
			
			break;

		case IMAGE_PIX_FORMAT_RGB32:
			if(DISP_DrawImage32(XOff, YOff, Image))
				THROW(FAIL);
			break;

		default:
			return ERR_NOT_SUPPORTED;
	}
	return SUCCESS;
}

/**
 * Draw rectangle
 *
 * StartX - Starting X
 * StartY - Starting Y
 * Width - Width of the rectangle
 * Height - Height of the rectangle
 * Color - Color of the rectangle
 *
 *  returns SUCCESS
 */
ErrorCode_t DISP_DrawRectangle(int StartX, int StartY, int Width, int Height, DISP_Color_t Color, BOOL or)
{
	int x;
	int y;
	ErrorCode_t Error = SUCCESS;
    DISP_Color_t screen_color;
    DISP_Color_t set_color;

	ERR_BLOCK_BEGIN
	{
		if(StartX < 0 || StartX + Width > DisplayWidth || 
				StartY < 0 || StartY + Height > DisplayHeight)
			ERR_THROW(Error = ERR_INVALID_PARAM);

		for(y = StartY; y < StartY + Height; y++)
		{
			DISP_PixIter_t Iter;

			if(DISP_GetPixIter(StartX, y, &Iter))
				ERR_THROW(Error = FAIL);

			for(x = 0; x < Width; x ++)
			{
                if(or == TRUE)
                {
                    DISP_GET_PIX(Iter, &screen_color);
                    set_color = Color | screen_color;
                    DISP_SET_PIX(Iter, set_color);
                }
                else
                    DISP_SET_PIX(Iter, Color);
				Iter = DISP_NEXT_PIX(Iter);
			}
		}
	}
	ERR_BLOCK_END;

	return Error;
}

ErrorCode_t DISP_DisplayColorBar(void)
{
	unsigned x;
	unsigned y;
	unsigned BandWidth = (DisplayWidth - 4)/DISP_BITS_PER_PIX;
	unsigned Border = (DisplayWidth - BandWidth * DISP_BITS_PER_PIX);
    unsigned val = 0;

	//DISP_FillScreen(DISP_COLOR_WHITE);

	//DISP_DrawRectangle(1, 1, DisplayWidth - 2, DisplayHeight - 2, DISP_COLOR_BLACK);

	for(y = Border/2; y < DisplayHeight-Border/2; y++)
	{
		DISP_PixIter_t Iter = { 0 };
		DISP_GetPixIter(Border/2, y, &Iter);
		for(x = 0; x < DisplayWidth - Border; x++)
		{
			DISP_SET_PIX(Iter, val);
			Iter = DISP_NEXT_PIX(Iter);
            if(x%80 == 0)
            {
                if(val==0)
                    val = 0xFFFFFFFF;
                else
                    val = 0;
            }
		}
	}

	return SUCCESS;
}

static ErrorCode_t DISP_DrawImage24(unsigned XOff, unsigned YOff, Image_t *Image,int flag)
{
	uint32 *ImagePtr=(uint32 *)Image->Buffer;
	uint32 *FramePtr=(uint32 *)FrameBuffer;
	//uint32 *TempPtr =(uint32 *)TempBuffer;
	unsigned Width = Image->Width;
	
	//printf("\nDrawImage24\n");
	if(flag==0)
	{
		memcpy(FramePtr, ImagePtr, FrameBuffSize);
		//memcpy(TempPtr, ImagePtr, size );
	}
	else
	{
		FramePtr += (XOff*2 +YOff*(Width*3));
		memcpy(FramePtr, ImagePtr, FrameBuffSize);
		//memcpy(TempPtr, ImagePtr, size);

	}
	return SUCCESS;
}


ErrorCode_t DISP_GetFrameBuffer(Image_t *Image)
{
	Image->Width = DisplayWidth;
	Image->Height = DisplayHeight;
	Image->LineWidth = DisplayLineWidth;
	Image->PixFormat = IMAGE_PIX_FORMAT_RGB24;
	Image->Buffer = FrameBuffer;

	return SUCCESS;
}

static ErrorCode_t DISP_DrawImage32(unsigned XOff, unsigned YOff, Image_t *Image)
{
#if DISP_BITS_PER_PIX >= 24
	unsigned Width = Image->Width;
	unsigned Height = Image->Height;
	uint8 *ImagePtr = Image->Buffer;
	uint8 *FramePtr = FrameBuffer;

	if(Width + XOff > DisplayWidth)
	{
		Width = DisplayWidth - XOff;
	}

	if(Height + YOff > DisplayHeight)
	{
		Height = DisplayHeight - YOff;
	}

	FramePtr += YOff * DisplayLineWidth + XOff * 4;


	if(XOff == 0 && Width == DisplayWidth && 
				Image->LineWidth == DisplayLineWidth)
	{
		memcpy(FramePtr, ImagePtr, DisplayLineWidth * Height);
	}
	else
	{
		while(Height--)
		{
			memcpy(FramePtr, ImagePtr, Width * 4);
			FramePtr += DisplayLineWidth;
			ImagePtr += Image->LineWidth;
		}
	}

	return SUCCESS;
#else
	return FAIL;
#endif
}
