/*
 * Display.h
 *
 * This module provides the interfaces for the display related functions
 *
 * Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
 * ALL RIGHTS RESERVED
 *
*/

#ifndef DISPLAY_H_
#define DISPLAY_H_

#include "Error.h"
#include "Common.h"

#ifdef __cplusplus
extern "C" {
#endif

#define DISP_BITS_PER_PIX		24
#define FB_NUM_BUFFERS  46

struct frame_update
{
    int num_images;
    int frames_per_image;
    int frame_iter;
    int frames_this_image[FB_NUM_BUFFERS];
};

#if DISP_BITS_PER_PIX >= 24
#define DISP_PACK_COLOR(r,g,b)	(DISP_Color_t)((((r) << 16) & 0xFF0000UL) | \
											   (((g) <<  8) & 0x00FF00UL) | \
											   (((b) <<  0) & 0x0000FFUL))
typedef uint32 DISP_Color_t;
#elif DISP_BITS_PER_PIX == 16
#define DISP_PACK_COLOR(r,g,b)	(DISP_Color_t)((((r) << 8) & 0xF100U) | \
											   (((g) << 3) & 0x07C0U) | \
											   (((b) >> 3) & 0x001FU)   )
typedef uint16 DISP_Color_t;
#elif  DISP_BITS_PER_PIX == 8
typedef uint8 DISP_Color_t;
#else
#error DISP_BITS_PER_PIX unsupported
#endif

#define DISP_COLOR_WHITE    DISP_PACK_COLOR(0xFF, 0xFF, 0xFF)
#define DISP_COLOR_BLACK    DISP_PACK_COLOR(0x00, 0x00, 0x00)
#define DISP_COLOR_RED      DISP_PACK_COLOR(0xFF, 0x00, 0x00)
#define DISP_COLOR_GREEN    DISP_PACK_COLOR(0x00, 0xFF, 0x00)
#define DISP_COLOR_BLUE     DISP_PACK_COLOR(0x00, 0x00, 0xFF)
#define DISP_COLOR_MAGENTA  DISP_PACK_COLOR(0xFF, 0xFF, 0x00)
#define DISP_COLOR_YELLOW   DISP_PACK_COLOR(0x1F, 0x00, 0xFF)
#define DISP_COLOR_CYAN     DISP_PACK_COLOR(0x00, 0xFF, 0xFF)
#define DISP_COLOR_GRAY(v)	DISP_PACK_COLOR(((v * 0xFF)/100), ((v * 0xFF)/100), ((v * 0xFF)/100))

#define DISP_SET_PIX(iter, val)			*(iter) = (val)
#define DISP_GET_PIX(iter, pval)		*(pval) = *(iter)
#define DISP_NEXT_PIX(iter)				((iter) + 1)
#define DISP_ADVANCE_PIX(iter, val)		((iter) + (val))

typedef DISP_Color_t *DISP_PixIter_t;

ErrorCode_t DISP_Init(void);
ErrorCode_t DISP_Close(int flag);
ErrorCode_t DISP_FillScreen(DISP_Color_t Color);
ErrorCode_t DISP_FillLine(unsigned Line, DISP_Color_t Color);
ErrorCode_t DISP_DrawLinePattern(unsigned Line, unsigned char *Pattern, 
															BOOL Invert);
ErrorCode_t DISP_SetPixel(unsigned X, unsigned Y, DISP_Color_t Color);
ErrorCode_t DISP_DrawRectangle(int StartX, int StartY, int Width, int Height, DISP_Color_t Color, BOOL or);
ErrorCode_t DISP_GetPixIter(unsigned X, unsigned Y, DISP_PixIter_t *Iter);

ErrorCode_t DISP_DrawImage(unsigned XOff, unsigned YOff, Image_t *Image,int flag);
ErrorCode_t DISP_DisplayColorBar(void);
ErrorCode_t DISP_GetFrameBuffer(Image_t *Image);
ErrorCode_t DISP_UpdateFrame(BOOL Update, BOOL Wait);
ErrorCode_t DISP_SetDisplayBufferAtBeginning(void);
ErrorCode_t DISP_SetWriteBufferAtBeginning(void);
ErrorCode_t DISP_StartFrameUpdates(struct frame_update *pUpdate);


unsigned DISP_GetWidth(void);
unsigned DISP_GetHeight(void);

#ifdef __cplusplus
}
#endif

#endif /* DISPLAY_H_ */
