/*
 * APIs to configure and communicate with ADS1256 ADC over SPI bus
 *
 * Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
 * ALL RIGHTS RESERVED

 */

#include <stdint.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <linux/types.h>
#include <linux/spi/spidev.h>
#include <string.h>

#include "ads1255.h"
extern uint32_t spi_clk_speed;

void ads1255_read_data(int fd, uint32_t *rx_buf, int num_samples)
{
	int ret;
	uint32_t tx[64*1024-1] = {0x03030303,0, }; //03 is the RDATAC command
	struct spi_ioc_transfer tr = {
		.tx_buf = (unsigned long)tx,
		.rx_buf = (unsigned long)rx_buf,
		.len = num_samples*4,
		.delay_usecs = 0,
		.speed_hz = spi_clk_speed,// >5 Mhz found not to work correctly with ADS1256 EVM
		.bits_per_word = 32,
        .cs_change = 0,
	};

    tx[num_samples-1] = 0x0f0f0f0f;//SDATAC command

	ret = ioctl(fd, SPI_IOC_MESSAGE(1), &tr);
	if (ret < 1)
		printf("can't send spi message\n");

	if(num_samples < 1024)
	{
		for (ret = 0; ret < num_samples; ret++) {
			if (!(ret % 4))
				puts("");
			printf("%.8X ", *(rx_buf+ret));
		}
		puts("");
	}
}

int ads1255_write_command(int fd, uint8_t cmd)
{
	struct spi_ioc_transfer xfer[2];
	unsigned char tx_buf[4] = {0x00, 0x00, 0x00, 0x00};
	unsigned char rx_buf[4] = {0x00, 0x00, 0x00, 0x00};
	int ret = 0;

    memset(&xfer[0], 0, sizeof(struct spi_ioc_transfer)*2);

	xfer[0].tx_buf= (unsigned long) &tx_buf;
	xfer[0].rx_buf= (unsigned long) &rx_buf;	
	tx_buf[0] = cmd;
	xfer[0].len = 1;
	xfer[0].delay_usecs = 7,
	xfer[0].speed_hz = spi_clk_speed;	// >5 Mhz found not to work correctly with ADS1256 EVM
	xfer[0].bits_per_word = 8;

	//Read first value
	xfer[1].tx_buf= (unsigned long) &(tx_buf[1]);
	xfer[1].rx_buf= (unsigned long) &(rx_buf[1]);
	xfer[1].len = 3;
	xfer[1].speed_hz = spi_clk_speed;	// >5 Mhz found not to work correctly with ADS1256 EVM
	xfer[1].bits_per_word = 8;
	

	ret = ioctl(fd, SPI_IOC_MESSAGE(2), xfer);
	if (ret < 1)
	{
		printf("can't write %d command\n", cmd);
		return ret;
	}

	return ret;
}

int ads1255_get_register(int fd, uint8_t reg_name, uint8_t *p_reg_val)
{
	struct spi_ioc_transfer xfer[2];
	unsigned char tx_buf[3];
	unsigned char rx_buf[3];
	
	int ret = 0;
    memset(&xfer[0], 0, sizeof(struct spi_ioc_transfer)*2);

	//RREG command	- Read back to verify
	tx_buf[0] = ADS1255_RREG | reg_name;
	tx_buf[1] = 0;
	xfer[0].tx_buf = (unsigned long) tx_buf;
	xfer[0].rx_buf = (unsigned long) rx_buf;
	xfer[0].len = 2;
	xfer[0].delay_usecs = 7,
	xfer[0].speed_hz = spi_clk_speed;	// >5 Mhz found not to work correctly with ADS1256 EVM
	xfer[0].bits_per_word = 8;

	tx_buf[2] = 0;
	rx_buf[2] = 0;
	xfer[1].tx_buf= (unsigned long) &(tx_buf[2]);
	xfer[1].rx_buf= (unsigned long) &(rx_buf[2]);
	xfer[1].len = 1;
	xfer[1].delay_usecs = 7,
	xfer[1].speed_hz = spi_clk_speed;	// >5 Mhz found not to work correctly with ADS1256 EVM
	xfer[1].bits_per_word = 8;

	ret = ioctl(fd, SPI_IOC_MESSAGE(2), xfer);
	if (ret < 1)
	{
		printf("can't write spi message\n");
		return ret;
	}

    printf("rx_buf = %x %x %x\n", rx_buf[0], rx_buf[1], rx_buf[2]);
    *p_reg_val = rx_buf[2];
    return 0;
}

int ads1255_set_register(int fd, uint8_t reg_name, uint8_t reg_val)
{
	struct spi_ioc_transfer xfer[2];
	unsigned char tx_buf[3];
	unsigned char rx_buf[3];
    uint8_t read_val;
	
	int ret = 0;
    memset(&xfer[0], 0, sizeof(struct spi_ioc_transfer)*2);

	//WREG command
	xfer[0].tx_buf = (unsigned long) tx_buf;
	xfer[0].rx_buf = (unsigned long) rx_buf;
	tx_buf[0] = ADS1255_WREG | reg_name;
	tx_buf[1] = 0;
	tx_buf[2] = reg_val;
	xfer[0].len = 3;
	xfer[0].delay_usecs = 0,
	xfer[0].speed_hz = spi_clk_speed;	// >5 Mhz found not to work correctly with ADS1256 EVM
	xfer[0].bits_per_word = 8;
	ret = ioctl(fd, SPI_IOC_MESSAGE(1), xfer);
	if (ret < 1)
	{
		printf("can't write spi message\n");
		return ret;
	}

    if(ads1255_get_register(fd, reg_name, &read_val) != 0)
    {
        printf("Register read back error\n");
        ret = -1;
    }
	else {
		if(read_val != reg_val)
		{
			printf("Data read back from register (0x%x) does not match the value written 0x%x\n", read_val, reg_val);
			ret = -1;
		}
	}

	return ret;
}

int ads1255_set_PGA_gain(int fd, int multiplier)
{
    uint8_t reg_val=0x20;

    if(ads1255_get_register(fd, ADS1255_ADCON, &reg_val) != 0)
        return -1;
    else
        printf("ADCON reg value is 0x%x\n", reg_val);

    reg_val &= 0xF8;    //Zero out bits 0-2
    switch(multiplier)
    {
        case 1:
            break;
        case 2:
            reg_val |= 1;
            break;
        case 4:
            reg_val |= 2;
            break;
        case 8:
            reg_val |= 3;
            break;
        case 16:
            reg_val |= 4;
            break;
        case 32:
            reg_val |= 5;
            break;
        case 64:
            reg_val |= 6;
            break;
        default:
            printf("Invalid Muliplier value - Valid values are 1, 2, 4, 8, 16, 32 and 64\n");
            return -1;
    }
    if(ads1255_set_register(fd, ADS1255_ADCON, reg_val) < 0)
        return -1;

    return 0;
}
uint8_t ads1255_convert_drate_to_regval(int drate)
{
	uint8_t ret_val;
	switch(drate) 
	{
		case 30000:
			ret_val= 0xF0;
			break;
		case 15000:
			ret_val= 0xE0;
			break;
		case 7500:
			ret_val= 0xD0;
			break;
		case 3750:
			ret_val= 0xC0;
			break;
		case 2000:
			ret_val= 0xB0;
			break;
		case 1000:
			ret_val= 0xA1;
			break;
		case 500:
			ret_val= 0x92;
			break;
		case 100:
			ret_val= 0x82;
			break;
		case 60:
			ret_val= 0x72;
			break;
		case 50:
			ret_val= 0x63;
			break;
		case 30:
			ret_val= 0x53;
			break;
		case 25:
			ret_val= 0x43;
			break;
		case 15:
			ret_val= 0x33;
			break;
		case 10:
			ret_val= 0x23;
			break;
		case 5:
			ret_val= 0x13;
			break;
		default:
            printf("Invalid value specified for ADC data rate\n");
			ret_val=0;
			break;
	}
	return ret_val;
}
