/*
 * dlp_nirscan.c
 *
 * This module handles the command line arguments and invokes the appropriate functions for requested action.
 *
 * Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
 * ALL RIGHTS RESERVED
 *
*/

#include <stdint.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/types.h>
#include <linux/spi/spidev.h>
#include <string.h>

#include "ads1255.h"
#include "dlpspectro.h"
#include "dlpc350_api.h"
#include "usb.h"
#include "Error.h"
#include "version.h"
#include "Display.h"

#define MAX_FILENAME_LEN 256
#define NUM_CAL_PEAKS 6
#define POLY_DEGREE   3
#define INTEGRATION_TIME 1500
#ifndef HOST_MODE
#define EEPROM_FILE "/sys/bus/i2c/devices/1-0050/eeprom"
#else
#define EEPROM_FILE "eeprom"
#endif
#define CALINFO_COEFFS_OFFSET 0xc00
#define CALINFO_CPS_OFFSET 0xc80

static const char *device = "/dev/spidev1.0";
static uint8_t mode = 1;
static uint8_t bits = 8;
uint32_t spi_clk_speed = 1500000; // must be between 10x sample rate and 1.92MHz. 750KHz is too slow for 30KHz data transfer
static uint8_t drate_reg = 0;
static uint8_t pga_gain = 0;
static bool set_pga_gain = false;
static bool perform_scan = false;
static bool perform_streaming = false;
static bool perform_load_for_streaming = false;
static bool compute_absorption = false;
static bool compute_coeffs = false;
static bool compute_pixel = false;
static bool compute_wavelength = false;
static bool find_peaks = false;
static bool calculate_hadamard_avg_readings = false;
static char firmware_file[MAX_FILENAME_LEN] = {0, };
static char adc_reading_filename[MAX_FILENAME_LEN] = " ";
static char ref_adc_reading_filename[MAX_FILENAME_LEN] = " ";
static char absorption_filename[MAX_FILENAME_LEN] = " ";
static char sdf_filename[MAX_FILENAME_LEN] = " ";
static char peaks_filename[MAX_FILENAME_LEN] = "peak_locations.txt";
static int num_images=0;
static bool perform_calibration=false;
static bool generate_csv_patterns=false;
static bool perform_flash_loadtime_measurement=false;
static bool write_calinfo = false;
static bool restore_calinfo = false;
static bool prepare_scan_solution=false;
static bool read_samples=false;
static int num_read_samples=0;
static int ptn_count=0;
static int first_img_idx=0;
static unsigned long int itrDelay = 0; //in ms
static int scanLoop = 1;
static int ptn_exposure_time_us = 0;
static int first_pixel = -1;
static int last_pixel = -1;
static int num_patterns = -1;
static int csv_begin_number=0;
static float threshold_factor = 0.001;
bool use_alternate_streaming_method = true;
bool use_hadamard_patterns = false;

static void pabort(const char *s)
{
	perror(s);
	abort();
}

static void print_usage(const char *prog)
{
	DEBUG_ERR("Usage: %s [-IvlSELdfGrTPfCpWRwxyAZNHoe]\n", prog);
	puts("  -I Initialize DLPC350\n"
		 "  -v print dlp_nirscan application version information\n"
	     "  -l <num_images> load scan_img_000.bmp thru scan_img_nnn to memory from SD card for streaming\n"
	     "  -S <ptn_count> stream n images via parallel RGB that are preloaded using the -l option below\n"
	     "  -V <sdf_file> stream n images via parallel RGB that are preloaded using the -l option below and use exposure times as defined in the sdf file\n"
         "  -E <exposure_time_in_microseconds> \n"
		 "  -L number iterations to be performed in a loop; default is treated as 1\n"
		 "  -d delay in milli second between each iterations\n"
		 "  -f <file_name> to store the mean ADC readings\n"
		 "  -G Generate absorption spectrum for the reference and sample adc reading file list. \n"
		 "  -r <file_name> file with calibration scan mean  readings with no object (open beam) \n"
		 "  -T <threshold_factor> values in readings file that are below threshold_factor*median_of_readings will be discarded. Default threshold_factor is 0.001\n"
	     "  -P <sdf_file> prepare scan solution as per the given Scan Description File\n"
		 "  -f <file_name> to store the mean ADC readings\n"
         "  -i <first_img_index> -c <ptn_count> - Performs a scan using prebuilt scan images in flash starting at first_img_index and goes thru ptn_count 1-bit patterns\n"
         "  -C perform calibration scan\n"
		 "  -p <file_name> find peaks in the data presented in <file_name>\n"
         "  -W Write calibration info to EEPROM\n"
         "  -R Restore calibration info from EEPROM\n"
         "  -w Compute lambda-pixel and pixel-lambda polynomial co-efficients from peak_locations.txt\n"
         "  -x Get DMD pixel positions for wavelengths given in wavelengths.txt\n"
         "  -y get Wavelengths for pixel values given in pixels.txt\n"
         "  -A <first pixel> input for generate_csv_patterns\n"
         "  -Z <last pixel> input for generate_csv_patterns\n"
         "  -b <begin_number> beginning number to use for generated csv file names\n"
         "  -N <num_patterns > input for generate_csv_patterns\n"
         "  -m <readings_filename> -f<output_transformed_lambda_readings_filename>\n"
         "  -t <num_samples > number of samples to read\n"
         "  -H <use_hadamard_patterns > input for generate_csv_patterns\n\n\n"
		 " Example-1: ./dlp_nirscan -G -r<adc_read_file_ref> -f<adc_read_file> (Generates absorption file from the sample readings in <adc_read_file> and reference readings in <adc_read_file_ref>)\n"
		 " Example-2: ./dlp_nirscan -A0 -Z1823 -N225 (Generates 225 patterns in CSV format with first patterns starting at pixel 0 and last one ending at pixel 1823)\n"
		 " Example-3: ./dlp_nirscan -l10 -S245 -E1500 (Loads scan_img_000.bmp thru scan_img_009.bmp to memory and performs scan using 245 patterns from those images the exposure time of each pattern being 1500 microseconds\n"
		 " Example-4: ./dlp_nirscan -i1 -C116 -E8000 (Performs scan with 116 patterns from flash starting with the image at index1 and exposure time of each pattern being 8000 microseconds)\n\n\n");
	exit(1);
}


static void parse_opts(int argc, char *argv[])
{
    while (1) {
        static const struct option lopts[] = {
            { "stream-images",  1, 0, 'S' },
            { "load-images",  1, 0, 'l' },
            { "do-calibration",  0, 0, 'C' },
            { "set-adc-data-rate",1, 0, 'D' },
            { "prep-scan-solution",   1, 0, 'P' },
            { "build-firmware",   1, 0, 'B' },
            { "upload-firmware",   1, 0, 'U' },
            { "num-images",   1, 0, 'n' },
            { "first-img-idx",   1, 0, 'i' },
            { "scan-img-count",   1, 0, 'c' },
            { "set-exposure-time",   1, 0, 'E' },
            { "init-dlpc350",   0, 0, 'I' },
            { "init-dlpc350-video-mode",   0, 0, 'V' },
            { "print-version",   0, 0, 'v' },
            { "measure-flash-loadtime",   0, 0, 'M' },
            { "generte-absorption-spectrum",   0, 0, 'G' },
            { "set spi clock speed", 1, 0, 's' },
            { "name of adc reading file", 1, 0, 'f' },
            { "name of reference reading file", 1, 0, 'r' },
            { "find peaks", 1, 0, 'p' },
            { "scan iterations", 1, 0, 'L' },
            { "delay b/w each iteration in milliseconds", 1, 0, 'd' },
            { "PGA multiplier value", 1, 0, 'g' },
            { "write-calinfo", 0, 0, 'W' },
            { "restore-calinfo", 0, 0, 'R' },
            { "compute-coefficients", 2, 0, 'w' },
            { "find pixel position", 1, 0, 'x' },
            { "find wavelength", 1, 0, 'y' },
            { "first-pixel", 1, 0, 'A' },
            { "last-pixel", 1, 0, 'Z' },
            { "num-patterns", 1, 0, 'N' },
            { "use-hadamard-patterns", 1, 0, 'H' },
            { "generate-odd-column-bands-only", 0, 0, 'o' },
            { "generate-even-column-bands-only", 0, 0, 'e' },
            { "find hadamard mean readings", 1, 0, 'm' },
            { "begin-number", 1, 0, 'b' },
            { "threshold-factor", 1, 0, 'T' },
            { "read-samples", 1, 0, 't' },
            { "use-alternate-streaming-method",   0, 0, 'a' },
            { NULL, 0, 0, 0 },
        };
        int c;

        if(argc == 1)
        {
            print_usage(argv[0]);
            return;
        }

        c = getopt_long(argc, argv, "S:V:l:D:P:B:U:n:i:c:E:s:f:r:p:L:d:g:w::A:Z:N:b:T:m:t:ICMvGxyWRaHoe", lopts, NULL);

        if (c == -1)
            break;

        switch (c) {
            case 'I':
                init_dlpc350();
                break;
            case 'C':
                perform_calibration = true;
                break;
            case 'M':
                perform_flash_loadtime_measurement = true;
                break;
            case 'i':
                first_img_idx = atoi(optarg);
                break;
            case 'c':
                ptn_count = atoi(optarg);
                perform_scan = true;
                break;
            case 'V':
                strcpy(sdf_filename, optarg);
                perform_streaming = true;
                break;
            case 'v':
                DEBUG_ERR("DLP NIRscan Application version %d.%d.%d\n", DLPNIRSCAN_VERSION_MAJOR, DLPNIRSCAN_VERSION_MINOR, DLPNIRSCAN_VERSION_BUILD);
                break;
            case 'E':
                ptn_exposure_time_us = atoi(optarg);
                break;
            case 'B':
            case 'U':
                strcpy(firmware_file, optarg);
                break;
            case 'S':
                ptn_count = atoi(optarg);
                perform_streaming = true;
                break;
            case 'l':
                num_images = atoi(optarg);
                perform_load_for_streaming = true;
                break;
            case 'n':
                num_images = atoi(optarg);
                break;
            case 's':
                spi_clk_speed = atoi(optarg);
                break;
            case 'g':
                set_pga_gain = true;
                pga_gain = atoi(optarg);
                break;
            case 'R':
                restore_calinfo = true;
                break;
            case 'W':
                write_calinfo = true;
                break;
            case 'w':
                compute_coeffs = true;
                if(optarg != NULL)
                    strcpy(peaks_filename, optarg);
                break;
            case 'x':
                compute_pixel = true;
                break;
            case 'y':
                compute_wavelength = true;
                break;
            case 'D':
                drate_reg = ads1255_convert_drate_to_regval(atoi(optarg));
                break;
            case 'P':
                strcpy(sdf_filename, optarg);
                prepare_scan_solution = true;
                break;
            case 'f':
                strcpy(&adc_reading_filename[0],optarg);
                DEBUG_MSG("Input file name: %s\n",adc_reading_filename);
                break;
            case 'r':
                strcpy(&ref_adc_reading_filename[0],optarg);
                DEBUG_MSG("Reference file name: %s\n",ref_adc_reading_filename);
                break;
            case 'p':
                strcpy(&absorption_filename[0],optarg);
                find_peaks = true;
                break;
            case 'G':
                compute_absorption = true;
                break;
            case 'L':
                scanLoop = atoi(optarg);
                break;
            case 'd':
                itrDelay = atol(optarg);
                break;
            case 'A':
                first_pixel = atol(optarg);
                break;
            case 'Z':
                last_pixel = atol(optarg);
                break;
            case 'N':
                num_patterns = atol(optarg);
                generate_csv_patterns = true;
                break;
            case 'b':
                csv_begin_number = atol(optarg);
                break;
            case 'T':
                threshold_factor = strtof(optarg, NULL);
                if(threshold_factor < 0.001 || threshold_factor >= 1)
                {
                    DEBUG_ERR("threshold_factor must be in the range 0.001 to 1. Default of .001 will be used in this case\n");
                    threshold_factor = 0.001;
                }
                break;
            case 'a':
                use_alternate_streaming_method = false;
                break;
            case 'H':
                use_hadamard_patterns = true;
                break;
            case 'm': 
                strcpy(&ref_adc_reading_filename[0],optarg);
                calculate_hadamard_avg_readings = true;
                break;
            case 't':
                read_samples = true;
                num_read_samples = atol(optarg);
                break;
            default:
                print_usage(argv[0]);
                break;
        }
    }
}

int main(int argc, char *argv[])
{
	int fd;
	char *chPtr;
	char readings_n_filename[256];
    char absorp_filename[256];
    char peak_string[256];
	char itoascii[8];
    //char eeprom_header[] = {0xAA,0x55,0x33,0xEE,'A','3','3','5','B','N','L','T'}; 
	int ret = 0;
    FILE *readings_fd;
    FILE *ref_readings_fd;
    FILE *raw_output_fd;
    FILE *absorp_spect_file;
    FILE *peaks_file;
    FILE *lambda_coeffs_file;
    FILE *pixel_coeffs_file;
    FILE *pixels_file;
    FILE *wavelengths_file;
    FILE *sdf_fp=NULL;
    FILE *eeprom_fp;
    FILE *control_points_fp;
    FILE *avg_readings_fp;
    long long *avg_readings;
    int i;
    int j;
    int num_peaks;
    unsigned int timingData;
    unsigned int max_loadtime=0;
    int index_at_max=0;
    unsigned char HWStatus, SysStatus, MainStatus;
    unsigned scan_img_count;
    float peaks[NUM_CAL_PEAKS];
    float peak_inds[NUM_CAL_PEAKS];
    double peak_pixels[NUM_CAL_PEAKS];
    double peak_lambdas[NUM_CAL_PEAKS] ={1446.6, 1693.1,2060.1, 2111, 2248.2, 2373.4};
    double coeffs[3];
	char *endPtr; //used in strtod function
    long double pixel;
    long double wavelength;
    float control_points[6*3*4];
    float fl_coeffs[3];
    char line[8192];
    char *pToken;
    float divisor=0.0;
    int ret_code;
    int32_t *p_readings;
    int num_raw_readings;
    int integration_time;

    ret = USB_Init();

    if(USB_IsConnected())
        USB_Close();

    if(USB_Open() != 0)
        DEBUG_ERR("USB connection to DLPC350 failed\n");
    else if (USB_IsConnected())
        DEBUG_MSG("USB connection to DLPC350 now open\n");

#ifndef HOST_MODE
    if( 0 != DISP_Init())
        return -1;
#endif

	parse_opts(argc, argv);

#ifndef HOST_MODE
	fd = open(device, O_RDWR);
	if (fd < 0)
		pabort("can't open device");
	/*
	 * spi mode
	 */
	ret = ioctl(fd, SPI_IOC_WR_MODE, &mode);
	if (ret == -1)
		pabort("can't set spi mode");

	ret = ioctl(fd, SPI_IOC_RD_MODE, &mode);
	if (ret == -1)
		pabort("can't get spi mode");

	/*
	 * bits per word
	 */
	ret = ioctl(fd, SPI_IOC_WR_BITS_PER_WORD, &bits);
	if (ret == -1)
		pabort("can't set bits per word");


	ret = ioctl(fd, SPI_IOC_RD_BITS_PER_WORD, &bits);
	if (ret == -1)
		pabort("can't get bits per word");

	/*
	 * max speed hz
	 */
	ret = ioctl(fd, SPI_IOC_WR_MAX_SPEED_HZ, &spi_clk_speed);
	if (ret == -1)
		pabort("can't set max speed hz");

	ret = ioctl(fd, SPI_IOC_RD_MAX_SPEED_HZ, &spi_clk_speed);
	if (ret == -1)
		pabort("can't get max speed hz");

	DEBUG_MSG("spi mode: %d\n", mode);
	DEBUG_MSG("bits per word: %d\n", bits);
	DEBUG_MSG("max speed: %d Hz (%d KHz)\n", spi_clk_speed, spi_clk_speed/1000);

	if(drate_reg != 0)
	{
		if(ads1255_set_register(fd, ADS1255_DRATE, drate_reg) < 0)
		{
			close(fd);
			return ret;
		}
	}

    if(set_pga_gain)
        if(ads1255_set_PGA_gain(fd, pga_gain) < 0)
        {
            DEBUG_ERR("Unable to set PGA multiplier\n");
            return -1;
        }
#endif

    if(num_images != 0)
    {
        if(firmware_file[0] != 0) //-B filename -n num_images option has been used for building firmware
        {
            ret = dlpspectro_add_images_to_firmware(firmware_file, num_images);
            if(ret < 0)
                DEBUG_ERR("Building firmware image failed\n");
        }
        if(perform_load_for_streaming)
        {
            if(dlpspectro_load_images_for_streaming(num_images, first_img_idx) < 0)
            {
                DEBUG_ERR("loading images for streaming failed\n");
                ret = -1;
            }
        }
    }
    else if(firmware_file[0] != 0) //-U firmware_file option has been used for updating the firmware
            dlpspectro_update_firmware(firmware_file);

    if(calculate_hadamard_avg_readings)
    {
        if(strcmp(adc_reading_filename," ") == 0)
        {
            strcpy(adc_reading_filename, "avg_readings_hadamard.txt");
        }
        readings_fd = fopen(adc_reading_filename, "w+");
        if(readings_fd == NULL)
        {
            DEBUG_ERR("Unable to open %s\n", adc_reading_filename);
            return -1;
        }
        ref_readings_fd = fopen(ref_adc_reading_filename, "r");
        if(ref_readings_fd == NULL)
        {
            DEBUG_ERR("Unable to open %s\n", ref_adc_reading_filename);
            fclose(readings_fd);
            return -1;
        }
        DEBUG_ERR("Output recoreded in %s\n", adc_reading_filename);
        return(dlpspectro_compute_hadamard_readings(ref_readings_fd, readings_fd));
    }

    if(prepare_scan_solution)
    {
        sdf_fp = fopen(sdf_filename, "r");
        if (sdf_fp == NULL)
        {
            DEBUG_ERR("can't open scan description file\n");
            return -1;
        }
        ret = dlpspectro_prep_scan_solution(sdf_fp, first_img_idx);
        if( 0 > ret )
        {
            DEBUG_ERR("Error preparing scan solution from SDF\n");
            return ret;
        }
        fclose(sdf_fp);
    }

    if(generate_csv_patterns)
    {
        if(ptn_exposure_time_us == 0)
            integration_time = INTEGRATION_TIME;
        else
            integration_time = ptn_exposure_time_us;

        if(strcmp(adc_reading_filename," ") == 0)
        {
            strcpy(adc_reading_filename, "patterns/scan.sdf");
        }
        sdf_fp=fopen(adc_reading_filename,"w+");
        if(sdf_fp == NULL)
        {
            DEBUG_ERR("Unable to open %s\n", adc_reading_filename);
            return -1;
        }

        ret = dlpspectro_generate_csv_sdf(first_pixel, last_pixel, num_patterns, csv_begin_number, integration_time, sdf_fp, use_hadamard_patterns, true);
        fclose(sdf_fp);
        if(ret < 0)
            return ret;
    }

    if(compute_coeffs)
    {
        peaks_file = fopen(peaks_filename, "r");
        if(peaks_file == NULL)
        {
            DEBUG_ERR("Unable to open %s\n", peaks_filename);
            return -1;
        }
        i=0;
        while((fscanf(peaks_file,"%s",peak_string) != EOF) && (i<NUM_CAL_PEAKS))
        {
            peak_pixels[i++] = strtod(peak_string ,&endPtr);
        }

        dlpspectro_polyfit(peak_pixels, peak_lambdas, coeffs);
        lambda_coeffs_file = fopen("lambda_coeffs.txt", "w+");
        if(lambda_coeffs_file == NULL)
        {
            DEBUG_ERR("Unable to open lambda_coeffs.txt\n");
            return -1;
        }
        for(i=0; i<POLY_DEGREE; i++)
        {
            fprintf(lambda_coeffs_file, "%.64f\n", coeffs[i]);
        }
        fclose(lambda_coeffs_file);
        dlpspectro_polyfit(peak_lambdas, peak_pixels, coeffs);
        pixel_coeffs_file = fopen("pixel_coeffs.txt", "w+");
        if(pixel_coeffs_file == NULL)
        {
            DEBUG_ERR("Unable to open pixel_coeffs.txt\n");
            return -1;
        }
        for(i=0; i<POLY_DEGREE; i++)
        {
            fprintf(pixel_coeffs_file, "%.64f\n", coeffs[i]);
        }
        fclose(pixel_coeffs_file);
    }

    if(compute_pixel)
    {
        pixel_coeffs_file = fopen("pixel_coeffs.txt", "r");
        if(pixel_coeffs_file == NULL)
        {
            DEBUG_ERR("Unable to open pixel_coeffs.txt\n");
            return -1;
        }
        for(i=0; i<POLY_DEGREE; i++)
        {
            fscanf(pixel_coeffs_file,"%s",peak_string);
            coeffs[i] = strtod(peak_string ,&endPtr);
        }
        fclose(pixel_coeffs_file);
        wavelengths_file = fopen("wavelengths.txt", "r");
        if(wavelengths_file == NULL)
        {
            DEBUG_ERR("Unable to open wavelengths.txt\n");
            return -1;
        }
        pixels_file = fopen("pixels.txt", "w+");
        if(pixels_file == NULL)
        {
            DEBUG_ERR("Unable to open pixels.txt\n");
            return -1;
        }
        while(fscanf(wavelengths_file, "%s", peak_string) != EOF)
        {
            wavelength = strtod(peak_string, &endPtr);
            pixel = wavelength*wavelength*coeffs[2] + wavelength*coeffs[1] + coeffs[0];
            fprintf(pixels_file,  "%Lf\n", pixel);
        }
        fclose(wavelengths_file);
        fclose(pixels_file);
    }

    if(compute_wavelength)
    {
        lambda_coeffs_file = fopen("lambda_coeffs.txt", "r");
        if(lambda_coeffs_file == NULL)
        {
            DEBUG_ERR("Unable to open lambda_coeffs.txt\n");
            return -1;
        }
        for(i=0; i<POLY_DEGREE; i++)
        {
            fscanf(lambda_coeffs_file,"%s",peak_string);
            coeffs[i] = strtod(peak_string ,&endPtr);
        }
        fclose(lambda_coeffs_file);
        pixels_file = fopen("pixels.txt", "r");
        if(pixels_file == NULL)
        {
            DEBUG_ERR("Unable to open pixels.txt\n");
            return -1;
        }
        wavelengths_file = fopen("wavelengths.txt", "w+");
        if(wavelengths_file == NULL)
        {
            DEBUG_ERR("Unable to open wavelengths.txt\n");
            return -1;
        }
        while(fscanf(pixels_file, "%s", peak_string) != EOF)
        {
            pixel = strtod(peak_string, &endPtr);
            wavelength = pixel*pixel*coeffs[2] + pixel*coeffs[1] + coeffs[0];
            fprintf(wavelengths_file,  "%Lf\n", wavelength);
        }
        fclose(wavelengths_file);
        fclose(pixels_file);
    }

    if(compute_absorption)
    {
        //If there are no sample files for processing
        if(strcmp(adc_reading_filename," ") == 0)
        {
            DEBUG_ERR("sample adc readings file not specified\n");
            print_usage(argv[0]);
            return -1;
        }
        if(strcmp(ref_adc_reading_filename," ") == 0)
        {
            DEBUG_ERR("reference adc readings file not specified\n");
            print_usage(argv[0]);
            return -1;
        }
        readings_fd = fopen(adc_reading_filename, "r");
        if(readings_fd == NULL)
        {
            DEBUG_ERR("Unable to open %s\n", adc_reading_filename);
            return -1;
        }
        ref_readings_fd = fopen(ref_adc_reading_filename, "r");
        if(ref_readings_fd == NULL)
        {
            DEBUG_ERR("Unable to open %s\n", ref_adc_reading_filename);
            fclose(readings_fd);
            return -1;
        }

        chPtr = (char*) memchr(adc_reading_filename, '.',strlen(adc_reading_filename));
        strncpy(absorp_filename,adc_reading_filename,(chPtr-adc_reading_filename));
        absorp_filename[(chPtr-adc_reading_filename)] = '\0';
        strcat(absorp_filename,"_absp.txt");

        absorp_spect_file = fopen(absorp_filename,"w+");
        if(absorp_spect_file == NULL)
        {
            DEBUG_ERR("ERROR: Couldn't create \"%s\" \n",absorp_filename);
            fclose(ref_readings_fd);
            fclose(readings_fd);
            return -1;
        }

        if((ret_code = dlpspectro_compute_absorption_spectrum(ref_readings_fd, readings_fd, absorp_spect_file, threshold_factor)) != 0)
        {
            if(ret_code > 0)
                DEBUG_ERR("WARNING: Some values discarded during generate_absorption_spectrum()\n");
            else
            {
                DEBUG_ERR("ERROR: Inside generate_absorption_spectrum()\n");
                fclose(absorp_spect_file);
                fclose(ref_readings_fd);
                fclose(readings_fd);
                return -1;
            }
        }
        fclose(absorp_spect_file);
        fclose(ref_readings_fd);
        fclose(readings_fd);
    }

    if(find_peaks)
    {
        absorp_spect_file = fopen(absorption_filename, "r");
        if(absorp_spect_file == NULL)
        {
            DEBUG_ERR("Unable to open %s\n", absorption_filename);
            return -1;
        }
        for(i=1; i<=30; i++)
        {
            num_peaks = dlpspectro_find_peaks(absorp_spect_file, i, peaks, peak_inds);
            rewind(absorp_spect_file);
            if(num_peaks > 6)
            {
                for(divisor=i-1; divisor<i; divisor+=0.1)
                {
                    num_peaks = dlpspectro_find_peaks(absorp_spect_file, divisor, peaks, peak_inds);
                    rewind(absorp_spect_file);
                    if(num_peaks == 6)
                        break;
                }
            }
            if(num_peaks == 6)
            {
                if(divisor == 0.0)
                    DEBUG_MSG("6 peaks found at (max-min)/divisor when divisor was %d\n", i);
                else
                    DEBUG_MSG("6 peaks found at (max-min)/divisor when divisor was %f\n", divisor);
                break;
            }
        }
        fclose(absorp_spect_file);
        peaks_file = fopen("peak_locations.txt", "w+");
        if(peaks_file == NULL)
        {
            DEBUG_ERR("Unable to open peak_locations.txt\n");
            return -1;
        }
        for(i=0; i<num_peaks; i++)
        {
            fprintf(peaks_file, "%f\n", peak_inds[i]);
        }
        DEBUG_ERR("Peak locations stored in peak_locations.txt\n");
        fclose(peaks_file);
    }


    if(perform_calibration)
        dlpspectro_setup_calib_scan_prgb(fd);

    if(read_samples)
    {
#if 0
        raw_output_fd = fopen("readings_raw.txt", "w+");
        if(raw_output_fd == NULL)
        {
            DEBUG_ERR("Unable to open %s\n", readings_n_filename);
            return -1;
        }
        dlpspectro_test_read_adc_samples(fd, num_read_samples, &p_readings);
        for(j=0; j<num_read_samples; j++)
        {
            fprintf(raw_output_fd, "0x%.8X\n", p_readings[j]);
        }
        fclose(raw_output_fd);
        DEBUG_MSG("Readings captured in readings_raw.txt\n");
#else
    ads1255_write_command(fd, ADS1255_SDATAC);
#endif
    }

    if(write_calinfo == true)
    {
        eeprom_fp = fopen(EEPROM_FILE, "wb");
        if(eeprom_fp == NULL)
        {
            DEBUG_ERR("Unable to open EEPROM_FILE\n");
            return -1;
        }
        fseek(eeprom_fp, CALINFO_COEFFS_OFFSET, SEEK_SET);
        pixel_coeffs_file = fopen("pixel_coeffs.txt", "r");
        if(pixel_coeffs_file == NULL)
        {
            DEBUG_ERR("Unable to open pixel_coeffs.txt\n");
            return -1;
        }
        for(i=0; i<POLY_DEGREE; i++)
        {
            fscanf(pixel_coeffs_file,"%s",peak_string);
            fl_coeffs[i] = strtof(peak_string ,&endPtr);
        }
        fwrite(fl_coeffs, sizeof(float), POLY_DEGREE, eeprom_fp);
        fclose(pixel_coeffs_file);
        lambda_coeffs_file = fopen("lambda_coeffs.txt", "r");
        if(lambda_coeffs_file == NULL)
        {
            DEBUG_ERR("Unable to open lambda_coeffs.txt\n");
            return -1;
        }
        for(i=0; i<POLY_DEGREE; i++)
        {
            fscanf(lambda_coeffs_file,"%s",peak_string);
            fl_coeffs[i] = strtof(peak_string ,&endPtr);
        }
        fwrite(fl_coeffs, sizeof(float), POLY_DEGREE, eeprom_fp);
        fclose(lambda_coeffs_file);

#if 1
        control_points_fp = fopen("control_points.txt", "r");
        if(control_points_fp == NULL)
        {
            DEBUG_ERR("Unable to open control_points.txt\n");
            return -1;
        }
        fgets(line, 8192, control_points_fp); // Read one line from SDF file

        i=0;
        strtok(line, ",");	//Get first value
        control_points[i++] = strtof(line, &endPtr);

        while((pToken = strtok(NULL, " ")))
        {
            control_points[i++] = strtof(pToken, &endPtr);
            if((pToken = strtok(NULL, ",")) == NULL)  //Get x
                break;
            control_points[i++] = strtof(pToken, &endPtr);
        }
        fseek(eeprom_fp, CALINFO_CPS_OFFSET, SEEK_SET);
        for(i=0; i<72; i+=8)
        {
            fwrite(&control_points[i], sizeof(float), 8, eeprom_fp);
            fwrite(&control_points[i], sizeof(float), 8, eeprom_fp); //dummy write to account for weird behavor of eeprom/driver
        }

        fclose(control_points_fp);
#endif
        fclose(eeprom_fp);
        DEBUG_ERR("pixel_coeffs.txt, lambda_coeffs.txt and control_points.txt files saved in eeprom\n");
    }

    if(restore_calinfo == true)
    {
        eeprom_fp = fopen(EEPROM_FILE, "rb");
        if(eeprom_fp == NULL)
        {
            DEBUG_ERR("Unable to open EEPROM_FILE\n");
            return -1;
        }

        fseek(eeprom_fp, CALINFO_COEFFS_OFFSET, SEEK_SET);

        pixel_coeffs_file = fopen("pixel_coeffs.txt", "w+");
        if(pixel_coeffs_file == NULL)
        {
            DEBUG_ERR("Unable to open pixel_coeffs.txt\n");
            return -1;
        }
        fread(fl_coeffs, sizeof(float), POLY_DEGREE, eeprom_fp);
        for(i=0; i<POLY_DEGREE; i++)
        {
            fprintf(pixel_coeffs_file, "%.64f\n", fl_coeffs[i]);
        }
        fclose(pixel_coeffs_file);

        lambda_coeffs_file = fopen("lambda_coeffs.txt", "w+");
        if(lambda_coeffs_file == NULL)
        {
            DEBUG_ERR("Unable to open lambda_coeffs.txt\n");
            return -1;
        }
        fread(fl_coeffs, sizeof(float), POLY_DEGREE, eeprom_fp);
        for(i=0; i<POLY_DEGREE; i++)
        {
            fprintf(lambda_coeffs_file, "%.64f\n", fl_coeffs[i]);
        }
        fclose(lambda_coeffs_file);

#if 1
        control_points_fp = fopen("control_points.txt", "w+");
        if(control_points_fp == NULL)
        {
            DEBUG_ERR("Unable to open control_points.txt\n");
            return -1;
        }
        fseek(eeprom_fp, CALINFO_CPS_OFFSET, SEEK_SET);
        for(i=0; i<72; i+=8)
        {
            fread(&control_points[i], sizeof(float), 8, eeprom_fp);
            fseek(eeprom_fp, 8*sizeof(float), SEEK_CUR); //skip 8 bytes to reciprocate dummy write behavior
        }
        for(i=0; i<6*4*3; )
        {
            fprintf(control_points_fp, "%f,", control_points[i++]);
            fprintf(control_points_fp, "%f ", control_points[i++]);
        }
        fprintf(control_points_fp, "\n");
        fclose(control_points_fp);
#endif

        fclose(eeprom_fp);
        DEBUG_ERR("pixel_coeffs.txt, lambda_coeffs.txt and control_points.txt files restored from eeprom\n");
    }
#ifndef HOST_MODE

    if( ((perform_scan == true) && (perform_flash_loadtime_measurement == false)) || (perform_streaming == true))
    {
        if(perform_streaming == true)
        {
            if((ptn_count < 0) || ptn_count > FB_NUM_BUFFERS*24)
            {
                DEBUG_ERR("max number of patterns supported in streaming %d\n", FB_NUM_BUFFERS*24);
                return -1;
            }
            else if(ptn_count == 0)
            {
                sdf_fp = fopen(sdf_filename, "r");
                if (sdf_fp == NULL)
                {
                    DEBUG_ERR("can't open scan description file\n");
                    return -1;
                }
            }

        }
        for(i=0; i<scanLoop; i++)
        {
            DEBUG_ERR("\n\n********DLP NIRSCAN BEGIN [%03d]*******************\n\n",i);

            /* Save the received data in file */
            if(strcmp(adc_reading_filename," ") == 0)
                strcpy(adc_reading_filename, "readings.txt");

            chPtr = (char*) memchr(adc_reading_filename, '.',strlen(adc_reading_filename));
            strncpy(readings_n_filename,adc_reading_filename,(chPtr-adc_reading_filename));
            readings_n_filename[(chPtr-adc_reading_filename)] = '\0';
            strcat(readings_n_filename,"_");
            sprintf(itoascii,"%02d",i);
            strcat(readings_n_filename,itoascii);
            strcat(readings_n_filename,"_raw.txt");

            raw_output_fd = fopen(readings_n_filename, "w+");
            if(raw_output_fd == NULL)
            {
                DEBUG_ERR("Unable to open %s\n", readings_n_filename);
                return -1;
            }

            chPtr = (char*) memchr(adc_reading_filename, '.',strlen(adc_reading_filename));
            strncpy(readings_n_filename,adc_reading_filename,(chPtr-adc_reading_filename));
            readings_n_filename[(chPtr-adc_reading_filename)] = '\0';
            strcat(readings_n_filename,"_");
            sprintf(itoascii,"%02d",i);
            strcat(readings_n_filename,itoascii);
            strcat(readings_n_filename,".txt");

            if(perform_scan)
            {
                if(ptn_exposure_time_us == 0)
                    ptn_exposure_time_us = 10000;
                num_raw_readings = dlpspectro_scan_flash_images(fd, first_img_idx, ptn_count, ptn_exposure_time_us, &p_readings, NULL);
            }
            if(perform_streaming)
            {
                if(ptn_exposure_time_us == 0)
                    ptn_exposure_time_us = 600;
                num_raw_readings = dlpspectro_stream_images(fd, &ptn_count, ptn_exposure_time_us, &p_readings, sdf_fp);
            }

            if(sdf_fp != NULL)
                rewind(sdf_fp);

            for(j=0; j<num_raw_readings; j++)
            {
                fprintf(raw_output_fd, "0x%.8X\n", p_readings[j]);
            }
            fclose(raw_output_fd);

            dlpspectro_compute_mean_readings(p_readings, num_raw_readings,  ptn_count);

            readings_fd = fopen(readings_n_filename, "w+");
            if(readings_fd == NULL)
            {
                DEBUG_ERR("Unable to open %s\n", readings_n_filename);
                return -1;
            }

            if(i==0)
            {
                avg_readings = (long long *)calloc(ptn_count,sizeof(long long));
                if(avg_readings == NULL)
                {
                    DEBUG_ERR("Memory allocation for averaging failed\n");
                    return -1;
                }
            }

            for(j=0; j<ptn_count; j++)
            {
                avg_readings[j] += p_readings[j];
                fprintf(readings_fd, "0x%.8X\n", p_readings[j]);
            }
            fclose(readings_fd);
            DEBUG_ERR("Values read from ADC captured in %s\n", readings_n_filename);

            DEBUG_ERR("\n\n********DLP NIRSCAN END*******************\n\n");

            if(itrDelay)
            {
                //Sleep before the next scan
                usleep((itrDelay*1000));
            }

        }

        avg_readings_fp = fopen("avg_readings.txt", "w+");
        if(avg_readings_fp == NULL)
        {
            DEBUG_ERR("Unable to open avg_readings.txt\n");
            return -1;
        }
        for (i = 0; i < ptn_count; i++) 
            fprintf(avg_readings_fp, "0x%.8X\n", avg_readings[i]/scanLoop);

        free(avg_readings);
        fclose(avg_readings_fp);
        if(sdf_fp != NULL)
            fclose(sdf_fp);
    }

    if(perform_flash_loadtime_measurement)
    {
        scan_img_count = (ptn_count+23)/24;

        for(i=0; i < scan_img_count; i++)
        {
            if(DLPC350_MeasureSplashLoadTiming(first_img_idx+i, 1) >= 0)
            {
                DLPC350_GetStatus(&HWStatus, &SysStatus, &MainStatus);
                if(DLPC350_ReadSplashLoadTiming(&timingData) >=0)
                {
                    timingData /= 18667;
                    if(timingData > max_loadtime)
                    {
                        max_loadtime = timingData;
                        index_at_max = first_img_idx+i;
                    }
                }
                else
                    DEBUG_ERR("Unable to read back flash load timing\n");
            }
            else
                DEBUG_ERR("Unable to instruct DLPC350 to measure flash load timing\n");
        }
        DEBUG_ERR("Longest flash load time is %d milliseconds for image at index %d\n", max_loadtime, index_at_max);
    }
	close(fd);
#endif

    USB_Close();
    DISP_Close(0);
	return ret;
}
