/*
 * dlpc350_api.h
 *
 * This module provides C callable APIs for each of the command supported by the DLPC350 controller and detailed in the programmer's guide.
 *
 * Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
 * ALL RIGHTS RESERVED
 *
*/

#ifndef API_H
#define API_H

#include <stdbool.h>

/* Bit masks. */
#define BIT0        0x01
#define BIT1        0x02
#define BIT2        0x04
#define BIT3        0x08
#define BIT4        0x10
#define BIT5        0x20
#define BIT6        0x40
#define BIT7        0x80
#define BIT8      0x0100
#define BIT9      0x0200
#define BIT10     0x0400
#define BIT11     0x0800
#define BIT12     0x1000
#define BIT13     0x2000
#define BIT14     0x4000
#define BIT15     0x8000
#define BIT16 0x00010000
#define BIT17 0x00020000
#define BIT18 0x00040000
#define BIT19 0x00080000
#define BIT20 0x00100000
#define BIT21 0x00200000
#define BIT22 0x00400000
#define BIT23 0x00800000
#define BIT24 0x01000000
#define BIT25 0x02000000
#define BIT26 0x04000000
#define BIT27 0x08000000
#define BIT28 0x10000000
#define BIT29 0x20000000
#define BIT30 0x40000000
#define BIT31 0x80000000

#define STAT_BIT_FLASH_BUSY     BIT3
#define HID_MESSAGE_MAX_SIZE    512
#define MAX_PAT_LUT_ENTRIES 1824

typedef struct _hidmessageStruct
{
    struct _hidhead
    {
        struct _packetcontrolStruct
        {
            unsigned char dest		:3; /* 0 - ProjCtrl; 1 - RFC; 7 - Debugmsg */
            unsigned char reserved	:2;
            unsigned char nack		:1; /* Command Handler Error */
            unsigned char reply	:1; /* Host wants a reply from device */
            unsigned char rw		:1; /* Write = 0; Read = 1 */
        }flags;
        unsigned char seq;
        unsigned short length;
    }head;
    union
    {
        unsigned short cmd;
        unsigned char data[HID_MESSAGE_MAX_SIZE];
    }text;
}hidMessageStruct;

typedef struct _readCmdData
{
    unsigned char CMD2;
    unsigned char CMD3;
    unsigned short len;
}CmdFormat;

typedef struct _rectangle
{
    unsigned short firstPixel;
    unsigned short firstLine;
    unsigned short pixelsPerLine;
    unsigned short linesPerFrame;
}rectangle;

typedef enum
{
    SOURCE_SEL,
    PIXEL_FORMAT,
    CLK_SEL,
    CHANNEL_SWAP,
    FPD_MODE,
    CURTAIN_COLOR,
    POWER_CONTROL,
    FLIP_LONG,
    FLIP_SHORT,
    TPG_SEL,
    PWM_INVERT,
    LED_ENABLE,
    GET_VERSION,
    SW_RESET,
    DMD_PARK,
    BUFFER_FREEZE,
    STATUS_HW,
    STATUS_SYS,
    STATUS_MAIN,
    CSC_DATA,
    GAMMA_CTL,
    BC_CTL,
    PWM_ENABLE,
    PWM_SETUP,
    PWM_CAPTURE_CONFIG,
    GPIO_CONFIG,
    LED_CURRENT,
    DISP_CONFIG,
    TEMP_CONFIG,
    TEMP_READ,
    MEM_CONTROL,
    I2C_CONTROL,
    LUT_VALID,
    DISP_MODE,
    TRIG_OUT1_CTL,
    TRIG_OUT2_CTL,
    RED_STROBE_DLY,
    GRN_STROBE_DLY,
    BLU_STROBE_DLY,
    PAT_DISP_MODE,
    PAT_TRIG_MODE,
    PAT_START_STOP,
    BUFFER_SWAP,
    BUFFER_WR_DISABLE,
    CURRENT_RD_BUFFER,
    PAT_EXPO_PRD,
    INVERT_DATA,
    PAT_CONFIG,
    MBOX_ADDRESS,
    MBOX_CONTROL,
    MBOX_DATA,
    TRIG_IN1_DELAY,
    TRIG_IN2_CONTROL,
    SPLASH_LOAD,
    SPLASH_LOAD_TIMING,
    MBOX_EXP_DATA,
    MBOX_EXP_ADDRESS,
    EXP_PAT_CONFIG,
    GPCLK_CONFIG,
    PULSE_GPIO_23,
    ENABLE_DLPC350_DEBUG,
    TPG_COLOR,
    PWM_CAPTURE_READ,
    PROG_MODE,
    BL_STATUS,
    BL_SPL_MODE,
    BL_GET_MANID,
    BL_GET_DEVID,
    BL_GET_CHKSUM,
    BL_SET_SECTADDR,
    BL_SECT_ERASE,
    BL_SET_DNLDSIZE,
    BL_DNLD_DATA,
    BL_FLASH_TYPE,
    BL_CALC_CHKSUM,
    BL_PROG_MODE,
}DLPC350_CMD;

int DLPC350_SetInputSource(unsigned int source, unsigned int portWidth);
int DLPC350_GetInputSource(unsigned int *pSource, unsigned int *portWidth);
int DLPC350_SetPixelFormat(unsigned int format);
int DLPC350_GetPixelFormat(unsigned int *pFormat);
int DLPC350_SetPortClock(unsigned int clock);
int DLPC350_GetPortClock(unsigned int *pClock);
int DLPC350_SetDataChannelSwap(unsigned int port, unsigned int swap);
int DLPC350_GetDataChannelSwap(unsigned int *pPort, unsigned int *pSwap);
int DLPC350_SetFPD_Mode_Field(unsigned int PixelMappingMode, bool SwapPolarity, unsigned int FieldSignalSelect);
int DLPC350_GetFPD_Mode_Field(unsigned int *pPixelMappingMode, bool *pSwapPolarity, unsigned int *pFieldSignalSelect);
int DLPC350_SetPowerMode(bool);
int DLPC350_SetLongAxisImageFlip(bool);
bool DLPC350_GetLongAxisImageFlip();
int DLPC350_SetShortAxisImageFlip(bool);
bool DLPC350_GetShortAxisImageFlip();
int DLPC350_SetTPGSelect(unsigned int pattern);
int DLPC350_GetTPGSelect(unsigned int *pPattern);
int DLPC350_SetLEDPWMInvert(bool invert);
int DLPC350_GetLEDPWMInvert(bool *inverted);
int DLPC350_SetLedEnables(bool SeqCtrl, bool Red, bool Green, bool Blue);
int DLPC350_GetLedEnables(bool *pSeqCtrl, bool *pRed, bool *pGreen, bool *pBlue);
int DLPC350_GetVersion(unsigned int *pApp_ver, unsigned int *pAPI_ver, unsigned int *pSWConfig_ver, unsigned int *pSeqConfig_ver);
int DLPC350_GetFirmwareVersion(unsigned int *pFW_ver);
int DLPC350_SoftwareReset(void);
int DLPC350_GetStatus(unsigned char *pHWStatus, unsigned char *pSysStatus, unsigned char *pMainStatus);
int DLPC350_SetPWMEnable(unsigned int channel, bool Enable);
int DLPC350_GetPWMEnable(unsigned int channel, bool *pEnable);
int DLPC350_SetPWMConfig(unsigned int channel, unsigned int pulsePeriod, unsigned int dutyCycle);
int DLPC350_GetPWMConfig(unsigned int channel, unsigned int *pPulsePeriod, unsigned int *pDutyCycle);
int DLPC350_SetPWMCaptureConfig(unsigned int channel, bool enable, unsigned int sampleRate);
int DLPC350_GetPWMCaptureConfig(unsigned int channel, bool *pEnabled, unsigned int *pSampleRate);
int DLPC350_SetGPIOConfig(unsigned int pinNum, bool enAltFunc, bool altFunc1, bool dirOutput, bool outTypeOpenDrain, bool pinState);
int DLPC350_GetGPIOConfig(unsigned int pinNum, bool *pEnAltFunc, bool *pAltFunc1, bool *pDirOutput, bool *pOutTypeOpenDrain, bool *pState);
int DLPC350_GetLedCurrents(unsigned char *pRed, unsigned char *pGreen, unsigned char *pBlue);
int DLPC350_SetLedCurrents(unsigned char RedCurrent, unsigned char GreenCurrent, unsigned char BlueCurrent);
int DLPC350_SetDisplay(rectangle croppedArea, rectangle displayArea);
int DLPC350_GetDisplay(rectangle *pCroppedArea, rectangle *pDisplayArea);
int DLPC350_MemRead(unsigned int addr, unsigned int *readWord);
int DLPC350_MemWrite(unsigned int addr, unsigned int data);
int DLPC350_ValidatePatLutData(unsigned int *pStatus);
int DLPC350_SetPatternDisplayMode(bool external);
int DLPC350_GetPatternDisplayMode(bool *external);
int DLPC350_SetTrigOutConfig(unsigned int trigOutNum, bool invert, unsigned int rising, unsigned int falling);
int DLPC350_GetTrigOutConfig(unsigned int trigOutNum, bool *pInvert,unsigned int *pRising, unsigned int *pFalling);
int DLPC350_SetRedLEDStrobeDelay(unsigned char rising, unsigned char falling);
int DLPC350_SetGreenLEDStrobeDelay(unsigned char rising, unsigned char falling);
int DLPC350_SetBlueLEDStrobeDelay(unsigned char rising, unsigned char falling);
int DLPC350_GetRedLEDStrobeDelay(unsigned char *, unsigned char *);
int DLPC350_GetGreenLEDStrobeDelay(unsigned char *, unsigned char *);
int DLPC350_GetBlueLEDStrobeDelay(unsigned char *, unsigned char *);
int DLPC350_EnterProgrammingMode(void);
int DLPC350_ExitProgrammingMode(void);
int DLPC350_GetProgrammingMode(bool *ProgMode);
int DLPC350_GetFlashManID(unsigned short *manID);
int DLPC350_GetFlashDevID(unsigned long long *devID);
int DLPC350_GetBLStatus(unsigned char *BL_Status);
int DLPC350_SetFlashAddr(unsigned int Addr);
int DLPC350_FlashSectorErase(void);
int DLPC350_SetDownloadSize(unsigned int dataLen);
int DLPC350_DownloadData(unsigned char *pByteArray, unsigned int dataLen);
void DLPC350_WaitForFlashReady(void);
int DLPC350_SetFlashType(unsigned char Type);
int DLPC350_CalculateFlashChecksum(void);
int DLPC350_GetFlashChecksum(unsigned int*checksum);
int DLPC350_SetMode(bool SLmode);
int DLPC350_GetMode(bool *pMode);
int DLPC350_LoadSplash(unsigned int index);
int DLPC350_GetSplashIndex(unsigned int *pIndex);
int DLPC350_SetTPGColor(unsigned short redFG, unsigned short greenFG, unsigned short blueFG, unsigned short redBG, unsigned short greenBG, unsigned short blueBG);
int DLPC350_GetTPGColor(unsigned short *pRedFG, unsigned short *pGreenFG, unsigned short *pBlueFG, unsigned short *pRedBG, unsigned short *pGreenBG, unsigned short *pBlueBG);
int DLPC350_ClearPatLut(void);
int DLPC350_ClearExpLut(void);
int DLPC350_AddToPatLut(int TrigType, int PatNum,int BitDepth,int LEDSelect,bool InvertPat, bool InsertBlack,bool BufSwap, bool trigOutPrev);
int DLPC350_AddToExpLut(int TrigType, int PatNum,int BitDepth,int LEDSelect,bool InvertPat, bool InsertBlack,bool BufSwap, bool trigOutPrev, unsigned int exp_time_us, unsigned int ptn_frame_period_us);
int DLPC350_GetPatLutItem(int index, int *pTrigType, int *pPatNum,int *pBitDepth,int *pLEDSelect,bool *pInvertPat, bool *pInsertBlack,bool *pBufSwap, bool *pTrigOutPrev);
int DLPC350_SendPatLut(void);
int DLPC350_SendVarExpPatLut(void);
int DLPC350_SendSplashLut(unsigned char *lutEntries, unsigned int numEntries);
int DLPC350_GetPatLut(int numEntries);
int DLPC350_GetSplashLut(unsigned char *pLut, int numEntries);
int DLPC350_SetPatternTriggerMode(int);
int DLPC350_GetPatternTriggerMode(int *);
int DLPC350_PatternDisplay(int Action);
int DLPC350_SetPatternConfig(unsigned int numLutEntries, bool repeat, unsigned int numPatsForTrigOut2, unsigned int numSplash);
int DLPC350_SetVarExpPatternConfig(unsigned int numLutEntries, bool repeat, unsigned int numPatsForTrigOut2, unsigned int numSplash);
int DLPC350_GetPatternConfig(unsigned int *pNumLutEntries, bool *pRepeat, unsigned int *pNumPatsForTrigOut2, unsigned int *pNumSplash);
int DLPC350_SetExposure_FramePeriod(unsigned int exposurePeriod, unsigned int framePeriod);
int DLPC350_GetExposure_FramePeriod(unsigned int *pExposure, unsigned int *pFramePeriod);
int DLPC350_SetTrigIn1Delay(unsigned int Delay);
int DLPC350_GetTrigIn1Delay(unsigned int *pDelay);
int DLPC350_SetInvertData(bool invert);
int DLPC350_PWMCaptureRead(unsigned int channel, unsigned int *pLowPeriod, unsigned int *pHighPeriod);
int DLPC350_SetGeneralPurposeClockOutFreq(unsigned int clkId, bool enable, unsigned int clkDivider);
int DLPC350_GetGeneralPurposeClockOutFreq(unsigned int clkId, bool *pEnabled, unsigned int *pClkDivider);
int DLPC350_MeasureSplashLoadTiming(unsigned int startIndex, unsigned int numSplash);
int DLPC350_ReadSplashLoadTiming(unsigned int *pTimingData);
int DLPC350_SetFreeze(bool Freeze);


#endif // API_H
