/*
 * Functions specific to DLP method of spectroscopy 
 *
 * Copyright (c) 2014 Texas Instruments Inc - http://www.ti.com/
 * ALL RIGHTS RESERVED
 */

#include <string.h>
#include <stdio.h>
#include <stdint.h>
#include <time.h>
#include <stdbool.h>
#include <pthread.h>
#include <math.h>
#include <unistd.h>
#include <stdlib.h>

#include "Common.h"
#include "BMPParser.h"
#include "firmware.h"
#include "FlashDevice.h"
#include "usb.h"
#include "Display.h"
#include "dlpc350_api.h"
#include "ads1255.h"
#include "Error.h"
#include "dlpspectro.h"

#define DMD_WIDTH 912
#define DMD_HEIGHT 1140
#define BYTES_PER_PIXEL 3
#define BLACK_PATTERN_EXPOSURE_US 1400

/* Trigger types */
#define INT_TRIG 0
#define EXT_POS 1
#define NO_TRIG 3

#define MAX_CSV_LINE_LEN 256
#define VSYNC_PERIOD_US 17700
#define MIN_EXPOSURE_TIME_US 230
#define DISCARD_WINDOW_FRONT 13
#define DISCARD_WINDOW_BACK 1
#define MAX_FLASH_PARAM_LINE_LEN 10*1024
#define COMPUTE_ABSORPTION_VAL(S , R)	((double)-1.0 * (double)log10( (double) ( (S) / (R) ) ) )
#define SIGN(x) ((x) >= 0 ? 1 : -1)

static unsigned char imageBuffer[DMD_WIDTH*DMD_HEIGHT*3];
static FlashDevice myFlashDevice;
extern bool use_alternate_streaming_method;
volatile static bool update_thread_started;
char vsync_sys_filename[] = "/sys/devices/platform/omap/da8xx_lcdc.0/vsync_usage";
static char *trimwhitespace(char *str);
int vsync_count_at_start, vsync_count_at_end;
static int32_t txrx[MAX_ADC_READ_SAMPLES] = {0, };
static int dlpspectro_setup_50_50_seq_prgb(void);

static int get_vsync_count()
{
    FILE *sys_fp = fopen(vsync_sys_filename, "r");
    char line[MAX_CSV_LINE_LEN];
    char *pToken;

    if(sys_fp == NULL)
    {
        DEBUG_ERR("File open error %s\n", vsync_sys_filename);
        return -1;
    }

    fgets(line, MAX_CSV_LINE_LEN, sys_fp); // Read one line from SDF file
    strtok(line, ":");	//Get first token
    pToken = strtok(NULL, "\n"); //Get next token

    fclose(sys_fp);
    return strtol(trimwhitespace(pToken), NULL, 10);

}

int init_dlpc350(void)
{
/**
 * Initializes DLPC350 controller in pattern mode, sets trigger out pulse width and reads the firmware version information
 *
 * @return   0 = SUCCESS
 *          <0 = FAIL
 *
 */
	int ret = 0;
    unsigned int FW_ver;

    if(DLPC350_GetFirmwareVersion(&FW_ver) == 0)
    {
        FW_ver &= 0xFFFFFF;
        DEBUG_ERR("DLPC350 Firmware version %d.%d.%d\n", (FW_ver >> 16), ((FW_ver << 16) >> 24), ((FW_ver << 24) >> 24));
    }
    else
    {
        return -1;
    }

    DLPC350_SetMode(true); //Set DLPC350 to pattern mode

    //Make the TRIG_OUT1 pulse as wide as possible so that the Sitara can "see" it
    // so give max -ve delay to the falling edge and max +ve delay to rising edge to make it as wide as possible
    DLPC350_SetTrigOutConfig(1, 0, 0xD5, 0);

#if 0
    /* Initial start command is required; so that the subsequent start stop can be controlled using TRIG_IN2 */
    DLPC350_PatternDisplay(2); //Start
    DLPC350_PatternDisplay(0); //Stop
#endif

    //Start a 50/50 dutycycle sequnce during idle time to reduce any hinge memory effects
    dlpspectro_setup_50_50_seq_prgb();

    return ret;
}

static int Read_from_File(void *Param, unsigned char *Data, unsigned int Size)
/**
 * This is the function that will be called from BMP and CSV Parse functions. This will read data from file and pass to the caller
 *
 * @param   Param - I - File pointer
 * @param   Data  - I - Pointer at which the data read will be filled
 * @param   Size  - I - Number of bytes to read
 *
 * @return  0 = PASS
 *          -1 = FAIL
 *
 */
{
    FILE *fd = (FILE *)Param;
    int ret;

    if(Size > 0)
    {
        if(Data == NULL)
        {
            //When BMPParser calls this function with Data=NULL; but non-zero size, it expects that this function reads Size bytes though returns nothing
            fseek(fd, Size, SEEK_CUR); 
        }
        else
        {
            if((ret = fread(Data, 1, Size, fd)) != Size)
            {
                DEBUG_ERR("fread returned %d when asked to read %d\n", ret, Size);
                return -1;
            }
        }
    }

    return 0;
}

static int Write_to_File(void *Param, unsigned char *Data, unsigned int Size)
/**
 * This is the function BMP_StoreImage will call to store the image to disk
 *
 * @param   Param - I - File pointer
 * @param   Data  - I - Pointer at which the data is present
 * @param   Size  - I - Number of bytes to write
 *
 * @return  0 = PASS
 *          -1 = FAIL
 *
 */
{
    FILE *pOutFile = (FILE *)Param;

    if(Data == NULL)
    {
        if(fseek(pOutFile, Size, SEEK_SET) != 0)
            return -1;
    }
    else if(Size > 0)
    {
        if(fwrite((char *)Data, 1, Size, pOutFile) != Size)
            return -1;
    }

    return 0;
}

static int Image_draw_in_mem(void *Param, unsigned int X, unsigned int Y, unsigned char *Pix, unsigned int Count)
/**
 * draws the pixels decoded from BMP file (by BMP_ParseImage) to a memory buffer. 
 * This function assumes a frame buffer of fixed height and width DMD_WIDTH*DMD_HEIGHT
 * DMD_WIDTH*2 is also supported as a special case to handle half pixel resolution images (for diamond DMDs) 1824x570 resolution images.
 *
 * @param   Param - I - Pointer to the memory buffer for drawing (Memory buffer is treated as 4 bytes per pixel format)
 * @param   X     - I - starting pixel X position
 * @param   Y     - I - starting pixel Y position
 * @param   Pix   - I - Pointer to the pixels coming from parser (Input is treated as 3 bytes per pixel format)
 * @param   Count - I - Number of pixels to write
 *
 * @return  0 = PASS
 *          -1 = FAIL
 *
 */
{
    char *Buffer = (char *)Param;
    char *Buffer1;
    int i,j;

    if(X > DMD_WIDTH*2 || Y > DMD_HEIGHT)
        return 0;

    if(X+Count > DMD_WIDTH*2)
        Count = DMD_WIDTH*2 - X;


    /* Special case where image width = DMD_WIDTH*2
    Here we copy Odd pixels in 1824x570 image to the even rows of a 912x1140 frame buffer
    and even pixels to the odd line 
    */
    if(X+Count > DMD_WIDTH)
    {
        Buffer = Buffer + Y*2*DMD_WIDTH*4 + X/2*4;
        Buffer1 = Buffer + DMD_WIDTH*4 + X/2*4;
        for(i=0;i<Count;i++)
        {
            if((X&1) == 1) //Odd X
            {
                for(j=0;j<3;j++)
                    *Buffer++ = *Pix++;
                *Buffer++ = 0;
            }
            else
            {
                for(j=0;j<3;j++)
                    *Buffer1++ = *Pix++;
                *Buffer1++ = 0;
            }
            X++;
        }
    }
    else
    {
        Buffer = Buffer + (X + Y*DMD_WIDTH)*4;
        for(i=0;i<Count;i++)
        {
            for(j=0;j<3;j++)
                *Buffer++ = *Pix++;

            Buffer++;
        }
    }

    return 0;
}

static int Image_stitch_in_mem(void *Param, unsigned int X, unsigned int Y, unsigned char *Pix, unsigned int Count)
/**
 * This function is used to pack 24 1-bit images into a single 24-bit image of 912x1140 dimension
 * Input stream at *Pix is always assumed to be of 8 bits per pixel
 *
 * @param   Param - I - LSB 5 bits represent where to place the pixel in stitched output image. MSB 3-bits represent bit depth of this image in the memory buffer (zero based)
 * @param   X     - I - starting pixel X position
 * @param   Y     - I - starting pixel Y position
 * @param   Pix   - I - Pointer to the pixels coming from parser (Input is always assumed to be 1 byte per pixel format)
 * @param   Count - I - Number of incoming pixels
 *
 * @return  0 = PASS
 *          -1 = FAIL
 *
 */
{
    unsigned char bitPos = *(unsigned char *)Param & 0x1F;
    unsigned char bitDepth = (*(unsigned char *)Param >> 5) + 1; 
    unsigned int i;
    unsigned char pixelByte, imgByte, bitMask;
    unsigned int bytePos=0;
    unsigned char *pImageBuffer = imageBuffer;

    if(bitPos > 23)
        return -1;
    else if(bitPos >15)
        bytePos = 2;
    else if(bitPos > 7)
        bytePos=1;

    bitPos -= bytePos*8; //relative position in the output Byte;

    if(X >= DMD_WIDTH || Y >= DMD_HEIGHT)
            return 0;

    if(X + Count > DMD_WIDTH)
    {
        Count = DMD_WIDTH - X;
    }

    for(i=0; i<Count; i++)
    {
        imgByte = *(pImageBuffer + (X + i + Y * DMD_WIDTH)*BYTES_PER_PIXEL + bytePos);
        pixelByte = *(Pix+i);
        //pixelByte <<= bitPos;
        bitMask = POW_OF_2(bitDepth)-1;
        bitMask <<= bitPos;
        imgByte &= ~bitMask;
        pixelByte &= bitMask;
        imgByte |= pixelByte;
        *(pImageBuffer + (X + i + Y * DMD_WIDTH)*BYTES_PER_PIXEL + bytePos) = imgByte;

        if(bitPos+bitDepth > 8)
        {
            imgByte = *(pImageBuffer + (X + i + Y * DMD_WIDTH)*BYTES_PER_PIXEL + bytePos + 1);
            pixelByte = *(Pix+i);
            //pixelByte >>= (8-bitPos);
            bitMask = POW_OF_2(8-bitPos)-1;
            imgByte &= ~bitMask;
            pixelByte &= bitMask;
            imgByte |= pixelByte;
            *(pImageBuffer + (X + i + Y * DMD_WIDTH)*BYTES_PER_PIXEL + bytePos + 1) = imgByte;
        }
    }
    //memcpy(pImageBuffer+(X + Y * DMD_WIDTH)*BYTES_PER_PIXEL, Pix, Count);

    return 0;
}

static int image_get(void *Param, unsigned int X, unsigned int Y, unsigned char *Pix, unsigned int Count)
/**
 * This function supplies pixels from the image buffer to BMP_StoreImage.
 *
 * @param   Param - I - Image width in pixels. If null default width is taken as DMD_WIDTH
 * @param   X     - I - starting pixel X position
 * @param   Y     - I - starting pixel Y position
 * @param   Pix   - I - Pointer at which the pixels are to be placed
 * @param   Count - I - Number of bytes required
 *
 * @return  0 = PASS
 *          -1 = FAIL
 *
 */
{
    unsigned char *pImageBuffer = imageBuffer;
    int image_width = DMD_WIDTH;

    if(Param != NULL)
        image_width = *(int *)Param;

    if(X >= image_width || Y >= DMD_HEIGHT)
        return 0;
    if(X + Count > image_width)
    {
        Count = image_width - X;
    }
    memcpy(Pix, pImageBuffer + (X + Y * image_width) * BYTES_PER_PIXEL, Count * BYTES_PER_PIXEL);
    return 0;
}

static int  find_num_lines_in_file(FILE *fp)
/**
 * This function reports the number of lines in a given file
 *
 * @param   fp    - I - File pointer.
 *
 * @return  Number of lines in the file
 *          -1 = FAIL
 *
 */
{
	char ch;
	int lines = 0;

	if(fp == NULL)
		return -1;

	while(!feof(fp))
	{
	  ch = fgetc(fp);
	  if(ch == '\n')
	  {
	    lines++;
	  }
	}

	rewind(fp);

	return lines;
}

static int dlpspectro_write_imagebuffer_to_file(char *outBmpFileName, int image_width)
/**
 * This function can be used to store a bitmap in BYTES_PER_PIXEL format in memory to a disk file in BMP format
 *
 * @param   outBmpFileName  - I - File name in which the image is to be stored.
 * @param   image_width     - I - width in pixels of the image in memory. Supports 912x1140 and 1824x570 resolution images.
 *
 * @return   0 = SUCCESS
 *          -1 = FAIL
 *
 */
{
	FILE *out_bmp_fd;
    BMP_Image_t splashImage;

    if(image_width == DMD_WIDTH*2)
        BMP_InitImage(&splashImage, DMD_WIDTH*2, DMD_HEIGHT/2, 8*BYTES_PER_PIXEL);
    else
        BMP_InitImage(&splashImage, DMD_WIDTH, DMD_HEIGHT, 8*BYTES_PER_PIXEL);

    out_bmp_fd = fopen(outBmpFileName, "w+");
    if(out_bmp_fd == NULL)
    {
        DEBUG_ERR("Unable to open file %s\n", outBmpFileName);
        return -1;
    }
    if (BMP_StoreImage(&splashImage,(BMP_DataFunc_t *)Write_to_File, out_bmp_fd, (BMP_PixelFunc_t *)image_get, &image_width) < 0)
    {
        DEBUG_ERR("BMP_StoreImage returned error");
        fclose(out_bmp_fd);
        return -1;
    }
    memset(imageBuffer, 0, DMD_WIDTH*DMD_HEIGHT*BYTES_PER_PIXEL); 
    fclose(out_bmp_fd);
    DEBUG_MSG("Pattern Image File %s generated\n", outBmpFileName);
    return 0;
}

static void matrix_mult(double *a, double*b, double*res, int p, int q, int r)
/**
 * Matrix multiplication function.
 *
 * @param   a  - I - multiplier matrix
 * @param   b  - I - multiplicand matrix
 * @param   res  - O - product matrix
 * @param   p  - I - number of rows in matrix a
 * @param   q  - I - number of columns in matrix a
 * @param   p  - I - number of columns in matrix b
 *
 * @return   0 = SUCCESS
 *          -1 = FAIL
 *
 */
{
    double sum=0;
    int i,j,k;
    for ( i = 0 ; i < p ; i++ )
    {
        for ( j = 0 ; j < r ; j++ )
        {
            for ( k = 0 ; k < q ; k++ )
            {
                sum = sum + a[i*q+k]*b[k*r+j];
            }

            res[i*r+j] = sum;
            sum = 0;
        }
    }	
}

#if 1
static void print_matrix(double *a, int p, int q)
{
    int i,j;
    for ( i = 0 ; i < p ; i++ )
    {
        for ( j = 0 ; j < q ; j++ )
        {
            printf("%f  ", a[i*q+j]);
        }
        printf("\n");
    }	
}
#endif

static void matrix_transpose(double*a, double*res, int row,int col)
/**
 * Matrix transpose function.
 *
 * @param   a  - I - input matrix
 * @param   res  - O - transposed matrix
 * @param   row  - I - number of rows in matrix a
 * @param   col  - I - number of columns in matrix a
 *
 * @return   none.
 *
 */
{
    int i,j;
    for (i=0;i<row;++i) 
        for (j=0;j<col;++j)
            res[j*row+i]=a[i*col + j];
}

int dlpspectro_get_exposure_times(FILE *sdf_fd, uint32_t *pExposureTimesUs)
/**
 *  Function that takes sdf file as input and spits out BMP files with patterns stitched into 24-bit BMP format 
 *  files scan_img_000.bmp thru scan_img_NNN.bmp
 *
 * @param   sdf_fd  - I - SDF file pointer. An SDF (Scan Description Files) is a text files in the following format.
 *                          pattern0.csv, exposure_time_pat0 (microseconds)
 *                          pattern1.bmp, exposure_time_pat1
 *                          pattern2.bmp, exposure_time_pat2
 *                          .......
 *                          .......
 *                          patternN.csv, exposure_time_patN
 *
 * @param   pExposureTimesUs  - I - Pointer at which the exposure time for each pattern is to be returned.
 *
 * @return   >=0 number of patterns successfully parsed from the sdf file
 *          -1 = FAIL
 *
 */
{
    int num_patterns=0;
	char line[MAX_CSV_LINE_LEN];
	char *pToken;

    if(sdf_fd == NULL)
        return -1;

	fgets(line, MAX_CSV_LINE_LEN, sdf_fd); // Read one line from SDF file
	while(!feof(sdf_fd))
	{
		strtok(line, ",");	//Get first token
		pToken = strtok(NULL, ","); //Get next token

        if(num_patterns == MAX_PAT_LUT_ENTRIES)
        {
            return -1;
        }
        pExposureTimesUs[num_patterns++] = strtoul(trimwhitespace(pToken), NULL,10);
        fgets(line, MAX_CSV_LINE_LEN, sdf_fd); // Read one line from SDF file
    }

    return num_patterns;

}

int dlpspectro_prep_scan_solution(FILE *sdf_fd, int num_bmp)
/**
 *  Function that takes sdf file as input and spits out BMP files with patterns stitched into 24-bit BMP format 
 *  files scan_img_num_bmp.bmp thru scan_img_NNN.bmp
 *
 * @param   sdf_fd  - I - SDF file pointer. An SDF (Scan Description Files) is a text files in the following format.
 *                          pattern0.csv, exposure_time_pat0 (microseconds)
 *                          pattern1.bmp, exposure_time_pat1
 *                          pattern2.bmp, exposure_time_pat2
 *                          .......
 *                          .......
 *                          patternN.csv, exposure_time_patN
 * @param   num_bmp  - I - Number from which the generated BMP filenames shall be numbered.
 *
 * @return   0 = SUCCESS
 *          -1 = FAIL
 *
 */
{
    int ret = 0;
    int outputBitDepth = 1;
    unsigned char drawParam = 0; 
    char line[MAX_CSV_LINE_LEN];
    char *pToken;
    FILE *img_fd;
    int line_count = 0;
    char outBmpFileName[64];
    int image_width=DMD_WIDTH;
    unsigned int color_order[]={ 16, 17, 18, 19, 20, 21, 22, 23, 8, 9, 10, 11, 12, 13, 14, 15, 0, 1, 2, 3, 4, 5, 6, 7 };

    if(sdf_fd == NULL)
        return -1;

    sprintf(outBmpFileName, "scan_img_%.3d.bmp", num_bmp);

    memset(imageBuffer, 0, DMD_WIDTH*DMD_HEIGHT*BYTES_PER_PIXEL); 
    fgets(line, MAX_CSV_LINE_LEN, sdf_fd); // Read one line from SDF file
    while(!feof(sdf_fd))
    {
        strtok(line, ",");  //Get first token
        img_fd = fopen(line, "r");
        if(img_fd == NULL)
        {
            DEBUG_ERR("Unable to open file %s\n", line);
            ret = -1;
            break;
        }
        pToken = strtok(NULL, ","); //Get next token
        DEBUG_MSG("Pattern File = %s Exposure time = %s\n", line, pToken); 

        /* Extract file extn */
        strtok(line, ".");
        pToken = strtok(NULL, ",");

        drawParam = color_order[line_count++]; // LSB 5 bits represent where to place the pixel in stitched output image
        drawParam |= ((outputBitDepth -1) << 5); //MSB 3-bits represent bit depth of input image (zero based)
        if(strcmp(pToken, "bmp") == 0)
        {
            ret = BMP_ParseImage((BMP_DataFunc_t *)Read_from_File, img_fd, (BMP_PixelFunc_t *)Image_stitch_in_mem, &drawParam, outputBitDepth);
            fclose(img_fd);
        }
        else if(strcmp(pToken, "csv") == 0)
        {
            ret = CSV_ParseImage((BMP_DataFunc_t *)Read_from_File, img_fd, (BMP_PixelFunc_t *)Image_stitch_in_mem, &drawParam, &image_width);
            fclose(img_fd);
        }
        else
        {
            DEBUG_ERR("Unsupported image file format found in input sdf file line number %d\n", line_count+1);
            fclose(img_fd);
            ret = -1;
            break;
        }
        if(ret != 0)
        {
            DEBUG_ERR("Image parsing returned %d\n", ret);
            fclose(img_fd);
            break;
        }

        if(line_count == 24)
        {
            if((ret = dlpspectro_write_imagebuffer_to_file(outBmpFileName, image_width)) < 0)
                break;

            sprintf(outBmpFileName, "scan_img_%.3d.bmp", ++num_bmp);
            line_count = 0;
        }
        fgets(line, MAX_CSV_LINE_LEN, sdf_fd); // Read next line from SDF file
    } //End While

    if(line_count != 0)
        ret = dlpspectro_write_imagebuffer_to_file(outBmpFileName, image_width);

    /* Clean up and Exit */
    return ret;
}

static void *read_file_into_mem(char *filename, int *file_size)
/**
 *  Helper function that reads a file into memory in binary format
 *
 * @param   filename  - I - Name/path of the file to be read
 * @param   file_size  - O - size in bytes of the file that was read
 *
 * @return  pointer to the memory buffer to which the file contents were read
 *
 */
{
    FILE *fp;
    char *pFileContents;
    int ret;

    fp = fopen(filename, "r");
    if(fp == NULL)
        return NULL;

    //Obtain file size
    fseek(fp, 0, SEEK_END);
    *file_size = ftell(fp);
    rewind(fp);

    pFileContents = (char *)malloc(*file_size);
    if(pFileContents == NULL)
    {
        DEBUG_ERR("Unable to allocate memory to hold file contents\n");
        fclose(fp);
        return NULL;
    }

    ret = fread(pFileContents, 1, *file_size, fp);
    if(ret != *file_size)
    {
        DEBUG_ERR("Unable to read file contents - Bytes read = %d\n", ret);
        fclose(fp);
        free(pFileContents);
        return NULL;
    }

    fclose(fp);
    return pFileContents;
}

void *dlpspectro_update_frames(void *upd_struct)
/**
 * This function runs in a thread separate from the main thread and manages the timing of the images being streamed out
 *
 * @param   upd_struct->frames_per_image - I - Number of frames for which each image is displayed. After that many framees,
 *                                               next buffer is to be displayed
 * @param   upd_struct->frame_iter - I - Number of frames already counted/elapsed before entering this function
 *
 * @return   none
 *
 */
{
    struct frame_update *pUpdate = (struct frame_update *)upd_struct;
    update_thread_started = true;
    int ret_val;
    int error_count = 0;

    /* Now display all the images one by one */
    for(; pUpdate->frame_iter<pUpdate->num_images*(pUpdate->frames_per_image); pUpdate->frame_iter++)
    {
        if(pUpdate->frame_iter%pUpdate->frames_per_image == 0)
            ret_val = DISP_UpdateFrame(true, true); //update to next buffer and wait for vysnc
        else
            ret_val = DISP_UpdateFrame(false, true); //just wait for vsync

        if(ret_val != 0)
            error_count++;

        if(pUpdate->frame_iter == pUpdate->num_images*pUpdate->frames_per_image-1)
        {
            vsync_count_at_end = get_vsync_count();
            DEBUG_MSG("Total frame task iterations until scan complete was %d\n", pUpdate->frame_iter+1);
            DEBUG_MSG("Total vsyncs until scan complete was %d\n", vsync_count_at_end - vsync_count_at_start);
            DEBUG_MSG("Number of times wait_for_vsync failed was %d\n", error_count);
        }
    }
}

int dlpspectro_load_images_for_streaming(int num_images, int first_img_idx)
/**
 * This function loads scan_img_first_img_idx.bmp thru scan_img_num_images.bmp into the consecutive frame buffers starting at first_img_idx
 *
 * @param   first_img_idx - I - Starting image to be loaded
 * @param   num_images - I - Number of images to be loaded to frame buffer.
 *
 * @return  0 = SUCCESS
 *          <0 = FAIL
 *
 */
{
    Image_t image;
    char bmp_filename[64];
    FILE *img_fd;
    int i;
    unsigned char *preload_buffer=NULL;
    int ret=0;

    preload_buffer = malloc(DMD_HEIGHT * DMD_WIDTH * 4);
    if(preload_buffer == NULL) 
    {
        DEBUG_ERR("Memory allocation for pre-load buffer failed\n");
        return -1;
    }
    image.Width = DMD_WIDTH;
    image.Height = DMD_HEIGHT;
    image.LineWidth = DMD_WIDTH * 4;
    image.PixFormat = IMAGE_PIX_FORMAT_RGB32;
    image.Buffer = preload_buffer;

    /* Preload all images  and copy them to framebuffer*/
    DISP_SetWriteBufferAtBeginning();

    for(i=0; i<num_images; i++)
    {
        sprintf(bmp_filename, "scan_img_%.3d.bmp", i+first_img_idx);
        img_fd = fopen(bmp_filename, "r");
        if(img_fd == NULL)
        {
            DEBUG_ERR("Unable to open file %s\n", bmp_filename);
            free(preload_buffer);
            return -1;
        }
        ret = BMP_ParseImage((BMP_DataFunc_t *)Read_from_File, img_fd, (BMP_PixelFunc_t *)Image_draw_in_mem, preload_buffer, 24);
        fclose(img_fd);
        if(ret != 0)
        {
            DEBUG_ERR("parsing %s returned %d\n", bmp_filename, ret);
            free(preload_buffer);
            return -1;
        }

        DISP_DrawImage(0, 0, &image, 0);
        DISP_UpdateFrame(true, false); //No need to wait for vysnc
    }
    free(preload_buffer);
    return 0;
}

static int dlpspectro_setup_50_50_seq_prgb(void)
/**
 * This function configures DLPC350 controller to operate a 50/50 dutycycle sequence on the DMD to minimize any hinge 
 * memory effects. It runs a full white pattern for 50% of frame time and full black for the rest 50%.
 *
 * @return  0 SUCCESS
 *          <0 = FAIL
 *
 */
{
    unsigned int status;

    DLPC350_SetPatternDisplayMode(true);//Patterns streamed via parallel rgb interface
    DLPC350_SetExposure_FramePeriod((VSYNC_PERIOD_US-MIN_EXPOSURE_TIME_US)/2, (VSYNC_PERIOD_US-MIN_EXPOSURE_TIME_US)/2);

    DLPC350_SetPatternConfig(2, true, 2, 1); //2 entries in LUT, repeat LUT, TrigOut2 frames 2 patterns
    DLPC350_SetPatternTriggerMode(0); //VSYNC triggers the pattern display sequence

    //Create the pattern LUT, send it and validate
    DLPC350_ClearPatLut();
    DLPC350_AddToPatLut(EXT_POS, 24, 1, 1, false, false, true, false);
    DLPC350_AddToPatLut(NO_TRIG, 24, 1, 1, true, true, false, false);

    DLPC350_SendPatLut();

    if(DLPC350_ValidatePatLutData(&status) < 0)
        return -1;

    if(status != 0)
        DEBUG_ERR("ValidatePatLutData returned 0x%x as status\n", status);

    DLPC350_PatternDisplay(2); //Start

    return 0;
}

static int dlpspectro_setup_scan_prgb(unsigned int exp_time_us, FILE *sdf_fd, struct frame_update *pUpdate )
/**
 * This function configures DLPC350 controller to operate in pattern mode to display patterns coming in via
 * the parallel RGB interface. Inserts a black pattern at the beginning of every frame.
 *
 * @param   exp_time_us - I - exposure time in microseconds for each pattern.
 * @param   sdf_fd - I - If not NULL, pointer to the sdf file from which to obtain the exposure time for each pattern
 * @param   pUpdate - I - If this and sdf_fd are not NULL, this function will fill the num_images and frames_this_image array after parsing the sdf file
 *
 * @return  On SUCCESS returns number of patterns found in sdf file
 *          <0 = FAIL
 *
 */
{
    int i;
    unsigned int status;
    int pat_reorder_lut[] = {24, 21, 18, 16, 8, 9, 10, 11, 12, 22, 19, 13, 14, 15, 0, 1, 2, 23, 20, 17, 3, 4, 5, 6, 7 };
    int pats_per_frame = (VSYNC_PERIOD_US - MIN_EXPOSURE_TIME_US)/exp_time_us; //subtract MIN_EXPOSURE_TIME_US because insertBlack requires that much time
    int trigger_type;
    bool bufSwap;
    bool insertBlack;
    int patNum;
    uint32_t exp_times_us[MAX_PAT_LUT_ENTRIES];
    int num_patterns=0;
    int index;
    int img_idx=0;
    uint32_t exp_this_frame;

    //First stop any already running sequence
    DLPC350_PatternDisplay(0); //Stop

    //Make the TRIG_OUT1 pulse as wide as possible so that the Sitara can "see" it
    // so give max -ve delay to the falling edge and max +ve delay to rising edge to make it as wide as possible
    DLPC350_SetTrigOutConfig(1, 0, 0xD5, 0);

    DLPC350_SetExposure_FramePeriod(exp_time_us, exp_time_us);
    DLPC350_SetPatternDisplayMode(true);//Patterns streamed via parallel rgb interface

    if(sdf_fd == NULL)
    {

        DLPC350_SetPatternConfig(25, true, 25, 1); //25 entries in LUT, repeat LUT, TrigOut2 frames 25 patterns
        DLPC350_SetPatternTriggerMode(0); //VSYNC triggers the pattern display sequence
        //Create the pattern LUT, send it and validate
        DLPC350_ClearPatLut();
        for(i=0; i<25; i++)
        {
            patNum = pat_reorder_lut[i];
            trigger_type = NO_TRIG;
            bufSwap = false;
            insertBlack = false;

            if(i==0) //First entry
            {
                bufSwap = true;
                trigger_type = EXT_POS;
            }
            else if (((i%pats_per_frame) == 0) && (use_alternate_streaming_method))
            {
                bufSwap = true;
                trigger_type = EXT_POS;
            }

            if(i==24)
                insertBlack = true;
            else if ((((i+1)%pats_per_frame) == 0) && (use_alternate_streaming_method)) //last sequence for this frame
                insertBlack = true;

            DLPC350_AddToPatLut(trigger_type, patNum, 1, 1, false, insertBlack, bufSwap, false);
        }

        DLPC350_SendPatLut();
    }
    else
    {
        if(pUpdate == NULL)
            return -1;

        if((num_patterns = dlpspectro_get_exposure_times(sdf_fd, &exp_times_us[0])) < 0)
            return -1;

        DLPC350_SetPatternTriggerMode(4); //Pattern trigger mode for vsync variable exposure

        pUpdate->num_images=0;
        DLPC350_ClearExpLut();
        for(i=0; i<num_patterns; i++)
        {
            index = i % 24;
            if(index == 0) //Insert black pattern as the first entry for each image
            {
                exp_this_frame = BLACK_PATTERN_EXPOSURE_US;
                pUpdate->frames_this_image[img_idx] = 1;
                if((exp_this_frame + exp_times_us[i]) > (VSYNC_PERIOD_US - MIN_EXPOSURE_TIME_US))
                    insertBlack = true; //if this is the only pattern that would fit in a vsync frame
                else
                    insertBlack = false;
                if(DLPC350_AddToExpLut(EXT_POS, 24, 1, 1, false, insertBlack, true, false, exp_this_frame, exp_this_frame) < 0)//First entry for each image
                    return -1;
                pUpdate->num_images++;
            }

            patNum = pat_reorder_lut[index+1];
            trigger_type = NO_TRIG;
            bufSwap = false;
            insertBlack = false;
            if((exp_times_us[i] > (VSYNC_PERIOD_US - MIN_EXPOSURE_TIME_US)) || (exp_times_us[i] < MIN_EXPOSURE_TIME_US))
            {
                return -1;
            }
            else
            {
                if((exp_this_frame + exp_times_us[i]) > (VSYNC_PERIOD_US - MIN_EXPOSURE_TIME_US)) //if adding this pattern overshoots the vsync frame period
                {
                    bufSwap = true;
                    trigger_type = EXT_POS;
                    exp_this_frame = exp_times_us[i];
                    pUpdate->frames_this_image[img_idx] += 1;
                }
                else
                {
                    exp_this_frame += exp_times_us[i];
                }
            }

            if((index==23) || (i==num_patterns-1)) //last pattern in each image
            {
                insertBlack = true;
                img_idx++;
            }
            else if( exp_this_frame + exp_times_us[i+1] > (VSYNC_PERIOD_US - MIN_EXPOSURE_TIME_US) ) //will next pattern fit in this same vsync frame?
                insertBlack = true;
            
            if(DLPC350_AddToExpLut(trigger_type, patNum, 1, 1, false, insertBlack, bufSwap, false, exp_times_us[i], exp_times_us[i]) < 0)
                return -1;
        }

        pUpdate->frames_per_image = 0;
        if(DLPC350_SendVarExpPatLut() < 0)
            return -1;
        if(DLPC350_SetVarExpPatternConfig(num_patterns+pUpdate->num_images, false, num_patterns+pUpdate->num_images, 1) < 0) //num_patterns entries in LUT, don't repeat LUT
            return -1;
    }

    if(DLPC350_ValidatePatLutData(&status) < 0)
        return -1;
    if(status != 0)
        DEBUG_ERR("ValidatePatLutData returned 0x%x as status\n", status);
    //DLPC350_SetPatternTriggerMode(0); //VSYNC triggers the pattern display sequence

    return num_patterns;
}


int dlpspectro_perform_scan(int spi_fd, int num_samples)
/**
 * This function collects num_samples from the ADC and stores them in the static arrary txrx
 *
 * @param   spi_fd - I - SPI device handle to communicate with the ADC
 * @param   num_samples - I - Total number of samples to be read from the ADC
 *
 * @return  0 = SUCCESS
 *          <0 = FAIL
 *
 */
{
    if(num_samples > MAX_ADC_READ_SAMPLES)
    {
        DEBUG_ERR("Max samples that can be read from the ADC is %d in this implementation\n", MAX_ADC_READ_SAMPLES);
        return -1;
    }

    //ads1255_write_command(spi_fd, ADS1255_RDATAC);

    ads1255_read_data(spi_fd, txrx, num_samples);
    //ads1255_write_command(spi_fd, ADS1255_SDATAC);

    //Start a 50/50 dutycycle sequnce during idle time to reduce any hinge memory effects
    dlpspectro_setup_50_50_seq_prgb();

    return 0;
}


int dlpspectro_compute_mean_readings(int32_t *p_raw_readings, int num_raw_readings, int expected_num_patterns)
/**
 * This function parses the raw ADC readings, computes the average ADC reading per pattern and stores them in the same array p_raw_readings
 *
 * @param   p_raw_readings - I - pointer at which raw ADC readings are stored.
 * @param   num_raw_readings -I - number of raw ADC readings
 * @param   expected_num_patterns - I - If the samples read correspond to more number of patterns than expected_num_patterns, discard those
 *
 * @return  0 = SUCCESS
 *          <0 = FAIL
 *
 */
{
    long long accumulator=0;
    int accu_start_idx;
    int accu_stop_idx;
    int num_patterns=0;
    int i,j;
    bool ptn_start_detected;
    int mean_black_level = 0;
    unsigned int num_black_patterns = 0;

    expected_num_patterns += (expected_num_patterns+23)/24; //Because we insert one black pattern per every set of 24 user defined patterns

    ptn_start_detected = false;
    accu_start_idx = 0;
    for (i = 0; i < num_raw_readings; i++) 
    {
        //Skip readings until the first trigger out1 HIGH
        if(ptn_start_detected == false)
        {
            if((p_raw_readings[i] & 2) == 2)
            {
                ptn_start_detected = true;
                accu_start_idx = i+ DISCARD_WINDOW_FRONT;
            }
        }
        else if((p_raw_readings[i] & 2) == 0)//Check if BIT1 is 0
        {
            ptn_start_detected = false;
            accu_stop_idx = i - DISCARD_WINDOW_BACK;
            if(accu_stop_idx > accu_start_idx)
            {
                for(j=accu_start_idx; j < accu_stop_idx; j++)
                    accumulator += p_raw_readings[j]>>8;
                accumulator /= (accu_stop_idx - accu_start_idx);
                p_raw_readings[num_patterns++] = accumulator;
                accumulator = 0;
                if(num_patterns == expected_num_patterns)
                    break;
            }
            else
                DEBUG_ERR("start > stop detected at pattern #%d start = %d stop =%d\n", num_patterns, accu_start_idx, accu_stop_idx);

        }
    }

    /*Every 25th pattern is a black pattern - That value needs to be used to compute average black level and 
      that pedestal needs to be subtracted from every other readings */
    for (i = 0; i < num_patterns; i+=25) 
    {
        mean_black_level += p_raw_readings[i];
        num_black_patterns++;
    }
    mean_black_level /= num_black_patterns;
    DEBUG_MSG("Mean black level = %d, num_black_patterns = %d\n", mean_black_level, num_black_patterns);

    for (i = 0, j=0; i < num_patterns; i++) 
    {
        if(i%25 != 0)
        {
            p_raw_readings[j++] = p_raw_readings[i] - mean_black_level;
        }
    }

    return 0;
}

int dlpspectro_stream_images(int spi_fd, int *p_num_patterns, unsigned int exp_time_us, int32_t **pp_raw_readings, FILE *sdf_fd)
/**
 * This function sets up the Sitara processor to stream the pattern images to DLPC350 via the parallel RGB interface in the deisred
 * order and frequencey. It also collects samples from the ADC, using the delimiter finds the average ADC reading per pattern and records
 * the mean as well as raw values in the files specified.
 *
 * @param   spi_fd - I - SPI device handle to communicate with the ADC
 * @param   p_num_patterns - I/O - Number of 1-bit patterns to be sequenced for the scan (given by the caller when sdf_fd==NULL; otherwise returned to the caller)
 * @param   exp_time_us - I - exposure time of each pattern in microseconds
 * @param   pp_raw_readings - O - Returns the pointer at which raw adc values are stored
 * @param   sdf_fd - I - NULL = uniform exposure time as per exp_time_us input; NOT NULL = derive pattern exposure times from the given sdf file
 *
 * @return  >0 number of samples collected from the ADC
 *          <0 = FAIL
 *
 */
{
    unsigned int num_samples;
    int pats_per_frame = (VSYNC_PERIOD_US - MIN_EXPOSURE_TIME_US)/exp_time_us; //subtract MIN_EXPOSURE_TIME_US because insertBlack requires that much time
    int frame_pipeline_delay = 2;
    struct frame_update fr_upd;
    int total_frames;
    int i;

    if(exp_time_us > (VSYNC_PERIOD_US - MIN_EXPOSURE_TIME_US))
    {
        printf("Exposure time specified %d greater than Vsync period %d\n", exp_time_us, VSYNC_PERIOD_US);
        return -1;
    }
    else if(exp_time_us < MIN_EXPOSURE_TIME_US)
    {
        DEBUG_ERR("Exposure time specified %d less than minimum limit %d\n", exp_time_us, MIN_EXPOSURE_TIME_US);
        return -1;
    }

    fr_upd.frame_iter = 0;
    if(use_alternate_streaming_method)
        fr_upd.frames_per_image = (25 + pats_per_frame - 1) / pats_per_frame; //rounded to whole frames
    else
        fr_upd.frames_per_image = ((25*exp_time_us+MIN_EXPOSURE_TIME_US) + VSYNC_PERIOD_US - 1) / VSYNC_PERIOD_US; //rounded to whole frames

    if(sdf_fd != NULL)
    {
        if((*p_num_patterns = dlpspectro_setup_scan_prgb(exp_time_us, sdf_fd, &fr_upd)) < 0)
        {
            DEBUG_MSG("Error configuring DLPC350\n");
            return -1;
        }
    }
    else
    {
        if(dlpspectro_setup_scan_prgb(exp_time_us, sdf_fd, &fr_upd) < 0)
            return -1;
    }

    fr_upd.num_images = (*p_num_patterns+23)/24;
    DEBUG_MSG("num_images = %d frames_per_image = %d\n", fr_upd.num_images, fr_upd.frames_per_image);
    if(fr_upd.frames_per_image != 0)
        total_frames = fr_upd.frames_per_image*fr_upd.num_images;
    else
    {
        total_frames = 0;
        for(i=0;i<fr_upd.num_images;i++)
        {
            total_frames += fr_upd.frames_this_image[i];
        }
    }

    num_samples = (total_frames+2)*(VSYNC_PERIOD_US/1000+1)*DEFAULT_ADC_SAMPLE_RATE_KSPS;

    DISP_SetWriteBufferAtBeginning();
    DISP_SetDisplayBufferAtBeginning(); //At next Vsync, the first display buffer contents will be streamed out
    vsync_count_at_start = get_vsync_count();

    DISP_StartFrameUpdates(&fr_upd);
    //Skip couple of frame until the output from Sitara gets to DLPC350 frame buffer
    while(get_vsync_count() < vsync_count_at_start + frame_pipeline_delay)
    {
        //wait
    }

    dlpspectro_perform_scan(spi_fd, num_samples);
    *pp_raw_readings = &txrx[0];

    return num_samples;

}


static int dlpspectro_setup_scan(int num_patterns, int num_images, int start_img_idx, int ptn_exposure_time_us, FILE *sdf_fd)
/**
 * This function sets up the DLPC350 to sequnce the pattern images from flash in the desired order and frequencey. 
 *
 * @param   num_patterns - I - Number of 1-bit patterns to be seuquenced for the scan
 * @param   num_images - I - Number of pattern images (each image containing 24 1-bit patterns) to be sequenced
 * @param   start_img_idx - I - Location/index of the first pattern image in flash. Thereafter the rest of the images are assumed to be in consecutive indices.
 * @param   ptn_exposure_time_us - I - exposure time of each pattern in microseconds
 * @param   sdf_fd - I - NULL. For future implementation of additional features.
 *
 * @return  total exposure time in microseconds required to display all the selected patterns.
 *
 */
{
    int i,j;
    unsigned int status;
    unsigned char imgLUT[64];
    int total_exp_time;
    int total_patterns_incl_black = num_patterns + (num_patterns+23)/24;

    //First stop any already running sequence
    DLPC350_PatternDisplay(0); //Stop

    //Make the TRIG_OUT1 pulse as wide as possible so that the Sitara can "see" it
    // so give max -ve delay to the falling edge and max +ve delay to rising edge to make it as wide as possible
    DLPC350_SetTrigOutConfig(1, 0, 0xD5, 0);
    DLPC350_SetPatternDisplayMode(false);//Patterns stored internally in flash

    //BUild and send image LUT
    for(i=start_img_idx,j=0; i<start_img_idx+num_images; i++)
        imgLUT[j++] = i;

    DLPC350_SendSplashLut(imgLUT, num_images);

    if(sdf_fd == NULL)
    {
        DLPC350_SetPatternTriggerMode(1);
        DLPC350_SetExposure_FramePeriod(ptn_exposure_time_us, ptn_exposure_time_us);
        DLPC350_SetPatternConfig(25, false, total_patterns_incl_black, num_images);
        //Create the pattern LUT, send it and validate
        DLPC350_ClearPatLut();
        //First entry - black pattern and bufSwap=true;
        DLPC350_AddToPatLut(0, 24, 1, 1, false, false, true, false);
        for(i=8;i<16;i++)
        {
            DLPC350_AddToPatLut(3, i, 1, 1, false, false, false, false);
        }
        for(i=0;i<8;i++)
        {
            DLPC350_AddToPatLut(3, i, 1, 1, false, false, false, false);
        }
        for(i=16;i<24;i++)
        {
            DLPC350_AddToPatLut(3, i, 1, 1, false, false, false, false);
        }
        DLPC350_SendPatLut();
        total_exp_time = total_patterns_incl_black*ptn_exposure_time_us;
    }
    else
    {
#if 0
        DLPC350_SetPatternTriggerMode(3);
        fgets(line, MAX_CSV_LINE_LEN, sdf_fd); // Read one line from SDF file
        DLPC350_ClearExpLut();
        i=0;
        total_exp_time = 0;
        while(!feof(sdf_fd))
        {
            strtok(line, ",");	//Get first token
            pToken = strtok(NULL, ","); //Get next token
            exp_time = strtol(trimwhitespace(pToken), NULL, 10);
            if((i % 24) == 0)
            {
                //First entry - black pattern and bufSwap=true;
                DLPC350_AddToExpLut(0, 24, 1, 1, false, false, true, false, exp_time, exp_time);
                total_exp_time += exp_time;
                num_entries += 24;
                i=0;
            }
            DLPC350_AddToExpLut(0, i, 1, 1, false, false, false, false, exp_time, exp_time);
            total_exp_time += exp_time;
        }
        num_entries += i;
        DLPC350_SetPatternConfigNew(num_entries, false, 25, num_images);
        DLPC350_SendVarExpPatLut();
#endif
    }

    DLPC350_ValidatePatLutData(&status);
    if(status != 0)
        DEBUG_ERR("ValidatePatLutData returned 0x%x as status\n", status);
    return total_exp_time;
}


int dlpspectro_test_read_adc_samples(int spi_fd, int num_samples, int32_t **pp_raw_readings)
/**
 * This function reads the desired number of samples from ADC and returns a pointer to the readings.
 *
 * @param   spi_fd - I - SPI device handle to communicate with the ADC
 * @param   num_samples - I - number of samples to read
 * @param   pp_raw_readings - O - Returns the pointer at which raw adc values are stored
 *
 * @return  total number of samples collected from the ADC
            <0 = FAIL
 *
 */
{
    dlpspectro_perform_scan(spi_fd, num_samples);
    *pp_raw_readings = &txrx[0];
    return num_samples;
}

int dlpspectro_scan_flash_images(int spi_fd, int start_img_idx, int ptn_count, int ptn_exposure_time_us, int32_t **pp_raw_readings, FILE *sdf_fd)
/**
 * This function sets up the DLPC350 to sequnce the pattern images from flash in the desired order and frequencey. 
 * It also collects samples from the ADC, then using the delimiter finds the average ADC reading per pattern and records them in the specified disk files.
 *
 * @param   spi_fd - I - SPI device handle to communicate with the ADC
 * @param   start_img_idx - I - Location/index of the first pattern image in flash. Thereafter the rest of the images are assumed to be in consecutive indices.
 * @param   ptn_count - I - Number of 1-bit patterns to be seuquenced for the scan
 * @param   ptn_exposure_time_us - I - exposure time of each pattern in microseconds
 * @param   pp_raw_readings - O - Returns the pointer at which raw adc values are stored
 * @param   sdf_fd - I - NULL. For future implementation of additional features.
 *
 * @return  total number of samples collected from the ADC
            <0 = FAIL
 *
 */
{
    int num_samples;
    int total_exp_time;
    unsigned int scan_img_count = (ptn_count+23)/24;

    total_exp_time = dlpspectro_setup_scan(ptn_count, scan_img_count, start_img_idx, ptn_exposure_time_us, sdf_fd);

    if ((total_exp_time <= 0) || (total_exp_time > 2000000))
    {
        DEBUG_ERR("For the given pattern count and exposure time, total scan time comes out to %d microseconds, max supported is 2000000 (2 seconds)\n", total_exp_time);
        return -1; 
    }

    num_samples = total_exp_time/1000*DEFAULT_ADC_SAMPLE_RATE_KSPS;
    num_samples += 100;
    dlpspectro_perform_scan(spi_fd, num_samples);
    *pp_raw_readings = &txrx[0];
    return num_samples;
}

int dlpspectro_setup_calib_scan_prgb(int spi_fd)
/**
 * This function draws the specific patterns required for a calibration scan and streams the pattern images to DLPC350
 * via the parallel RGB interface in the deisred order and frequencey. It also collects samples from the ADC, 
 * then using the delimiter finds the average ADC reading per pattern and records the mean as well as raw values in disk files
 * mean_readings_xx.txt and raw_readings_xx.txt one each for top(00), mid(01) and bottom(02) scans.
 *
 * @param   spi_fd - I - SPI device handle to communicate with the ADC
 *
 * @return  0 = SUCCESS
 *          <0 = FAIL
 *
 */
{
    int j;
    int i;
    int dmd_width = DMD_WIDTH;
    char output_filename[256];
    FILE *mean_output_fd;
    FILE *raw_output_fd;
    int StartX, StartY;
    int cal_pattern_width = 3;
    int cal_pattern_height = 160;
    DISP_Color_t color;
    int pattern_shift = 1;
    int num_patterns=0;
    int32_t *p_readings;
    int num_raw_readings;
    unsigned int color_order[]={   1<<16, 1<<17, 1<<18, 1<<19, 1<<20, 1<<21, 1<<22, 1<<23, \
                                   1<<8, 1<<9, 1<<10, 1<<11, 1<<12, 1<<13, 1<<14, 1<<15, \
                                   1<<0, 1<<1, 1<<2, 1<<3, 1<<4, 1<<5, 1<<6, 1<<7 };
                                
    
    //One iteration each for top, center and bottom scan
    for(j=0; j<3; j++)
    {
        sprintf(output_filename, "mean_readings%d.txt", j);
        mean_output_fd = fopen(output_filename, "w+");

        if(mean_output_fd == NULL)
        {
            printf("Unable to open %s\n", output_filename);
            return -1;
        }

        sprintf(output_filename, "raw_readings%d.txt", j);
        raw_output_fd = fopen(output_filename, "w+");

        if(raw_output_fd == NULL)
        {
            DEBUG_ERR("Unable to open %s\n", output_filename);
            return -1;
        }

        /* Fill the frame buffer with calibration patterns 
           each pattern being 3 pixel wide and 160 rows tall */
        DISP_SetWriteBufferAtBeginning();

        StartX = 0;
        StartY = j*cal_pattern_height*3;
        color = 1;
        num_patterns=0;
        DISP_DrawRectangle(0, 0, dmd_width, DMD_HEIGHT, 0, false);
        while(StartX + cal_pattern_width <= dmd_width)
        {
            if( (num_patterns%24 == 0) && (num_patterns!=0) )
            {
                DISP_UpdateFrame(true, false); //No need to wait for vysnc
                DISP_DrawRectangle(0, 0, dmd_width, DMD_HEIGHT, 0, false);
            }
            color = color_order[num_patterns%24];
            DISP_DrawRectangle(StartX, StartY, cal_pattern_width, cal_pattern_height, color, true);
            StartX += pattern_shift;
            num_patterns++;
        }

        num_raw_readings = dlpspectro_stream_images(spi_fd, &num_patterns, 1500, &p_readings, NULL);
        for(i=0; i<num_raw_readings; i++)
        {
            fprintf(raw_output_fd, "0x%.8X\n", p_readings[i]);
        }
        dlpspectro_compute_mean_readings(p_readings, num_raw_readings, num_patterns);

        for(i=0; i<num_patterns; i++)
        {
            fprintf(mean_output_fd, "0x%.8X\n", p_readings[i]);
        }

        fclose(mean_output_fd);
        fclose(raw_output_fd);
    }
    return 0;
}

#if 0
int dlpspectro_setup_calib_scan(int fd)
{
    unsigned int regval;
    int imgs_per_segment = 7;
    int i;
    int j;
    int segment_height = 180;
    int dmd_height = 1140;
    char output_filename[256];
    FILE *mean_output_fd;
    FILE *raw_output_fd;

    //One iteration each for top, center and bottom scan
    for(j=0; j<3; j++)
    {
        sprintf(output_filename, "mean_readings%d.txt", j);
        mean_output_fd = fopen(output_filename, "w+");

        if(mean_output_fd == NULL)
        {
            DEBUG_ERR("Unable to open %s\n", output_filename);
            return -1;
        }

        sprintf(output_filename, "raw_readings%d.txt", j);
        raw_output_fd = fopen(output_filename, "w+");

        if(raw_output_fd == NULL)
        {
            DEBUG_ERR("Unable to open %s\n", output_filename);
            return -1;
        }
        
        imgs_per_segment = 7;
        for(i=0; i<38; i+=imgs_per_segment)
        {
            if(38-i < imgs_per_segment)
                imgs_per_segment = 38-i;

            dlpspectro_setup_scan(24*imgs_per_segment, imgs_per_segment, i, NULL);

            if(j==0)
            {
                DLPC350_MemWrite(0x11010044, 0); //first display line position
                DLPC350_MemWrite(0x1101003C, segment_height); //total input line select
            }
            else if (j==1)
            {
                DLPC350_MemWrite(0x11010044, dmd_height/2 - segment_height/2); //first display line position
                DLPC350_MemWrite(0x1101003C, segment_height); //total input line select
            }
            else
            {
                DLPC350_MemWrite(0x11010044, dmd_height - segment_height); //first display line position
                DLPC350_MemWrite(0x1101003C, segment_height); //total input line select
            }

            dlpspectro_perform_scan(fd, (25*imgs_per_segment + 1)*ptn_exposure_time_us/1000*30, raw_output_fd, mean_output_fd);

        }
        fclose(mean_output_fd);
        fclose(raw_output_fd);
    }
}
#endif

static int compare_double(void *left, void *right)
{
    return *(double *)left - *(double *)right;
}

static int convert_sanitize_readings(FILE *pReadingsFile, double *pReadings, int *pFrontNumReadingsToDiscard, int *pBackNumReadingsToDiscard, float threshold_factor)
/**
 * After a scan is peformed using dlpspectro_scan_flash_images() or dlpspectro_stream_images() or dlpspectro_setup_calib_scan_prgb()
 * and mean_readings.txt is obtained, this function is used to discard the samples in the beginning or end of file that are below a certain
 * threshold compared to the median. This step is sometimes necessary before computing the absorption spectrum to discard the samples corresponding
 * to regions that may be outside of the illuminated region due to mechanical mis-alignment.
 *
 * @param   pReadingsFile - I - Pointer to the file that contains the mean adc readings
 * @param   pReadings - O - Pointer to the memory buffer in which the ADC readings read from the file shall be stored.
 * @param   pFrontNumReadingsToDiscard - O - Number of samples found in the beginning of the file that are below the threshold and hence need to be discarded
 * @param   pBackNumReadingsToDiscard - O - Number of samples found in the end of the file that are below the threshold and hence need to be discarded
 * @param   threshold_factor - I - If threshold_factor is 0.001, all samples <= .001*median_value will be discarded
 *
 * @return  0 = SUCCESS
 *          <0 = FAIL
 *
 */
{
	char RdBuf[50];
    int numReadings=0;
    double *pSortedReadings;
    double median_val;
    int i;

    while(fscanf(pReadingsFile,"%s",RdBuf) != EOF) 
    {
        *(pReadings + numReadings++) = strtod(RdBuf,NULL);
    }
    pSortedReadings = (double *)malloc(numReadings * sizeof(double));
    if(pSortedReadings == NULL)
    {
        DEBUG_ERR("ERROR: Unable to allocate memory\n");
        return -1;
    }

    memcpy(pSortedReadings, pReadings, numReadings*sizeof(double));
    qsort(pSortedReadings, numReadings, sizeof(double), compare_double);

#if 1
    //median value is the mid point of sorted array
    if(numReadings & 1) //odd number of elements in the array
        median_val = pSortedReadings[numReadings/2];
    else
        median_val = pSortedReadings[numReadings/2-1]/2 + pSortedReadings[numReadings/2]/2;
#else
        median_val = pSortedReadings[numReadings-1]/2;
#endif

    DEBUG_MSG("In the readings file - Min value = %f, Max value = %f and Median value = %f\n", pSortedReadings[0], pSortedReadings[numReadings-1], median_val);

    //Traverse the readings and discard values below the threshold at the beginning and end of the array
    *pFrontNumReadingsToDiscard = 0;
    *pBackNumReadingsToDiscard = 0;

    for(i=0; i<numReadings; i++)
    {
        if(i < numReadings/2) //first half of the data
        {
            if(pReadings[i] <= median_val*threshold_factor)
                *pFrontNumReadingsToDiscard = i+1;
        }
        else
        {
            if(pReadings[i] <= median_val*threshold_factor)
            {
                *pBackNumReadingsToDiscard = numReadings-i;
                break;
            }
        }
    }
    
    free(pSortedReadings);
}

/*
* This function generates the absorption spectrum
*/

int dlpspectro_compute_absorption_spectrum(FILE *calRefSpectFilePtr, FILE *inputSampleSpectFilePtr, FILE *abspSpeFilePtr, float threshold_factor)
/**
 * After a scan is peformed using dlpspectro_scan_flash_images() or dlpspectro_stream_images() or dlpspectro_setup_calib_scan_prgb()
 * and mean_readings.txt is obtained, this function is used to compute the absorption spectrum computed using one reference and one sample set of readings.
 *
 * @param   calRefSpectFilePtr - I - Pointer to the file that contains the mean adc readings obtained from scanning the reference object.
 * @param   inputSampleSpectFilePtr - I - Pointer to the file that contains the mean adc readings obtained from scanning the sample object.
 * @param   abspSpeFilePtr - O - Pointer to the file that will contain the computed absorption spectrum from the above two inputs.
 * @param   threshold_factor - I - If threshold_factor is 0.001, all samples <= .001*median_value will be discarded
 *
 * @return  0 = SUCCESS
 *          <0 = FAIL
 *
 */
{
	int i;
	double tempDoubleVal;
    int frontNumReadingsToDiscardSample;
    int backNumReadingsToDiscardSample;
    int frontNumReadingsToDiscardRef;
    int backNumReadingsToDiscardRef;
    int numReadings;
    double *pRefReadings;
    double *pSampleReadings;
    int ret = 0;

	//Find if number of lines in each file matching
	if((numReadings = find_num_lines_in_file(calRefSpectFilePtr)) != find_num_lines_in_file(inputSampleSpectFilePtr) )
	{
		DEBUG_ERR("ERROR: Mismatch in number of lines in reference and sample reading files\n");
		return -1;
	}

    pRefReadings = (double *)malloc(numReadings * sizeof(double));
    pSampleReadings = (double *)malloc(numReadings * sizeof(double));

    if((pRefReadings == NULL) || (pSampleReadings == NULL))
    {
        DEBUG_ERR("ERROR: Unable to allocate memory\n");
        return -1;
    }

    convert_sanitize_readings(calRefSpectFilePtr, pRefReadings, &frontNumReadingsToDiscardRef, &backNumReadingsToDiscardRef, threshold_factor);
    convert_sanitize_readings(inputSampleSpectFilePtr, pSampleReadings, &frontNumReadingsToDiscardSample, &backNumReadingsToDiscardSample, threshold_factor);

	// Setting discarded samples for sample measurement to the same as ref measurement, since ref is a better judge of where the spectrum begins and ends.
	frontNumReadingsToDiscardSample = frontNumReadingsToDiscardRef;
	backNumReadingsToDiscardSample = backNumReadingsToDiscardRef;

    if(frontNumReadingsToDiscardRef)
    {
        DEBUG_WRN("%d samples at the beginning of reference readings file discarded as noise\n", frontNumReadingsToDiscardRef);
        ret = frontNumReadingsToDiscardRef;
    }
    if(backNumReadingsToDiscardRef)
    {
        DEBUG_WRN("%d samples at the end of reference readings file discarded as noise\n", backNumReadingsToDiscardRef);
        ret = backNumReadingsToDiscardRef;
    }
    if(frontNumReadingsToDiscardSample)
    {
        DEBUG_WRN("%d samples at the beginning of sample readings file discarded as noise\n", frontNumReadingsToDiscardSample);
        ret = frontNumReadingsToDiscardSample;
    }
    if(backNumReadingsToDiscardSample)
    {
        DEBUG_WRN("%d samples at the end of sample readings file discarded as noise\n", backNumReadingsToDiscardSample);
        ret = backNumReadingsToDiscardSample;
    }

    if((frontNumReadingsToDiscardRef != frontNumReadingsToDiscardSample) || (backNumReadingsToDiscardRef != backNumReadingsToDiscardSample))
    {
        DEBUG_ERR("Mismatch in number of readings from reference and sample after discarding noise\n");
        free(pRefReadings);
        free(pSampleReadings);
        return -1;
    }
    
    for(i=0; i<numReadings-backNumReadingsToDiscardRef; i++)
	{
		if(i<frontNumReadingsToDiscardRef)
		{
			tempDoubleVal = COMPUTE_ABSORPTION_VAL(*(pSampleReadings+frontNumReadingsToDiscardRef), *(pRefReadings+frontNumReadingsToDiscardSample));
		}
		else
		{
			tempDoubleVal = COMPUTE_ABSORPTION_VAL(*(pSampleReadings+i), *(pRefReadings+i));
		}
		if(isnan(tempDoubleVal) || isinf(tempDoubleVal) || (tempDoubleVal <= 0))
        {
            DEBUG_WRN("ERROR: Invalid absorption value computed at reading #%d.. ref=%f, sample=%f, abs = %f\n", i, *(pRefReadings+i), *(pSampleReadings+i), tempDoubleVal);
            ret = 1;
        }
        fprintf(abspSpeFilePtr,"%f\n",tempDoubleVal);

	}

    free(pRefReadings);
    free(pSampleReadings);
	return ret;
}

static float dlpspectro_find_peaks3(float y1, float y2, float y3)
/*
FINDPEAK3 - parabolic interpolation of peak location given 3
equally-spaced points
Returns a relative delta-x value to be added to the x-value of y(2).
i.e., xpeak = x(2) + findpeak(y) * (x(3) - x(2));
*/
{
    return (0.5 * (y1 - y3) / (y1 - 2 * y2 + y3));
}

int dlpspectro_find_peaks(FILE *absorp_spect_file, float peak_sel_divisor, float *peaks, float *peak_inds)
/**
 * Finds and retuns the peak values and the locations of the peak for a given set of values.
 *
 * @param   absorp_spect_file - I - Pointer to the file that contains the computed absorption spectrum values.
 * @param   peak_sel_divisor - I - This input will decide the criteria to select peaks. Values that are higher than the preceding dip/valley by
 *                                  an amount >= (max-min)/peak_sel_divisor will be counted as a peak.
 * @param   peaks - O - Peak values in the absorption spectrum file input that matches the condition above.
 * @param   peak_inds - O - Location/index/position of the Peak values in the absorption spectrum file input that matches the condition above.
 *
 * @return  number of peaks found in the input data set
 *          <0 = FAIL
 *
 */
{
	char line[MAX_CSV_LINE_LEN];
    float *values=NULL;
    float *diff_vals=NULL;
    float *peak_vals=NULL;
    char *endPtr; //used in strtod function
    int i;
    int j;
    int file_len;
    int *peak_valley_indices=NULL;
    int *peak_locs=NULL;
    int num_peaks_vallies;
    float *peaks_vallies=NULL;
    float min_val;
    float max_val;
    float sel;
    int ret_val=0;
    int start_index;
    float left_min;
    float temp_val;
    int temp_loc;
    int max_peaks;
    bool found_peak;
    int peak_index;
    float start_offset = 2.5;

    file_len = find_num_lines_in_file(absorp_spect_file);
    values = (float *)malloc(file_len*sizeof(float));
    diff_vals = (float *)malloc((file_len-1)*sizeof(float));
    peak_valley_indices = (int *)malloc(((file_len+1)/2+2) * sizeof(int));
    peaks_vallies = (float *)malloc(((file_len+1)/2+2)*sizeof(float));
    if(values == NULL || diff_vals == NULL || peaks_vallies == NULL || peak_valley_indices == NULL)
    {
        DEBUG_ERR("Unable to allocate memory\n");
        ret_val = -1;
        goto cleanup_and_exit;
    }

    // Read all values from file
    for(i=0; i<file_len; i++)
    {
        fgets(line, MAX_CSV_LINE_LEN, absorp_spect_file); // Read one line from file
        values[i] = strtof(line, &endPtr);
    }

    //Find derivatives or diffs 
    for(i=0; i < file_len-1; i++)
    {
        diff_vals[i] = values[i+1] - values[i];
    }

    //Find indices where derivate changes sign - these are our potential peaks and valleys
    j=0;
    /* Include end points in potential peaks and vallies */
    //disabling end points because last pattern (910) was commonly causing a peak
	//peak_valley_indices[j++] = 0;
    num_peaks_vallies = 0;
    for(i=1; i<file_len-1; i++)
    {
        if(SIGN(diff_vals[i]) != SIGN(diff_vals[i-1]))
        {
            peak_valley_indices[j++] = i;
        }
    }

    /* Include end points in potential peaks and vallies */
	//disabling end points because last pattern (910) was commonly causing a peak
	//peak_valley_indices[j++] = file_len-1;
	//num_peaks_vallies = j;
	num_peaks_vallies = j-1;
    if(num_peaks_vallies <= 2)
    {
        DEBUG_ERR("The data does not have peaks/valleys\n");
        ret_val = -1;
        goto cleanup_and_exit;
    }
    min_val = values[0];
    max_val = values[0];
    for(i=0; i<num_peaks_vallies; i++)
    {
        peaks_vallies[i] = values[peak_valley_indices[i]];
        if(peaks_vallies[i] < min_val)
            min_val = peaks_vallies[i];
        if(peaks_vallies[i] > max_val)
            max_val = peaks_vallies[i];
    }
    sel = (max_val - min_val)/peak_sel_divisor;
    //DEBUG_MSG("min_val = %f, max_val = %f\n", min_val, max_val);

    /* Deal with first point - since we just took it, it may not alternate like the rest*/
    start_index=0;
    /* if first value is larger than second and second is larger than third, we can remove the second point from potential peaks */
    if( peaks_vallies[0] >= peaks_vallies[1] )
    {
        if(peaks_vallies[1] >= peaks_vallies[2])
        {
            peaks_vallies[1] = peaks_vallies[0];
            start_index = 1;
        }
    }
    else
    {
        /* if first value is smaller than second and second is smaller than third, we can remove the first two points from potential peaks */
        if( peaks_vallies[1] < peaks_vallies[2] )
            start_index = 2;
        else
            start_index = 1;
    }

    /* Initialize loop variables */
#if 0
    DEBUG_MSG("num_peaks_vallies = %d\n", num_peaks_vallies);
    DEBUG_MSG("we are starting at the peak at index %d\n", start_index);
    DEBUG_MSG("Selectivity divisor = %d\n", peak_sel_divisor);
#endif
    max_peaks = (num_peaks_vallies-start_index+1)/2;
    peak_locs = (int *)malloc(max_peaks*sizeof(int));
    peak_vals = (float *)malloc(max_peaks*sizeof(float));
    if(peak_locs == NULL || peak_vals == NULL)
    {
        DEBUG_ERR("Unable to allocate memory\n");
        ret_val = -1;
        goto cleanup_and_exit;
    }
    found_peak = false;
    temp_val = min_val;
    left_min = min_val;
    j=0;

    for(i=start_index; i<num_peaks_vallies-1; i++)
    {
        /* i is at a peak */
        /* Reset peak finding if we had a peak and next peak is larger than this or if the left min was small enough */
        if(found_peak)
        {
            temp_val = min_val;
            found_peak = false;
        }

        if((peaks_vallies[i] > temp_val) && (peaks_vallies[i] > left_min + sel))
        {
            temp_loc = i;
            temp_val = peaks_vallies[i];
        }

        i++; //Move on to the valley
        if(i == num_peaks_vallies-1)
            break; //Make sure we don't read past the array

        /* down at least sel from peak */
        if((found_peak == false) && (temp_val > peaks_vallies[i] + sel))
        {
            found_peak = true;
            left_min = peaks_vallies[i];
            peak_locs[j] = temp_loc; //Add peak to index
            peak_vals[j] = temp_val;
            j++;
        }
        else if(peaks_vallies[i] < left_min) //new left min
            left_min = peaks_vallies[i];
    }

    //Handle last point
    if((peaks_vallies[num_peaks_vallies-1] > temp_val) && (peaks_vallies[num_peaks_vallies-1] > left_min + sel))
    {
        peak_locs[j] = num_peaks_vallies-1; 
        peak_vals[j] = peaks_vallies[num_peaks_vallies-1];
        j++;
    }
    else if((found_peak == false) && (temp_val > min_val))
    {
        peak_locs[j] = temp_loc;
        peak_vals[j] = temp_val;
        j++;
    }

    for(i=0; i<j; i++)
    {
        peaks[i] = peak_vals[i];
        peak_index = peak_valley_indices[peak_locs[i]];
        //Do parabolic interpolation of the peaks
        peak_inds[i] = start_offset + 2*(peak_index + dlpspectro_find_peaks3(values[peak_index-1], values[peak_index], values[peak_index+1]));
    }
    ret_val = j;

cleanup_and_exit:
    if(values != NULL)
        free(values);
    if(diff_vals != NULL)
        free(diff_vals);
    if(peak_valley_indices != NULL)
        free(peak_valley_indices);
    if(peaks_vallies != NULL)
        free(peaks_vallies);

    if(peak_locs != NULL)
        free(peak_locs);
    if(peak_vals != NULL)
        free(peak_vals);
    return ret_val;
}

int dlpspectro_polyfit(double *x_peaks, double *l_peaks, double*x_to_l_coeffs)
/**
 * Finds a second order polynomial that fits the given x and y input values and returs the three co-efficients for that polynomial.
 *
 * @param   x_peaks - I - Pointer to the x values
 * @param   l_peaks - I - Pointer to the y values
 * @param   x_to_l_coeffs - O - Pointer to the computed polynomial coefficients in the following order c, b, a which should be used to compute any given y value as
 *                                       y = ax2 + bx + c
 *
 * @return  number of peaks found in the input data set
 *          <0 = FAIL
 *
 */
{
    double x_trans[3][6],a[3][3],res[3][6], x[6][3];
    double a_inv[3][3];
    int i,j;
    double determinant=0;

    //Create X matrix
    for(i=0;i<6;i++){
        x[i][0]=1;
        x[i][1]=x_peaks[i];
        x[i][2]=x_peaks[i]*x_peaks[i];
    }

    //get x transpose
    matrix_transpose(&x[0][0],&x_trans[0][0],6,3);

    //x_trans * x
    matrix_mult(&x_trans[0][0],&x[0][0],&a[0][0],3,6,3);


    //Get inverse of (X_trans*X) matrix
    determinant = a[0][0]*((a[1][1]*a[2][2]) - (a[2][1]*a[1][2])) -a[0][1]*(a[1][0]*a[2][2] - a[2][0]*a[1][2]) + a[0][2]*(a[1][0]*a[2][1] - a[2][0]*a[1][1]);
    for(j=0;j<3;j++){
        for(i=0;i<3;i++)
            a_inv[i][j]= ((a[(i+1)%3][(j+1)%3] * a[(i+2)%3][(j+2)%3]) - (a[(i+1)%3][(j+2)%3]*a[(i+2)%3][(j+1)%3]))/ determinant;
    }

    matrix_mult(&a_inv[0][0], &x_trans[0][0], &res[0][0],3, 3, 6);

    //Get beta matrix
    matrix_mult(&res[0][0], l_peaks, x_to_l_coeffs,3,6,1);

    return 0;
}

int dlpspectro_add_images_to_firmware(char *fw_filename, int num_images)
/**
 *  Adds/Replaces the images in given firmware image with images from scan_img_000.bmp thru scan_img_num_images.bmp
 *
 * @param   fw_filename  - I - Name/path to the firmware file to be modified.
 * @param   num_images   - I - Number of images to be added/replaced in the firmware file. Replaces images starting with the one at index 1
 *
 * @return   0 = SUCCESS
 *          <0 = FAIL
 *
 */
{
    int file_size;
    unsigned int new_fw_size;
    unsigned char *pFw_file_contents;
    unsigned char *pNewFw_contents;
    int ret;
    char inBmpFileName[64];
    int i;
    int splash_idx;
    int start_splash_index = 1;
    FILE *out_fw_fd;
    FILE *tmp_bmp_fd;
    unsigned char *pImageBuffer = imageBuffer;
    unsigned char *pByteArray;
    unsigned char compression;
    unsigned int compSize;
    BMP_Image_t splashImage;
    int num_splash_in_fwimage;

    pFw_file_contents = read_file_into_mem(fw_filename, &file_size);

    if( (pFw_file_contents == NULL) || (num_images <= 0))
        return -1;

    ret = Frmw_CopyAndVerifyImage(pFw_file_contents, file_size);
    if(ret == ERROR_FRMW_FLASH_TABLE_SIGN_MISMATCH)
    {
        DEBUG_ERR("Flash table signature mismatch : Bad firmware image\n");
        goto cleanup_and_exit;
    }
    else if (ret == ERROR_NO_MEM_FOR_MALLOC)
    {
        DEBUG_ERR("Malloc failed\n");
        goto cleanup_and_exit;
    }

    num_splash_in_fwimage = Frmw_GetSplashCount();
    if(start_splash_index > num_splash_in_fwimage)
    {
        DEBUG_ERR("Incorrect start splash index specified %d\n", start_splash_index);
        goto cleanup_and_exit;
    }
    if(start_splash_index + num_images > num_splash_in_fwimage)
        num_splash_in_fwimage = start_splash_index + num_images;

    Frmw_SPLASH_InitBuffer(num_splash_in_fwimage);

    for(splash_idx = 0; splash_idx < Frmw_GetSplashCount(); )
    {
        if(splash_idx == start_splash_index)
        {
            for(i=0; i<num_images; i++)
            {
                DEBUG_MSG("splash_index = %d replaced\n", splash_idx+i);
                sprintf(inBmpFileName, "scan_img_%.3d.bmp", i);
                pByteArray = read_file_into_mem(inBmpFileName, &file_size);

                compression = SPLASH_NOCOMP_SPECIFIED;
                ret = Frmw_SPLASH_AddSplash(pByteArray, &compression, &compSize);
                if (ret < 0)
                {
                    switch(ret)
                    {
                        case ERROR_NOT_BMP_FILE:
                            DEBUG_ERR("Error building firmware - %s not in BMP format\n", inBmpFileName);
                            break;
                        case ERROR_NOT_24bit_BMP_FILE:
                            DEBUG_ERR("Error building firmware - %s not in 24-bit format\n", inBmpFileName);
                            break;
                        case ERROR_NO_MEM_FOR_MALLOC:
                            DEBUG_ERR("Error building firmware with %s - Insufficient memory\n", inBmpFileName);
                            break;
                        default:
                            DEBUG_ERR("Error building firmware with %s - error code %d\n", inBmpFileName, ret);
                            break;
                    }
                    goto cleanup_and_exit;
                }

                free(pByteArray);
            }
            splash_idx += num_images;
        }
        //Get and put the same splash at splash_index 
        else // if(splash_index != start_splash_index
        {
            DEBUG_MSG("splash_index = %d\n", splash_idx);
            memset(pImageBuffer, 0, DMD_WIDTH*DMD_HEIGHT*BYTES_PER_PIXEL);
            Frmw_GetSpashImage(pImageBuffer, splash_idx);

            tmp_bmp_fd = fopen("tmp.bmp", "w+");
            if(tmp_bmp_fd == NULL)
            {
                DEBUG_ERR("Unable to open tmp.bmp\n");
                ret = -1;
                break;
            }

            BMP_InitImage(&splashImage, DMD_WIDTH, DMD_HEIGHT, 8*BYTES_PER_PIXEL);
            ret = BMP_StoreImage(&splashImage,(BMP_DataFunc_t *)Write_to_File, tmp_bmp_fd, (BMP_PixelFunc_t *)image_get, NULL);
            if(ret != 0)
            {
                DEBUG_ERR("BMP_StoreImage returned %d\n", ret);
                break;
            }
            fclose(tmp_bmp_fd);

            pByteArray = read_file_into_mem("tmp.bmp", &file_size);

            compression = SPLASH_NOCOMP_SPECIFIED;
            ret = Frmw_SPLASH_AddSplash(pByteArray, &compression, &compSize);
            free(pByteArray);
            if (ret < 0)
            {
                DEBUG_ERR("Frmw_SPLASH_AddSplash returned %d\n", ret);
                break;
            }

            splash_idx++;
        }
    }

    if(ret >= 0)
    {
        Frmw_Get_NewFlashImage(&pNewFw_contents, &new_fw_size);
        if((pNewFw_contents == NULL) || (new_fw_size <= 0))
        {
            DEBUG_ERR("New firmware image invalid\n");
            ret = -1;
            goto cleanup_and_exit;
        }
        out_fw_fd = fopen("new_firmware.bin", "w+");
        fwrite(pNewFw_contents, 1, new_fw_size, out_fw_fd);
        fclose(out_fw_fd);
    }

cleanup_and_exit:
    free(pFw_file_contents);
    return ret;
}

static char *trimwhitespace(char *str)
/**
 * Helper function that trims the given character string of the whitespace at either end. 
 *
 * @param   str  - I - Pointer to input string.
 *
 * @return   pointer to the trimmed string
 *
 */
{
    char *end;

    //Trim leading white spaces
    while(isspace(*str))
        str++;

    if(*str == 0)
        return str; //reached end of line

    //Trim trailing white spaces
    end = str + strlen(str) - 1;
    while((end > str) && (isspace(*end)))
        end--;

    *(end+1) = 0;//Terminate string here

    return str;
}

static bool process_flash_params_line(char *line)
{
    unsigned int MfgID, DevID, i;
    char *pToken;

    line = trimwhitespace(line);
    if(line[0] == '/')
        return false;

    if(line[0] == 0) //if line is empty
        return false;

    strtok(line, ",");	//Get first token and discar - this is Mfg Name

    pToken = strtok(NULL, ","); //Get token #1
    if(*pToken == 0)
        return false;

    MfgID = strtol(trimwhitespace(pToken), NULL, 16);
    pToken = strtok(NULL, ","); //Get token #2 and discard - this is Device Name
    if(*pToken == 0)
        return false;
    pToken = strtok(NULL, ","); //Get token #3
    if(*pToken == 0)
        return false;
    DevID = strtol(trimwhitespace(pToken), NULL, 16);

    if((MfgID == myFlashDevice.Mfg_ID) && (DevID == myFlashDevice.Dev_ID))
    {
        pToken = strtok(NULL, ","); //Get token #4
        if(*pToken == 0)
            return false;
        myFlashDevice.Size_MBit = strtol(trimwhitespace(pToken), NULL, 10);
        pToken = strtok(NULL, ","); //Get token #5
        if(*pToken == 0)
            return false;
        myFlashDevice.Type = strtol(trimwhitespace(pToken), NULL, 10);
        pToken = strtok(NULL, ","); //Get token #6 and discard
        if(*pToken == 0)
            return false;
        pToken = strtok(NULL, ","); //Get token #7
        if(*pToken == 0)
            return false;
        myFlashDevice.numSectors = strtol(trimwhitespace(pToken), NULL, 10);

        for(i=0; i<myFlashDevice.numSectors; i++)
        {
            pToken = strtok(NULL, ","); //Get next token
            if(*pToken == 0)
                return false;
            myFlashDevice.SectorArr[i] = strtol(trimwhitespace(pToken), NULL, 16);
        }

        return true;
    }
    return false;
}

static int GetSectorNum(unsigned int Addr)
{
    unsigned int i;
    for(i=0; i < myFlashDevice.numSectors; i++)
    {
        if(myFlashDevice.SectorArr[i] > Addr)
            break;
    }
    return i-1;
}

int dlpspectro_update_firmware(char *fw_filename)
{
    unsigned short manID;
    unsigned long long devID;
    int startSector=0, i, BLsize=0, lastSectorToErase;
    unsigned char *pByteArray=NULL;
    unsigned int dataLen, dataLen_full;
    int fw_file_size;
    int bytesSent;
    unsigned int expectedChecksum=0, checksum;
    long long percent_completion = 0;
    char line[MAX_FLASH_PARAM_LINE_LEN];
    FILE *flparam_fp;
    int ret;

    ret = USB_Init();

    if(USB_IsConnected())
        USB_Close();

    if(USB_Open() != 0)
        DEBUG_ERR("USB connection to DLPC350 failed\n");
    else if (USB_IsConnected())
        DEBUG_MSG("USB connection to DLPC350 now open\n");

    pByteArray = read_file_into_mem(fw_filename, &fw_file_size);
    if(pByteArray == NULL)
        return -1;

    if(DLPC350_EnterProgrammingMode() < 0)
    {
        DEBUG_ERR("Unable to enter Programming mode\n");
        ret = -1;
        goto cleanup_and_exit;
    }

    USB_Close();
    DEBUG_ERR("Waiting to enter programming mode\n");

    sleep(5); //Wait for 5 seconds
    DEBUG_ERR("Wait over to enter programming mode\n");

    if(USB_Open() != 0)
        DEBUG_ERR("USB connection to DLPC350 failed\n");
    else if (USB_IsConnected())
        DEBUG_MSG("USB connection to DLPC350 now open\n");

    if(DLPC350_GetFlashManID(&manID) < 0)
    {
        DEBUG_ERR("Unable to read Flash Manufacturer ID");
        ret = -1;
        goto cleanup_and_exit;
    }
    if(DLPC350_GetFlashDevID(&devID) < 0)
    {
        DEBUG_ERR("Unable to read Flash Device ID");
        ret = -1;
        goto cleanup_and_exit;
    }
    devID &= 0xFFFF;

    myFlashDevice.Mfg_ID = manID;
    myFlashDevice.Dev_ID = devID;

    flparam_fp = fopen("FlashDeviceParameters.txt", "r");
    if(flparam_fp == NULL)
    {
        DEBUG_ERR("Unable to open FlashDeviceParameters.txt");
        ret = -1;
        goto cleanup_and_exit;
    }

    bool found = false;
    while (!feof(flparam_fp))
    {
        fgets(line, MAX_FLASH_PARAM_LINE_LEN, flparam_fp);
        if(process_flash_params_line(line))
        {
            found = true;
            break;
        }
    }

    if(found == false)
    {
        DEBUG_ERR("Unsupported Flash Device : Manufacturer ID = 0x%x & Device ID = 0x%llx", manID, devID);
        ret = -1;
        goto cleanup_and_exit;
    }

    if(true) //Always skip bootloader 
    {
        BLsize = 128 * 1024;
    }

    startSector = GetSectorNum(BLsize);
    lastSectorToErase = GetSectorNum(fw_file_size);
    if(fw_file_size == myFlashDevice.SectorArr[lastSectorToErase]) //If perfectly aligned with last sector start addr, no need to erase last sector.
        lastSectorToErase -= 1;

    DLPC350_SetFlashType(myFlashDevice.Type);
    i=0;
    DEBUG_ERR("Erasing Flash Sectors %d", i);
    fflush(stdout);

    for(i=startSector; i <= lastSectorToErase; i++)
    {
        DLPC350_SetFlashAddr(myFlashDevice.SectorArr[i]);
        DLPC350_FlashSectorErase();
        DLPC350_WaitForFlashReady();    //Wait for flash busy flag to go off
        DEBUG_ERR("\b\b\b %d", (i*100/lastSectorToErase));
        fflush(stdout);
    }

    DEBUG_ERR("\nErasing Flash Sectors Complete\n");
    dataLen = fw_file_size;
    dataLen -= BLsize;

    DLPC350_SetFlashAddr(BLsize);
    DLPC350_SetDownloadSize(dataLen);

    dataLen_full = dataLen;
    i=0;
    DEBUG_ERR("Downloading Firmware Image %d", i);
    fflush(stdout);

    while(dataLen > 0)
    {
        bytesSent = DLPC350_DownloadData(pByteArray+BLsize+dataLen_full-dataLen, dataLen);

        if(bytesSent < 0)
        {
            DEBUG_ERR("Flash Data Download Failed");
            ret = -1;
            goto cleanup_and_exit;
        }
        for(i=0; i<bytesSent; i++)
        {
            expectedChecksum += pByteArray[BLsize+dataLen_full-dataLen+i];
        }

        dataLen -= bytesSent;
        if(percent_completion != (((dataLen_full-dataLen)*100)/dataLen_full))
        {
            percent_completion = (((dataLen_full-dataLen)*100)/dataLen_full);
            DEBUG_ERR(" \b\b\b%d", (int)percent_completion);
            fflush(stdout);
        }
    }
    DEBUG_ERR("Waiting for checksum verification");
    DLPC350_CalculateFlashChecksum();

    DLPC350_WaitForFlashReady();

    if(DLPC350_GetFlashChecksum(&checksum) < 0)
    {
        DEBUG_ERR("Error reading checksum from target");
    }
    else  if(checksum != expectedChecksum)
    {
        DEBUG_ERR("Checksum mismatch: Expected %x; Received %x", expectedChecksum, checksum);
    }
    else
    {
        DLPC350_ExitProgrammingMode(); //Exit programming mode; Start application.
        DEBUG_ERR("Download Complete");
    }

cleanup_and_exit:
    USB_Close();
    free(pByteArray);
    return ret;

}

static void read_matrix_from_bin_file(FILE *fp, double *a, int n)
{
    int i,j;
    unsigned int bit_mask;
    unsigned int hadamard_word;
    int bit_index=0;

    for(j=0; j < n; j++)
    {
        for(i = 0; i < n; i++)
        {
            if(bit_index++ % 32 == 0)
            {
                fread(&hadamard_word, sizeof(unsigned int), 1, fp);
                bit_mask = 1<<31;
            }
            if(hadamard_word & bit_mask)
                a[i*n+j] = 1;
            else
                a[i*n+j] = 0;

            bit_mask >>= 1;
        }
    }
}

int dlpspectro_compute_hadamard_readings(FILE *readings_fp,FILE *output_fp)
/**
 *  Transforms mean readings input into wavelength domain using inverse hadmard matrix. 
 *
 * @param   readings_fp  - I - ADC readings using a scan based on hadamard matrix patterns. Number of entries in this file must correspond to a valid hadamard matrix available
 * @param   output_fp  - I - After transformation, the values corresponding to wavelengths will be recorded in this file
 *
 * @return   0 = SUCCESS
 *          <0 = FAIL
 *
 */
{
    double *mean_readings;
    double *hadamard_inv;
    double *hadamard_matrix;
    double *result;
    double *result2;
    double temp;
    char mean_val[10];
    int32_t mean_value;
    int i;
    FILE *hadamard_fp;
    char hadamard_filename[2048];
    int num_patterns;
    bool split=true;

    if((readings_fp == NULL) || (output_fp == NULL))
        return -1;

    num_patterns = find_num_lines_in_file(readings_fp);
    mean_readings= malloc(num_patterns*sizeof(double));
    i=0;
    while((fscanf(readings_fp,"%s",mean_val) != EOF) && (i<num_patterns))
    {
        sscanf(mean_val, "%X", &mean_value);
        mean_readings[i++]=(double)mean_value;
    }

    if(split)
        num_patterns /= 2;
    
    sprintf(hadamard_filename, "Hadamard_matrices/s_mat_%d.bin", num_patterns);
    hadamard_fp = fopen(hadamard_filename, "r");
    if(hadamard_fp == NULL)
    {
       DEBUG_ERR("Unable to open %s\n", hadamard_filename);
       return -1;
    }

    hadamard_matrix = malloc(sizeof(double)*num_patterns*num_patterns);
    hadamard_inv = malloc(sizeof(double)*num_patterns*num_patterns);

    if((hadamard_matrix == NULL) || (hadamard_inv == NULL))
    {
        DEBUG_ERR("Memory allocation for hadamard matrices failed\n");
        return -1;
    }

    read_matrix_from_bin_file(hadamard_fp, hadamard_matrix, num_patterns);
    matrix_transpose(hadamard_matrix, hadamard_inv, num_patterns, num_patterns);
    for(i=0; i<num_patterns*num_patterns; i++)
    {
        hadamard_inv[i] = (hadamard_inv[i]-0.5)/((num_patterns+1)/4);
    }

    result = malloc(num_patterns*sizeof(double));
    matrix_mult(mean_readings,hadamard_inv,result,1,num_patterns,num_patterns);

    if(split)
    {
        result2 = malloc(num_patterns*sizeof(double));
        matrix_mult(&mean_readings[num_patterns],hadamard_inv,result2,1,num_patterns,num_patterns);
    }

    for(i=0;i<num_patterns;i++)
    {
        fprintf(output_fp, "0x%.8X\n",(unsigned int)result[i]);
        if(split)
            fprintf(output_fp, "0x%.8X\n",(unsigned int)result2[i]);
    }

    fclose(readings_fp);
    fclose(output_fp);

    free(hadamard_inv);
    free(hadamard_matrix);
    free(mean_readings);
    free(result);
    if(split)
        free(result2);
    fclose(hadamard_fp);
	
    return 0;

}

int checkPrime(int n)
{
    int i,isPrime=1;
    for(i=2;i<=n/2;++i)
    {
        if(n%i==0)
        {
            isPrime=0;
            break;
        }
    }

    return isPrime;
}


static int dlpspectro_getPaleyOrder(int n)
{
    n++;
    if(n%4!=0)
        n += (4-(n%4));

    while(!checkPrime(n-1))
        n += 4;

    return n;
}

static int fprint_singleband_row(int band_first_pixel, int band_last_pixel, int num_pixels_per_row, FILE *pattern_fp)
{
    int j;

    for(j=0;j<num_pixels_per_row;j++)
    {
        if(j <= band_last_pixel)
        {
            if(j >= band_first_pixel)
                fprintf(pattern_fp,"1,");
            else
                fprintf(pattern_fp,"0,");	
        }
        else
            fprintf(pattern_fp,"0,");
    }
    return 0;
}

#define HADAMARD_FILL_EVEN_BANDS 0
#define HADAMARD_FILL_ODD_BANDS 1
#define HADAMARD_FILL_ALL_BANDS 2

static int fprint_hadamard_row(int first_pixel, int last_pixel, int num_bands, int hadamard_row_num, int num_hadamard_elements, int num_pixels_per_row, int odd_even_all, FILE *hadamard_fp, FILE *pattern_fp)
{
    int extra_columns;
    int columns_per_band;
    int extra_column_freq;
    int band_last_pixel;
    unsigned int hadamard_word;
    unsigned int bit_mask;
    int hadamard_element = 0;
    int i;
    int j;
    bool skip_this_band;

    if((odd_even_all < 0) || (odd_even_all > HADAMARD_FILL_ALL_BANDS))
        return -1;

    if((odd_even_all == HADAMARD_FILL_ALL_BANDS) && (num_hadamard_elements < num_bands))
        return -1;

    if((odd_even_all != HADAMARD_FILL_ALL_BANDS) && (num_hadamard_elements > num_bands))
        return -1;

    extra_columns = (last_pixel - first_pixel +1) % num_bands;
    columns_per_band=(last_pixel-first_pixel+1)/num_bands;
    if(extra_columns)
        extra_column_freq = num_bands/extra_columns;
    else
        extra_column_freq = 0;

    rewind(hadamard_fp);
    //Skip "hadamard_row_num" rows
    for(i=0; i < num_hadamard_elements*hadamard_row_num; i++)
    {
        if(hadamard_element++ % 32 == 0)
        {
            fread(&hadamard_word, sizeof(unsigned int), 1, hadamard_fp);
            bit_mask = 1<<31;
        }
        bit_mask >>= 1;
    }

    for(j=0;j<first_pixel;j++)
        fprintf(pattern_fp,"0,");

    for(i = 0; i < num_bands; i++)
    {
        band_last_pixel = j+columns_per_band;
        if(extra_columns)
        {
            if(i % extra_column_freq == 0)
            {
                band_last_pixel++;
                extra_columns--;
            }
        }

        skip_this_band = false;
        if((odd_even_all == HADAMARD_FILL_EVEN_BANDS) && ((i&1) == 1))
            skip_this_band = true;
        if((odd_even_all == HADAMARD_FILL_ODD_BANDS) && ((i&1) == 0))
            skip_this_band = true;

        if(skip_this_band==true)
        {
            for(;j<band_last_pixel;j++)
                fprintf(pattern_fp,"0,");
        }
        else
        {
            if(hadamard_element++ % 32 == 0)
            {
                fread(&hadamard_word, sizeof(unsigned int), 1, hadamard_fp);
                bit_mask = 1<<31;
            }
            if(hadamard_word & bit_mask)
            {
                for(;j<band_last_pixel;j++)
                    fprintf(pattern_fp,"1,");
            }
            else
            {
                for(;j<band_last_pixel;j++)
                    fprintf(pattern_fp,"0,");
            }

            bit_mask >>= 1;
        }
    }

    for(;j<num_pixels_per_row;j++)
        fprintf(pattern_fp,"0,");

    return 0;
}

int dlpspectro_generate_csv_sdf(int first_pixel, int last_pixel, int num_bands, int csv_begin_number, int integration_time, FILE *sdf_fp, bool use_hadamard_patterns, bool split)
/**
 *  Generates csv files and sdf file based on the input parameters. 
 *
 * @param   first_pixel  - I - pixel on the DMD corresponding to the start wavelength of interest 
 * @param   last_pixel   - I - pixel on the DMD corresponding to the last wavelength of interest (last pixel is inclusive)
 * @param   num_bands   - I - number of column bands to create between the first and last pixel
 * @param   csv_begin_number   - I - Generated csv files are named as pattern_N.csv beginning with N=csv_begin_number and incremented for each file
 * @param   integration_time   - I - fixed integration time to be specified for each pattern in the sdf file
 * @param   sdf_fp   - I - Pointer to the sfd file
 * @param   use_hadamard_patterns   - I - TRUE=Patterns generated as per hadamard matrix
 *                                        FALSE=Patterns will have only one column band tuned ON
 * @param   split   - I - TRUE=Generate patterns that have only even bands turned ON followed by patterns that have only odd bands turned ON to avoid boundaary interference effect
 *                        FALSE=Output pattern will have even and odd bands enabled as per hadamard matrix
 *
 * @return   0 = SUCCESS
 *          <0 = FAIL
 *
 */
{
    int cur_band;
    int cur_band_first_pixel;
    int next_band_first_pixel;
    int extra_column_freq;
    int extra_columns;
    int columns_per_band;
    FILE *pattern_fp;
	char filename[1024];
    FILE *hadamard_fp= NULL;
    int num_hadamard_elements;
    int num_csv_patterns;

    if(first_pixel < 0 || first_pixel >= last_pixel || first_pixel > 1823)
    {
        DEBUG_ERR("first_pixel value %d invalid\n", first_pixel);
        return -1;
    }
    if(last_pixel < 0 || last_pixel > 1823)
    {
        DEBUG_ERR("last_pixel value %d invalid\n", last_pixel);
        return -1;
    }
    if(num_bands <=0 || num_bands > 1824)
    {
        DEBUG_ERR("num_bands value %d invalid\n", num_bands);
        return -1;
    }
    if(csv_begin_number <0 || csv_begin_number > 1823)
    {
        DEBUG_ERR("csv file number %d invalid\n", csv_begin_number);
        return -1;
    }
    if(sdf_fp == NULL)
    {
        DEBUG_ERR("SDF file pointer is NULL\n");
        return -1;
    }

    if(use_hadamard_patterns == true)
    {
        if(split)
        {
            num_hadamard_elements = dlpspectro_getPaleyOrder((num_bands+1)/2)-1;
            sprintf(filename, "Hadamard_matrices/s_mat_%d.bin", num_hadamard_elements);
            num_csv_patterns = num_hadamard_elements*2;
        }
        else
        {
            num_hadamard_elements = dlpspectro_getPaleyOrder(num_bands)-1;
            sprintf(filename, "Hadamard_matrices/s_mat_%d.bin", num_hadamard_elements);
            num_csv_patterns = num_hadamard_elements;
        }

        hadamard_fp = fopen(filename, "r");
        if(hadamard_fp == NULL)
        {
            DEBUG_ERR("Unable to open %s\n", filename);
            return -1;
        }
        else
            DEBUG_ERR("opened %s\n", filename);
    }
	else
		num_csv_patterns = num_bands;

    columns_per_band=(last_pixel-first_pixel+1)/num_bands;
    extra_columns = (last_pixel - first_pixel +1) % num_bands; //Total pixels is not exactly divisible by num_bands. Extra columns to be distributed in various bands

    if(extra_columns)
        extra_column_freq = num_bands/extra_columns;
    else
        extra_column_freq = 0;

    cur_band_first_pixel = first_pixel;

    for(cur_band=0;cur_band<num_csv_patterns;cur_band++)
    {
        sprintf(filename, "patterns/pattern_%d.csv", cur_band+csv_begin_number);
        fprintf(sdf_fp,"pattern_%d.csv,%d\n", cur_band+csv_begin_number,integration_time);
        pattern_fp=fopen(filename, "w+");
        if(pattern_fp == NULL)
        {
            DEBUG_ERR("Unable to open %s\n", filename);
            return -1;
        }

        next_band_first_pixel = cur_band_first_pixel + columns_per_band;

        if(extra_columns)
        {
            if((cur_band % extra_column_freq) == 0) //Add an extra column to the patterns at extra_column_freq until we are finished with extra columns
            {
                next_band_first_pixel++;
                extra_columns--;
            }
        }

        if(use_hadamard_patterns)
        {
            if(split)
            {
                if(cur_band < num_hadamard_elements)
                    fprint_hadamard_row(first_pixel, last_pixel, num_bands, num_hadamard_elements, cur_band, 1824, HADAMARD_FILL_EVEN_BANDS, hadamard_fp, pattern_fp);
                else
                    fprint_hadamard_row(first_pixel, last_pixel, num_bands, num_hadamard_elements, cur_band-num_hadamard_elements, 1824, HADAMARD_FILL_ODD_BANDS, hadamard_fp, pattern_fp);
            }
            else
                fprint_hadamard_row(first_pixel, last_pixel, num_bands, num_hadamard_elements, cur_band, 1824, HADAMARD_FILL_ALL_BANDS, hadamard_fp, pattern_fp);
        }
        else
            fprint_singleband_row(cur_band_first_pixel, next_band_first_pixel-1, 1824, pattern_fp);

        fclose(pattern_fp);
        cur_band_first_pixel = next_band_first_pixel;
    }
    if(use_hadamard_patterns == true)
        fclose(hadamard_fp);

    return 0;
}
