/*
 * \file bl_copy.c
 *
 * \brief  Initializes the Spi. Copies the application from SPI flash to DDR.
 *
 */

/* Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/
 * ALL RIGHTS RESERVED
 */

#include "bl.h"
#include "bl_config.h"
#include "bl_copy.h"
//#include "uartStdio.h"
#ifdef NAND

#include "nandlib.h"
#include "bl_nand.h"

#endif

/******************************************************************************
**                     Local function Declaration 
*******************************************************************************/
extern void bl_UARTPuts(char* buffer);
#ifdef SPI
static int SPIBootCopy(void);
extern unsigned int SPIBootRprc(void);
#elif defined(NAND)
static unsigned int NANDBootCopy(void);
#endif

#ifdef MMCSD
static int MMCSDBootCopy(void);
extern void HSMMCSDInit(void);
extern int HSMMCSDImageCopy(void);
#endif

/******************************************************************************
**                     External variable Declaration 
*******************************************************************************/
extern unsigned int entryPoint;

/******************************************************************************
**                       Function Definition 
*******************************************************************************/
/*
 * \brief This function copies Image 
 *
 * \param  none
 *
 * \return none 
*/

unsigned int ImageCopy(void)
{
    int ret = E_FAIL;
#ifdef MMCSD
    ret = MMCSDBootCopy();
#endif
#ifdef SPI
    #ifdef HW_ICE
        #ifdef RPRC_IMAGE
            return SPIBootRprc();
        #else
            if (ret != E_PASS)
                return SPIBootCopy();
        #endif
    #endif    
#elif defined(NAND)
    return NANDBootCopy();
#endif
    return ret;
}

/*
 * \brief This function Initializes and configures SPI and copies 
 *                                           data from SPI FLASH 
 *
 * \param  none
 *
 * \return unsigned int: Status (Success or Failure) 
*/
#ifdef SPI
static int SPIBootCopy(void)
{
#if defined(am1808) || defined(omapl138) || defined(c6748)
    SPIBootHeader spiHeader;
 
    /* Spi Initialization */ 
    SPISetup(); 

    /* Spi read function to read the headder */
    SPIReadFlash(IMAGE_OFFSET, sizeof(spiHeader), (unsigned char*)&spiHeader); 
    
    if ((spiHeader.magicNum != MAGIC_NUM_SF) && 
        (spiHeader.magicNum != MAGIC_NUM_GF))
    {
        UARTPuts("Invalid Image Found \r\n", -1);
        UARTPuts("Aborting Boot\r\n", -1);
        BootAbort();
    }
   
    /* Copies application from SPI flash to DDR RAM */
    SPIReadFlash (spiHeader.memAddress, spiHeader.appSize,
                  (unsigned char *) spiHeader.ldAddress);

    entryPoint = spiHeader.ldAddress;
#elif (defined am335x_15x15) || (defined am335x_13x13)
    ti_header imageHdr;
    
    UARTPuts("Copying application image from SPI to RAM\r\n", -1);

    SPIReadFlash(IMAGE_OFFSET, 8, (unsigned char *)&imageHdr);
    SPIReadFlash (IMAGE_OFFSET + 8, imageHdr.image_size, (unsigned char *)IMAGE_LOAD_ADDR);
    entryPoint = IMAGE_LOAD_ADDR;

#endif
    return E_PASS;
}
#endif

#ifdef MMCSD
static int MMCSDBootCopy(void)
{
#if  (defined am335x_15x15) || (defined am335x_13x13)
    HSMMCSDInit();
        UARTPuts("Copying application image from MMCSD to RAM\r\n", -1);
    if(HSMMCSDImageCopy() == E_FAIL)
        return E_FAIL;
    entryPoint = IMAGE_LOAD_ADDR;
#endif
    return E_PASS;
}
#endif

#ifdef NAND
/*
 * \brief This function Initializes and configures NAND and copies
 *        data from NAND
 *
 * \param  none
 *
 * \return unsigned int: Status (Success or Failure)
*/

static unsigned int NANDBootCopy(void)
{
    NANDInfo nandInfo;

    /* NAND Configuration */
    NANDSetup();
    NANDConfigure(&nandInfo);

    UARTPuts("Copying application image from NAND to RAM\r\n", -1);

    NANDReadFlash (&nandInfo, IMAGE_OFFSET);

    entryPoint = IMAGE_LOAD_ADDR;

    return E_PASS;
}
#endif

/***************************** End Of File ***********************************/

