/*
 * \file nand.c
 *
 * \brief  NAND Initialization functions.  And a funciton to copy data 
*          from NAND to the given address.
 *  
 */

/* Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/
 * ALL RIGHTS RESERVED
 */

#include "gpmc.h"
#include "nandlib.h"
#include "bl.h"
#include "bl_config.h"
#include "bl_nand.h"
//#include "uartStdio.h"
#include "hw_types.h"
#include <string.h>

/******************************************************************************
**                     Macro Defination 
*******************************************************************************/

/******************************************************************************
**                     Function Prototype Declaration.
*******************************************************************************/

extern void NANDInfoInit(NANDInfo *nandInfo, unsigned int cs);
extern void bl_UARTPuts(char* buffer);
/******************************************************************************
**                    Local  Declaration 
*******************************************************************************/
unsigned char eccData[NAND_ECC_DATA_SIZE];


/*
* \brief - NAND Configures.
* \param - none.
*
* \return  none.
*/
unsigned int NANDConfigure(NANDInfo *nandInfo)
{
    unsigned int retVal;
    unsigned int cs;
    NANDDevInfo *nandDevInfo;

    retVal = E_PASS;
    NANDDevInfoDefaultValSet(nandInfo);
    cs = NANDCSGet();
    /* Initialize the nandInfo struct */
    NANDInfoInit(nandInfo, cs);
    /* Initialize the controller */
    NANDCtrlInit(nandInfo);
    /* ECC related initialization */
    retVal = NANDECCInit(nandInfo);
    if(retVal == NAND_STATUS_PASSED)
    {
        NANDResetDevice(nandInfo);
        NANDIdRead(nandInfo);
    }

    nandDevInfo = NANDDevInfoGet(nandInfo, nandInfo->nandCtrlInfo.curCS);
    if(nandDevInfo->devId != NAND_DEVICE_ID)
    {
        retVal = E_FAIL;
    }

    return (retVal);
}

/**
* \brief - Reads bytes from NAND.
* \param - offset - NAND Flash address.\n.
* \param - size - Indicates the total size needs to be read from flash.\n.
* \param - dst - Destination address where data needs to be copy.\n.
*
* \return none
**/

void NANDReadFlash (NANDInfo *nandInfo, unsigned int offset)
{
    unsigned int blkNum;
    unsigned int numOfPages;
    unsigned int page;
    unsigned int i;
    unsigned int retVal;
    NANDDevInfo *nandDevInfo;
    unsigned char *ptrHeaderInfo;
    unsigned char *dstAddr;
    unsigned int size;

    nandDevInfo = NANDDevInfoGet(nandInfo, nandInfo->nandCtrlInfo.curCS);
    ptrHeaderInfo = (unsigned char *) (IMAGE_LOAD_ADDR + 3000);

    /* Conver the offset to block, page address */

    blkNum = (offset / (nandDevInfo->blkSize));

    retVal = NANDBadBlockCheck(nandInfo, blkNum);
    if(retVal == NAND_BLOCK_GOOD)
    {
        page = 0;
        retVal = NANDPageRead(nandInfo, blkNum, page,
                              ptrHeaderInfo,
                              &eccData[0]);
        if(retVal != NAND_STATUS_PASSED)
        {
            UARTPuts("\r\n Reading Image From NAND ...NAND Read Failed.", -1);
            BootAbort();
        }

        page++;
        size = (*((unsigned int *)ptrHeaderInfo));
        ptrHeaderInfo = (ptrHeaderInfo + 4);
        dstAddr = (unsigned char *)(*((unsigned int*)ptrHeaderInfo));
        ptrHeaderInfo = (ptrHeaderInfo + 4);

        memcpy(dstAddr, ptrHeaderInfo, (NAND_PAGE_SIZE - 8) );
        dstAddr += (NAND_PAGE_SIZE - 8);

        numOfPages  = (size / (nandDevInfo->pageSize)) ;
        for(i = 1; i <= numOfPages; i++)
        {
            retVal = NANDPageRead(nandInfo, blkNum, page,
                                  dstAddr,
                                  &eccData[0]);
            if(retVal != NAND_STATUS_PASSED)
            {
                UARTPuts("\r\n Reading Image From NAND ...NAND Read Failed.", -1);
                BootAbort();
            }
            if(page > (nandDevInfo->pagesPerBlk))
            {
                page = 0;
                blkNum++;
            }
            page++;
            dstAddr = (dstAddr + nandDevInfo->pageSize);
        }
    }
    else if(retVal == NAND_BLOCK_SPARE_AREA_READ_FAILED)
    {
        UARTPuts("\r\n NAND Spare Aread Read Failed...", -1);
        BootAbort();
    }
    else
    {
        UARTPuts("\r\n NAND Block Is Bad...", -1);
        BootAbort();
    }
}
/***************************** End Of File ***********************************/
