/**
 * \file  bl_rprc.c
 *
 * \brief Parses application image from SPI flash in RPRC format.
 */

/* Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/
 * ALL RIGHTS RESERVED */

#include "bl.h"
#include "bl_config.h"
#include "bl_copy.h"
//#include "uartStdio.h"

/******************************************************************************
**                     Local function Declaration 
*******************************************************************************/
unsigned int SPIBootRprc(void);
static void SPI_readBytes(void *value, int *cursor, int size);

/******************************************************************************
**                     External variable Declaration 
*******************************************************************************/
extern unsigned int entryPoint;

/******************************************************************************
**                       Function Definition 
*******************************************************************************/
/**
 * \brief  This function parses an application image from SPI flash
 *
 * \param  none
 *
 * \return unsigned int: Status (success or failure)
*/

unsigned int SPIBootRprc(void)
{
    SPIBootHeader image;
    rprcFileHeader header;
    rprcSectionHeader section;
    int offset = IMAGE_OFFSET, size;
    
    /* Spi Initialization */ 
    SPISetup();
    SPIConfigure();

    // check magic number and read image size from SPI header
    SPI_readBytes(&image, &offset, sizeof(SPIBootHeader));
    if ((image.magicNum != MAGIC_NUM_SF) &&
        (image.magicNum != MAGIC_NUM_GF))
    {
        UARTPuts("Invalid magic number in boot image\r\n", -1);
        BootAbort();
    }
    size = image.appSize;

    // read application image header
    SPI_readBytes(&header, &offset, sizeof(rprcFileHeader));

    // check magic number
    if (header.magic != RPRC_MAGIC_NUMBER)
    {
        UARTPuts("Invalid magic number in boot image\r\n", -1);
        BootAbort();
    }

    // skip over text header
    offset += header.text_len;

    // read entrypoint and copy sections to memory
    while (offset - IMAGE_OFFSET < size + sizeof(SPIBootHeader))
    {
        // read new section header
        SPI_readBytes(&section, &offset, sizeof(rprcSectionHeader));

        if (section.type == RPRC_RESOURCE)
        {
            // check that resource has BOOTADDR type (ignore other resources)
            SPI_readBytes(&section.type, &offset, sizeof(int));
            if (section.type == RPRC_BOOTADDR)
                entryPoint = section.addr;

            // skip the rest of the resource section
            offset += section.size - sizeof(int);
        }
        else
        {
            // copy section to memory
            SPI_readBytes((void *)section.addr, &offset, section.size);
        }
    }

    return E_PASS;
}

/**
 * \brief  This function reads N bytes from SPI flash and advances a cursor
 *
 * \param  void *value:
 *
 * \return none
*/

static void SPI_readBytes(void *value, int *cursor, int size)
{
    SPIReadFlash(*cursor, size, value);
    *cursor += size;
}

/***************************** End Of File ***********************************/
