/**
 * \file  bl_spi.c
 *
 * \brief Spi Initialization functions.  And a funciton to copy data from Flash
 *        to the given address.
 */

/*  Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
 *  ALL RIGHTS RESERVED. */

#include "spi.h"
#include "bl.h"
#include "bl_config.h"
#include "bl_spi.h"

 
/******************************************************************************
**                     Macro Defination 
*******************************************************************************/

/* value to configure SMIO,SOMI,CLK and CS pin as functional pin */
#define SIMO_SOMI_CLK_CS        0x00000E01
#define CHAR_LENGTH             0x8

/* flash data read command */
#define SPI_FLASH_READ          0x03

/******************************************************************************
**                    Local  Declaration 
*******************************************************************************/
static void SPIConfigDataFmtReg(unsigned int dataFormat);
static void SpiTransfer(unsigned char *p_tx,
                         unsigned char *p_rx,
                         unsigned int len);
/*
* \brief - SPI Configures.
* \param - none.
*
* \return  none.
*/

void SPIConfigure(void)
{
    unsigned int val;

    SPIReset(SPI_BASE);

    SPIOutOfReset(SPI_BASE);

    SPIModeConfigure(SPI_BASE, SPI_MASTER_MODE);

    SPIClkConfigure(SPI_BASE, SPI_INPUT_FREQ, SPI_SCLK_FREQ,
					      SPI_DATA_FORMAT0);

    val = SIMO_SOMI_CLK_CS;
    SPIPinControl(SPI_BASE, 0, 0, &val);

    SPIDefaultCSSet(SPI_BASE, SPI_CS_INACTIVE_STATE);

    /* Configures SPI Data Format Register */
    SPIConfigDataFmtReg(SPI_DATA_FORMAT0);

    /* Selects the SPI Data format register to used and Sets CSHOLD
     * to assert CS pin(line)
     */
    SPIDat1Config(SPI_BASE, (SPI_CSHOLD | SPI_DATA_FORMAT0), SPI_CS);

    /* Enable SPI communication */
    SPIEnable(SPI_BASE);
} 

/*
* \brief - Reads bytes from SPI Flash.
* \param - dataFormat - The data format that needs to configured
*
* \return none
*/

static void SPIConfigDataFmtReg(unsigned int dataFormat)
{
    /* Configures the polarity and phase of SPI clock */
    SPIConfigClkFormat(SPI_BASE, (SPI_CLK_POL_HIGH | SPI_CLK_INPHASE),
                       dataFormat);

    /* Configures SPI to transmit MSB bit First during data transfer */
    SPIShiftMsbFirst(SPI_BASE, dataFormat);

    /* Sets the Charcter length */
    SPICharLengthSet(SPI_BASE, CHAR_LENGTH, dataFormat);
}

/**
* \brief - Reads bytes from SPI Flash.
* \param - offset - SPI Flash address.\n.
* \param - size - Indicates the total size needs to be read from flash.\n.
* \param - dst - Destination address where data needs to be copy.\n.
*
* \return none
**/
void SPIReadFlash (unsigned int offset,
                   unsigned int size, 
                   unsigned char *dst)
{
    unsigned char tx_data;
    unsigned char rx_data;
    unsigned char addr[3];
    unsigned int len;

    /* The process of reading the data from the flash involves asserting
     * proper chipselect line, asserting CSHOLD, selecting correct data format.
     * Then the flash needs a command to indicate start of read and then
     * any number of bytes can be read. After the required number of bytes
     * are read, the CS needs to be de-asserted to indicate the end of transfer
     */
    SPIDat1Config(SPI_BASE, (SPI_CSHOLD | SPI_DATA_FORMAT0), 0x1);

    /* Send read command to the flash (one byte) */
    tx_data =  SPI_FLASH_READ;
    SpiTransfer(&tx_data, &rx_data, 1);

    /* Send the address to start reading from (3 bytes) */
    addr[0] = (unsigned char)(offset >> 16);
    addr[1] = (unsigned char)(offset >> 8);
    addr[2] = (unsigned char)offset;
    len = 0;

    while (len < sizeof(addr))
    {
        SpiTransfer(&addr[len], &rx_data, 1);
        len++;
    }

    /* Read all the bytes */
    len = 0;

    while(len < size)
    {
        SpiTransfer(&tx_data, dst, 1);
        dst++;
        len++;
    }

    /* De-assert chip select line at the end of transfer */
    SPIDat1Config(SPI_BASE, SPI_DATA_FORMAT0, 0x1);
}

/**
* \brief - Spi Write and Read.
* \param - p_tx - SPI transmit data address.\n.
* \param - p_rx - SPI data reception address.\n.
* \param - len - Indicates the total length the read and write has to do.\n.
*
* \return none
**/

static void  SpiTransfer(unsigned char *p_tx, unsigned char *p_rx,
                         unsigned int len)
{
	int i;

	for (i = 0; i < len; i++)
	{
		while (!SPIIntStatus(SPI_BASE, SPI_TRANSMIT_INT));
		SPITransmitData1(SPI_BASE, p_tx[i]);

		while (!SPIIntStatus(SPI_BASE, SPI_RECV_INT));
		p_rx[i] = SPIDataReceive(SPI_BASE);
	}
}

/***************************** End Of File ***********************************/
