/*
 * \file   bl_uartDebug.c
 *
 * \brief  UART debug code - modified from Sitaraware example
 *         IDC uses UART3
 */

/*
* Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
*
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright 
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the 
*    documentation and/or other materials provided with the   
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/

#include "uart_irda_cir.h"
#include "hw/soc_AM335x.h"
#include "armv7a/am335x/evmAM335x.h"
#include "hw/hw_types.h"
#include "hw/hw_control_AM335x.h"

#include "hw_cm_per.h"
#include "soc_AM335x.h"

#include <string.h>

/******************************************************************************
**              INTERNAL MACRO DEFINITIONS
******************************************************************************/
#define BAUD_RATE_115200          (115200)
#define UART_MODULE_INPUT_CLK     (48000000)

#ifdef HW_ICE
#define BL_UART_BASE              SOC_UART_5_REGS
#else
#define BL_UART_BASE              SOC_UART_3_REGS
#endif

/******************************************************************************
**              INTERNAL FUNCTION PROTOTYPES
******************************************************************************/
static void UartFIFOConfigure(void);
static void UartBaudRateSet(void);
static void PowerUART(void);
static void PinMuxUART(void);
/******************************************************************************
**              GLOBAL VARIABLE DEFINITIONS
******************************************************************************/

/******************************************************************************
**              FUNCTION DEFINITIONS
******************************************************************************/

void bl_UARTInit(void)
{
	/* Configuring the system clocks for UART3 instance. */
    PowerUART();

    /* Performing the Pin Multiplexing for UART3 instance. */
	PinMuxUART();
    /* Performing a module reset. */
    UARTModuleReset(BL_UART_BASE);

    /* Performing FIFO configurations. */
    UartFIFOConfigure();

    /* Performing Baud Rate settings. */
    UartBaudRateSet();

    /* Switching to Configuration Mode B. */
    UARTRegConfigModeEnable(BL_UART_BASE, UART_REG_CONFIG_MODE_B);

    /* Programming the Line Characteristics. */
    UARTLineCharacConfig(BL_UART_BASE,
                         (UART_FRAME_WORD_LENGTH_8 | UART_FRAME_NUM_STB_1), 
                         UART_PARITY_NONE);

    /* Disabling write access to Divisor Latches. */
    UARTDivisorLatchDisable(BL_UART_BASE);

    /* Disabling Break Control. */
    UARTBreakCtl(BL_UART_BASE, UART_BREAK_COND_DISABLE);

    /* Switching to UART16x operating mode. */
    UARTOperatingModeSelect(BL_UART_BASE, UART16x_OPER_MODE);

    /* Performing Interrupt configurations. */
    //UartInterruptEnable();
}

/*
** PowerUART - for uart3
*/
void PowerUART(void)
{
#ifdef HW_ICE
    HWREG( SOC_PRCM_REGS + CM_PER_UART5_CLKCTRL )  |= 0x2; 		// UART5  //0x44E00038
#elif defined(HW_IDK)
	HWREG( SOC_PRCM_REGS + CM_PER_UART3_CLKCTRL )  |= 0x2; 	    // UART3  //0x44E00074 
#endif    
}

/*
** PinMuxUART - pin muxing for uart3
*/
void PinMuxUART(void)
{


#ifdef HW_ICE
	//uart5_txd_mux
	HWREG(SOC_CONTROL_REGS + CONTROL_CONF_LCD_DATA(8)) = 4 | CONTROL_CONF_LCD_DATA8_CONF_LCD_DATA8_PUDEN | 
														CONTROL_CONF_LCD_DATA8_CONF_LCD_DATA8_RXACTIVE;
	//uart5_rxd_mux
	HWREG(SOC_CONTROL_REGS + CONTROL_CONF_LCD_DATA(9)) = 4;
#elif defined(HW_IDK)
	//IDK
    //uart3_rxd_mux1
	HWREG(SOC_CONTROL_REGS + CONTROL_CONF_SPI0_CS1) = 1 | CONTROL_CONF_SPI0_CS0_CONF_SPI0_CS0_PUDEN | 
														CONTROL_CONF_SPI0_CS0_CONF_SPI0_CS0_RXACTIVE;
	//uart3_txd_mux1
	HWREG(SOC_CONTROL_REGS + CONTROL_CONF_ECAP0_IN_PWM0_OUT) = 1;
#endif 


}

/*
** A wrapper function performing FIFO configurations.
*/
static void UartFIFOConfigure(void)
{
    unsigned int fifoConfig = 0;

    /* Setting the TX and RX FIFO Trigger levels as 1. No DMA enabled. */
    fifoConfig = UART_FIFO_CONFIG(UART_TRIG_LVL_GRANULARITY_1,
                                  UART_TRIG_LVL_GRANULARITY_1,
                                  1,
                                  1,
                                  1,
                                  1,
                                  UART_DMA_EN_PATH_SCR,
                                  UART_DMA_MODE_0_ENABLE);

    /* Configuring the FIFO settings. */
    UARTFIFOConfig(BL_UART_BASE, fifoConfig);
}

/*
** A wrapper function performing Baud Rate settings.
*/

static void UartBaudRateSet(void)
{
    unsigned int divisorValue = 0;

    /* Computing the Divisor Value. */
    divisorValue = UARTDivisorValCompute(UART_MODULE_INPUT_CLK,
                                         BAUD_RATE_115200,
                                         UART16x_OPER_MODE,
                                         UART_MIR_OVERSAMPLING_RATE_42);

    /* Programming the Divisor Latches. */
    UARTDivisorLatchWrite(BL_UART_BASE, divisorValue);
}

/*
** A wrapper function performing UART send string.
*/
void bl_UARTPuts(char* buffer)
{
    while (*buffer) 
    {
		UARTCharPut(BL_UART_BASE, *buffer++);
    }
}


/******************************* End of file *********************************/
