/*
 * main.c
 */

#include "uart_irda_cir.h"
#include "soc_AM335x.h"
#include "interrupt.h"
#include "evmAM335x.h"
#include "hw_types.h"
#include "hw_cm_per.h"
#include "hw_control_AM335x.h"

/******************************************************************************
**			  INTERNAL MACRO DEFINITIONS
******************************************************************************/
#define BAUD_RATE_115200		  (115200)
#define UART_MODULE_INPUT_CLK	 (48000000)

/******************************************************************************
**			  INTERNAL FUNCTION PROTOTYPES
******************************************************************************/
static void PowerUART(void);
static void PinMuxUART(void);
static void UartInterruptEnable(void);
static void UART5AINTCConfigure(void);
static void UartFIFOConfigure(void);
static void UartBaudRateSet(void);
static void UARTIsr(void);

/******************************************************************************
**			  GLOBAL VARIABLE DEFINITIONS
******************************************************************************/
unsigned char txArray[] = "StarterWare AM335X UART Interrupt application\r\n";

/******************************************************************************
**			  FUNCTION DEFINITIONS
******************************************************************************/


void main(void)
{
	/* Configuring the system clocks for UART5 instance. */
	PowerUART();

	/* Performing the Pin Multiplexing for UART5 instance. */
	PinMuxUART();

	/* Performing a module reset. */
	UARTModuleReset(SOC_UART_5_REGS);

	/* Performing FIFO configurations. */
	UartFIFOConfigure();

	/* Performing Baud Rate settings. */
	UartBaudRateSet();

	/* Switching to Configuration Mode B. */
	UARTRegConfigModeEnable(SOC_UART_5_REGS, UART_REG_CONFIG_MODE_B);

	/* Programming the Line Characteristics. */
	UARTLineCharacConfig(SOC_UART_5_REGS, 
						 (UART_FRAME_WORD_LENGTH_8 | UART_FRAME_NUM_STB_1), 
						 UART_PARITY_NONE);

	/* Disabling write access to Divisor Latches. */
	UARTDivisorLatchDisable(SOC_UART_5_REGS);

	/* Disabling Break Control. */
	UARTBreakCtl(SOC_UART_5_REGS, UART_BREAK_COND_DISABLE);

	/* Switching to UART16x operating mode. */
	UARTOperatingModeSelect(SOC_UART_5_REGS, UART16x_OPER_MODE);

	/* Performing Interrupt configurations. */
	UartInterruptEnable();

	while(1);
}

/*
** PowerUART - for uart3
*/
void PowerUART(void)
{
#ifdef HW_ICE
	HWREG( SOC_PRCM_REGS + CM_PER_UART5_CLKCTRL )  |= 0x2; 		// UART5  //0x44E00038
#elif defined(HW_IDK)
	HWREG( SOC_PRCM_REGS + CM_PER_UART3_CLKCTRL )  |= 0x2; 		// UART3  //0x44E00074 
#endif
}

/*
** PinMuxUART - pin muxing for uart3
*/
void PinMuxUART(void)
{
#ifdef HW_ICE
	//uart5_txd_mux
	HWREG(SOC_CONTROL_REGS + CONTROL_CONF_LCD_DATA(8)) = 4 | CONTROL_CONF_LCD_DATA8_CONF_LCD_DATA8_PUDEN | 
														CONTROL_CONF_LCD_DATA8_CONF_LCD_DATA8_RXACTIVE;
	//uart5_rxd_mux
	HWREG(SOC_CONTROL_REGS + CONTROL_CONF_LCD_DATA(9)) = 4;
#elif defined(HW_IDK)
	//IDK
	//uart3_rxd_mux1
	HWREG(SOC_CONTROL_REGS + CONTROL_CONF_SPI0_CS1) = 1 | CONTROL_CONF_SPI0_CS0_CONF_SPI0_CS0_PUDEN | 
														CONTROL_CONF_SPI0_CS0_CONF_SPI0_CS0_RXACTIVE;
	//uart3_txd_mux1
	HWREG(SOC_CONTROL_REGS + CONTROL_CONF_ECAP0_IN_PWM0_OUT) = 1;
#endif 
}

/*
** A wrapper function performing FIFO configurations.
*/

static void UartFIFOConfigure(void)
{
	unsigned int fifoConfig = 0;

	/* Setting the TX and RX FIFO Trigger levels as 1. No DMA enabled. */
	fifoConfig = UART_FIFO_CONFIG(UART_TRIG_LVL_GRANULARITY_1,
								  UART_TRIG_LVL_GRANULARITY_1,
								  1,
								  1,
								  1,
								  1,
								  UART_DMA_EN_PATH_SCR,
								  UART_DMA_MODE_0_ENABLE);

	/* Configuring the FIFO settings. */
	UARTFIFOConfig(SOC_UART_5_REGS, fifoConfig);
}

/*
** A wrapper function performing Baud Rate settings.
*/

static void UartBaudRateSet(void)
{
	unsigned int divisorValue = 0;

	/* Computing the Divisor Value. */
	divisorValue = UARTDivisorValCompute(UART_MODULE_INPUT_CLK,
										 BAUD_RATE_115200,
										 UART16x_OPER_MODE,
										 UART_MIR_OVERSAMPLING_RATE_42);

	/* Programming the Divisor Latches. */
	UARTDivisorLatchWrite(SOC_UART_5_REGS, divisorValue);
}

/*
** A wrapper function performing Interrupt configurations.
*/

static void UartInterruptEnable(void)
{
	/* Enabling IRQ in CPSR of ARM processor. */
	IntMasterIRQEnable();

	/* Configuring AINTC to receive UART5 interrupts. */
	UART5AINTCConfigure();

	/* Enabling the specified UART interrupts. */
	UARTIntEnable(SOC_UART_5_REGS, (UART_INT_LINE_STAT | UART_INT_THR |
									UART_INT_RHR_CTI));
}

/*
** Interrupt Service Routine for UART.
*/

static void UARTIsr(void)
{
	static unsigned int txStrLength = sizeof(txArray);
	static unsigned int count = 0;
	unsigned char rxByte = 0;
	unsigned int intId = 0;

	/* Checking ths source of UART interrupt. */
	intId = UARTIntIdentityGet(SOC_UART_5_REGS);

	switch(intId)
	{
		case UART_INTID_TX_THRES_REACH:

			if(0 != txStrLength)
			{
				UARTCharPut(SOC_UART_5_REGS, txArray[count]);
				txStrLength--;
				count++;
			}
			else
			{
				/* Disabling the THR interrupt. */
				UARTIntDisable(SOC_UART_5_REGS, UART_INT_THR);
			}

		break;

		case UART_INTID_RX_THRES_REACH:
			rxByte = UARTCharGetNonBlocking(SOC_UART_5_REGS);
			UARTCharPutNonBlocking(SOC_UART_5_REGS, rxByte);
		break;

		case UART_INTID_RX_LINE_STAT_ERROR:
		case UART_INTID_CHAR_TIMEOUT:
			UARTCharGetNonBlocking(SOC_UART_5_REGS);
		break;
	
		default:
		break;	
	}

}

/*
** This function configures the AINTC to receive UART interrupts.
*/

static void UART5AINTCConfigure(void)
{
	/* Initializing the ARM Interrupt Controller. */
	IntAINTCInit();

	/* Registering the Interrupt Service Routine(ISR). */
	IntRegister(SYS_INT_UART5INT, UARTIsr);

	/* Setting the priority for the system interrupt in AINTC. */
	IntPrioritySet(SYS_INT_UART5INT, 0, AINTC_HOSTINT_ROUTE_IRQ);

	/* Enabling the system interrupt in AINTC. */
	IntSystemEnable(SYS_INT_UART5INT);	
}

/******************************* End of file *********************************/

